#version 430 core

in VertexData {
	vec3 position_world;
	vec3 normal_world;
	vec2 uv;
} vert;

layout (location = 0) out vec4 color;
layout (location = 1) out vec4 brightColor;

uniform vec3 camera_world;

uniform vec3 materialCoefficients; // x = ambient, y = diffuse, z = specular 
uniform float specularAlpha;
uniform float expConst;
uniform sampler2D diffuseTexture;
uniform samplerCube depthMap;

uniform float far_plane;

uniform struct PointLight {
	vec3 color;
	vec3 position;
	vec3 attenuation;
} pointL;

// array of offset direction for sampling
vec3 gridSamplingDisk[20] = vec3[]
(
   vec3(1, 1,  1), vec3( 1, -1,  1), vec3(-1, -1,  1), vec3(-1, 1,  1), 
   vec3(1, 1, -1), vec3( 1, -1, -1), vec3(-1, -1, -1), vec3(-1, 1, -1),
   vec3(1, 1,  0), vec3( 1, -1,  0), vec3(-1, -1,  0), vec3(-1, 1,  0),
   vec3(1, 0,  1), vec3(-1,  0,  1), vec3( 1,  0, -1), vec3(-1, 0, -1),
   vec3(0, 1,  1), vec3( 0, -1,  1), vec3( 0, -1, -1), vec3( 0, 1, -1)
);

float ShadowCalculation(vec3 fragPos)
{
	vec3 lightPos = pointL.position;
    vec3 fragToLight = fragPos - lightPos;
    float currentDepth = length(fragToLight);
    float shadow = 0.0;
	float bias = max(0.05 * (1.0 - dot(vert.normal_world, fragToLight)), 0.005);
    int samples = 20;
    float viewDistance = length(camera_world - fragPos);
    float diskRadius = (1.0 + (viewDistance / far_plane)) / 25.0;
    for(int i = 0; i < samples; ++i)
    {
        float closestDepth = texture(depthMap, fragToLight + gridSamplingDisk[i] * diskRadius).r;
        closestDepth *= far_plane;   // undo mapping [0;1]
        if(currentDepth - bias < closestDepth)
            shadow += 1.0;
    }
    shadow /= float(samples);
        
    // debug (to visualize depth cubemap)
    // color = vec4(vec3(closestDepth / far_plane), 1.0);    
        
    return shadow;
}

vec3 phong(vec3 n, vec3 l, vec3 v, vec3 diffuseC, float diffuseF, vec3 specularC, float specularF, float alpha, bool attenuate, vec3 attenuation) {
	float d = length(l);
	l = normalize(l);
	float att = 1.0;	
	if(attenuate) att = 1.0f / (attenuation.x + d * attenuation.y + d * d * attenuation.z);
	vec3 r = reflect(-l, n);
	return (diffuseF * diffuseC * max(0, dot(n, l)) + specularF * specularC * pow(max(0, dot(r, v)), alpha)) * att; 
}

float hash(vec2 p) {
    int n = int(p.x*3 + p.y*113);

    // 1D hash inspired by Hugo Elias
	n = (n << 7) ^ n;
    n = n * (n * n * 15731 + 789221) + 1376312589;
    return -1.0+2.0*float( n & 0x0fffffff)/float(0x0fffffff);
}

float noise(vec2 p) {
    vec2 i = vec2(floor( p ));
    vec2 f = fract( p );
	
    // cubic interpolant
    vec2 u = f*f*(3.0-2.0*f);

    return mix( mix( hash( i + vec2(0,0) ), 
                     hash( i + vec2(1,0) ), u.x),
                mix( hash( i + vec2(0,1) ), 
                     hash( i + vec2(1,1) ), u.x), u.y);
}


vec3 lerp(float mix, vec3 a, vec3 b)
{
    return mix * a + (1.0 - mix) * b;
}

vec3 sampleProcTexture(vec2 uvIn) {
    vec2 uv = uvIn;
	uv *= 8.0;
    mat2 m = mat2( 1.6,  1.2, -1.2,  1.6 );
	float f = 0.0;
	f  = 0.5000*noise( uv ); uv = m*uv;
	f += 0.2500*noise( uv ); uv = m*uv;
	f += 0.1250*noise( uv ); uv = m*uv;
	f += 0.0625*noise( uv ); uv = m*uv;
    
	f = 0.5 + 0.5*f;
	
    f *= smoothstep( 0.0, 0.005, abs(uv.x-0.6) );

    vec3 color1 = vec3(1.0);
    vec3 color2 = vec3(0.2);
    vec3 color3 = vec3(0.5);
    vec3 color4 = vec3(0.1);

    vec3 col = lerp(0.5 * sin(uv.x + f * 20) + 0.5, color1, color2);

    if (mod(floor(uvIn.x*2) + floor(uvIn.y*2), 2.0) == 0.0) {
		col = lerp(0.5 * sin((0.25*uv.x+0.75*uv.y) + f * 40) + 0.5, color3, color4);
	}

	return col;
}


void main() {	
	vec3 n = normalize(vert.normal_world);
	vec3 v = normalize(camera_world - vert.position_world);

	vec3 texColor = sampleProcTexture(vert.uv*8).rgb;
	color = vec4(texColor * materialCoefficients.x * expConst, 1); // ambient
	
	// add point light contribution
	color.rgb += phong(n, pointL.position - vert.position_world, v, pointL.color * texColor, materialCoefficients.y, pointL.color, materialCoefficients.z, specularAlpha, true, pointL.attenuation);

	// add shadow
	float shadow = ShadowCalculation(vert.position_world); 
	color.rgb *= shadow;

    float brightness = dot(color.rgb, vec3(0.2126, 0.7152, 0.0722));
    if(brightness > 1.0)
        brightColor = vec4(color.rgb, 1.0);
    else
        brightColor = vec4(0.0, 0.0, 0.0, 1.0);

	//color.rgb = (vert.normal_world + vec3(1)) / 2.0; //showing normals for debugging
}

