#version 430 core
/*
* Copyright 2019 Vienna University of Technology.
* Institute of Computer Graphics and Algorithms.
* This file is part of the ECG Lab Framework and must not be redistributed.
*/

#define MAX_LIGHT_COUNT 64

in VertexData {
    vec3 position_world;
    vec3 normal_world;
    vec4 posLightSpace;
    vec2 uv;
} vert;

out vec4 color;

const float levels = 12.0;

uniform vec3 camera_world;

uniform vec3 materialCoefficients;// x = ambient, y = diffuse, z = specular
uniform float specularAlpha;

uniform sampler2D diffuseTexture;
uniform sampler2D surfaceNormalTexture;
uniform sampler2D depthTexture;
uniform sampler2D shadowDepthTexture;

uniform struct DirectionalLight {
    vec3 color;
    vec3 direction;
} dirL;

uniform uint directionalLightCount;
uniform DirectionalLight directionalLights[MAX_LIGHT_COUNT];

uniform struct PointLight {
    vec3 color;
    vec3 position;
    vec3 attenuation;
} pointL;

uniform uint pointLightCount;
uniform PointLight pointLights[MAX_LIGHT_COUNT];

uniform float globalBrightness;

vec3 phongDiffuse(vec3 n, vec3 l, vec3 v, vec3 diffuseC, float diffuseF, vec3 specularC, float specularF, float alpha, bool attenuate, vec3 attenuation) {
    float d = length(l);
    l = normalize(l);
    float att = 1.0;
    if (attenuate) att = 1.0f / (attenuation.x + d * attenuation.y + d * d * attenuation.z);
    vec3 r = reflect(-l, n);
    return diffuseF * diffuseC * max(0, dot(n, l)) * att;
}

vec3 phongSpecular(vec3 n, vec3 l, vec3 v, vec3 diffuseC, float diffuseF, vec3 specularC, float specularF, float alpha, bool attenuate, vec3 attenuation) {
    float d = length(l);
    l = normalize(l);
    float att = 1.0;
    if (attenuate) att = 1.0f / (attenuation.x + d * attenuation.y + d * d * attenuation.z);
    vec3 r = reflect(-l, n);
    return specularF * specularC * pow(max(0, dot(r, v)), alpha) * att;
}

float calcShadow(vec4 posLightSpace, vec3 n, vec3 lightDir) {
    vec3 projCoords = posLightSpace.xyz / posLightSpace.w;
    projCoords = projCoords * 0.5 + 0.5;

    float closestDepth = texture(shadowDepthTexture, projCoords.xy).x;

    float currentDepth = projCoords.z;

    lightDir = normalize(lightDir);
    float bias = max(0.001 * (1.0 - dot(n, lightDir)), 0.0001);
    // float bias = 0;

    float shadow = 0.0;

    vec2 texelSize = 1.0 / textureSize(shadowDepthTexture, 0);
    for (int i=-2; i<=2; i++) {
        for (int j=-2; j<=2; j++) {
            float pcfDepth = texture(shadowDepthTexture, projCoords.xy + vec2(i, j) * texelSize).x;
            shadow += currentDepth - bias > pcfDepth ? 1.0 : 0.0;
        }
    }
    shadow /= 25.0;
    return shadow;
}

void main() {
    vec3 texColor = texture(diffuseTexture, vert.uv).rgb;

    vec3 n = normalize(vert.normal_world);
    vec3 v = normalize(camera_world - vert.position_world);

    vec3 diffuseColor = materialCoefficients.x * texColor;
    vec3 specularColor = vec3(0.0, 0.0, 0.0);

    vec3 diffuseShaodowSubstract = vec3(0); // Do not include shadow to cel shade devide.
    vec3 specularShaodowSubstract = vec3(0); // Do not include shadow to cel shade devide.

    for (uint i = 0; i < directionalLightCount; i++){
        // add directional light contribution

        vec3 d = phongDiffuse(n, -directionalLights[i].direction, v, directionalLights[i].color * texColor, materialCoefficients.y, directionalLights[i].color, materialCoefficients.z, specularAlpha, false, vec3(0));
        vec3 s = phongSpecular(n, -directionalLights[i].direction, v, directionalLights[i].color * texColor, materialCoefficients.y, directionalLights[i].color, materialCoefficients.z, specularAlpha, false, vec3(0));

        float shadowFactor;
        if (i == 0) {
            float shadowFactor = calcShadow(vert.posLightSpace, n, directionalLights[i].direction);
            diffuseShaodowSubstract = shadowFactor * d;
            specularShaodowSubstract = shadowFactor * s;
        }

        diffuseColor += d;
        specularColor += s;
    }

    for (uint i = 0; i < pointLightCount; i++){
        // add point light contribution
        diffuseColor += phongDiffuse(n, pointLights[i].position - vert.position_world, v, pointLights[i].color * texColor, materialCoefficients.y, pointLights[i].color, materialCoefficients.z, specularAlpha, true, pointLights[i].attenuation);
        specularColor += phongSpecular(n, pointLights[i].position - vert.position_world, v, pointLights[i].color * texColor, materialCoefficients.y, pointLights[i].color, materialCoefficients.z, specularAlpha, true, pointLights[i].attenuation);
    }

    diffuseColor = round(diffuseColor * levels) / levels - diffuseShaodowSubstract;
    specularColor = specularColor - specularShaodowSubstract;

    color = vec4(globalBrightness * (diffuseColor + specularColor), 1.0);
}

