#version 440

struct Material
{
	vec3 ambient;
	vec3 diffuse;
	vec3 specular;
	sampler2D diffuseTex;
	sampler2D specularTex;
	float transparency;
};

struct PointLight
{
	vec3 position;
	float intensity;
	vec3 color;
	float constant;
	float linear;
	float quadratic;
};

in vec3 vs_position;
in vec3 vs_color;
in vec2 vs_texcoord;
in vec3 vs_normal;

layout (location = 0) out vec4 finalColor;

#define NR_POINT_LIGHTS 9
// Uniforms
uniform Material material;
uniform PointLight pointLights[NR_POINT_LIGHTS];
uniform vec3 cameraPos;
uniform vec3 skyColor;
uniform float brightness;
uniform int celShading;
uniform int specular;

const float levels = 3.0;

// Functions
vec3 calculateAmbient(Material material, PointLight pointLight)
{
	return material.ambient * pointLight.color * pointLight.intensity;
}

vec3 calculateDiffuse(Material material, vec3 vs_position, vec3 vs_normal, PointLight pointLight)
{
	vec3 diffuse;
	vec3 posToLightDirVec = normalize(pointLight.position - vs_position);
	float diffuseConstant = clamp(dot(posToLightDirVec, normalize(vs_normal)), 0, 1);
	diffuse = material.diffuse * diffuseConstant * pointLight.color * pointLight.intensity;
	return diffuse;
}

vec3 calculateSpecular(Material material, vec3 vs_position, vec3 vs_normal, PointLight pointLight, vec3 cameraPos)
{
	vec3 specular;
	vec3 lightToPosDirVec = normalize(vs_position - pointLight.position);
	vec3 reflectDirVec = normalize(reflect(lightToPosDirVec, normalize(vs_normal)));
	vec3 posToViewDirVec = normalize(cameraPos - vs_position);
	float specularConstant = pow(max(dot(posToViewDirVec, reflectDirVec), 0), 35);
	specular = material.specular * specularConstant * texture(material.specularTex, vs_texcoord).rgb * pointLight.color * pointLight.intensity;
	return specular;
}

vec3 rgbToHsv(vec3 rgb)
{
	float hue;
	float saturation;
	float value;

	float cmax = max(rgb.r, max(rgb.g, rgb.b));
	float cmin = min(rgb.r, min(rgb.g, rgb.b));
	float diff = cmax - cmin;

	if (diff == 0) {
		hue = 0;
	} else if (cmax == rgb.r) {
		hue = int(60 * ((rgb.g - rgb.b) / diff)) % 360;
	} else if (cmax == rgb.g) {
		hue = int(60 * (2 + ((rgb.b - rgb.r) / diff))) % 360;
	} else if (cmax == rgb.b) {
		hue = int(60 * (4 + ((rgb.r - rgb.g) / diff))) % 360;
	}

	if (hue < 0) {
		hue = hue + 360;
	}

	if (cmax == 0 || (rgb.r == rgb.g && rgb.g == rgb.b && rgb.r == 0)) {
		saturation = 0.0f;
	} else {
		saturation = diff / cmax;
	}

	value = cmax;

	return vec3(hue, saturation, value);
}

vec3 hsvToRgb(vec3 hsv)
{
	vec3 rgb;

	float h_i = floor(hsv.x / 60);
	float f = (hsv.x / 60 - h_i);

	float p = hsv.z * (1 - hsv.y);
	float q = hsv.z * (1 - (hsv.y * f));
	float t = hsv.z * (1 - (hsv.y * (1 - f)));

	if (h_i == 0 || h_i == 6)
		rgb = vec3(hsv.z, t, p);
	else if (h_i == 1)
		rgb = vec3(q, hsv.z, p);
	else if (h_i == 2)
		rgb = vec3(p, hsv.z, t);
	else if (h_i == 3)
		rgb = vec3(p, q, hsv.z);
	else if (h_i == 4)
		rgb = vec3(t, p, hsv.z);
	else if (h_i == 5)
		rgb = vec3(hsv.z, p, q);

	return rgb;
}

vec4 calculatePointLight(PointLight pointLight)
{
	vec4 result;

	// Ambient Light
	vec3 ambientFinal = calculateAmbient(material, pointLight);

	// Diffuse Light
	vec3 diffuseFinal = calculateDiffuse(material, vs_position, vs_normal, pointLight);

	// Specular Light
	vec3 specularFinal = calculateSpecular(material, vs_position, vs_normal, pointLight, cameraPos);

	// Attenuation - Fading with distance
	float distance = length(pointLight.position - vs_position);

	float attenuation;
	if (celShading == 1)
		attenuation = 1.f / ((pointLight.constant) + (pointLight.linear * distance));
	else 
		attenuation = 
			1.f / (
				(pointLight.constant) + 
				(pointLight.linear * distance) +
				(pointLight.quadratic * (distance * distance))
			);

	// Final Light
	ambientFinal *= attenuation;
	diffuseFinal *= attenuation;
	specularFinal *= attenuation;

	// Final Color
	result = 
	brightness +
	texture(material.diffuseTex, vs_texcoord) * 
	vec4(vs_color, material.transparency) *
	(
		vec4(ambientFinal, 1.f)
		+ vec4(diffuseFinal, 1.f)
		+ (specular == 1 ? vec4(specularFinal, 1.f) : vec4(0))
	);

	return result;
}

void main()
{
	for (int i = 0; i < NR_POINT_LIGHTS; i++)
	{
		finalColor += calculatePointLight(pointLights[i]);
	}

	// CelShading
	if (celShading == 1)
	{
		// Convert to HSV and modify brightness
		vec3 hsv = rgbToHsv(finalColor.rgb);

		float level = floor(hsv.z * levels);
		hsv.z = level / levels;

		// Increase brightness if 0
		if (hsv.z == 0)
			hsv.z = 0.1;

		// Convert back to RGB
		finalColor = vec4(hsvToRgb(hsv), finalColor.a);
	}

	// Discard transparent pixels
	if (finalColor.a < 0.1)
	{
		discard;
	}
}