#version 330 core
in vec2 TexCoords;
uniform sampler2D positionTexture;    // View-space depth texture
uniform sampler2D normalTexture;   // View-space normal texture
uniform sampler2D sceneTexture;   // View-space normal texture
uniform mat4 projection;
out vec4 SSR;

#define MAX_RAY_STEPS 256
#define REFINEMENT_STEPS 16
#define STEP_SIZE 0.05
#define THICKNESS 0.15
#define MIN_BORDER_DISTANCE 0.01
#define MAX_BORDER_DISTANCE 0.1

void main() {
    vec3 fragViewPos = texture(positionTexture, TexCoords).xyz; // View-space position
    vec3 normal = normalize(texture(normalTexture, TexCoords).xyz);
    vec3 viewDir = -normalize(fragViewPos);
    vec3 reflectedDir = normalize(reflect(-viewDir, normal));

    vec4 result = vec4(0, 0, 0, 0);

    for(int i = 1; i < MAX_RAY_STEPS; i++) {
        vec3 rayPos = i * reflectedDir * STEP_SIZE + fragViewPos;

        vec4 projectedRay = projection * vec4(rayPos, 1.0);
        vec2 uv = (projectedRay.xyz / projectedRay.w).xy * 0.5 + 0.5;

        if (uv.x < 0.0 || uv.y < 0.0 || uv.x > 1.0 || uv.y > 1.0) {
            break;
        }

        float sceneDepth = length(texture(positionTexture, uv).xyz);
        float rayDepth = length(rayPos);

        if (rayDepth < sceneDepth + THICKNESS && rayDepth > sceneDepth) {
            float lower = (i - 1) * STEP_SIZE; // not reflecting
            float upper = (i) * STEP_SIZE; // reflecting

            for(int j = 0; j < REFINEMENT_STEPS; j++) {
                float mid = lower + (upper - lower) / 2;
                rayPos = mid * reflectedDir + fragViewPos;

                projectedRay = projection * vec4(rayPos, 1.0);
                vec2 candidateUv = (projectedRay.xyz / projectedRay.w).xy * 0.5 + 0.5;

                sceneDepth = length(texture(positionTexture, candidateUv).xyz);
                rayDepth = length(rayPos);

                if (rayDepth > sceneDepth) {
                    upper = mid;
                } else {
                    uv = candidateUv;
                    lower = mid;
                }
            }
            

            float distanceToBorder = min(min(min(uv.x, 1.0 - uv.x), uv.y), 1.0 - uv.y);
            float borderFade = clamp((distanceToBorder - MIN_BORDER_DISTANCE) / (MAX_BORDER_DISTANCE - MIN_BORDER_DISTANCE), 0, 1);

            result.xyz = borderFade * texture(sceneTexture, uv).xyz;
            result.w = 1;

            break;
        }
    }


    SSR = result;
}
