import bpy
import os
import json
import mathutils

# ============================================================
# CONFIG
# ============================================================

BASE_NAME = "ParticleCube"

EXPORT_ROOT_NAME = "export"
JSON_FILENAME = "particleScene.json"

# ============================================================
# SAFE ROOT (Windows/Blender)
# ============================================================

if not bpy.data.filepath:
    raise RuntimeError("❌ Save the .blend file before running the script")

ROOT_DIR = os.path.dirname(bpy.data.filepath)
EXPORT_ROOT = os.path.join(ROOT_DIR, EXPORT_ROOT_NAME)
os.makedirs(EXPORT_ROOT, exist_ok=True)

# ============================================================
# HELPER: get base color as vec3
# ============================================================

def get_base_color(material):
    if material and material.use_nodes:
        nodes = material.node_tree.nodes
        principled = next((n for n in nodes if n.type == 'BSDF_PRINCIPLED'), None)
        if principled:
            color = principled.inputs['Base Color'].default_value  # vec4
            return [color[0], color[1], color[2]]  # rgb only
    return [1.0, 1.0, 1.0]  # default white

# ============================================================
# MAIN EXPORT
# ============================================================

scene_data = []

for obj in bpy.context.scene.objects:
    if obj.name != BASE_NAME and not obj.name.startswith(BASE_NAME + "."):
        continue

    print(f"Processing {obj.name}")

    # --- compute world matrix ---
    world_mat = obj.matrix_world.copy()
    # convert to list of 16 floats (column-major for glm)
    mat4_list = []
    for col in world_mat.col:
        mat4_list.extend([col[0], col[1], col[2], col[3]])

    # --- base color ---
    mat = obj.active_material
    color = get_base_color(mat)

    # --- append to JSON ---
    scene_data.append({
        "name": obj.name,
        "transform": mat4_list,   # mat4 for glm
        "baseColor": color        # vec3 for glm
    })

# ============================================================
# WRITE JSON
# ============================================================

json_path = os.path.join(EXPORT_ROOT, JSON_FILENAME)

with open(json_path, "w", encoding="utf-8") as f:
    json.dump(scene_data, f, indent=4)

print("\n✅ PARTICLE SCENE JSON EXPORT COMPLETE")
print("JSON path:", json_path)
