import bpy
import os
import json
import shutil

# ============================================================
# CONFIG
# ============================================================

BASE_NAME = "NormalMappingObject"

EXPORT_ROOT_NAME = "export"
TEXTURES_SUBDIR = os.path.join("textures", "vgpNormal")
MODELS_SUBDIR = os.path.join("models", "vgpNormal")

# ============================================================
# SAFE ROOT (WINDOWS / BLENDER)
# ============================================================

if not bpy.data.filepath:
    raise RuntimeError("❌ Save the .blend file before running the script")

ROOT_DIR = os.path.dirname(bpy.data.filepath)

EXPORT_ROOT = os.path.join(ROOT_DIR, EXPORT_ROOT_NAME)
TEXTURES_DIR = os.path.join(EXPORT_ROOT, TEXTURES_SUBDIR)
MODELS_DIR = os.path.join(EXPORT_ROOT, MODELS_SUBDIR)

os.makedirs(TEXTURES_DIR, exist_ok=True)
os.makedirs(MODELS_DIR, exist_ok=True)

print("Export root:", EXPORT_ROOT)

# ============================================================
# MATERIAL PARSING
# ============================================================

def get_albedo_and_normal(material):
    albedo = []
    normal = []

    if not material or not material.use_nodes:
        return albedo, normal

    nodes = material.node_tree.nodes
    links = material.node_tree.links

    principled = next(
        (n for n in nodes if n.type == 'BSDF_PRINCIPLED'),
        None
    )
    if not principled:
        return albedo, normal

    # --- Albedo ---
    for link in links:
        if (
            link.to_node == principled
            and link.to_socket.name == "Base Color"
            and link.from_node.type == 'TEX_IMAGE'
            and link.from_node.image
        ):
            albedo.append(link.from_node.image)

    # --- Normal ---
    for link in links:
        if (
            link.to_node == principled
            and link.to_socket.name == "Normal"
            and link.from_node.type == 'NORMAL_MAP'
        ):
            normal_node = link.from_node
            for l in links:
                if (
                    l.to_node == normal_node
                    and l.from_node.type == 'TEX_IMAGE'
                    and l.from_node.image
                ):
                    normal.append(l.from_node.image)

    return albedo, normal


# ============================================================
# TEXTURE COPY
# ============================================================

def copy_texture(image):
    src = bpy.path.abspath(image.filepath)

    if not os.path.exists(src):
        raise RuntimeError(f"Texture file not found: {src}")

    name = os.path.basename(src)
    dst = os.path.join(TEXTURES_DIR, name)

    if not os.path.exists(dst):
        shutil.copy(src, dst)

    return os.path.join(TEXTURES_SUBDIR, name).replace("\\", "/")


# ============================================================
# MAIN EXPORT
# ============================================================

scene_data = []

for obj in bpy.context.scene.objects:
    if obj.name != BASE_NAME and not obj.name.startswith(BASE_NAME + "."):
        continue

    print(f"\nProcessing: {obj.name}")

    if not obj.material_slots:
        raise RuntimeError(f"{obj.name} has no materials")

    mat = obj.material_slots[0].material
    albedo_imgs, normal_imgs = get_albedo_and_normal(mat)

    if len(albedo_imgs) != 1 or len(normal_imgs) != 1:
        raise RuntimeError(
            f"{obj.name}: expected exactly 1 albedo and 1 normal, "
            f"got {len(albedo_imgs)} / {len(normal_imgs)}"
        )

    albedo_path = copy_texture(albedo_imgs[0])
    normal_path = copy_texture(normal_imgs[0])

    # --- mesh stats ---
    mesh = obj.data
    vertices_count = len(mesh.vertices)
    indices_count = sum(len(p.vertices) - 2 for p in mesh.polygons) * 3

    # --- export OBJ ---
    bpy.ops.object.select_all(action='DESELECT')
    obj.select_set(True)
    bpy.context.view_layer.objects.active = obj

    obj_filename = f"{obj.name}.obj"
    obj_export_path = os.path.join(MODELS_DIR, obj_filename)

    bpy.ops.wm.obj_export(
        filepath=obj_export_path,
        export_selected_objects=True,
        export_materials=False,
        export_triangulated_mesh=True,
        forward_axis='NEGATIVE_Z',
        up_axis='Y'
    )

    scene_data.append({
        "name": obj.name,
        "obj_file": f"{MODELS_SUBDIR}/{obj_filename}".replace("\\", "/"),
        "vertices_count": vertices_count,
        "indices_count": indices_count,
        "albedoTexture": albedo_path,
        "normalMapTexture": normal_path,
        "shader_type": "normalMapping"
    })

# ============================================================
# WRITE JSON
# ============================================================

json_path = os.path.join(EXPORT_ROOT, "normalScene.json")

with open(json_path, "w", encoding="utf-8") as f:
    json.dump(scene_data, f, indent=4)

print("\n✅ EXPORT FINISHED")
print("JSON:", json_path)
