import bpy
import json
import mathutils
import math
import os

# Output JSON path (next to the .blend file)
output_path = bpy.path.abspath("//export//camera.json")

camera = bpy.context.scene.camera
if camera is None:
    raise Exception("No active camera in the scene!")

# Collect all keyframes from animation data
keyframes = set()
anim = camera.animation_data
if anim and anim.action:
    for fcurve in anim.action.fcurves:
        for key in fcurve.keyframe_points:
            keyframes.add(int(key.co.x))

keyframes = sorted(list(keyframes))

data = {
    "camera_params": {
        "fovy": camera.data.angle,  # Blender's fov in radians
        "aspect": camera.data.sensor_width / camera.data.sensor_height,
        "near": camera.data.clip_start,
        "far": camera.data.clip_end
    },
    "keyframes": []
}

# Coordinate system adjustment: Blender -> GLM/OpenGL
# Blender: Z-up, Y-forward
# GLM: Y-up, -Z-forward
rot_fix = mathutils.Matrix.Rotation(math.radians(-90.0), 4, 'X')

for frame in keyframes:
    bpy.context.scene.frame_set(frame)
    
    mat_world = camera.matrix_world.copy()
    mat_converted = rot_fix @ mat_world  # Apply rotation fix
    
    loc = mat_converted.to_translation()
    rot = mat_converted.to_euler('XYZ')

    frame_data = {
        "frame": frame,
        "location": [loc.x, loc.y, loc.z],
        "rotation_euler": [rot.x, rot.y, rot.z]
    }
    data["keyframes"].append(frame_data)

# Save JSON
with open(output_path, 'w') as f:
    json.dump(data, f, indent=4)

print(f"Exported camera animation to {output_path}")
