#include <tamashii/core/io/io.hpp>
#include <tamashii/core/scene/light.hpp>
#include <tamashii/core/scene/image.hpp>
#include <tiny_ies.hpp>
#include <stb_image.h>
#include <stb_image_write.h>
T_USE_NAMESPACE

template <typename T>
float lerp(T a, T b, T t) {
    return a + t * (b - a);
}

std::pair<float, float> anglesToImageCoords(float aVertAngle, float aHorizAngle, int aWidth, int aHeight) {
    // Convert angles to latitude and longitude
    float latitude = 180.0f - aVertAngle;
    float longitude = aHorizAngle;

    // Convert latitude and longitude to image coordinates
    float x = (longitude / 360.0f) * static_cast<float>(aWidth);
    float y = (latitude / 180.0f) * static_cast<float>(aHeight);

    return {x, y};
}

// Calculate luminance from RGB values using the Rec. 709 standard
float calculateLuminance(const std::vector<float>& aRGB) {
    return 0.2126f * aRGB[0] + 0.7152f * aRGB[1] + 0.0722f * aRGB[2];
}


void generateAnglesSampling (float aHorizStartAngle, float aHorizEndAngle, float aVertStartAngle, float aVertEndAngle, int aHorizSize, int aVertSize, std::vector<float>& aHorizAngles, std::vector<float>& aVertAngles) {
    float horizMiddle = (aHorizEndAngle + aHorizStartAngle) / 2.0f;
    float vertMiddle = (aVertEndAngle + aVertStartAngle) / 2.0f;

    // Create 256 horizontal angles
    for (int i = 0; i < aHorizSize/4; ++i) {
        float t = static_cast<float>(i) / (static_cast<float>(aHorizSize) / 4 - 1);

        aHorizAngles[0 * aHorizSize/4 + i] = lerp(horizMiddle, aHorizEndAngle, t);
        aHorizAngles[1 * aHorizSize/4 + i] = lerp(aHorizEndAngle, horizMiddle, t);
        aHorizAngles[2 * aHorizSize/4 + i] = lerp(horizMiddle, aHorizStartAngle, t);
        aHorizAngles[3 * aHorizSize/4 + i] = lerp(aHorizStartAngle, horizMiddle, t);
    }

    // Create 1024 vertical angles in total
    for (int i = 0; i < aVertSize; ++i) {
        float t = static_cast<float>(i) / (static_cast<float>(aVertSize) - 1);

        aVertAngles[0 * aVertSize + i] = lerp(vertMiddle, aVertEndAngle, t);
        aVertAngles[1 * aVertSize + i] = lerp(vertMiddle, aVertStartAngle, t);
        aVertAngles[2 * aVertSize + i] = lerp(vertMiddle, aVertStartAngle, t);
        aVertAngles[3 * aVertSize + i] = lerp(vertMiddle, aVertEndAngle, t);
    }
}

// Convert a hdr file to a window light with an ies profile
std::unique_ptr<WindowLight> io::Import::loadHdriToIes(const std::filesystem::path &aFile) {

    // Load HDR file
    int width, height, desiredChannels = 3, nrComponents;
    float *data = stbi_loadf(aFile.string().c_str(), &width, &height, &nrComponents, desiredChannels);
    if (!data) {
        spdlog::error("Failed to load HDR image.");
        return nullptr;
    }

    // Define the size of the candela texture
    constexpr int vertSize = 256, horizSize = 256;
    // Angles for sampling
    std::vector<float> verticalAnglesSampling(4 * vertSize), horizontalAnglesSampling(horizSize);
    // Angles for emission
    std::vector<float> verticalAnglesEmitting, horizontalAnglesEmitting;
    // Sampling angle range
    float sampleVertAngleStart = 90, sampleVertAngleEnd = 180, sampleHorizAngleStart = 0, sampleHorizAngleEnd = 360;
    // Emission angle range
    float emitVertAngleStart = 0, emitVertAngleEnd = 35, emitHorizAngleStart = 0, emitHorizAngleEnd = 360;
    // Initialize candela values
    std::vector candelaValues(vertSize * horizSize, 0.0f);
    // Generate the sampling angles
    generateAnglesSampling(sampleHorizAngleStart, sampleHorizAngleEnd, sampleVertAngleStart ,sampleVertAngleEnd, horizSize, vertSize, horizontalAnglesSampling, verticalAnglesSampling);

    // Extra index to pick correct vertical Angles for sampling
    int vertAngleIndex = 0;
    // Sample the HDRI and calculate luminance
    for (int h = 0; h < horizSize; ++h) {
        float horizAngle = horizontalAnglesSampling[h];
        if(h == horizSize / 4) {
            vertAngleIndex = 1;
        } else if (h == horizSize / 2) {
            vertAngleIndex = 2;
        } else if (h == 3 * horizSize / 4) {
            vertAngleIndex = 3;
        }
        for (int v = 0; v < vertSize; ++v) {
            float vertAngle = verticalAnglesSampling[vertAngleIndex * (vertSize - 1) + v];

            // Get pixel values from sampling angles
            auto [x, y] = anglesToImageCoords(vertAngle, horizAngle, width, height);

            // Calculate the non-fractional and fractional part of the coordinates
            float yFloor = std::floor(y);
            float xFloor = std::floor(x);
            float fx = x - xFloor;
            float fy = y - yFloor;

            // Determine the integer coordinates of the surrounding pixels
            int x0 = std::clamp(static_cast<int>(xFloor),0, width - 1);
            int x1 = std::clamp(x0 + 1, 0, width - 1);
            int y0 = std::clamp(static_cast<int>(yFloor), 0, height - 1);
            int y1 = std::clamp(y0 + 1, 0, height - 1);

            // Interpolate the color values
            std::vector<float> rgb(desiredChannels, 0.0f);
            for (int c = 0; c < desiredChannels; ++c) {

                float p00 = data[(y0 * width + x0) * desiredChannels + c];
                float p01 = data[(y1 * width + x0) * desiredChannels + c];
                float p10 = data[(y0 * width + x1) * desiredChannels + c];
                float p11 = data[(y1 * width + x1) * desiredChannels + c];

                rgb[c] = lerp(lerp(p00, p10, fx), lerp(p01, p11, fx), fy);
            }
            // Calculate the index in the one-dimensional list
            int index = h * vertSize + v;
            // Calculate luminance and store it in the array
            candelaValues[index] = calculateLuminance(rgb);
        }
    }
    float intensity = *std::max_element(candelaValues.begin(), candelaValues.end());

    // Calculate the average color value
    glm::vec3 totalColor(0, 0, 0);
    for (size_t i = 0; i < width * (height/2) * desiredChannels; i += 3) {
        float r = data[i], g = data[i+1], b = data[i+2];
        totalColor[0] += r;     // Red channel
        totalColor[1] += g;     // Green channel
        totalColor[2] += b;     // Blue channel
    }
    glm::vec3 averageColor = totalColor / (static_cast<float>(width) * (static_cast<float>(height)/2));

    // Make sure the highest value is 1.0 and scale the others accordingly
    if (float maxColorValue = glm::compMax(averageColor); maxColorValue > 1.0f) {
        averageColor /= maxColorValue;
    }

    // Create sampler
    constexpr Sampler sampler = {
        Sampler::Filter::LINEAR, Sampler::Filter::LINEAR, Sampler::Filter::LINEAR,
        Sampler::Wrap::CLAMP_TO_BORDER, Sampler::Wrap::MIRRORED_REPEAT,
        Sampler::Wrap::CLAMP_TO_EDGE, 0, 0
    };

    // Create image
    Image* img = Image::alloc(aFile.filename().string());
    img->init(horizSize, vertSize, Image::Format::R32_FLOAT, candelaValues.data());

    //stbi_write_hdr(R"(C:\Users\morit\Documents\Moritz\TU_Wien\Bachelorarbeit\image.png)", img->getWidth(), img->getHeight(), 1, candelaValues.data());

    // Create texture with image and sampler
    Texture* texture = Texture::alloc();
    texture->image = img;
    texture->sampler = sampler;

    // Create the WindowLight object
    auto light = std::make_unique<WindowLight>();
    light->setFilepath(aFile.string());
    light->setIntensity(intensity);
    light->setColor(averageColor);
    light->setCandelaTexture(texture);
    light->generateHorizontalEmissionAngles(emitHorizAngleStart, emitHorizAngleEnd, horizSize);
    light->generateVerticalEmissionAngles(emitVertAngleStart, emitVertAngleEnd, vertSize);
    light->setConnectedModel(nullptr);

    stbi_image_free(data); // Free the original data
    return std::move(light);
}

std::unique_ptr<WindowLight> io::Import::loadHdriToIesRGB(const std::filesystem::path &aFile) {

    // Load HDR file
    int width, height, desiredChannels = 3, nrComponents;
    float *data = stbi_loadf(aFile.string().c_str(), &width, &height, &nrComponents, desiredChannels);
    if (!data) {
        spdlog::error("Failed to load HDR image.");
        return nullptr;
    }

    // Convert HDR to LDR
    for (int i = 0; i < width * height * desiredChannels; ++i) {
        // Clamp to 1.0 if higher
        data[i] = std::clamp(data[i], 0.0f, 1.0f);
    }

    // Define the size of the candela texture
    constexpr int vertSize = 512, horizSize = 512;
    // Angles for sampling
    std::vector<float> verticalAnglesSampling(4 * vertSize), horizontalAnglesSampling(horizSize);
    // Angles for emission
    std::vector<float> verticalAnglesEmitting, horizontalAnglesEmitting;
    // Sampling angle range
    float sampleVertAngleStart = 90, sampleVertAngleEnd = 178, sampleHorizAngleStart = 2, sampleHorizAngleEnd = 358;
    // Emission angle range
    float emitVertAngleStart = 0, emitVertAngleEnd = 35, emitHorizAngleStart = 0, emitHorizAngleEnd = 360;
    // Initialize candela values
    std::vector candelaValues(vertSize * horizSize * 4, 0.0f);
    // Generate the sampling angles
    generateAnglesSampling(sampleHorizAngleStart, sampleHorizAngleEnd, sampleVertAngleStart ,sampleVertAngleEnd, horizSize, vertSize, horizontalAnglesSampling, verticalAnglesSampling);

    // Extra index to pick correct vertical Angles for sampling
    int vertAngleIndex = 0;
    // Sample the HDRI and calculate luminance
    for (int h = 0; h < horizSize; ++h) {
        float horizAngle = horizontalAnglesSampling[h];
        if(h == horizSize / 4) {
            vertAngleIndex = 1;
        } else if (h == horizSize / 2) {
            vertAngleIndex = 2;
        } else if (h == 3 * horizSize / 4) {
            vertAngleIndex = 3;
        }
        for (int v = 0; v < vertSize; ++v) {
            float vertAngle = verticalAnglesSampling[vertAngleIndex * (vertSize - 1) + v];

            // Get pixel values from sampling angles
            auto [x, y] = anglesToImageCoords(vertAngle, horizAngle, width, height);

            // Calculate the non-fractional and fractional part of the coordinates
            float yFloor = std::floor(y);
            float xFloor = std::floor(x);
            float fx = x - xFloor;
            float fy = y - yFloor;

            // Determine the integer coordinates of the surrounding pixels
            int x0 = std::clamp(static_cast<int>(xFloor),0, width - 1);
            int x1 = std::clamp(x0 + 1, 0, width - 1);
            int y0 = std::clamp(static_cast<int>(yFloor), 0, height - 1);
            int y1 = std::clamp(y0 + 1, 0, height - 1);

            // Calculate the index in the one-dimensional list
            int index = (h * vertSize + v) * 4;

            // Interpolate the color values
            glm::vec3 rgb(0.0f);
            for (int c = 0; c < desiredChannels; ++c) {

                float p00 = data[(y0 * width + x0) * desiredChannels + c];
                float p01 = data[(y1 * width + x0) * desiredChannels + c];
                float p10 = data[(y0 * width + x1) * desiredChannels + c];
                float p11 = data[(y1 * width + x1) * desiredChannels + c];

                rgb[c] = lerp(lerp(p00, p10, fx), lerp(p01, p11, fx), fy);
            }

            candelaValues[index] = rgb[0];
            candelaValues[index + 1] = rgb[1];
            candelaValues[index + 2] = rgb[2];
            // set alpha
            candelaValues[index + 3] = 0.0f;
        }
    }
    float intensity = *std::max_element(candelaValues.begin(), candelaValues.end());

    // Create sampler
    constexpr Sampler sampler = {
        Sampler::Filter::LINEAR, Sampler::Filter::LINEAR, Sampler::Filter::LINEAR,
        Sampler::Wrap::CLAMP_TO_BORDER, Sampler::Wrap::MIRRORED_REPEAT,
        Sampler::Wrap::CLAMP_TO_EDGE, 0, 0
    };

    // Create image
    Image* img = Image::alloc(aFile.filename().string());
    img->init(horizSize, vertSize, Image::Format::RGBA32_FLOAT, candelaValues.data());

    // Create texture with image and sampler
    Texture* texture = Texture::alloc();
    texture->image = img;
    texture->sampler = sampler;

    // Create the WindowLight object
    auto light = std::make_unique<WindowLight>();
    light->setFilepath(aFile.string());
    light->setIntensity(intensity);
    light->setColor({0,0,0});
    light->setCandelaTexture(texture);
    light->generateHorizontalEmissionAngles(emitHorizAngleStart, emitHorizAngleEnd, horizSize);
    light->generateVerticalEmissionAngles(emitVertAngleStart, emitVertAngleEnd, vertSize);
    light->setConnectedModel(nullptr);

    stbi_image_free(data); // Free the original data
    return std::move(light);
}