#pragma once
#include <tamashii/public.hpp>
#include <tamashii/core/forward.h>

#include <string>
#include <deque>

T_BEGIN_NAMESPACE

// data for the rendering backend
struct SceneBackendData
{
	// unique data of this scene
	std::deque<Image*>&							images;
	std::deque<Texture*>&						textures;
	std::deque<std::shared_ptr<Model>>&			models;
	std::deque<Material*>&						materials;
	// instantiated data of this scene 
	// e.g. one model from SceneBackendData.models can be instantiated multible times at different position but with the same geometry
	std::deque<std::shared_ptr<RefModel>>&		refModels;
	std::deque<std::shared_ptr<RefLight>>&		refLights;
	std::deque<std::shared_ptr<RefCamera>>&		refCameras;
};
enum class HitMask {
	Geometry = 1,
	Light = 2,
	All = Geometry | Light
};
enum class CullMode {
	None = 0,
	Front = 1,
	Back = 2,
	Both = 3
};
struct IntersectionSettings
{
	IntersectionSettings() : mCullMode{ CullMode::None }, mHitMask{ HitMask::All } {}
	IntersectionSettings(const CullMode	aCullMode, const HitMask aHitMask) : mCullMode{ aCullMode }, mHitMask{ aHitMask } {}
	CullMode								mCullMode;
	HitMask									mHitMask;
};
struct Intersection {
	std::shared_ptr<Ref>					mHit;				// object that was hit
	RefMesh*								mRefMeshHit;		// if object was a model then this also includes the mesh that was hit
	uint32_t								mMeshIndex;			// if object was a model/mesh, this is the mesh index that was hit
	uint32_t								mPrimitiveIndex;	// if object was a model/mesh, this is the triangle index that was hit
	glm::vec3								mOriginPos;			// ray start pos in worldspace
	glm::vec3								mHitPos;			// hit pos in worldspace
	glm::vec2								mBarycentric;		// bary coords if hit was triangle mesh
	float									mTmin;				// distance
};
struct DrawInfo {
	enum class Target : uint32_t {
		VERTEX_COLOR,
		CUSTOM
	};
	Target									mTarget;
	glm::vec3								mCursorColor;
	bool									mDrawMode;			// draw mode off/on
	bool									mHoverOver;			// is mouse hovering over mesh
	Intersection							mHitInfo;
	glm::vec3								mPositionWs;		// position of draw pos
	glm::vec3								mNormalWsNorm;		// normal of draw pos
    glm::vec3								mTangentWsNorm;		// tangent of draw pos
	float									mRadius;			// draw influence radius
	glm::vec4								mColor0;			// color for left mouse button
	glm::vec4								mColor1;			// color for right mouse button
	bool									mDrawRgb;
	bool									mDrawAlpha;
	bool									mSoftBrush;
	bool									mDrawAll;
};
struct Selection
{
	std::shared_ptr<Ref>	reference;
	uint32_t				meshOffset;
	uint32_t				primitiveOffset;
	uint32_t				vertexOffset;
};

struct SceneUpdateInfo
{
	bool									mImages;
	bool									mTextures;
	bool									mMaterials;
	bool									mModelInstances;	// the instantiations or the model matrix of the scene geometry has changed
	bool									mModelGeometries;	// the overall scene is the same but the geometry has changed
	bool									mLights;
	bool									mCamera;

	[[nodiscard]] bool any() const { return mImages || mTextures || mMaterials || mModelInstances || mModelGeometries || mLights || mCamera; }
	[[nodiscard]] bool all() const { return mImages && mTextures && mMaterials && mModelInstances && mModelGeometries && mLights && mCamera; }
	void reset() {
		mImages = false;
		mTextures = false;
		mMaterials = false;
		mModelInstances = false;
		mModelGeometries = false;
		mLights = false;
		mCamera = false;
	}
	SceneUpdateInfo operator| (const SceneUpdateInfo& aSceneUpdates) const
	{
		SceneUpdateInfo su = {};
		su.mImages = mImages || aSceneUpdates.mImages;
		su.mTextures = mTextures || aSceneUpdates.mTextures;
		su.mMaterials = mMaterials || aSceneUpdates.mMaterials;
		su.mModelInstances = mModelInstances || aSceneUpdates.mModelInstances;
		su.mModelGeometries = mModelGeometries || aSceneUpdates.mModelGeometries;
		su.mLights = mLights || aSceneUpdates.mLights;
		su.mCamera = mCamera || aSceneUpdates.mCamera;
		return su;
	}
	SceneUpdateInfo operator& (const SceneUpdateInfo& aSceneUpdates) const
	{
		SceneUpdateInfo su = {};
		su.mImages = mImages && aSceneUpdates.mImages;
		su.mTextures = mTextures && aSceneUpdates.mTextures;
		su.mMaterials = mMaterials && aSceneUpdates.mMaterials;
		su.mModelInstances = mModelInstances && aSceneUpdates.mModelInstances;
		su.mModelGeometries = mModelGeometries && aSceneUpdates.mModelGeometries;
		su.mLights = mLights && aSceneUpdates.mLights;
		su.mCamera = mCamera && aSceneUpdates.mCamera;
		return su;
	}
};

class RenderScene final {
public:
											RenderScene();
											~RenderScene();

	bool									initFromFile(const std::string& aFile);
	bool									initFromData(io::SceneData& aSceneInfo);
	bool									addSceneFromFile(const std::string& aFile);
	bool									addSceneFromData(io::SceneData& aSceneInfo);

	void									destroy();

											// since the scene can be loaded/reloaded from different thread
											// it is necessary to synchronize the access
	void									readyToRender(bool aReady);
	bool									readyToRender() const;

	std::shared_ptr<RefLight>				addLightRef(const std::shared_ptr<Light>& aLight, glm::vec3 aPosition = glm::vec3(0), glm::vec4 aRotation = glm::vec4(0), glm::vec3 aScale = glm::vec3(1));
	void									addMaterial(Material* aMaterial);
	std::shared_ptr<RefModel>				addModelRef(const std::shared_ptr<Model>& aModel, glm::vec3 aPosition = glm::vec3(0), glm::vec4 aRotation = glm::vec4(0), glm::vec3 aScale = glm::vec3(1));

	void									removeModel(const std::shared_ptr<RefModel>& aRefModel);
	void									removeLight(const std::shared_ptr<RefLight>& aRefLight);

											// call to indicate data changes
	void									requestImageUpdate();
	void									requestTextureUpdate();
	void									requestMaterialUpdate();
	void									requestModelInstanceUpdate();
	void									requestModelGeometryUpdate();
	void									requestLightUpdate();
	void									requestCameraUpdate();

	void									intersect(glm::vec3 aOrigin, glm::vec3 aDirection, IntersectionSettings aSettings, Intersection *aHitInfo) const;

	bool									animation() const;
	void									setAnimation(bool aPlay);
	void									resetAnimation();

	void									setSelection(const Selection& aSelection);

	void									update(float aMilliseconds);
	void									draw();

	std::string								getSceneFileName();

	std::deque<std::shared_ptr<RefCamera>>& getAvailableCameras();
	RefCamera&								getCurrentCamera() const;
	std::shared_ptr<RefCamera>				referenceCurrentCamera() const;
	void									setCurrentCamera(const std::shared_ptr<RefCamera>& aCamera);
	glm::vec3								getCurrentCameraPosition() const;
	glm::vec3								getCurrentCameraDirection() const;
	Selection&						getSelection();

	std::deque<std::shared_ptr<RefModel>>& getModelList();
	std::deque<std::shared_ptr<RefLight>>& getLightList();
	std::deque<std::shared_ptr<RefCamera>>& getCameraList();

											// animation
	float									getCurrentTime() const;
	float									getCycleTime() const;

	SceneBackendData						getSceneData();
	io::SceneData							getSceneInfo();
private:
	static void								filterSceneInfo(io::SceneData& aSceneInfo);
	void									traverseSceneGraph(Node& aNode, glm::mat4 aMatrix = glm::mat4(1.0f), bool aAnimatedPath = false);

	std::string								mSceneFile;
	std::atomic<bool>						mReady;	// scene is ready to render

											// assets used in this scene referenced from global asset manager
	std::shared_ptr<Node>					mSceneGraph;
	std::deque<std::shared_ptr<Model>>		mModels;
	std::deque<std::shared_ptr<Camera>>		mCameras;
	std::deque<std::shared_ptr<Light>>		mLights;
	std::deque<Material*>					mMaterials;
	std::deque<Texture*>					mTextures;
	std::deque<Image*>						mImages;

											// instantiations of models
	std::deque<std::shared_ptr<RefModel>>	mRefModels;
	std::deque<std::shared_ptr<RefCamera>>	mRefCameras;
	std::deque<std::shared_ptr<RefLight>>	mRefLights;

	std::shared_ptr<Camera>					mDefaultCamera;
	std::shared_ptr<RefCamera>				mDefaultCameraRef;

	std::shared_ptr<RefCamera>				mCurrentCamera;
	Selection							mSelection;

	bool									mPlayAnimation;
											// time of one animation cycle in seconds
	float									mAnimationCycleTime;
											// current time of this scene in seconds
	float									mAnimationTime;
											// was there an update since the last frame
	SceneUpdateInfo							mUpdateRequests;
											// added since last cycle
	std::deque<std::shared_ptr<Ref>>		mNewlyAddedRef;
	std::deque<std::shared_ptr<Ref>>		mNewlyRemovedRef;
	std::deque<std::shared_ptr<Asset>>		mNewlyRemovedAsset;
};

T_END_NAMESPACE
