//? #version 460
#ifndef GLSL_RANDOM_GLSL
#define GLSL_RANDOM_GLSL

#ifndef __cplusplus
	#extension GL_EXT_shader_explicit_arithmetic_types : require //? #extension GL_NV_gpu_shader5 : enable
	//#extension GL_EXT_control_flow_attributes : require

	#ifndef REF
		#define REF(type) inout type
	#endif
	#ifndef OREF
		#define OREF(type) out type
	#endif
#endif


// Simple Random
// Based on http://byteblacksmith.com/improvements-to-the-canonical-one-liner-glsl-rand-for-opengl-es-2-0/
// from https://www.shadertoy.com/view/4t2SDh
// Uniform noise in [0..1[ range
float randomFloat(const vec2 uv) {
	return fract(sin(dot(uv, vec2(12.9898f, 78.233f))) * 43758.5453f);
}

// Using a triangle distribution which gives a more final uniform noise.
// See Banding in Games:A Noisy Rant(revision 5) Mikkel Gj�l, Playdead (slide 27) https://loopit.dk/banding_in_games.pdf
// triangle noise
// from https://www.shadertoy.com/view/4t2SDh
float randomFloatTriDist(const vec2 uv) {
	// Original code from https://www.shadertoy.com/view/4t2SDh
	// Uniform noise in [0..1[ range
	float nrnd0 = randomFloat(uv);
	// Convert uniform distribution into triangle-shaped distribution.
	// https://en.wikipedia.org/wiki/Triangular_distribution
	const float orig = nrnd0 * 2.0f - 1.0f;		// range [-1..1[
	nrnd0 = orig * inversesqrt(abs(orig));
	nrnd0 = max(-1.0f, nrnd0); 				// removes NaN's generated by 0*rsqrt(0). Thanks @FioraAeterna!
	// Vulkan is rounding a float before writting it to an image with UNORM so we center the distribution around 0.0, instead of 0.5
	// https://www.khronos.org/registry/vulkan/specs/1.2/html/chap3.html#fundamentals-fpfixedconv
	return nrnd0 - sign(orig); 				// result is range [-1..1[
	//return nrnd0 - sign(orig) + 0.5; 		// result is range [-0.5,1.5[
}

/*****************************/
/* Tiny Encryption Algorithm */
/*****************************/
// https://en.wikipedia.org/wiki/Tiny_Encryption_Algorithm
// https://github.com/nvpro-samples/vk_raytracing_tutorial_KHR/tree/master/ray_tracing_jitter_cam
// https://github.com/nvpro-samples/optix_prime_baking/blob/332a886f1ac46c0b3eea9e89a59593470c755a0e/random.h
// Generates a seed from 2 inputs plus a backoff
uint32_t tea_init(uint32_t v0, uint32_t v1) {
	uint32_t sum = 0u;

	//[[unroll]]
	for (uint32_t n = 0u; n < 16u; n++) {
		sum += 0x9e3779b9U;
		v0 += ((v1 << 4u) + 0xa341316cU) ^ (v1 + sum) ^ ((v1 >> 5u) + 0xc8013ea4U);
		v1 += ((v0 << 4u) + 0xad90777dU) ^ (v0 + sum) ^ ((v0 >> 5u) + 0x7e95761eU);
	}

	return v0;
}

// https://www.unf.edu/~cwinton/html/cop4300/s09/class.notes/LCGinfo.pdf
// Generate a random unsigned int in [0, 2^24) given the previous RNG state
uint32_t tea_nextUInt(REF(uint32_t) seed)
{
	// LCG values from Numerical Recipes
	return (seed = 1664525u * seed + 1013904223u);
}

// Generate a random float in [0, 1) given the previous RNG state
float tea_nextFloat(REF(uint32_t) seed)
{
	// Float version using bitmask from Numerical Recipes
	const uint32_t one = 0x3f800000;
	const uint32_t msk = 0x007fffff;
	return uintBitsToFloat(0x3f800000u | (0x007fffffu & tea_nextUInt(seed))) - 1.0f;
}
vec2 tea_nextFloat2(REF(uint32_t) seed) { return vec2(tea_nextFloat(seed), tea_nextFloat(seed)); }
vec3 tea_nextFloat3(REF(uint32_t) seed) { return vec3(tea_nextFloat(seed), tea_nextFloat(seed), tea_nextFloat(seed)); }
vec4 tea_nextFloat4(REF(uint32_t) seed) { return vec4(tea_nextFloat(seed), tea_nextFloat(seed), tea_nextFloat(seed), tea_nextFloat(seed)); }

// Faster version from NVIDIA examples
// https://github.com/nvpro-samples/vk_raytracing_tutorial_KHR/tree/master/ray_tracing_jitter_cam
float tea_nextFloatFast(REF(uint32_t) seed) {
    return uintBitsToFloat(tea_nextUInt(seed) & 0x00FFFFFFu) / uintBitsToFloat(0x01000000u);
}
vec2 tea_nextFloatFast2(REF(uint32_t) seed) { return vec2(tea_nextFloatFast(seed), tea_nextFloatFast(seed)); }
vec3 tea_nextFloatFast3(REF(uint32_t) seed) { return vec3(tea_nextFloatFast(seed), tea_nextFloatFast(seed), tea_nextFloatFast(seed)); }
vec4 tea_nextFloatFast4(REF(uint32_t) seed) { return vec4(tea_nextFloatFast(seed), tea_nextFloatFast(seed), tea_nextFloatFast(seed), tea_nextFloatFast(seed)); }

/*
 * PCG Random Number Generation for C.
 *
 * Copyright 2014 Melissa O'Neill <oneill@pcg-random.org>
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * For additional information about the PCG random number generation scheme,
 * including its license and other licensing options, visit
 *
 *       http://www.pcg-random.org
 */

/*
 * This code is derived from the full C implementation, which is in turn
 * derived from the canonical C++ PCG implementation. The C++ version
 * has many additional features and is preferable if you can use C++ in
 * your project.
 */

// Generate a uniformly distributed 32-bit random number
uint32_t pcg32_nextUInt(REF(u64vec2) seed) {
    uint64_t oldstate = seed.x;
    seed.x = oldstate * 0x5851f42d4c957f2dUL + seed.y;
    uint32_t xorshifted = uint32_t(((oldstate >> 18ul) ^ oldstate) >> 27ul);
    uint32_t rot = uint32_t(oldstate >> 59ul);
    return (xorshifted >> rot) | (xorshifted << ((~rot + 1u) & 31u));
}

// Generate a single precision floating point value on the interval [0, 1)
float pcg32_nextFloat(REF(u64vec2) seed) {
    // Trick from MTGP: generate an uniformly distributed
    // double precision number in [1,2) and subtract 1.
    const uint32_t u = (pcg32_nextUInt(seed) >> 9u) | 0x3f800000u;
    return uintBitsToFloat(u) - 1.0f;
}
vec2 pcg32_nextFloat2(REF(u64vec2) seed) { return vec2(pcg32_nextFloat(seed), pcg32_nextFloat(seed)); }
vec3 pcg32_nextFloat3(REF(u64vec2) seed) { return vec3(pcg32_nextFloat(seed), pcg32_nextFloat(seed), pcg32_nextFloat(seed)); }
vec4 pcg32_nextFloat4(REF(u64vec2) seed) { return vec4(pcg32_nextFloat(seed), pcg32_nextFloat(seed), pcg32_nextFloat(seed), pcg32_nextFloat(seed)); }

// Generate a double precision floating point value on the interval [0, 1)
double pcg32_nextDouble(REF(u64vec2) seed) {
    // Trick from MTGP: generate an uniformly distributed
    // double precision number in [1,2) and subtract 1.
    const uint64_t u = (uint64_t(pcg32_nextUInt(seed)) << 20ul) | 0x3ff0000000000000ul;
    return uint64BitsToDouble(u) - 1.0;
}

/* vec2(state, inc) */
u64vec2 pcg32_init(uint64_t initstate, uint64_t initseq) {
	const uint64_t one = 1;
    initseq = max(one, initseq);
    u64vec2 s;
    s.x = 0u;
    s.y = (initseq << 1u) | 1u;
    pcg32_nextUInt(s);
    s.x += initstate;
    pcg32_nextUInt(s);
    return s;
}

#endif /* GLSL_RANDOM_GLSL */