//? #version 460

#ifndef GLSL_BSDF_VOLUME
#define GLSL_BSDF_VOLUME

#ifndef M_2PI
#define M_2PI 6.28318530717959f
#endif
#ifndef M_INV_2PI
#define M_INV_2PI 0.159154943091895f
#endif
#ifndef M_INV_4PI
#define M_INV_4PI 0.0795774715459477f
#endif

#ifndef GLSL_RENDERING_UTILS
#include "../rendering_utils.glsl"
#endif

// simgma_a absorption coefficient
// simgma_s scattering coefficient
// simgma_t extinction coefficient (transmittance (attenuation))

// https://www.pbr-book.org/3ed-2018/Light_Transport_II_Volume_Rendering/Sampling_Volume_Scattering
// https://en.wikipedia.org/wiki/Beer%E2%80%93Lambert_law
// Beer's law (absorption)
float transmittance(const float sigma_t, const float dist) {
    return exp(-sigma_t * dist);
}
float sampleDistance(const float sigma_t, const float rand) {
    return -log(1.0f - rand) / sigma_t;
}
float sampleDistancePdf(const float sigma_t, const float dist, const float rand) {
    return sigma_t * transmittance(sigma_t, dist);
}

// henyey greenstein (anisotropic phase functions)
// https://www.astro.umd.edu/~jph/HG_note.pdf
// g in [-1..1]
// g > 0 = forward scattering is dominant
// g < 0 = backscattering predominates
// https://www.pbr-book.org/3ed-2018/Light_Transport_II_Volume_Rendering/Volumetric_Light_Transport
// phase function and pdf is the same
float hgPhaseFunction(const float cosTheta, const float g)
{
    const float denom = 1.0f + g * g - 2.0f * g * cosTheta;
    return M_INV_4PI * (1.0f - g * g) / (denom * sqrt(denom));
}

#define hgPdf (M_INV_2PI)

// https://www.pbr-book.org/3ed-2018/Light_Transport_II_Volume_Rendering/Sampling_Volume_Scattering#SamplingPhaseFunctions
vec2 hgSample(vec3 w, float g, const vec2 rand)
{
    float cosTheta;
    if (abs(g) < 0.001f) cosTheta = 1.0f - 2.0f * rand[0];
    else {
        const float sqrTerm = (1.0f - g * g) / (1.0f - g + 2.0f * g * rand[0]);
        cosTheta = (1.0f + g * g - sqrTerm * sqrTerm) / (2.0f * g);
    }

    const float theta = acos(cosTheta);
    const float phi = M_2PI * rand[1];
    return vec2(theta, phi);
    // const float sinTheta = clamp(sqrt(1.0f - (cosTheta * cosTheta)), 0.0f, 1.0f);
    // const float sinPhi = sin(phi);
    // const float cosPhi = cos(phi);

    // vec3 T, B;
    // Onb(w, T, B);

    // return sinTheta * cosPhi * T + sinTheta * sinPhi * B + cosTheta * w;
}


#endif /* GLSL_BSDF_VOLUME */