﻿
// Copyright (c) 2011-2012, Daniel Müller <dm@g4t3.de>
// Computer Graphics Systems Group at the Hasso-Plattner-Institute, Germany
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without 
// modification, are permitted provided that the following conditions are met:
//   * Redistributions of source code must retain the above copyright notice, 
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright 
//     notice, this list of conditions and the following disclaimer in the 
//     documentation and/or other materials provided with the distribution.
//   * Neither the name of the Computer Graphics Systems Group at the 
//     Hasso-Plattner-Institute (HPI), Germany nor the names of its 
//     contributors may be used to endorse or promote products derived from 
//     this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
// LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
// CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
// SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
// INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
// CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
// ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
// POSSIBILITY OF SUCH DAMAGE.

#include "earth.h"
#include <assert.h>
#include "math.h"

#define _deg(rad) \
    ((rad) * 180.0L / _PI)

#define _rad(deg) \
    ((deg) * _PI / 180.0L)

#define _PI 3.1415926535897932384626433832795L

// same as _hour
// note: if d is negative use -_decimal(d, m, s) instead of _decimal(-d, m, s)
#define _decimal(d, m, s) \
    ((d) + ((m) + (s) / 60.0L) / 60.0L)

// The linear eccentricity of the earth orbit is about 2.5 * 10^6 km.
// Compared to the avg. distance of 149.6 * 10^6 km this is not much.
// http://www.greier-greiner.at/hc/ekliptik.htm

// P. Bretagnon, "Théorie du mouvement de l'ensamble des planètes. Solution VSOP82", 1982


// This is, if required, approximatelly refraction corrected...

const double osgCloudyDay::Earth::viewDistanceWithinAtmosphere(
    const double y
,   const bool refractionCorrected)
{
    const double t = atmosphereThickness();
    const double r = meanRadius();

    // This works, since dot product of [0, 1, 0] and 
    // eye with [x, y, z] gives y.

    double h = asin(y * (1.0 - 1.e-12)); // correction is required to 
                       // gain t at y = 1.0 - fix for double accuracy.

    if(refractionCorrected)
        h += _rad(atmosphericRefraction(_deg(asin(y))));

    const double cosa = cos(h);
    const double rt = r + t;

    // Using law of sine for arbitrary triangle with two sides and one angle known.
    // Since the angle is (π/2 + a), cos is used instead of sine.

    const double distance = cos(h + asin(cosa * r / rt)) * rt / cosa;

    return distance;
}


// Effect of refraction for true altitudes (AA.15.4).
// G.G. Bennet, "The Calculation of the Astronomical Refraction in marine Navigation", 1982
// and Porsteinn Saemundsson, "Sky and Telescope" 1982

const double osgCloudyDay::Earth::atmosphericRefraction(const double altitude)
{
    double R = 1.02 / 
        tan(_rad(altitude + 10.3 / (altitude + 5.11))) + 0.0019279;

    return _decimal(0, R, 0); // (since R is in minutes)
}


const double osgCloudyDay::Earth::meanRadius()
{
    // http://nssdc.gsfc.nasa.gov/planetary/factsheet/earthfact.html

    return  6371.0; // in kilometers
}


double osgCloudyDay::Earth::atmosphereThickness()
{
    // Thickness of atmosphere if the density were uniform.
    
    // 8000 ("Precomputed Atmospheric Scattering" - 2008 - Bruneton, Neyret)
    // 7994 ("Display of the earth taking into account atmospheric scattering" - 1993 - Nishita et al.)
    return 7.994;
}


const double osgCloudyDay::Earth::atmosphereThicknessNonUniform()
{
    // Thickness of atmosphere.
    return 85.0; // ~
}


const double osgCloudyDay::Earth::apparentMagnitudeLimit()
{
    // http://www.astronomynotes.com/starprop/s4.htm
    return 6.5;
}