#include "VoxelizeMesh.h"
#include <osgDB/ReadFile>
#include <osg/CullFace>
#include <osg/MatrixTransform>
#include <osg/Depth>

float* osgCloudyDay::ReadBufferCallback::m_data;

osgCloudyDay::VoxelizeMesh::VoxelizeMesh(std::string path, int slice_width, int slice_height, int slice_depth) :  viewportWidth(slice_width), viewportHeight(slice_height), m_slice_depth(slice_depth)
{	 
	group = new osg::Group();

	osg::CullFace* cullFront = new osg::CullFace();
	cullFront->setMode(osg::CullFace::FRONT);
	osg::CullFace* cullBack = new osg::CullFace();
	cullBack->setMode(osg::CullFace::BACK);			

	{
		osg::ref_ptr<osg::Node> quad_tmp = osgDB::readNodeFile(path);
		quad_tmp->getOrCreateStateSet()->setAttributeAndModes(cullFront, osg::StateAttribute::ON); 
		quad_tmp->getOrCreateStateSet()->addUniform(new osg::Uniform("color", osg::Vec4(1.f, 1.f, 1.f, 1.f)));
		osg::MatrixTransform* t = new osg::MatrixTransform(osg::Matrixd(1.f, 0.f, 0.f, 0.f, 
																		0.f, 1.f, 0.f, 0.f, 
																		0.f, 0.f, 1.f, 0.f, 
																		0.f, 0.f, 0.f, 1.f));
		t->addChild(quad_tmp);		
		group->addChild(t);
	}
	{
		osg::ref_ptr<osg::Node> quad = osgDB::readNodeFile(path);
		quad->getOrCreateStateSet()->addUniform(new osg::Uniform("color", osg::Vec4(0.f, 0.f, 0.f, 0.f)));	
		quad->getOrCreateStateSet()->setAttributeAndModes(cullBack, osg::StateAttribute::ON); 		
		osg::MatrixTransform* t = new osg::MatrixTransform(osg::Matrixd(1.f, 0.f, 0.f, 0.f, 
																		0.f, 1.f, 0.f, 0.f, 
																		0.f, 0.f, 1.f, 0.f, 
																		0.f, 0.f, 0.f, 1.f));
		t->addChild(quad);		
		group->addChild(t);

		CcalculateBoundingBox bbox ;
		quad->accept( bbox  );
		m_boundingbox = bbox.getBoundBox();
	}	

	/*m_boundingbox.xMin() -= 5.f;
	m_boundingbox.yMin() -= 5.f;
	m_boundingbox.zMin() -= 5.f;
	m_boundingbox.xMax() += 5.f;
	m_boundingbox.yMax() += 5.f;
	m_boundingbox.zMax() += 5.f;*/

/////////
	voxels_tex = setupTexture3D("Voxel", GL_RGBA, GL_RGBA, GL_FLOAT, slice_width, slice_height, slice_depth);

////////////////
	m_shader=(new osg::Program);
	osg::ref_ptr<osg::Shader> cloudShadowMapVertexShader(osg::Shader::readShaderFile (osg::Shader::VERTEX, "shaders/basic.vert"));	
	osg::ref_ptr<osg::Shader> cloudShadowMapFragShader(osg::Shader::readShaderFile (osg::Shader::FRAGMENT, "shaders/basic.frag"));
	
	//Binding the box shaders to its program
	m_shader->addShader(cloudShadowMapVertexShader.get());
	m_shader->addShader(cloudShadowMapFragShader.get());
	m_shader->addBindFragDataLocation("out_color", 0);
}


osgCloudyDay::VoxelizeMesh::~VoxelizeMesh(void)
{
	delete[] m_data;
	//delete[] ReadBufferCallback::m_data;
}

osg::Texture3D* osgCloudyDay::VoxelizeMesh::setupTexture3D(const char *name,   const GLenum internalFormat,   const GLenum pixelFormat,   const GLenum dataType,   const int width,   const int height,   const int depth,   osg::Image *image)
{
    osg::Texture3D* texture = (new osg::Texture3D);

    texture->setName(name);
    texture->setTextureSize(width, height, depth);

    if(image)
    {
        image->setInternalTextureFormat(internalFormat);
        image->allocateImage(width, height, depth, pixelFormat, dataType);

        texture->setImage(image);
    }
    else
    {
        texture->setInternalFormat(internalFormat);
        texture->setSourceFormat(pixelFormat);
        texture->setSourceType(dataType);
    }

    texture->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR);
    texture->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR);
    texture->setWrap(osg::Texture::WRAP_S, osg::Texture::CLAMP_TO_EDGE);
    texture->setWrap(osg::Texture::WRAP_T, osg::Texture::CLAMP_TO_EDGE);
    texture->setWrap(osg::Texture::WRAP_R, osg::Texture::CLAMP_TO_EDGE);

    return texture;
}

osg::Vec4 osgCloudyDay::VoxelizeMesh::GetData(int x, int y, int z)
{
	return osg::Vec4(	ReadBufferCallback::m_data[(z*viewportHeight*viewportWidth+y*viewportWidth+x)*4+3], 
						ReadBufferCallback::m_data[(z*viewportHeight*viewportWidth+y*viewportWidth+x)*4+3], 
						ReadBufferCallback::m_data[(z*viewportHeight*viewportWidth+y*viewportWidth+x)*4+3],
						ReadBufferCallback::m_data[(z*viewportHeight*viewportWidth+y*viewportWidth+x)*4+3]);
}

void osgCloudyDay::VoxelizeMesh::Perform()
{
	 //if(ifDirtyOnly && !m_dirty)        return false;

    //m_dirty = false;

    osg::Timer_t t = osg::Timer::instance()->tick();

    // Setup Viewer
    viewer = new osgViewer::CompositeViewer;
    osgViewer::View *view = new osgViewer::View();
    viewer->addView(view);

    // Setup Context and Camera
    osg::GraphicsContext *gc(setupContext());

    if(!gc->valid())
    {
        OSG_FATAL << "Initialize PBuffer graphics context failed" << std::endl;
        return;
    }


    osg::Camera *camera = view->getCamera();
    camera->setGraphicsContext(gc);
    camera->setClearColor(osg::Vec4f(0.f, 0.f, 0.f, 0.f));
    camera->setViewport(0, 0, 1, 1);

    osg::Group *group(new osg::Group);
    view->setSceneData(group);
	
	targets3D = t_tex3DsByUnit();
	targets3D.insert(std::pair<int,osg::Texture3D*>(0, voxels_tex));

	osg::Depth* depth = new osg::Depth(osg::Depth::LEQUAL);
	group->getOrCreateStateSet()->setAttribute(depth);
	
	//Back faces => white
    Render3D(osg::Vec4(1.f,1.f,1.f,1.f), true); ///*viewer, quad, targets3D, samplers2D, samplers3D, uniforms, glsl_bruneton_f_inscatter1().c_str()*/);
	

	//Front faces=> black
	//Render3D(osg::Vec4(0.f,0.f,0.f,1.f), false);	

	ReadBufferCallback::m_data = new float[viewportWidth * viewportHeight * m_slice_depth * 4];
	Read3DTexture();

    // Unref
    delete viewer;
    OSG_NOTICE << "Atmopshere Precomputed (took "  << osg::Timer::instance()->delta_s(t,  osg::Timer::instance()->tick()) << " s)" << std::endl;	
}

void osgCloudyDay::VoxelizeMesh::Read3DTexture()
{
	osg::Group *group = setupGroup(viewer);   
	
	for(int layer = 0; layer < m_slice_depth; ++layer)
	{
		osg::ref_ptr<osg::Camera> camera(new osg::Camera);		

		camera->setViewport(0, 0, viewportWidth, viewportHeight);
		camera->setRenderOrder(osg::Camera::POST_RENDER, layer);
		camera->setCullingActive(false);
	
		camera->setClearColor(osg::Vec4(0.f, 0.f, 0.f, 0.f));
		camera->setFinalDrawCallback(new ReadBufferCallback(voxels_tex, viewportWidth, viewportHeight, layer));

		camera->setRenderTargetImplementation(osg::Camera::FRAME_BUFFER_OBJECT);

		group->addChild(camera.get());

		for(t_tex3DsByUnit::const_iterator i3 = targets3D.begin(); i3 != targets3D.end(); ++i3)
			camera->attach(static_cast<osg::Camera::BufferComponent>(osg::Camera::COLOR_BUFFER0 + i3->first), i3->second, 0U, layer);
	}

	viewer->frame();
}

void osgCloudyDay::VoxelizeMesh::Render3D(osg::Vec4 color, bool clear)
{    
	t_tex3DsByUnit::const_iterator i3;
	const t_tex3DsByUnit::const_iterator t3End = targets3D.end();

    i3 = targets3D.begin();	
    const int width  = voxels_tex->getTextureWidth();
    const int height = voxels_tex->getTextureHeight();
    const int depth  = voxels_tex->getTextureDepth();	
    
    // Setup graph	
    osg::Group *group = setupGroup(viewer);   

    osg::StateSet *ss(group->getOrCreateStateSet());
    ss->setAttributeAndModes(m_shader);
	//ss->addUniform(new osg::Uniform("color", color));

    for(int layer = 0; layer < depth; ++layer)
    {
        // Setup local camera

        osg::ref_ptr<osg::Camera> camera = setupCamera(layer, clear);
        group->addChild(camera.get());

        //setupLayerUniforms(camera->getOrCreateStateSet(), depth, layer);
		
		osg::Matrix proj = osg::Matrix::ortho(
			m_boundingbox.xMin(), m_boundingbox.xMax(), 
			m_boundingbox.yMin(), m_boundingbox.yMax(), 
			0.f, (m_boundingbox.zMax()-m_boundingbox.zMin()));

		osg::Matrix view = osg::Matrix::lookAt(osg::Vec3f(0.f, 0.f, m_boundingbox.zMin() + (m_boundingbox.zMax()-m_boundingbox.zMin())*((float)(layer)/(float)(m_slice_depth))), 
											   osg::Vec3f(0.f, 0.f, m_boundingbox.zMax()), 
											   osg::Vec3f(0.f, 1.f, 0.f));

		camera->getOrCreateStateSet()->addUniform(new osg::Uniform("Projection", proj));
		camera->getOrCreateStateSet()->addUniform(new osg::Uniform("View", view));		

        // Assign Textures and Samplers
        for(i3 = targets3D.begin(); i3 != t3End; ++i3)
        {
            /*if(i3->second->getImage())
            {
                // workaround: use a slice here instead of the whole image, since osg does not support this directly...
                osg::Image *slice = getLayerFrom3DImage(i3->second->getImage(), layer);
                camera->attach(static_cast<osg::Camera::BufferComponent>(osg::Camera::COLOR_BUFFER0 + i3->first), slice);
            }
            else*/
			camera->attach(static_cast<osg::Camera::BufferComponent>(osg::Camera::COLOR_BUFFER0 + i3->first), i3->second, 0U, layer);
        }
    }

    viewer->frame(); // Render single frame   


	cleanUp(viewer);
    dirtyTargets(targets3D);
    targets3D.clear();
}

void osgCloudyDay::VoxelizeMesh::cleanUp(osgViewer::CompositeViewer *viewer)
{
    osg::Group *root(dynamic_cast<osg::Group*>(viewer->getView(0)->getSceneData()));
    root->removeChildren(0, root->getNumChildren());
}

void osgCloudyDay::VoxelizeMesh::dirtyTargets(t_tex3DsByUnit &targets3D)
{
    t_tex3DsByUnit::const_iterator i3 = targets3D.begin();
    const t_tex3DsByUnit::const_iterator t3End = targets3D.end();

    for(; i3 != t3End; ++i3)
    {
        if(i3->second->getImage())
            i3->second->getImage()->dirty();
    }
}

osg::Camera* osgCloudyDay::VoxelizeMesh::setupCamera(const int orderNum, bool clear)
{
    osg::Camera *camera(new osg::Camera);

    camera->setViewport(0, 0, viewportWidth, viewportHeight);
    camera->setRenderOrder(osg::Camera::PRE_RENDER, orderNum);
	camera->setCullingActive(false);
	
	camera->setClearColor(osg::Vec4(0.f, 0.f, 0.f, 0.f));
	
	camera->setComputeNearFarMode( osgUtil::CullVisitor::DO_NOT_COMPUTE_NEAR_FAR );
	camera->getOrCreateStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF);

	if(clear)	camera->setClearMask(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
	else		camera->setClearMask(GL_DEPTH_BUFFER_BIT);
	//if(!clear) 	camera->setFinalDrawCallback(new ReadBufferCallback(voxels_tex, orderNum));

    camera->setRenderTargetImplementation(osg::Camera::FRAME_BUFFER_OBJECT);
	
	
	camera->addChild(group);	

    return camera;
}

// (from Emmanuel Roche e.g. here http://www.mail-archive.com/osg-users@lists.openscenegraph.org/msg42832.html)
// Creates a "view" to a single layer of an 3d image. This is 
// required for rendering into a Texture3D based on its image
// (at least when using one camera per layer...).

osg::Image *osgCloudyDay::VoxelizeMesh::getLayerFrom3DImage(osg::Image *source,   const int layer)
{
    osg::Image *image(new osg::Image());

    unsigned char *data = source->data(0, 0, layer);

    image->setImage(source->s(), source->t(), 1
        , source->getInternalTextureFormat()
        , source ->getPixelFormat()
        , source->getDataType()
        , data, osg::Image::NO_DELETE, source->getPacking());

    return image;
}

osg::GraphicsContext *osgCloudyDay::VoxelizeMesh::setupContext()
{
    // (pbuffer by http://forum.openscenegraph.org/viewtopic.php?t=9025)

    osg::GraphicsContext::Traits *traits = new osg::GraphicsContext::Traits;

    traits->screenNum = 0;
    traits->x = 0;
    traits->y = 0;
    traits->width = 1;
    traits->height = 1;
    traits->windowDecoration = false;
    traits->doubleBuffer = false;
    traits->sharedContext = NULL;
    traits->pbuffer = true;
    //traits->samples = 16;

    return osg::GraphicsContext::createGraphicsContext(traits);
}

osg::Group *osgCloudyDay::VoxelizeMesh::setupGroup(osgViewer::CompositeViewer *viewer)
{
    osg::Group *root(dynamic_cast<osg::Group*>(viewer->getView(0)->getSceneData()));       

    osg::ref_ptr<osg::Group> group(new osg::Group);
    root->addChild(group.get());

    return group;
}