#include "Cloud2D.h"
#include <osg/PolygonMode>

osgCloudyDay::Cloud2D::Cloud2D(void) : m_tesselation(true)
{
}


osgCloudyDay::Cloud2D::~Cloud2D(void)
{
}

void osgCloudyDay::Cloud2D::SetGeometry()
{
	SetGeometry(osg::Vec3(0.f, 0.f, 5000.f), osg::Vec2(10000.0f, 10000.0f));
}

void osgCloudyDay::Cloud2D::SetGeometry(osg::Vec3 position, osg::Vec2 size)
{
	//std::cout << std::endl << std::endl << "2D CLoud" << std::endl;
	geode = new osg::Geode;	

	// set up the Geometry.
	geom = new osg::Geometry;

	osg::Vec3 width, depth, topleft;
	topleft = osg::Vec3( -size.x()/2.0, -size.y()/2.0, 4000.0 );
	width = osg::Vec3( size.x(), 0.0, 0.0 );
	depth = osg::Vec3( 0.0, size.y(), 0.0 );

	osg::Vec3Array* coords = new osg::Vec3Array(100*100);		
	osg::Vec2Array* tcoords = new osg::Vec2Array(100*100);
	osg::UIntArray* indices = new osg::UIntArray(99*99*6);	

	SetupGeometry(position, size, coords, tcoords, indices);
	
	geom->setVertexArray(coords);	

	geom->setTexCoordArray( 0, tcoords );
	geom->setTexCoordArray( 1, tcoords );
	geom->setUseDisplayList(false);

	//geom->addPrimitiveSet( new osg::DrawArrays( osg::PrimitiveSet::QUADS, 0 , coords->size() ) );
	//geom->addPrimitiveSet( new osg::DrawElementsUInt( osg::PrimitiveSet::PATCHES, indices->size(), &indices->at(0)));
	geom->addPrimitiveSet( new osg::DrawElementsUInt( osg::PrimitiveSet::TRIANGLES, indices->size(), &indices->at(0)));
	geode->addDrawable( geom );	

	// Compute smoothed normals
	osgUtil::SmoothingVisitor smoother;
	smoother.apply( *geode );
	
	osg::ref_ptr< osgUtil::TangentSpaceGenerator > tsg = new osgUtil::TangentSpaceGenerator;
	tsg->generate( geom, 1 );
	geom->setVertexAttribData( 0, osg::Geometry::ArrayData( coords, osg::Geometry::BIND_PER_VERTEX, GL_FALSE ) );
	geom->setVertexAttribData( 1, osg::Geometry::ArrayData( tcoords, osg::Geometry::BIND_PER_VERTEX, GL_FALSE ) );
	geom->setVertexAttribData( 5, osg::Geometry::ArrayData( tsg->getNormalArray(), osg::Geometry::BIND_PER_VERTEX, GL_FALSE ) );
	geom->setVertexAttribData( 6, osg::Geometry::ArrayData( tsg->getTangentArray(), osg::Geometry::BIND_PER_VERTEX, GL_FALSE ) );
	geom->setVertexAttribData( 7, osg::Geometry::ArrayData( tsg->getBinormalArray(), osg::Geometry::BIND_PER_VERTEX, GL_FALSE ) );		

	for(unsigned int k = 0; k < geom->getNumPrimitiveSets() && m_tesselation; k++)					
	{						
		if(geom->getPrimitiveSet(k)->getMode() == 5 || geom->getPrimitiveSet(k)->getType() == 4)
		{						
			//osg::PrimitiveSet::Type::			
			if(geom->getPrimitiveSet(k)->getType() == 4)
			{
				GLushort* oldindices = (GLushort*)(geom->getPrimitiveSet(k)->getDataPointer());																
				GLushort* indices = new GLushort[(geom->getPrimitiveSet(k)->getNumIndices()-2)*3];
							
				//std::cout << geom->getPrimitiveSet(k)->getNumIndices() << " SIZE: "<< ((geom->getPrimitiveSet(k)->getNumIndices()-2)*3) << std::endl;
				for(unsigned int l = 2; l < geom->getPrimitiveSet(k)->getNumIndices(); l++)
				{									
					indices[((l-2)*3)+0] = oldindices[l-2];
					indices[((l-2)*3)+1] = oldindices[l-1];
					indices[((l-2)*3)+2] = oldindices[l-0];																
				}

				geom->setPrimitiveSet(k, new osg::DrawElementsUShort( osg::PrimitiveSet::PATCHES, (geom->getPrimitiveSet(k)->getNumIndices()-2)*3, indices));								
			}
			else if(geom->getPrimitiveSet(k)->getType() == 1)
			{
				osg::DrawArrays* oldindices = static_cast<osg::DrawArrays*>(geom->getPrimitiveSet(k));																
				GLuint* indices = new GLuint[(geom->getPrimitiveSet(k)->getNumIndices()-2)*3];

				for(unsigned int l = 2; l < geom->getPrimitiveSet(k)->getNumIndices(); l++)
				{
					indices[((l-2)*3)+0] = (l-2);
					indices[((l-2)*3)+1] = (l-1);
					indices[((l-2)*3)+2] = (l-0);					
				}
															
				geom->setPrimitiveSet(k, new osg::DrawElementsUInt(osg::PrimitiveSet::PATCHES, (geom->getPrimitiveSet(k)->getNumIndices()-2)*3, &indices[0]));
			}			
		}						
		else	geom->getPrimitiveSet(k)->setMode(osg::PrimitiveSet::PATCHES);						
		geom->getPrimitiveSet(k)->dirty();
		geom->dirtyBound();
		geom->getOrCreateStateSet()->setMode(GL_BLEND, osg::StateAttribute::ON);
		geom->getOrCreateStateSet()->setAttributeAndModes(new osg::BlendFunc);
	}
	geode->getOrCreateStateSet()->setMode(GL_BLEND, osg::StateAttribute::ON);
	geode->getOrCreateStateSet()->setAttributeAndModes(new osg::BlendFunc);

	geode->setCullCallback(new ViewerLightCallback);
	//osg::ref_ptr<osg::PolygonMode> mode(new osg::PolygonMode());
	//mode->setMode(  osg::PolygonMode::FRONT_AND_BACK, osg::PolygonMode::LINE );
	//geode->getOrCreateStateSet()->setAttributeAndModes(mode);
}

void osgCloudyDay::Cloud2D::Create(osg::Geode* geode, osg::Image* perlinImg)
{
	osg::Image* perlinNormalImg = CreateNormalMap(perlinImg);
	Create(geode, perlinImg, perlinNormalImg, perlinImg);
}

void osgCloudyDay::Cloud2D::Create(osg::Geode* geode, osg::Image* perlinImg, 
													 osg::Image* perlinNormalImg)
{
	Create(geode, perlinImg, perlinNormalImg, perlinImg);
}

void osgCloudyDay::Cloud2D::Create(osg::Geode* geode, osg::Image* perlinImg, 
									    osg::Image* perlinNormalImg, 
										osg::Image* perlinHeightImg)
{	
	// model texture
	osg::Texture2D * diffuse = new osg::Texture2D;	
	diffuse->setImage(perlinImg);
	diffuse->setWrap( osg::Texture2D::WRAP_S, osg::Texture2D::MIRROR );
	diffuse->setWrap( osg::Texture2D::WRAP_T, osg::Texture2D::MIRROR );
	diffuse->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
	diffuse->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
	diffuse->setResizeNonPowerOfTwoHint( false );
	diffuse->setUseHardwareMipMapGeneration(true);

	// bump map texture
	osg::Texture2D * normal = new osg::Texture2D;	
	normal->setImage( perlinNormalImg );	
	normal->setWrap( osg::Texture2D::WRAP_S, osg::Texture2D::MIRROR );
	normal->setWrap( osg::Texture2D::WRAP_T, osg::Texture2D::MIRROR );
	normal->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
	normal->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
	normal->setResizeNonPowerOfTwoHint( false );
	normal->setUseHardwareMipMapGeneration(true);

	osg::Texture2D * height = new osg::Texture2D;
	height->setImage(perlinHeightImg);
	height->setWrap( osg::Texture2D::WRAP_S, osg::Texture2D::MIRROR );
	height->setWrap( osg::Texture2D::WRAP_T, osg::Texture2D::MIRROR );
	height->setFilter(osg::Texture::MIN_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
	height->setFilter(osg::Texture::MAG_FILTER, osg::Texture::LINEAR_MIPMAP_LINEAR);
	height->setResizeNonPowerOfTwoHint( false );
	height->setUseHardwareMipMapGeneration(true);

	// Set up the shader stateset 
	osg::StateSet * stateset = geode->getOrCreateStateSet();
	stateset->setTextureAttribute( 0, diffuse );
	stateset->setTextureAttribute( 1, normal );
	stateset->setTextureAttribute( 2, height );
	
	osg::ref_ptr<osg::Program> cloudPerlinProg (new osg::Program);
	
	if(!m_tesselation)
	{
		osg::ref_ptr<osg::Shader> cloudPerlinVertShader(osg::Shader::readShaderFile (osg::Shader::VERTEX, "shaders/perlinCloud.vert"));
		osg::ref_ptr<osg::Shader> cloudPerlinFragShader(osg::Shader::readShaderFile (osg::Shader::FRAGMENT, "shaders/perlinCloud.frag"));
	
		//Binding the box shaders to its program
		cloudPerlinProg->addShader(cloudPerlinVertShader.get());
		cloudPerlinProg->addShader(cloudPerlinFragShader.get());
	}
	else
	{	
		osg::ref_ptr<osg::Shader> cloudPerlinVertShader2(osg::Shader::readShaderFile (osg::Shader::VERTEX,			"shaders/cloud2DHimmel.vert"));
		osg::ref_ptr<osg::Shader> cloudPerlinConShader2(osg::Shader::readShaderFile (osg::Shader::TESSCONTROL,		"shaders/cloud2DHimmel.con"));
		osg::ref_ptr<osg::Shader> cloudPerlinEvaShader2(osg::Shader::readShaderFile (osg::Shader::TESSEVALUATION,	"shaders/cloud2DHimmel.eva"));
		osg::ref_ptr<osg::Shader> cloudPerlinFragShader2(osg::Shader::readShaderFile (osg::Shader::FRAGMENT,		"shaders/cloud2DHimmel.frag"));	

		cloudPerlinProg->addShader(cloudPerlinVertShader2.get());
		cloudPerlinProg->addShader(cloudPerlinConShader2.get());
		cloudPerlinProg->addShader(cloudPerlinEvaShader2.get());
		cloudPerlinProg->addShader(cloudPerlinFragShader2.get());
		cloudPerlinProg->setParameter(GL_PATCH_VERTICES, 3);
	}

	cloudPerlinProg->addBindFragDataLocation("out_color", 0);
	cloudPerlinProg->addBindFragDataLocation("out_color2", 1);	

	stateset->addUniform(new osg::Uniform("color_tex", 0));		
	stateset->addUniform(new osg::Uniform("normal_tex", 1));
	stateset->addUniform(new osg::Uniform("height_tex", 2));
	stateset->addUniform(new osg::Uniform("ModelMatrix", osg::Matrixd()));
	stateset->addUniform(new osg::Uniform("ViewMatrix", osg::Matrixd()));
	stateset->addUniform(new osg::Uniform("NormalMatrix", osg::Matrixd()));
	stateset->addUniform(new osg::Uniform("ProjectionMatrix", osg::Matrixd()));
	stateset->addUniform(new osg::Uniform("LightMatrix", osg::Matrixd()));	

	stateset->addUniform(new osg::Uniform("light_proj_matrix", osg::Matrixd()));	
	stateset->addUniform(new osg::Uniform("light_mv_matrix", osg::Matrixd()));	

	
	//std::cout << "SETUP Uniform" << std::endl;
	stateset->addUniform(new osg::Uniform("transmittanceSampler", 9));			
	stateset->addUniform(new osg::Uniform("skyIrradianceSampler", 10));			
	stateset->addUniform(new osg::Uniform("inscatterSampler", 11));							
	stateset->addUniform(new osg::Uniform("glareSampler", 12));	

	stateset->setTextureAttributeAndModes(9, SkydomeHimmel::m_transmittance);		
	stateset->setTextureAttributeAndModes(10, SkydomeHimmel::m_irradiance);		
	stateset->setTextureAttributeAndModes(11, SkydomeHimmel::m_inscatter);		
	stateset->setTextureAttributeAndModes(12, SkydomeHimmel::m_glare);		
	//std::cout << "SETUP Uniform done" << std::endl;


	stateset->setAttribute(cloudPerlinProg.get());	
	stateset->setMode(GL_BLEND, osg::StateAttribute::ON);
	stateset->setAttributeAndModes(new osg::BlendFunc);
	stateset->setRenderingHint(osg::StateSet::TRANSPARENT_BIN); 
	stateset->setRenderBinDetails(99, "DepthSortedBin");	
}

osg::Image* osgCloudyDay::Cloud2D::CreateNormalMap(osg::Image* img)
{	
	float* fdata = (float*)img->data();	
	int width = img->s();
	int height = img->t();

	int size = 0;
	switch(img->getPixelFormat())
	{
	case GL_RGB: size = 3;
		break;
	case GL_RGBA: size = 4;
		break;
	}

	//Create NormalMap	
	float* du = new float[width*height];
	float* dv = new float[width*height];
	for(int i = 0; i < height*width; i++)
		du[i] = dv[i] = 0.f;	

	float* normalmap = new float[width*height*3];
	for(int i = 0; i < height*width*3; i++)		
		normalmap[i] = 0.f;			

	float* kernel1 = new float[3*3];
	float* kernel2 = new float[3*3];
	kernel1[(0*3)+0] =-1.f;	kernel1[(0*3)+1] =-2.f;	kernel1[(0*3)+2] =-1.f;
	kernel1[(1*3)+0] = 0.f;	kernel1[(1*3)+1] = 0.f;	kernel1[(1*3)+2] = 0.f;
	kernel1[(2*3)+0] = 1.f;	kernel1[(2*3)+1] = 2.f;	kernel1[(2*3)+2] = 1.f;	

	kernel2[(0*3)+0] = kernel1[(0*3)+0];	kernel2[(0*3)+1] = kernel1[(1*3)+0];	kernel2[(0*3)+2] = kernel1[(2*3)+0];
	kernel2[(1*3)+0] = kernel1[(0*3)+1];	kernel2[(1*3)+1] = kernel1[(1*3)+1];	kernel2[(1*3)+2] = kernel1[(2*3)+1];
	kernel2[(2*3)+0] = kernel1[(0*3)+2];	kernel2[(2*3)+1] = kernel1[(1*3)+2];	kernel2[(2*3)+2] = kernel1[(2*3)+2];

	for(int y = 1; y < height-1; y++)
	{
		for(int x = 1; x < width-1; x++)
		{			
			for(int y1 = -1; y1 <= 1; y1++)
			{
				for(int x1 = -1; x1 <= 1; x1++)
				{
					du[(y*height)+x] += (fdata[((y+y1)*height)+(x+x1)]) * 0.5 * kernel1[(y1*3)+x1];
					dv[(y*height)+x] += (fdata[((y+y1)*height)+(x+x1)]) * 0.5 * kernel2[(y1*3)+x1];					
				}
			}

			du[(y*height)+x] = abs(du[(y*height)+x]);
			dv[(y*height)+x] = abs(dv[(y*height)+x]);
		}
	}	

	for(int y = 1; y < height-1; y++)
	{
		for(int x = 1; x < width-1; x++)
		{
			float norm_factor = sqrt(powf(du[(height*y)+x],2.f) +  powf(dv[(height*y)+x],2.f) + 1.f);
			normalmap[(height*y*3)+ (x*3) + 0] = du[(height*y)+x]/norm_factor;
			normalmap[(height*y*3)+ (x*3) + 1] = dv[(height*y)+x]/norm_factor;
			normalmap[(height*y*3)+ (x*3) + 2] = 1.f			 /norm_factor;
			//std::cout << "N: " << normalmap[(height*y*3)+ (x*3) + 0] << " " << normalmap[(height*y*3)+ (x*3) + 1] << " " << normalmap[(height*y*3)+ (x*3) + 2] << std::endl;								  
		}
	}
			
	for(int i = 0; i < height*width*3; i++)			
		normalmap[i] = (normalmap[i]+1.f)/2.f;	
	
	//delete[] fdata;
	//delete[] normalmap;
	delete[] du;
	delete[] dv;	

	osg::Image* normal = new osg::Image();
	normal->allocateImage(width, height, 1,  GL_RGB, GL_FLOAT); 
	normal->setOrigin(osg::Image::BOTTOM_LEFT);
	normal->setImage(width, height, 1, GL_RGB,GL_RGB,GL_FLOAT, (unsigned char*)normalmap,osg::Image::NO_DELETE); 
	return normal;
}