#include "CelluarAutomataGenerator.h"


osgCloudyDay::CelluarAutomataGenerator::CelluarAutomataGenerator(unsigned int width, unsigned int depth, unsigned int height, int type, osg::Vec4 layer) : CloudGenerator(layer), t(0), m_width(width), m_height(height), m_depth(depth), m_elapsedSteps(0), m_time(0.f), tdens(0)
{		
	f_width = (float)(width);
	f_height = (float)(height);
	f_depth = (float)(depth);

	m_category = type;
	//m_category = CloudScene::CT_Cumulus;

	m_positions = (new osg::Vec3Array());
	m_radius = (new osg::Vec3Array());
	m_lifetimes = (new osg::UIntArray());

	bb_min = osg::Vec3(0.f,0.f,0.f);
	bb_max = osg::Vec3(1.f,1.f,1.f);
	windVector = osg::Vec3(0.f,0.f,1.5f);

	SetupArrays();		
	Perform();
	CalcDensity();
}


osgCloudyDay::CelluarAutomataGenerator::~CelluarAutomataGenerator(void)
{
	
	delete[] m_hum[1];
	delete[] m_hum[0];	
	delete[] m_hum;
	delete[] m_act[1];
	delete[] m_act[0];
	delete[] m_act;
	delete[] m_cld[1];
	delete[] m_cld[0];
	delete[] m_cld;
	
	delete[] m_density2[1];
	delete[] m_density2[0];
	delete[] m_density2;

	delete[] m_fact;	
	delete[] m_pact;	
}

void osgCloudyDay::CelluarAutomataGenerator::SetupArrays()
{
	m_hum = new unsigned int*[2];
	m_act = new unsigned int*[2];
	m_cld = new unsigned int*[2];
	m_density2 = new float*[2];

	for(int i = 0; i < 2; i++)
	{
		m_hum[i] = new unsigned int[m_width*m_height*m_depth];
		m_act[i] = new unsigned int[m_width*m_height*m_depth];
		m_cld[i] = new unsigned int[m_width*m_height*m_depth];

		m_density2[i] = new float[m_width*m_height*m_depth];
	}

	m_current_density = new float[m_width*m_height*m_depth];

	m_fact = new unsigned int[m_width*m_height*m_depth];

	m_phum = new float[m_width*m_height*m_depth];
	m_pext = new float[m_width*m_height*m_depth];
	m_pact = new float[m_width*m_height*m_depth];

	DefineInitValues();
}

void osgCloudyDay::CelluarAutomataGenerator::DefineInitValues()
{
	for(int i = 0; i < m_width*m_height*m_depth; i++)
	{
		for(unsigned int j = 0; j < 2; j++)
		{
			m_hum[j][i] = 0;
			m_act[j][i] = 0;
			m_cld[j][i] = 0;
			m_density2[j][i] = 0.f;
		}

		m_current_density[i] = 0.f;
		m_fact[i] = 0;

		m_phum[i] = 0.f;		
		m_pact[i] = 0.f;
	}
	
	for(int z = 0; z < m_height; z++)	
	{
		for(int y = 0; y < m_depth; y++)	
		{
			for(int x = 0; x < m_width; x++)			
			{
				unsigned int index = Index(x,y,z);
				m_pext[index] = (float)exp( -(float)(z)*0.05f );					
			}
		}
	}
}

void osgCloudyDay::CelluarAutomataGenerator::Perform(void)
{		
	t = !t;

	//if( m_elapsedSteps <= 1 )
	{		
		UpdateEllisoides();
		ShapeVolume();
	}

	//CALCULATE STEP	
	Grow();			
	Extrinction();			
	AdvectionWind();
		
	if( m_elapsedSteps <= 1 )
	{				
		//SupplyVapor();
		m_elapsedSteps = 20;		
	}
	else	m_elapsedSteps--;

}

bool osgCloudyDay::CelluarAutomataGenerator::IsCellInSpace(int x, int y, int z)
{
	if(x < 0) return false;
	if(y < 0) return false;
	if(z < 0) return false;
	if(x >= m_width)  return false;
	if(z >= m_height) return false;
	if(y >= m_depth)  return false;
	return true;
}
int osgCloudyDay::CelluarAutomataGenerator::Index(int x, int y, int z)
{
	if(x < 0) x = 0;
	if(y < 0) y = 0;
	if(z < 0) z = 0;
	if(x >= m_width) x = m_width-1;	
	if(z >= m_height) z = m_height-1;	
	if(y >= m_depth) y = m_depth-1;

	return (m_width*m_depth*z)+(m_width*y)+x;
}
float random()
{
	return rand() / (float) RAND_MAX;
}

void osgCloudyDay::CelluarAutomataGenerator::Grow()
{	
	for(int z = 0; z < m_height; z++)
	{
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{
				unsigned int index = Index(x,y,z);
				m_fact[index] = m_act[!t][Index(x+1, y, z)] || m_act[!t][Index(x-1, y, z)] ||
								m_act[!t][Index(x, y+1, z)] || m_act[!t][Index(x, y-1, z)] ||
								m_act[!t][Index(x, y, z+1)] || m_act[!t][Index(x, y, z-1)] ||
								m_act[!t][Index(x+2, y, z)] || m_act[!t][Index(x-2, y, z)] ||
								m_act[!t][Index(x, y+2, z)] || m_act[!t][Index(x, y-2, z)] ||
								m_act[!t][Index(x, y, z+2)] || m_act[!t][Index(x, y, z-2)];	
			}
		}
	}

	for(int z = 0; z < m_height; z++)
	{
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{		
				if(!IsCellInSpace(x,y,z))	continue;

				unsigned int index = Index(x,y,z);
				m_hum[!t][index] =  m_hum[!t][index] && !m_act[!t][index];
				m_cld[!t][index] =  m_cld[!t][index] ||  m_act[!t][index];
				m_act[!t][index] = !m_act[!t][index] &&  m_hum[!t][index] && m_fact[index];
			}
		}
	}
}
void osgCloudyDay::CelluarAutomataGenerator::Extrinction()
{
	for(int z = 0; z < m_height; z++)
	{
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{
				if(!IsCellInSpace(x,y,z))	continue;
				
				int index = Index(x,y,z);
				m_cld[!t][index] = m_cld[!t][index] && (random() > (m_pext[index])); //OTHERWISE, RESET SET TO 0
				m_hum[!t][index] = m_hum[!t][index] || (random() < (m_phum[index])); //OTHERWISE, NEVER SET TO 1
				m_act[!t][index] = m_act[!t][index] || (random() < (m_pact[index]));
			}
		}
	}
}
void osgCloudyDay::CelluarAutomataGenerator::AdvectionWind()
{	
	for(int z = 0; z < m_height; z++)
	{
		int vel = 0;
		if(m_height > m_height/2)	vel = 2;
		else						vel = 1;

		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{			
				if(!IsCellInSpace(x,y,z))		continue;				
								
				unsigned int index = Index(x,y,z);	
				unsigned int index2 = Index(x,y+vel,z);					
				
				if(IsCellInSpace(x,y+vel,z))	
				{
					m_cld[t][index] = m_cld[!t][index2]; 
					m_hum[t][index] = m_hum[!t][index2]; 
					m_act[t][index] = m_act[!t][index2];
				}							
				else
				{
					m_cld[t][index2] = 0; 
					m_hum[t][index2] = 0; 
					m_act[t][index2] = 0;			
				}
			}
		}
	}

	/*for(int z = 0; z < m_height; z++)
	{		
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{			
				unsigned int index = Index(x,y,z);	
				m_cld[!t][index] = m_cld[t][index]; 
				m_hum[!t][index] = m_hum[t][index]; 
				m_act[!t][index] = m_act[t][index];
			}
		}
	}*/
}
void osgCloudyDay::CelluarAutomataGenerator::SupplyVapor()
{
	for(int z = 0; z < m_height; z++)
	{
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{
				if(!IsCellInSpace(x,y,z) )	continue;
				
				unsigned int index = Index(x,y,z);
				m_hum[!t][index] = m_hum[t][index] || ( random() < m_phum[index] );
				m_act[!t][index] = m_hum[t][index] && ( m_act[t][index] || ( random() < m_pact[index] ) );
							
			}
		}
	}
}

void osgCloudyDay::CelluarAutomataGenerator::InitCell( int x, int y, int z, float fProbSeed )
{
	if( !IsCellInSpace( x,y,z ) ) return;
	
	int index = Index(x,y,z);	
			
	
	//Good for Cumulus
	float fCurExtProb =  0.3f * ( 1.f - fProbSeed );
	float fCurHumProb =  0.2f  * fProbSeed;
	float fCurActProb =	 0.2f * fProbSeed;
	/*/	
	/*
	//stratus, very fin 
	float fCurExtProb =  0.01f * ( 1.f - fProbSeed );
	float fCurHumProb =  0.1f  * fProbSeed;
	float fCurActProb =	 0.01f * fProbSeed;
	//*/
	/*
	float fCurExtProb =  0.01f * ( 1.f - fProbSeed );
	float fCurHumProb =  0.1f  * fProbSeed;
	float fCurActProb =	 0.2f * fProbSeed;
	*/
	m_pext[index] *= fCurExtProb;
	m_phum[index] = m_phum[index]*(1.f-fCurHumProb)+fCurHumProb;
	m_pact[index] = m_pact[index]*(1.f-fCurActProb)+fCurActProb;
}

void osgCloudyDay::CelluarAutomataGenerator::AddEllipsoide(osg::Vec3 radius, osg::Vec3 position, float lifetime)
{
	m_positions->push_back(position);
	m_radius->push_back(radius);
	m_lifetimes->push_back(lifetime);
}

bool osgCloudyDay::CelluarAutomataGenerator::IsOutside(osg::Vec3 pos)
{
	if(pos.x() < bb_min.x()) return true;
	if(pos.y() < bb_min.y()) return true;
	if(pos.z() < bb_min.z()) return true;
	
	if(pos.x() > bb_max.x()) return true;
	if(pos.y() > bb_max.y()) return true;
	if(pos.z() > bb_max.z()) return true;
	return false;
}

void osgCloudyDay::CelluarAutomataGenerator::UpdateEllisoides()
{
	osg::Vec3 cellSize(1.f/f_width, 1.f/f_depth, 1.f/f_height);
	osg::Vec3 offset= osg::Vec3(0.f, 0.f, 0.f); 		

	//std::cout << "m_lifetimes->size(): " << m_lifetimes->size() << std::endl;
	if(m_lifetimes->size() > 0) 
	{
		for(int i = m_lifetimes->size()-1; i >= 0; i--)
		{
			m_lifetimes->at(i)--;
			float tmprand = random();
			m_positions->at(i) += osg::Vec3(windVector.x()*cellSize.x()*tmprand,
											windVector.y()*cellSize.y()*tmprand,
											windVector.z()*cellSize.z()*tmprand);

			if(m_lifetimes->at(i) <= 0 || IsOutside(m_positions->at(i)))
			{
				m_lifetimes->erase(m_lifetimes->begin()+i);
				m_radius->erase(m_radius->begin()+i);
				m_positions->erase(m_positions->begin()+i);
			}
		}	
	}
	else
	{
		//windDirection = windDirection + osg::Vec3(0.f, 0.f, 0.2f);
		m_positions->clear();
		m_radius->clear();
		m_lifetimes->clear();		

		
		if(m_category == CloudScene::CT_Stratus || m_category == CloudScene::CT_AltStratus)
		{
			for(int n = 0; n < 5; n++)
			{
				osg::Vec3 Pos = osg::Vec3(random(), random(), 0.2f);

				for(int m = 0; m < 5; m++)
				{
					float life = 20.f*random();
					osg::Vec3 Pos2 = osg::Vec3((random()-0.5f)*0.2f, (random()-0.5f)*0.2f, (random()-0.5f)*0.2f);
					AddEllipsoide(osg::Vec3(cellSize.x()*6.f, cellSize.y()*6.f, cellSize.z()*4.5f), Pos + Pos2 + osg::Vec3(-cellSize.x()*12.f,-cellSize.y()*12.f, 0.f),	life*(random()*0.5f+0.5)); //Stratus
					AddEllipsoide(osg::Vec3(cellSize.x()*6.f, cellSize.y()*6.f, cellSize.z()*4.5f), Pos + Pos2 + osg::Vec3(-cellSize.x()*12.f, cellSize.y()*12.f, 0.f),	life*(random()*0.5f+0.5));
					AddEllipsoide(osg::Vec3(cellSize.x()*6.f, cellSize.y()*6.f, cellSize.z()*4.5f), Pos + Pos2 + osg::Vec3( cellSize.x()*12.f, cellSize.y()*12.f, 0.f),	life*(random()*0.5f+0.5));						
					AddEllipsoide(osg::Vec3(cellSize.x()*6.f, cellSize.y()*6.f, cellSize.z()*4.5f), Pos + Pos2 + osg::Vec3( cellSize.x()*12.f,-cellSize.y()*12.f, 0.f),	life*(random()*0.5f+0.5));									 
					AddEllipsoide(osg::Vec3(cellSize.x()*6.f, cellSize.y()*6.f, cellSize.z()*4.5f), Pos + Pos2,															life*(random()*0.5f+0.5));									 
				}
			}
		}
		else
		{
			for(int n = 0; n < 15; n++)
			{
				float life = 20.f*random()+5.f;
				osg::ref_ptr<osg::Vec3Array> box_triangles(new osg::Vec3Array());

				osg::Vec3 randVer = osg::Vec3(random(), random(), 2.f*cellSize.z());
				osg::Vec3 size;								
				size = osg::Vec3(	cellSize.x()*3.f,
									cellSize.y()*3.f,
									cellSize.z()*3.f);
				AddEllipsoide(size, randVer		+ osg::Vec3(cellSize.x()*-4.f, cellSize.y()*-4.f, -1.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer	+ osg::Vec3(cellSize.x()*-4.f, cellSize.y()*-4.f, -1.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer		+ osg::Vec3(cellSize.x()*-4.f, cellSize.y()* 4.f, -1.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer	+ osg::Vec3(cellSize.x()*-4.f, cellSize.y()* 4.f, -1.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer		+ osg::Vec3(cellSize.x()* 4.f, cellSize.y()* 4.f, -1.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer	+ osg::Vec3(cellSize.x()* 4.f, cellSize.y()* 4.f, -1.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer		 + osg::Vec3(cellSize.x() *4.f, cellSize.y()*-4.f, -1.f*cellSize.z()), life);	
				AddBox(box_triangles, randVer	 + osg::Vec3(cellSize.x() *4.f, cellSize.y()*-4.f, -1.f*cellSize.z()), size);




				AddEllipsoide(size, randVer		+ osg::Vec3(cellSize.x()*-4.f, cellSize.y(), -1.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer	+ osg::Vec3(cellSize.x()*-4.f, cellSize.y(), -1.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer		+ osg::Vec3(cellSize.x()* 4.f, cellSize.y(), -1.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer	+ osg::Vec3(cellSize.x()* 4.f, cellSize.y(), -1.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer		+ osg::Vec3(cellSize.x(), cellSize.y()* 4.f, -1.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer	+ osg::Vec3(cellSize.x(), cellSize.y()* 4.f, -1.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer		 + osg::Vec3(cellSize.x(), cellSize.y()*-4.f, -1.f*cellSize.z()), life);	
				AddBox(box_triangles, randVer	 + osg::Vec3(cellSize.x(), cellSize.y()*-4.f, -1.f*cellSize.z()), size);

				size = osg::Vec3(	cellSize.x()*4.f, 
									cellSize.y()*4.f,
									cellSize.z()*4.f);
				AddEllipsoide(size, randVer,	life);	
				AddBox(box_triangles, randVer, size);

				/*
				size = osg::Vec3(	cellSize.x()*2.f,
									cellSize.y()*2.f,
									cellSize.z()*2.f);
			
				AddEllipsoide(size, randVer+ windDirection + osg::Vec3(cellSize.x()  *-7.f, cellSize.y() * -9.f, -2.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer              + osg::Vec3(cellSize.x()  *-7.f, cellSize.y() * -9.f, -2.f*cellSize.z()), size);

				AddEllipsoide(size, randVer  + windDirection + osg::Vec3(cellSize.x()*-7.f, cellSize.y()* 9.f, -2.f*cellSize.z()),	life);	
				AddBox(box_triangles, randVer				 + osg::Vec3(cellSize.x()*-7.f, cellSize.y()* 9.f, -2.f*cellSize.z()), size);
			
				AddEllipsoide(size, randVer  + windDirection + osg::Vec3(cellSize.x()* 7.f, cellSize.y()* 9.f, -2.f*cellSize.z()),	life);
				AddBox(box_triangles, randVer				 + osg::Vec3(cellSize.x()* 7.f, cellSize.y()* 9.f, -2.f*cellSize.z()), size);

				AddEllipsoide(size, randVer  + windDirection + osg::Vec3(cellSize.x()* 7.f, cellSize.y()*-7.f, -2.f*cellSize.z()),	(life*0.5f+0.5));	
				AddBox(box_triangles, randVer				 + osg::Vec3(cellSize.x()* 7.f, cellSize.y()*-7.f, -2.f*cellSize.z()), size);
				*/

				/*
				size = osg::Vec3(	cellSize.x()*1.5f,
									cellSize.y()*1.5f,
									cellSize.z()*1.5f);
			
				AddEllipsoide(size, randVer   + windDirection + osg::Vec3(cellSize.x()*-6.f, cellSize.y()*-6.f, cellSize.z()*5.f),	life);				
				AddBox(box_triangles, randVer +					osg::Vec3(cellSize.x()*-6.f, cellSize.y()*-6.f, cellSize.z()*11.f), size);

				AddEllipsoide(size, randVer   + windDirection + osg::Vec3(cellSize.x()*-6.f, cellSize.y()* 6.f, cellSize.z()*5.f),	life);	
				AddBox(box_triangles, randVer +					osg::Vec3(cellSize.x()*-6.f, cellSize.y()* 6.f, cellSize.z()*11.f), size);
			
				AddEllipsoide(size, randVer	  + windDirection + osg::Vec3(cellSize.x()* 6.f, cellSize.y()* 6.f, cellSize.z()*5.f),	life);
				AddBox(box_triangles, randVer +					osg::Vec3(cellSize.x()* 6.f, cellSize.y()* 6.f, cellSize.z()*11.f), size);

				AddEllipsoide(size, randVer   + windDirection + osg::Vec3(cellSize.x()* 6.f, cellSize.y()*-6.f, cellSize.z()*5.f),	life);	
				AddBox(box_triangles, randVer				  + osg::Vec3(cellSize.x()* 6.f, cellSize.y()*-6.f, cellSize.z()*11.f), size);
				*/
				/*/
				osg::Vec3 triangle[3];
				triangle[0] = osg::Vec3();
				triangle[1] = osg::Vec3();
				triangle[2] = osg::Vec3();
		
				for(int i = 0; i < 20; i++)
				{
					osg::Vec3 res;

					osg::Vec3* _ray = new osg::Vec3[2];
					_ray[0] = osg::Vec3(0.f, 0.f, 0.f);
				
					_ray[1] = osg::Vec3(
								(frand()-0.5f)*10.f, 
								(frand()-0.5f)*10.f,
								frand()*5.f*((float)i/10.f));									
					_ray[1].normalize();				
					_ray[1] *= 100.f;

					for(int j = 0; j < ((int)box_triangles->size()-2); j+=3)
					{
						triangle[0] = osg::Vec3(box_triangles->at(j+0).x(), box_triangles->at(j+0).y(), box_triangles->at(j+0).z());
						triangle[1] = osg::Vec3(box_triangles->at(j+1).x(), box_triangles->at(j+1).y(), box_triangles->at(j+1).z());
						triangle[2] = osg::Vec3(box_triangles->at(j+2).x(), box_triangles->at(j+2).y(), box_triangles->at(j+2).z());
				
						osg::Vec3 res2;
						int intersection_result = intersect_RayTriangle(_ray, triangle, res2);
						if(intersection_result == 1 && res2.length() > res.length())	res = res2;
					}
						
					float asdf = (random()*2.f+1.f);
					float asdf1 = (random()*4.f+1.f);				
					size = osg::Vec3(cellSize.x()*asdf1, cellSize.y()*asdf1, cellSize.z()*asdf);
			
					osg::Vec3 posHelper = randVer+ windDirection+res+size;			
					if(posHelper.x()+size.x() > 1.f || posHelper.x() < 0.f) continue;
					if(posHelper.y()+size.y() > 1.f || posHelper.y() < 0.f) continue;
					if(posHelper.z()+size.z() > 1.f || posHelper.z() < 0.f) continue;

					AddEllipsoide(size, posHelper,	life);									

					posHelper -= randVer;			
					AddBox(box_triangles, res+size, size);
				}*/
			}
		}
		/*/			
			
			/*
			//Pos = osg::Vec3(0.5f, 0.5f, 0.2f);
			AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3(-cellSize.x()*2.f,-cellSize.y()*2.f, 0.f),	50); //Cumulus
			AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3(-cellSize.x()*2.f, cellSize.y()*2.f, 0.f),	50);
			AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3( cellSize.x()*2.f, cellSize.y()*2.f, 0.f),	50);						
			AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3( cellSize.x()*2.f,-cellSize.y()*2.f, 0.f),	50);									 
			
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f,-cellSize.y()*4.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f, cellSize.y()*4.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f, cellSize.y()*4.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f,-cellSize.y()*4.f, 0.f),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f,-cellSize.y()*4.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f,-cellSize.y()*6.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*6.f,-cellSize.y()*6.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*6.f,-cellSize.y()*4.f, 0.f),	50); //Cumulus

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f, cellSize.y()*4.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f, cellSize.y()*6.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*6.f, cellSize.y()*6.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*6.f, cellSize.y()*4.f, 0.f),	50);

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f, cellSize.y()*4.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f, cellSize.y()*6.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*6.f, cellSize.y()*6.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*6.f, cellSize.y()*4.f, 0.f),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f,-cellSize.y()*4.f, 0.f),	50);							
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f,-cellSize.y()*6.f, 0.f),	50);							
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*6.f,-cellSize.y()*6.f, 0.f),	50);							
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*6.f,-cellSize.y()*4.f, 0.f),	50);							


			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*8.f,-cellSize.y()*8.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*8.f,-cellSize.y()*10.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f,-cellSize.y()*10.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f,-cellSize.y()*8.f, 0.f),	50); //Cumulus			

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*8.f, cellSize.y()*8.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*8.f, cellSize.y()*10.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f, cellSize.y()*10.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f, cellSize.y()*8.f, 0.f),	50);

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*8.f, cellSize.y()*8.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*8.f, cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f, cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f, cellSize.y()*8.f, 0.f),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*8.f,-cellSize.y()*8.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*8.f,-cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f,-cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f,-cellSize.y()*8.f, 0.f),	50);						

			
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f,-cellSize.y()*4.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f,-cellSize.y()*10.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f,-cellSize.y()*10.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f,-cellSize.y()*4.f, 0.f),	50); //Cumulus			

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f, cellSize.y()*4.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*4.f, cellSize.y()*10.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f, cellSize.y()*10.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f, cellSize.y()*4.f, 0.f),	50);

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f, cellSize.y()*4.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f, cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f, cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f, cellSize.y()*4.f, 0.f),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f,-cellSize.y()*4.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*4.f,-cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f,-cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f,-cellSize.y()*4.f, 0.f),	50);						


			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*2.f,-cellSize.y()*2.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*2.f,-cellSize.y()*10.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f,-cellSize.y()*10.f, 0.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f,-cellSize.y()*2.f, 0.f),	50); //Cumulus			

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*2.f, cellSize.y()*2.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*2.f, cellSize.y()*10.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f, cellSize.y()*10.f, 0.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*10.f, cellSize.y()*2.f, 0.f),	50);

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*2.f, cellSize.y()*2.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*2.f, cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f, cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f, cellSize.y()*2.f, 0.f),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*2.f,-cellSize.y()*2.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*2.f,-cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f,-cellSize.y()*10.f, 0.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*10.f,-cellSize.y()*2.f, 0.f),	50);						


			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 1.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 1.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 1.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 1.5f*cellSize.z()),	50);		
			
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f,-cellSize.y()*5.f, 1.5f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f, cellSize.y()*5.f, 1.5f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f, cellSize.y()*5.f, 1.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f,-cellSize.y()*5.f, 1.5f*cellSize.z()),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f,-cellSize.y()*7.f, 1.5f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f, cellSize.y()*7.f, 1.5f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f, cellSize.y()*7.f, 1.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f,-cellSize.y()*7.f, 1.5f*cellSize.z()),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*9.f,-cellSize.y()*9.f, 1.5f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*9.f, cellSize.y()*9.f, 1.5f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*9.f, cellSize.y()*9.f, 1.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*9.f,-cellSize.y()*9.f, 1.5f*cellSize.z()),	50);						



			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 3.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 3.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 3.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, 3.5f*cellSize.z()),	50);		

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f,-cellSize.y()*5.f, 3.5f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f, cellSize.y()*5.f, 3.5f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f, cellSize.y()*5.f, 3.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f,-cellSize.y()*5.f, 3.5f*cellSize.z()),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f,-cellSize.y()*7.f, 3.5f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f, cellSize.y()*7.f, 3.5f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f, cellSize.y()*7.f, 3.5f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f,-cellSize.y()*7.f, 3.5f*cellSize.z()),	50);						



			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*6.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*6.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*6.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*6.f),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f,-cellSize.y()*5.f, 6.f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f, cellSize.y()*5.f, 6.f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f, cellSize.y()*5.f, 6.f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f,-cellSize.y()*5.f, 6.f*cellSize.z()),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f,-cellSize.y()*7.f, 6.f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f, cellSize.y()*7.f, 6.f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f, cellSize.y()*7.f, 6.f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f,-cellSize.y()*7.f, 6.f*cellSize.z()),	50);						



			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*8.f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*8.f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*8.f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*8.f),	50);									

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f,-cellSize.y()*5.f, 8.f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f, cellSize.y()*5.f, 8.f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f, cellSize.y()*5.f, 8.f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f,-cellSize.y()*5.f, 8.f*cellSize.z()),	50);						

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f,-cellSize.y()*7.f, 8.f*cellSize.z()),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f, cellSize.y()*7.f, 8.f*cellSize.z()),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f, cellSize.y()*7.f, 8.f*cellSize.z()),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*2.f, cellSize.y()*2.f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f,-cellSize.y()*7.f, 8.f*cellSize.z()),	50);						



			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*9.5f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*9.5f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*9.5f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*9.5f),	50);					

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f,-cellSize.y()*5.f, cellSize.z()*9.5f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f, cellSize.y()*5.f, cellSize.z()*9.5f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f, cellSize.y()*5.f, cellSize.z()*9.5f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f,-cellSize.y()*5.f, cellSize.z()*9.5f),	50);					

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f,-cellSize.y()*7.f, cellSize.z()*9.5f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f, cellSize.y()*7.f, cellSize.z()*9.5f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f, cellSize.y()*7.f, cellSize.z()*9.5f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f,-cellSize.y()*7.f, cellSize.z()*9.5f),	50);					



			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*11.5f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*11.5f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f, cellSize.y()*3.f, cellSize.z()*11.5f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*3.f,-cellSize.y()*3.f, cellSize.z()*11.5f),	50);					

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f,-cellSize.y()*5.f, cellSize.z()*11.5f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*5.f, cellSize.y()*5.f, cellSize.z()*11.5f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f, cellSize.y()*5.f, cellSize.z()*11.5f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*5.f,-cellSize.y()*5.f, cellSize.z()*11.5f),	50);					

			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f,-cellSize.y()*7.f, cellSize.z()*11.5f),	50); //Cumulus
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3(-cellSize.x()*7.f, cellSize.y()*7.f, cellSize.z()*11.5f),	50);
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f, cellSize.y()*7.f, cellSize.z()*11.5f),	50);						
			if(rand()%5 < 2) AddEllipsoide(osg::Vec3(cellSize.x()*1.5f, cellSize.y()*1.5f, cellSize.z()*1.5f), Pos + osg::Vec3((frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x(),(frand()-0.5f)*cellSize.x()) +osg::Vec3( cellSize.x()*7.f,-cellSize.y()*7.f, cellSize.z()*11.5f),	50);					
			*/
		}	//*/
		
		/*if((rand()%2)==0)	
		{
							AddEllipsoide(osg::Vec3(0.125f/2.f, 0.125f/2.f, 0.125f/2.f), osg::Vec3(0.25f, 0.25f, 0.5f),	50);
			if(rand()%2==0)	AddEllipsoide(osg::Vec3(0.125f/2.f, 0.125f/2.f, 0.125f/2.f), osg::Vec3(0.25f, 0.25f, 0.75f),	50);
		}
		if((rand()%2)==0)	
		{
							AddEllipsoide(osg::Vec3(0.125f/2.f, 0.125f/2.f, 0.125f/2.f), osg::Vec3(0.25f, 0.75f, 0.5f),	50);
			if(rand()%2==0)	AddEllipsoide(osg::Vec3(0.125f/2.f, 0.125f/2.f, 0.125f/2.f), osg::Vec3(0.25f, 0.75f, 0.75f),	50);
		}
		if((rand()%2)==0)	
		{
							AddEllipsoide(osg::Vec3(0.125f, 0.125f, 0.125f), osg::Vec3(0.75f, 0.25f, 0.5f),	50);
			if(rand()%2==0)	AddEllipsoide(osg::Vec3(0.125f, 0.125f, 0.125f), osg::Vec3(0.75f, 0.25f, 0.75f),	50);
		}
		
		if((rand())%2==0)	
		{			
							AddEllipsoide(osg::Vec3(0.125f, 0.125f, 0.125f), osg::Vec3(0.75f, 0.75f, 0.5f),	50);
			if(rand()%2==0)	AddEllipsoide(osg::Vec3(0.125f, 0.125f, 0.125f), osg::Vec3(0.75f, 0.75f, 0.75f),	50);
		}*/

		//AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.2f), osg::Vec3(0.75f, 0.25f, 0.7f),	5);
		

		/*/
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.25f+0.25f*(frand()-0.5f), 0.25f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);				
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.25f+0.25f*(frand()-0.5f), 0.75f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.25f+0.25f*(frand()-0.5f), 0.50f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);

		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.5f+0.25f*(frand()-0.5f), 0.25f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);				
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.5f+0.25f*(frand()-0.5f), 0.75f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.5f+0.25f*(frand()-0.5f), 0.50f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);		
		
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f),
			osg::Vec3(0.75f+0.25f*(frand()-0.5f), 0.25f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);				
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.75f+0.25f*(frand()-0.5f), 0.75f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);
		if(rand() % 4 == 0) AddEllipsoide(osg::Vec3(0.2f, 0.2f, 0.1f), 
			osg::Vec3(0.75f+0.25f*(frand()-0.5f), 0.50f+0.25f*(frand()-0.5f), 0.5f+0.5f*frand()), 5);

		/*
		AddEllipsoide(osg::Vec3(0.0625f*(frand()/0.5f+0.5f), 0.0625f*(frand()/0.5f+0.5f), 0.0625f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.25f, 0.25f, 0.25f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f), 5);
		
		AddEllipsoide(osg::Vec3(0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.35f, 0.25f, 0.25f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);		
		AddEllipsoide(osg::Vec3(0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.15f, 0.25f, 0.25f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);
		AddEllipsoide(osg::Vec3(0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.25f, 0.25f, 0.15f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);
		AddEllipsoide(osg::Vec3(0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f), 0.0625f*0.5f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.25f, 0.25f, 0.35f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);

		AddEllipsoide(osg::Vec3(0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.35f, 0.35f, 0.25f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);		
		AddEllipsoide(osg::Vec3(0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.15f, 0.35f, 0.25f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);
		AddEllipsoide(osg::Vec3(0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.25f, 0.35f, 0.15f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);
		AddEllipsoide(osg::Vec3(0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.25f, 0.35f, 0.35f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);
		
		AddEllipsoide(osg::Vec3(0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f), 0.0625f*0.25f*(frand()/0.5f+0.5f)), 
			osg::Vec3(0.25f, 0.50f, 0.25f)+osg::Vec3(frand()*0.2f, frand()*0.2f, frand()*0.2f)*(frand()-0.5f), 5);		
	}
	//*/
}

void osgCloudyDay::CelluarAutomataGenerator::ShapeVolume()
{	
	bool setto0=false;
	for(int z = 0; z < m_height; z++)
	{
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{
				bool setted=false;
				for(unsigned int i = 0; i < m_radius->size() && !setted; i++)
				{
					if(m_lifetimes->at(i) > 0)
					{
						osg::Vec3 pos = m_positions->at(i);
						pos.x() = pos.x()*f_width;//  / (bb_max.x()-bb_min.x());
						pos.y() = pos.y()*f_depth;//  / (bb_max.y()-bb_min.y());
						pos.z() = pos.z()*f_height;// / (bb_max.z()-bb_min.z());

						osg::Vec3 rad = m_radius->at(i);
						rad.x() = rad.x()*f_width;//  / (bb_max.x()-bb_min.x());
						rad.y() = rad.y()*f_depth;//  / (bb_max.y()-bb_min.y());
						rad.z() = rad.z()*f_height;// / (bb_max.z()-bb_min.z());				

						float fx = (float)(x)-pos.x();
						float fy = (float)(y)-pos.y();
						float fz = (float)(z)-pos.z();
						
						if( -rad.x() < fx && fx < rad.x() && 
							-rad.y() < fy && fy < rad.y() && 
							-rad.z() < fz && fz < rad.z())
						{
							float distance  = (fx*fx) / (rad.x()*rad.x()) 
											+ (fy*fy) / (rad.y()*rad.y()) 
											+ (fz*fz) / (rad.z()*rad.z());
							if(distance < 1.f)
							{
								setted=true;
								InitCell( x, y, z, (float)exp(-distance));
							}
						}
					}
				}
				if(!setted && IsCellInSpace(x,y,z))
				{
					int index = Index(x,y,z);	

					m_pext[index] = 0.f;
					m_phum[index] = 0.f;
					m_pact[index] = 0.f;	
					m_cld[!t][index] = 0;
					setto0=true;
				}
			}
		}
	}

	if(!setto0) std::cerr << "NET SET TO 0" << std::endl;
}

void osgCloudyDay::CelluarAutomataGenerator::CalcDensity()
{	
	tdens = !tdens;
	for(int z = 0; z < m_height; z++)
	{
		for(int y = 0; y < m_depth; y++)
		{
			for(int x = 0; x < m_width; x++)
			{
				if(!IsCellInSpace(x,y,z) ) continue;
				
				int index = Index(x,y,z);
				m_density2[tdens][index] = 0.f;
					
				for(int p = z-1; p<=z+1; p++)
					for(int q = y-1; q<=y+1; q++)
						for(int r = x-1; r<=x+1; r++)
							m_density2[tdens][index] += m_cld[t][Index(r, q, p)];
				m_density2[tdens][index] /= 27.0f;					
				
			}
		}
	}
}

float osgCloudyDay::CelluarAutomataGenerator::GetCellDensity( int i, int j, int k )
{
	int index = Index( i, j, k );	
	return m_current_density[index];
}

float osgCloudyDay::CelluarAutomataGenerator::GetPointDensity(osg::Vec3 Point)	
{
	osg::Vec3 PointTex = osg::Vec3(Point.x(), 
								   Point.y(),  
								   Point.z());

	int i = (int)PointTex.x();
	int j = (int)PointTex.y();
	int k = (int)PointTex.z();
	float r = PointTex.x() - (float)(i);
	float s = PointTex.y() - (float)(j);
	float t = PointTex.z() - (float)(k);
	
	// get the densities of 8 points around the point.
	float d0 = GetCellDensity( i,   j,   k	);
	float d1 = GetCellDensity( i,	j,   k+1);
	float d2 = GetCellDensity( i,   j+1, k	);
	float d3 = GetCellDensity( i,	j+1, k+1);
	float d4 = GetCellDensity( i+1, j,   k	);
	float d5 = GetCellDensity( i+1, j,   k+1);
	float d6 = GetCellDensity( i+1, j+1, k	);
	float d7 = GetCellDensity( i+1, j+1, k+1);

	// interpolate densities
	float z01 = (d1 - d0)*t + d0;
	float z23 = (d3 - d2)*t + d2;
	float z45 = (d5 - d4)*t + d4;
	float z67 = (d7 - d6)*t + d6;
	float x0145 = (z45 - z01)*r + z01;
	float x2367 = (z67 - z23)*r + z23;
	float result = ((x2367 - x0145)*s + x0145);
	return result;
}

void osgCloudyDay::CelluarAutomataGenerator::Initialize()
{
	randomPos = new osg::Vec3[m_height*m_depth*m_width];
	for(int i = 0; i < (m_height*m_depth*m_width); i++)
	{
		randomPos[i] = osg::Vec3(random()*2.f-1.f, random()*2.f-1.f, random()*2.f-1.f);
		randomPos[i].normalize();
	}

	m_iterFloatRand=0;
	for(int i = 0; i < 5; i++)	
	{
		Perform(); //ACHTUNG; BEI MANCHEN ITERATIONEN GEHT ES NICHT (WENN DURCH 5 TEILBAR)		
	}
	CalcDensity();
	m_time += 0.05f;

	osg::Vec3 size(15.f, 15.f, 15.f);
	int num_boxes = 0;	
	for(unsigned int i = 0; i < m_radius->size(); i++)
	{
		osg::Vec3 position = m_positions->at(i);

		osg::Vec3 rad = m_radius->at(i);
				  rad.x() = rad.x()*(f_width)   / (bb_max.x()-bb_min.x());
				  rad.y() = rad.y()*(f_depth)   / (bb_max.y()-bb_min.y());
				  rad.z() = rad.z()*(f_height)  / (bb_max.z()-bb_min.z());				
		
		osg::Vec3	pos = m_positions->at(i);
					pos.x() = pos.x()*(f_width)  / (bb_max.x()-bb_min.x());
					pos.y() = pos.y()*(f_depth)  / (bb_max.y()-bb_min.y());
					pos.z() = pos.z()*(f_height) / (bb_max.z()-bb_min.z());
		
		osg::Vec3 sizes = osg::Vec3(1.f, 1.f, 1.f);
		osg::Vec3 randVer = m_middlepoint+osg::Vec3(-1000.f, -1000.f, 2000.f);

		int max_z = 0;
		
		for(int z = -rad.z(); z < rad.z(); z+=1)		//height
		{	
			for(int y = -rad.y(); y < rad.y(); y+=1)	//depth
			{
				for(int x = -rad.x(); x < rad.x(); x+=1)	//width
				{	
					/*
					for(int z1 = 0; z1 < 2; z1++)
					{
						for(int y1 = 0; y1 < 2; y1++)
						{
							for(int x1 = 0; x1 < 2; x1++)
							{
								
								osg::Vec3 p = osg::Vec3((float)x + (0.25f+0.5f*(float)(x1)), 
														(float)y + (0.25f+0.5f*(float)(y1)), 
														(float)z + (0.25f+0.5f*(float)(z1)));
					/*/
								osg::Vec3 p = osg::Vec3((float)x, 
														(float)y, 
														(float)z);
								//*/
								float density = GetPointDensity(osg::Vec3(p.x(),p.y(),p.z()));										

								if(density >= 0.005f)
								{					
									//std::cout << "DENSITY: " << density << std::endl;					
									osg::Vec3 posHelper = randVer + osg::Vec3(	p.x()*size.x(), 
																				p.y()*size.y(), 
																				p.z()*size.z());			

									osg::Vec3 jitter_sizes;
									jitter_sizes.x() = size.x() / sizes.x(); 
									jitter_sizes.y() = size.y() / sizes.y(); 
									jitter_sizes.z() = size.z() / sizes.z(); 
												
									max_z = std::max(max_z, z);

									AddCloudBox(num_boxes, posHelper/*+randomPos[x+y*m_width+z*m_width*m_depth]*500.f*/, sizes, size, jitter_sizes, false, 300.f, osg::Vec4(5.f, 10.f, 5.f, 10.f), osg::Vec4(m_density.x(), m_density.y(), m_density.z(), density*4.5f*m_density.w()));
									num_boxes++;
									AddBox(dBoxes_vertices, posHelper, size);
								}
							/*}
						}
					}*/
				}
			}
		}

		dBoxes_indices.resize(dBoxes_vertices->size());
		for(unsigned int i = 0; i < dBoxes_vertices->size(); i++)	dBoxes_indices[i] = i;		

		dBoxes_geometry->setVertexArray (dBoxes_vertices.get());
		dBoxes_geometry->addPrimitiveSet(new osg::DrawElementsUShort(osg::PrimitiveSet::TRIANGLES, dBoxes_vertices->size(), &dBoxes_indices[0]));

		dBoxes_geometry->getPrimitiveSet(0)->getDrawElements()->setDataVariance(osg::Object::STATIC);
		dBoxes_geometry->setUseDisplayList(false);
		dBoxes->addDrawable(dBoxes_geometry.get());
		dBoxes->getOrCreateStateSet()->setMode(GL_CULL_FACE, osg::StateAttribute::OFF );
		dBoxes->setCullingActive(false);			
	}
	
	ex->AddCloud(m_cloud->GetVertices(), m_cloud->GetRotation(), m_cloud->GetColor(), m_cloud->GetCenter(), m_cloud->GetIds(), m_cloud->GetBoxCenters(), num_boxes);
}

bool osgCloudyDay::CelluarAutomataGenerator::Update()
{	
	float multi_dens = 1.f;
	if(m_category != CloudScene::CT_Stratus && m_category != CloudScene::CT_AltStratus )
		multi_dens *=2.0f;

	m_iterFloatRand=0;
	m_cloud->ClearBoundingBox();
	m_cloud->GetVertices()->clear();
	m_cloud->GetRotation()->clear();
	m_cloud->GetCenter()->clear();
	m_cloud->GetIds()->clear();
	m_cloud->GetColor()->clear();
	m_cloud->GetBoxCenters()->clear();
	m_cloud->GetNumberOfIndices()=0;	

	m_time += 0.1f;
	Interpolate(m_time);

	int num_boxes = 0;	

	osg::Vec3 sizes = osg::Vec3(1.f, 1.f, 1.f);
	osg::Vec3 size(100.f, 100.f, 100.f);
	osg::Vec3 randVer = m_middlepoint+osg::Vec3(-size.x()*64.f, -size.y()*64.f, 2000.f);	

	int n = 0;
#if 1
	for(int z = 0; z < m_height; z+=1)		//height
	{	
		for(int y = 0; y < m_depth; y+=1)	//depth
		{
			for(int x = 0; x < m_width; x+=1)	//width
			{	
				/*
				for(int z1 = 0; z1 < 2; z1++)
				{
					for(int y1 = 0; y1 < 2; y1++)
					{
						for(int x1 = 0; x1 < 2; x1++)
						{

							osg::Vec3 p = osg::Vec3((float)x + (random()*-0.5f+random()*(float)(x1)), 
													(float)y + (random()*-0.5f+random()*(float)(y1)), 
													(float)z + (random()*-0.5f+random()*(float)(z1)));
						/*/		
							osg::Vec3 p = osg::Vec3((float)x+randomPos[x+y*m_width+z*m_width*m_depth].x()-0.5f, 
													(float)y+randomPos[x+y*m_width+z*m_width*m_depth].y()-0.5f, 
													(float)z+randomPos[x+y*m_width+z*m_width*m_depth].z()-0.5f);
													//*/
							float density = GetPointDensity(osg::Vec3(p.x(),p.y(),p.z()));				
				
							n++;

							if(density >= 0.005f)
							{								
								//if(density > 0.0f) std::cout << "DENSITY: " << density << std::endl;					
								osg::Vec3 posHelper = randVer + osg::Vec3(	p.x() * size.x(), 
																			p.y() * size.y(), 
																			p.z() * size.z());			
																

								osg::Vec3 jitter_sizes;

								m_iterFloatRand=0;					
								AddCloudBox(num_boxes, posHelper/*randomPos[x+y*m_width+z*m_width*m_depth]*500.f*/, sizes, size, jitter_sizes, false, 200.f, osg::Vec4(5.f, 10.f, 5.f, 10.f), osg::Vec4(m_density.x(), m_density.y(), m_density.z(), density*multi_dens*1.f*m_density.w()));
								num_boxes++;
								//AddBox(dBoxes_vertices, posHelper, size);
							}
					/*	}
					}
				}*/
			}
		}
	}	
	//EliminateRedudantSprites();
#else	
	for(unsigned int i = 0; i < m_radius->size(); i++)
	{
		osg::Vec3 position = m_positions->at(i);

		osg::Vec3 rad = m_radius->at(i);
				  rad.x() = rad.x()*(float)(m_width)   / (bb_max.x()-bb_min.x());
				  rad.y() = rad.y()*(float)(m_depth)   / (bb_max.y()-bb_min.y());
				  rad.z() = rad.z()*(float)(m_height)  / (bb_max.z()-bb_min.z());				
		
		osg::Vec3	pos = m_positions->at(i);
					pos.x() = pos.x()*(float)(m_width)  / (bb_max.x()-bb_min.x());
					pos.y() = pos.y()*(float)(m_depth)  / (bb_max.y()-bb_min.y());
					pos.z() = pos.z()*(float)(m_height) / (bb_max.z()-bb_min.z());
		
		
		
		for(int z = -rad.z(); z < rad.z(); z+=1)		//height
		{	
			for(int y = -rad.y(); y < rad.y(); y+=1)	//depth
			{
				for(int x = -rad.x(); x < rad.x(); x+=1)	//width
				{	
					osg::Vec3 p = osg::Vec3((pos.x()+(float)x)/(float)m_width, 
											(pos.y()+(float)y)/(float)m_depth, 
											(pos.z()+(float)z)/(float)m_height);
					float density = GetPointDensity(osg::Vec3(x,y,z));					

					osg::Vec3 size(250.f, 250.f, 250.f);

					//if(density > 0.0f) std::cout << "DENSITY: " << density << std::endl;					
					osg::Vec3 posHelper = randVer + osg::Vec3(	p.x()*size.x()*(float)(m_width), 
																p.y()*size.y()*(float)(m_depth), 
																p.z()*size.z()*(float)(m_height));

					/*osg::Vec3 size = osg::Vec3(	(0.5f*frand()+0.5f)*200.f,
												(0.5f*frand()+0.5f)*200.f,
												(0.5f*frand()+0.5f)*200.f);	*/					

					osg::Vec3 jitter_sizes;
					jitter_sizes.x() = size.x() / sizes.x(); 
					jitter_sizes.y() = size.y() / sizes.y(); 
					jitter_sizes.z() = size.z() / sizes.z(); 

					if(density >= 0.124f)
					{
						//std::cout << "ADD BOX: "<< p.x() << " " << p.y() << " " << p.z() << std::endl;						
						AddCloudBox(num_boxes, posHelper, sizes, size, jitter_sizes, false, 200.f, osg::Vec4(10.f, 20.f, 10.f, 20.f), density*1.5f);
						num_boxes++;
						AddBox(dBoxes_vertices, posHelper, size);
					}				
				}
			}
		}
		//std::cout << "NUM BOXES: " << num_boxes << std::endl;
	}	
#endif

	return true;
}

void osgCloudyDay::CelluarAutomataGenerator::Interpolate(float& time)
{
	//time*=2.f;
	if( time >= 1.0 ) 
	{       		
		// Create the next DensitySpace of the next phase		
		Perform();		
		CalcDensity();	

		time=0.f;		
	}

 	for(int z = 0; z < m_height; z++ )
 	{
 		for(int y = 0; y < m_depth; y++)
 		{
 			for(int x = 0; x < m_width; x++ )
 			{
 				if(!IsCellInSpace(x,y,z) ) continue;				
					
				unsigned int index = Index(x,y,z);							
 				m_current_density[index] = ((float)(1.0f-time) * m_density2[!tdens][index]) + 
					 									 (time * m_density2[tdens][index]);			
 			}
 		}
 	}
}