#version 330 compatibility 
uniform mat4 ProjectionMatrix;
uniform mat4 ViewMatrix;
uniform mat4 ModelMatrix;
uniform mat4 light_proj_matrix;
uniform mat4 light_mv_matrix;

// uniform shader-parameters
uniform vec3 v3InvWavelength;	// 1 / pow(wavelength, 4) for the red, green, and blue channels
uniform float fOuterRadius;		// The outer (atmosphere) radius
uniform float fOuterRadius2;	// fOuterRadius^2
uniform float fInnerRadius;		// The inner (planetary) radius
uniform float fInnerRadius2;	// fInnerRadius^2
uniform float fKrESun;			// Kr * ESun
uniform float fKmESun;			// Km * ESun
uniform float fKr4PI;			// Kr * 4 * PI
uniform float fKm4PI;			// Km * 4 * PI
uniform float fScale;			// 1 / (fOuterRadius - fInnerRadius)
uniform float fScaleDepth;		// The scale depth (i.e. the altitude at which the atmosphere's average density is found)
uniform float fScaleOverScaleDepth;	// fScale / fScaleDepth

uniform mat4 osg_ViewMatrix;
uniform mat4 osg_ModelViewMatrix;

layout (location = 0) in vec4 in_position;
layout (location = 1) in vec4 in_texCoord;
layout (location = 5) in vec4 in_normal;
layout (location = 6) in vec4 in_tangent;
layout (location = 7) in vec4 in_binormal;

out float ex_lightdistance;
out vec2 ex_texCoord;
out vec4 ex_shadowCoord;

out vec4 ambientColor;
out vec4 diffuseColor;
out vec4 specularColor;

out vec3 normalEye;
out vec3 lightDir;
out vec3 positionEye;

out vec4 ex_secondary_color;
out vec4 ex_front_color;
out vec3 ex_v3Direction;

const int nSamples = 2000;
const float fSamples = 2000.0;

out vec3 lightVec;
out vec3 eyeVec;


//Ground to space
float PI = 3.147;
float SkydomeRadius = 1000000.0;
float InnerRadius = 1.0;
float OuterRadius = 1.025;
float InnerRadius2=pow(InnerRadius,2.0);
float OuterRadius2=pow(OuterRadius,2.0);
float ESun = 10.0;
float Km = 0.015;
float Kr = Km/2.0;
float KrESun =  Kr * ESun;
float KmESun =  Km * ESun;
float Kr4PI = Kr * 4.0 * PI;
float Km4PI = Km * 4.0 * PI;
float Scale = 1.0/(OuterRadius - InnerRadius);	
float ScaleDepth = (OuterRadius - InnerRadius) / 2.0f;
float ScaleOverScaleDepth = Scale / ScaleDepth;
vec3 InvWavelength = vec3(0.65f, 0.57f, 0.475f);
float scale(float fCos)
{
	float x = 1.0 - fCos;
	return ScaleDepth * 1.0 * exp(-0.00287 + x*(0.459 + x*(3.83 + x*(-6.80 + x*5.25))));
}

float getNearIntersection(vec3 v3Pos, vec3 v3Ray, float fDistance2, float fRadius2)
{
   float B = 2.0 * dot(v3Pos, v3Ray);
   float C = fDistance2 - fRadius2;
   float fDet = max(0.0, B*B - 4.0 * C);
   return 0.5 * (-B - sqrt(fDet));
}

void main(void)
{ 
	mat4 Model = inverse(osg_ViewMatrix)*gl_ModelViewMatrix;

	vec4 lightPosition = inverse(light_mv_matrix) * vec4(0.0,0.0,0.0,1.0);

	vec4 lightAmbientColor=vec4(0.1);
	vec4 lightDiffuseColor=vec4(1.0);
	vec4 lightSpecularColor=vec4(1.0);

	vec4 materialAmbientColor=vec4(1.0);
	vec4 materialDiffuseColor=vec4(1.0);
	vec4 materialSpecularColor=vec4(0.1);
	
	vec4 cameraPos = inverse(osg_ViewMatrix) * vec4(0.0,0.0,0.0,1.0);
	
	ambientColor = materialAmbientColor * lightAmbientColor;
	diffuseColor = materialDiffuseColor * lightDiffuseColor;
	specularColor = materialSpecularColor * lightSpecularColor;
	
	mat4 mvpMatrix = gl_ProjectionMatrix * gl_ModelViewMatrix;	
	vec4 worldPos = Model * in_position;

	positionEye = (osg_ViewMatrix*worldPos).xyz;
	
	normalEye = normalize((osg_ViewMatrix * Model * vec4(in_normal.xyz, 0.0)).xyz);
	lightDir = normalize((lightPosition - worldPos).xyz);

	
	ex_shadowCoord = light_proj_matrix * light_mv_matrix * worldPos;
	ex_texCoord = in_texCoord.xy;
	gl_Position = mvpMatrix * in_position;	

	ex_lightdistance = abs((light_proj_matrix * light_mv_matrix * worldPos).z); 
	
	mat3 tbnMatrix = mat3(in_tangent.x, in_binormal.x, in_normal.x,
	                      in_tangent.y, in_binormal.y, in_normal.y,
	                      in_tangent.z, in_binormal.z, in_normal.z);
	
	mat4 invModelMatrix = inverse(ModelMatrix);
	eyeVec = tbnMatrix*(invModelMatrix * vec4(cameraPos.xyz - worldPos.xyz,1.0)).xyz;
	lightVec = tbnMatrix*(invModelMatrix * vec4(lightDir,1.0)).xyz;
	
	

	
	vec3 v3CameraPos = (inverse(osg_ViewMatrix) * vec4(0.0, 0.0, 0.0, 1.0)).xyz;		
	//vec3 v3CameraPos = ( vec4(0.0, 0.0, 0.0, 1.0)).xyz;		
	vec3 vecCamera = v3CameraPos;		
	vecCamera /= SkydomeRadius;	
	vecCamera *= InnerRadius;	
	vecCamera.z += InnerRadius;//*1.01250;	
	
	float fCameraHeight = length(vecCamera);
	float fCameraHeight2 = pow(fCameraHeight, 2.f);	
	
	vec4 lightP = vec4(lightPosition.xyz, 1.0);
	vec3 lightPos = ((lightP)).xyz;		
	lightPos = normalize(lightPos);
	
	
	// Get the ray from the camera to the vertex, and its length (which is the far point of the ray passing through the atmosphere)
	vec3 v3Pos = (Model*in_position).xyz;		
	vec3 Position = v3Pos;
	v3Pos /= SkydomeRadius;		
	v3Pos *= InnerRadius;	
	v3Pos.z += InnerRadius;//*1.25;	
	
	
	vec3 v3Ray = v3Pos-vecCamera;
	float fFar = length(v3Ray);
	v3Ray /= fFar;

	float near = 0.0; //getNearIntersection(v3Pos, v3Ray, length(v3Ray), InnerRadius2);
	
	// Calculate the ray's starting position, then calculate its scattering offset
	vec3 v3Start = vecCamera;// + (v3Ray * near);		
	fFar -= near;
		
	vec3 v3FrontColor = vec3(0.0);
	vec3 v3Attenuate = vec3(1.0);
	
	if(v3Pos.z < vecCamera.z)
	{		
		float fDepth = exp(1.0/ScaleDepth *(InnerRadius - fCameraHeight));		// exp(fScaleOverScaleDepth * (InnerRadius - fCameraHeight));//
		float fLightAngle = dot(lightPos.xyz, v3Pos) / length(v3Pos);
		float fCameraAngle = dot(-v3Ray, v3Pos) / length(v3Pos);
		float fCameraScale = scale(fCameraAngle);
		float fLightScale = scale(fLightAngle);
		float fCameraOffset = fDepth*fCameraScale;		

		// Initialize the scattering loop variables
		float fSampleLength = fFar / fSamples;
		float fScaledLength = fSampleLength * Scale;
		vec3 v3SampleRay = v3Ray * fSampleLength;	
		vec3 v3SamplePoint = v3Start + v3SampleRay * 0.5;

		float fHeight = length(v3SamplePoint);												//OK
		float fStartAngle = dot(v3Ray, v3Start) / fHeight;
		float fStartOffset = fDepth*scale(fStartAngle);
			
		// Now loop through the sample rays	
		for(int i=0; i<nSamples; i++)
		{
			float fHeight = length(v3SamplePoint);												//OK
			float fDepth = exp(1.0/ScaleDepth * (InnerRadius - fHeight));					//OK		
			
			float fScatter = fDepth*(scale(fLightAngle) + scale(fCameraAngle)) - fCameraOffset;
			
			v3Attenuate = exp(-fScatter * (InvWavelength * Kr4PI + Km4PI));				//OK				
			v3FrontColor += v3Attenuate* (fDepth * fScaledLength);		
			v3SamplePoint += v3SampleRay;
		}
	}
	else
	{
		// Calculate the ray's starting position, then calculate its scattering offset
		vec3 v3Start = vecCamera;
		float fHeight = length(v3Start);
		float fDepth = exp(1.0/ScaleDepth * (InnerRadius - fCameraHeight));
		float fStartAngle = dot(v3Ray, v3Start) / fHeight;
		float fStartOffset = fDepth*scale(fStartAngle);

		// Initialize the scattering loop variables
		float fSampleLength = fFar / fSamples;
		float fScaledLength = fSampleLength * Scale;
		vec3 v3SampleRay = v3Ray * fSampleLength;		
		vec3 v3SamplePoint = v3Start + (v3SampleRay * 0.5);

		// Now loop through the sample rays	
		for(int i=0; i<nSamples; i++)
		{
			float fHeight = length(v3SamplePoint);													//OK
			float fDepth = exp(1.0/ScaleDepth * (InnerRadius - fHeight));					//OK
			
			float fLightAngle = dot(lightPos.xyz, v3SamplePoint.xyz) / fHeight;								//OK
			float fCameraAngle = dot(v3Ray, v3SamplePoint) / fHeight;								//OK
			float fScatter = (fDepth*(scale(fLightAngle) - scale(fCameraAngle))) + fStartOffset;	//OK
			
			v3Attenuate = exp(-fScatter * (InvWavelength * Kr4PI + Km4PI));				//OK				
			v3FrontColor += v3Attenuate* (fDepth * fScaledLength);		
			v3SamplePoint += v3SampleRay;
		}
	}

	// Finally, scale the Mie and Rayleigh colors and set up the varying variables for the pixel shader	
	ex_secondary_color.w = 1.0;
	ex_secondary_color.xyz = v3Attenuate*0.750;
	ex_front_color.w = 1.0;	
	ex_front_color.rgb = v3FrontColor * (v3InvWavelength  * KrESun+ KmESun);
	ex_v3Direction = -vecCamera+v3Pos;	
}
