#version 400 core

#define OPTIMIZE
//#define HORIZON_HACK
#define ATMO_FULL


#define SUN_INTENSITY 100.0
#define PLANET_RADIUS 6360000.0

#define TRANSMITTANCE_NON_LINEAR
#define INSCATTER_NON_LINEAR

// ----------------------------------------------------------------------------
// PHYSICAL MODEL PARAMETERS
// ----------------------------------------------------------------------------

const float SCALE = 1000.0;

const float Rg = 6360.0 * SCALE;
const float Rt = 6420.0 * SCALE;
const float RL = 6421.0 * SCALE;

const float AVERAGE_GROUND_REFLECTANCE = 0.1;

// Rayleigh
const float HR = 8.0 * SCALE;
const vec3 betaR = vec3(5.8e-3, 1.35e-2, 3.31e-2) / SCALE;

// Mie
// DEFAULT
const float HM = 1.2 * SCALE;
const vec3 betaMSca = vec3(4e-3) / SCALE;
const vec3 betaMEx = (vec3(4e-3) / SCALE) / 0.9;
const float mieG = 0.8;
// CLEAR SKY
/*const float HM = 1.2 * SCALE;
const vec3 betaMSca = vec3(20e-3) / SCALE;
const vec3 betaMEx = betaMSca / 0.9;
const float mieG = 0.76;*/
// PARTLY CLOUDY
/*const float HM = 3.0 * SCALE;
const vec3 betaMSca = vec3(3e-3) / SCALE;
const vec3 betaMEx = betaMSca / 0.9;
const float mieG = 0.65;*/

const float M_PI = 3.141592657;

// ----------------------------------------------------------------------------
// NUMERICAL INTEGRATION PARAMETERS
// ----------------------------------------------------------------------------

const int TRANSMITTANCE_INTEGRAL_SAMPLES = 500;
const int INSCATTER_INTEGRAL_SAMPLES = 50;
const int IRRADIANCE_INTEGRAL_SAMPLES = 32;
const int INSCATTER_SPHERICAL_INTEGRAL_SAMPLES = 16;

// ----------------------------------------------------------------------------
// PARAMETERIZATION OPTIONS
// ----------------------------------------------------------------------------

const int TRANSMITTANCE_W = 256;
const int TRANSMITTANCE_H = 64;

const int SKY_W = 64;
const int SKY_H = 16;

const int RES_R = 32;
const int RES_MU = 128;
const int RES_MU_S = 32;
const int RES_NU = 8;


// ----------------------------------------------------------------------------
// PARAMETERIZATION FUNCTIONS
// ----------------------------------------------------------------------------

uniform sampler2D transmittanceSampler;

uniform sampler2D skyIrradianceSampler;

uniform sampler3D inscatterSampler;

uniform sampler2D glareSampler;

float getHdrExposure() {
    return 0.4;
}

vec3 hdr(vec3 L) {
#ifndef NOHDR
    L = L * getHdrExposure();
    L.r = L.r < 1.413 ? pow(L.r * 0.38317, 1.0 / 2.2) : 1.0 - exp(-L.r);
    L.g = L.g < 1.413 ? pow(L.g * 0.38317, 1.0 / 2.2) : 1.0 - exp(-L.g);
    L.b = L.b < 1.413 ? pow(L.b * 0.38317, 1.0 / 2.2) : 1.0 - exp(-L.b);
#endif
    return L;
}


vec2 getTransmittanceUV(float r, float mu) {
    float uR, uMu;
#ifdef TRANSMITTANCE_NON_LINEAR
    uR = sqrt((r - Rg) / (Rt - Rg));
    uMu = atan((mu + 0.15) / (1.0 + 0.15) * tan(1.5)) / 1.5;
#else
    uR = (r - Rg) / (Rt - Rg);
    uMu = (mu + 0.15) / (1.0 + 0.15);
#endif
    return vec2(uMu, uR);
}

void getTransmittanceRMu(out float r, out float muS) {
    r = gl_FragCoord.y / float(TRANSMITTANCE_H);
    muS = gl_FragCoord.x / float(TRANSMITTANCE_W);
#ifdef TRANSMITTANCE_NON_LINEAR
    r = Rg + (r * r) * (Rt - Rg);
    muS = -0.15 + tan(1.5 * muS) / tan(1.5) * (1.0 + 0.15);
#else
    r = Rg + r * (Rt - Rg);
    muS = -0.15 + muS * (1.0 + 0.15);
#endif
}

vec2 getIrradianceUV(float r, float muS) {
    float uR = (r - Rg) / (Rt - Rg);
    float uMuS = (muS + 0.2) / (1.0 + 0.2);
    return vec2(uMuS, uR);
}

void getIrradianceRMuS(out float r, out float muS) {
    r = Rg + (gl_FragCoord.y - 0.5) / (float(SKY_H) - 1.0) * (Rt - Rg);
    muS = -0.2 + (gl_FragCoord.x - 0.5) / (float(SKY_W) - 1.0) * (1.0 + 0.2);
}

vec4 texture4D(sampler3D table, float r, float mu, float muS, float nu)
{
    float H = sqrt(Rt * Rt - Rg * Rg);
    float rho = sqrt(r * r - Rg * Rg);
#ifdef INSCATTER_NON_LINEAR
    float rmu = r * mu;
    float delta = rmu * rmu - r * r + Rg * Rg;
    vec4 cst = rmu < 0.0 && delta > 0.0 ? vec4(1.0, 0.0, 0.0, 0.5 - 0.5 / float(RES_MU)) : vec4(-1.0, H * H, H, 0.5 + 0.5 / float(RES_MU));
    float uR = 0.5 / float(RES_R) + rho / H * (1.0 - 1.0 / float(RES_R));
    float uMu = cst.w + (rmu * cst.x + sqrt(delta + cst.y)) / (rho + cst.z) * (0.5 - 1.0 / float(RES_MU));
    // paper formula
    //float uMuS = 0.5 / float(RES_MU_S) + max((1.0 - exp(-3.0 * muS - 0.6)) / (1.0 - exp(-3.6)), 0.0) * (1.0 - 1.0 / float(RES_MU_S));
    // better formula
    float uMuS = 0.5 / float(RES_MU_S) + (atan(max(muS, -0.1975) * tan(1.26 * 1.1)) / 1.1 + (1.0 - 0.26)) * 0.5 * (1.0 - 1.0 / float(RES_MU_S));
#else
    float uR = 0.5 / float(RES_R) + rho / H * (1.0 - 1.0 / float(RES_R));
    float uMu = 0.5 / float(RES_MU) + (mu + 1.0) / 2.0 * (1.0 - 1.0 / float(RES_MU));
    float uMuS = 0.5 / float(RES_MU_S) + max(muS + 0.2, 0.0) / 1.2 * (1.0 - 1.0 / float(RES_MU_S));
#endif
    float lerp = (nu + 1.0) / 2.0 * (float(RES_NU) - 1.0);
    float uNu = floor(lerp);
    lerp = lerp - uNu;
    return texture(table, vec3((uNu + uMuS) / float(RES_NU), uMu, uR)) * (1.0 - lerp) +
           texture(table, vec3((uNu + uMuS + 1.0) / float(RES_NU), uMu, uR)) * lerp;
}

void getMuMuSNu(float r, vec4 dhdH, out float mu, out float muS, out float nu) {
    float x = gl_FragCoord.x - 0.5;
    float y = gl_FragCoord.y - 0.5;
#ifdef INSCATTER_NON_LINEAR
    if (y < float(RES_MU) / 2.0) {
        float d = 1.0 - y / (float(RES_MU) / 2.0 - 1.0);
        d = min(max(dhdH.z, d * dhdH.w), dhdH.w * 0.999);
        mu = (Rg * Rg - r * r - d * d) / (2.0 * r * d);
        mu = min(mu, -sqrt(1.0 - (Rg / r) * (Rg / r)) - 0.001);
    } else {
        float d = (y - float(RES_MU) / 2.0) / (float(RES_MU) / 2.0 - 1.0);
        d = min(max(dhdH.x, d * dhdH.y), dhdH.y * 0.999);
        mu = (Rt * Rt - r * r - d * d) / (2.0 * r * d);
    }
    muS = mod(x, float(RES_MU_S)) / (float(RES_MU_S) - 1.0);
    // paper formula
    //muS = -(0.6 + log(1.0 - muS * (1.0 -  exp(-3.6)))) / 3.0;
    // better formula
    muS = tan((2.0 * muS - 1.0 + 0.26) * 1.1) / tan(1.26 * 1.1);
    nu = -1.0 + floor(x / float(RES_MU_S)) / (float(RES_NU) - 1.0) * 2.0;
#else
    mu = -1.0 + 2.0 * y / (float(RES_MU) - 1.0);
    muS = mod(x, float(RES_MU_S)) / (float(RES_MU_S) - 1.0);
    muS = -0.2 + muS * 1.2;
    nu = -1.0 + floor(x / float(RES_MU_S)) / (float(RES_NU) - 1.0) * 2.0;
#endif
}

// ----------------------------------------------------------------------------
// UTILITY FUNCTIONS
// ----------------------------------------------------------------------------

// nearest intersection of ray r,mu with ground or top atmosphere boundary
// mu=cos(ray zenith angle at ray origin)
float limit(float r, float mu) {
    float dout = -r * mu + sqrt(r * r * (mu * mu - 1.0) + RL * RL);
    float delta2 = r * r * (mu * mu - 1.0) + Rg * Rg;
    if (delta2 >= 0.0) {
        float din = -r * mu - sqrt(delta2);
        if (din >= 0.0) {
            dout = min(dout, din);
        }
    }
    return dout;
}

// optical depth for ray (r,mu) of length d, using analytic formula
// (mu=cos(view zenith angle)), intersections with ground ignored
// H=height scale of exponential density function
float opticalDepth(float H, float r, float mu, float d) {
    float a = sqrt((0.5/H)*r);
    vec2 a01 = a*vec2(mu, mu + d / r);
    vec2 a01s = sign(a01);
    vec2 a01sq = a01*a01;
    float x = a01s.y > a01s.x ? exp(a01sq.x) : 0.0;
    vec2 y = a01s / (2.3193*abs(a01) + sqrt(1.52*a01sq + 4.0)) * vec2(1.0, exp(-d/H*(d/(2.0*r)+mu)));
    return sqrt((6.2831*H)*r) * exp((Rg-r)/H) * (x + dot(y, vec2(1.0, -1.0)));
}

// transmittance(=transparency) of atmosphere for infinite ray (r,mu)
// (mu=cos(view zenith angle)), intersections with ground ignored
vec3 transmittance(float r, float mu) {
    vec2 uv = getTransmittanceUV(r, mu);
    return texture2D(transmittanceSampler, uv).rgb;
}

// transmittance(=transparency) of atmosphere for ray (r,mu) of length d
// (mu=cos(view zenith angle)), intersections with ground ignored
// uses analytic formula instead of transmittance texture
vec3 analyticTransmittance(float r, float mu, float d) {
    return exp(- betaR * opticalDepth(HR, r, mu, d) - betaMEx * opticalDepth(HM, r, mu, d));
}

// transmittance(=transparency) of atmosphere for infinite ray (r,mu)
// (mu=cos(view zenith angle)), or zero if ray intersects ground
vec3 transmittanceWithShadow(float r, float mu) {
    return mu < -sqrt(1.0 - (Rg / r) * (Rg / r)) ? vec3(0.0) : transmittance(r, mu);
}

// transmittance(=transparency) of atmosphere between x and x0
// assume segment x,x0 not intersecting ground
// r=||x||, mu=cos(zenith angle of [x,x0) ray at x), v=unit direction vector of [x,x0) ray
vec3 transmittance(float r, float mu, vec3 v, vec3 x0) {
    vec3 result;
    float r1 = length(x0);
    float mu1 = dot(x0, v) / r;
    if (mu > 0.0) {
        result = min(transmittance(r, mu) / transmittance(r1, mu1), 1.0);
    } else {
        result = min(transmittance(r1, -mu1) / transmittance(r, -mu), 1.0);
    }
    return result;
}

// transmittance(=transparency) of atmosphere between x and x0
// assume segment x,x0 not intersecting ground
// d = distance between x and x0, mu=cos(zenith angle of [x,x0) ray at x)
vec3 transmittance(float r, float mu, float d) {
    vec3 result;
    float r1 = sqrt(r * r + d * d + 2.0 * r * mu * d);
    float mu1 = (r * mu + d) / r1;
    if (mu > 0.0) {
        result = min(transmittance(r, mu) / transmittance(r1, mu1), 1.0);
    } else {
        result = min(transmittance(r1, -mu1) / transmittance(r, -mu), 1.0);
    }
    return result;
}

vec3 irradiance(sampler2D sampler, float r, float muS) {
    vec2 uv = getIrradianceUV(r, muS);
    return texture2D(sampler, uv).rgb;
}

// Rayleigh phase function
float phaseFunctionR(float mu) {
    return (3.0 / (16.0 * M_PI)) * (1.0 + mu * mu);
}

// Mie phase function
float phaseFunctionM(float mu) {
    return 1.5 * 1.0 / (4.0 * M_PI) * (1.0 - mieG*mieG) * pow(1.0 + (mieG*mieG) - 2.0*mieG*mu, -3.0/2.0) * (1.0 + mu * mu) / (2.0 + mieG*mieG);
}

// approximated single Mie scattering (cf. approximate Cm in paragraph "Angular precision")
vec3 getMie(vec4 rayMie) { // rayMie.rgb=C*, rayMie.w=Cm,r
    return rayMie.rgb * rayMie.w / max(rayMie.r, 1e-4) * (betaR.r / betaR);
}

float SQRT(float f, float err) {
#ifdef OPTIMIZE
    return sqrt(f);
#else
    return f >= 0.0 ? sqrt(f) : err;
#endif
}

// ----------------------------------------------------------------------------
// PUBLIC FUNCTIONS
// ----------------------------------------------------------------------------

// assumes sundir=vec3(0.0, 0.0, 1.0)
vec3 outerSunRadiance(vec3 viewdir)
{
    vec3 data = viewdir.z > 0.0 ? texture2D(glareSampler, vec2(0.5) + viewdir.xy * 4.0).rgb : vec3(0.0);
    return pow(data, vec3(2.2)) * SUN_INTENSITY;
}

// incident sun light at given position (radiance)
// r=length(x)
// muS=dot(x,s) / r
vec3 sunRadiance(float r, float muS) {
#if defined(ATMO_SUN_ONLY) || defined(ATMO_FULL)
    return transmittanceWithShadow(r, muS) * SUN_INTENSITY;
#elif defined(ATMO_NONE)
    return vec3(SUN_INTENSITY);
#else
    return vec3(0.0);
#endif
}

// incident sky light at given position, integrated over the hemisphere (irradiance)
// r=length(x)
// muS=dot(x,s) / r
vec3 skyIrradiance(float r, float muS) {
#if defined(ATMO_SKY_ONLY) || defined(ATMO_FULL)
    return irradiance(skyIrradianceSampler, r, muS) * SUN_INTENSITY;
#else
    return vec3(0.0);
#endif
}

// single scattered sunlight between two points
// camera=observer
// viewdir=unit vector towards observed point
// sundir=unit vector towards the sun
// return scattered light and extinction coefficient
vec3 skyRadiance(vec3 camera, vec3 viewdir, vec3 sundir, out vec3 extinction, float shaftWidth)
{
#if defined(ATMO_INSCATTER_ONLY) || defined(ATMO_FULL)
    vec3 result;
    camera += viewdir * max(shaftWidth, 0.0);
    float r = length(camera);
    float rMu = dot(camera, viewdir);
    float mu = rMu / r;
    float r0 = r;
    float mu0 = mu;

    float deltaSq = SQRT(rMu * rMu - r * r + Rt*Rt, 1e30);
    float din = max(-rMu - deltaSq, 0.0);
    if (din > 0.0) {
        camera += din * viewdir;
        rMu += din;
        mu = rMu / Rt;
        r = Rt;
    }

    if (r <= Rt) {
        float nu = dot(viewdir, sundir);
        float muS = dot(camera, sundir) / r;

        vec4 inScatter = texture4D(inscatterSampler, r, rMu / r, muS, nu);
        if (shaftWidth > 0.0) {
            if (mu > 0.0) {
                inScatter *= min(transmittance(r0, mu0) / transmittance(r, mu), 1.0).rgbr;
            } else {
                inScatter *= min(transmittance(r, -mu) / transmittance(r0, -mu0), 1.0).rgbr;
            }
        }
	
		
        extinction = transmittance(r, mu);

        vec3 inScatterM = getMie(inScatter);
        float phase = phaseFunctionR(nu);
        float phaseM = phaseFunctionM(nu);
        result = inScatter.rgb * phase + inScatterM * phaseM;
    } else {
        result = vec3(0.0);
        extinction = vec3(1.0);
    }

    return result * SUN_INTENSITY;
#else
    extinction = vec3(1.0);
    return vec3(0.0);
#endif
}

void sunRadianceAndSkyIrradiance(vec3 worldP, vec3 worldN, vec3 worldS, out vec3 sunL, out vec3 skyE)
{
    float r = length(worldP);
    if (r < 0.9 * PLANET_RADIUS) {
        worldP.z += PLANET_RADIUS;
        r = length(worldP);
    }
    vec3 worldV = worldP / r; // vertical vector
    float muS = dot(worldV, worldS);

    float sunOcclusion = 1.0;// - sunShadow;
    sunL = sunRadiance(r, muS) * sunOcclusion;

    // ambient occlusion due only to slope, does not take self shadowing into account
    float skyOcclusion = (1.0 + dot(worldV, worldN)) * 0.5;
    // factor 2.0 : hack to increase sky contribution (numerical simulation of
    // "precompued atmospheric scattering" gives less luminance than in reality)
    skyE = 2.0 * skyIrradiance(r, muS) * skyOcclusion;
}

void sunRadianceAndSkyIrradiance(vec3 worldP, vec3 worldN, vec3 worldS, float pixelScale, vec4 ambientAperture, out vec3 sunL, out vec3 skyE)
{
    float r = length(worldP);
    if (r < 0.9 * PLANET_RADIUS) {
        worldP.z += PLANET_RADIUS;
        r = length(worldP);
    }
    vec3 worldV = worldP / r; // vertical vector
    float muS = dot(worldV, worldS);

    const float rl = 0.00465;
    float rp = acos(1.0 - ambientAperture.x);
    float d = acos(dot(worldS, ambientAperture.yzw));
    float sunOcclusion = 1.0 - smoothstep(rp - rl, rp + rl, d);
    // a "proper" way to filter the sunOcclusion is to compute the mean
    // occlusion due to max and min aperture over the visible pixel area.
    // We only have the mean apearture (precomputed with mipmaping), but the max
    // aperture can be set to one in the limit of a pixel covering a large area
    // (several kilometers). So we interpolate between the occlusion due to the
    // mean aperture, and a mean occlusion using mean and max aperture, based
    // on the pixel size in kilomters.
    float lerp = smoothstep(1000.0, 8000.0, pixelScale);
    sunOcclusion = sunOcclusion * (1.0 - lerp) + (1.0 + sunOcclusion) * 0.5 * lerp;

    float skyOcclusion = ambientAperture.x;

    sunL = sunRadiance(r, muS) * sunOcclusion;

    // factor 2.0 : hack to increase sky contribution (numerical simulation of
    // "precompued atmospheric scattering" gives less luminance than in reality)
    skyE = 2.0 * skyIrradiance(r, muS) * skyOcclusion;
}

// single scattered sunlight between two points
// camera=observer
// point=point on the ground
// sundir=unit vector towards the sun
// return scattered light and extinction coefficient
vec3 inScattering(vec3 camera, vec3 point, vec3 sundir, out vec3 extinction, float shaftWidth) {
#if defined(ATMO_INSCATTER_ONLY) || defined(ATMO_FULL)
    vec3 result;
    vec3 viewdir = point - camera;
    float d = length(viewdir);
    viewdir = viewdir / d;
    float r = length(camera);
    if (r < 0.9 * Rg) {
        camera.z += Rg;
        point.z += Rg;
        r = length(camera);
    }
    float rMu = dot(camera, viewdir);
    float mu = rMu / r;
    float r0 = r;
    float mu0 = mu;
    point -= viewdir * clamp(shaftWidth, 0.0, d);

    float deltaSq = SQRT(rMu * rMu - r * r + Rt*Rt, 1e30);
    float din = max(-rMu - deltaSq, 0.0);
    if (din > 0.0 && din < d) {
        camera += din * viewdir;
        rMu += din;
        mu = rMu / Rt;
        r = Rt;
        d -= din;
    }

    if (r <= Rt) {
        float nu = dot(viewdir, sundir);
        float muS = dot(camera, sundir) / r;

        vec4 inScatter;

        if (r < Rg + 600.0) {
            // avoids imprecision problems in aerial perspective near ground
            float f = (Rg + 600.0) / r;
            r = r * f;
            rMu = rMu * f;
            point = point * f;
        }

        float r1 = length(point);
        float rMu1 = dot(point, viewdir);
        float mu1 = rMu1 / r1;
        float muS1 = dot(point, sundir) / r1;

#ifdef ANALYTIC_TRANSMITTANCE
        extinction = min(analyticTransmittance(r, mu, d), 1.0);
#else
        if (mu > 0.0) {
            extinction = min(transmittance(r, mu) / transmittance(r1, mu1), 1.0);
        } else {
            extinction = min(transmittance(r1, -mu1) / transmittance(r, -mu), 1.0);
        }
#endif

#ifdef HORIZON_HACK
        const float EPS = 0.004;
        float lim = -sqrt(1.0 - (Rg / r) * (Rg / r));
        if (abs(mu - lim) < EPS) {
            float a = ((mu - lim) + EPS) / (2.0 * EPS);

            mu = lim - EPS;
            r1 = sqrt(r * r + d * d + 2.0 * r * d * mu);
            mu1 = (r * mu + d) / r1;
            vec4 inScatter0 = texture4D(inscatterSampler, r, mu, muS, nu);
            vec4 inScatter1 = texture4D(inscatterSampler, r1, mu1, muS1, nu);
            vec4 inScatterA = max(inScatter0 - inScatter1 * extinction.rgbr, 0.0);

            mu = lim + EPS;
            r1 = sqrt(r * r + d * d + 2.0 * r * d * mu);
            mu1 = (r * mu + d) / r1;
            inScatter0 = texture4D(inscatterSampler, r, mu, muS, nu);
            inScatter1 = texture4D(inscatterSampler, r1, mu1, muS1, nu);
            vec4 inScatterB = max(inScatter0 - inScatter1 * extinction.rgbr, 0.0);

            inScatter = mix(inScatterA, inScatterB, a);
        } else {
            vec4 inScatter0 = texture4D(inscatterSampler, r, mu, muS, nu);
            vec4 inScatter1 = texture4D(inscatterSampler, r1, mu1, muS1, nu);
            inScatter = max(inScatter0 - inScatter1 * extinction.rgbr, 0.0);
        }
#else
        vec4 inScatter0 = texture4D(inscatterSampler, r, mu, muS, nu);
        vec4 inScatter1 = texture4D(inscatterSampler, r1, mu1, muS1, nu);
        inScatter = max(inScatter0 - inScatter1 * extinction.rgbr, 0.0);
#endif

        //cancels inscatter when sun hidden by mountains
        //TODO smoothstep values depend on horizon angle in sun direction
        //inScatter.w *= smoothstep(0.035, 0.07, muS);

        // avoids imprecision problems in Mie scattering when sun is below horizon
        inScatter.w *= smoothstep(0.00, 0.02, muS);

        vec3 inScatterM = getMie(inScatter);
        float phase = phaseFunctionR(nu);
        float phaseM = phaseFunctionM(nu);
        result = inScatter.rgb * phase + inScatterM * phaseM;
    } else {
        result = vec3(0.0);
        extinction = vec3(1.0);
    }

    return result * SUN_INTENSITY;
#else
    extinction = vec3(1.0);
    return vec3(0.0);
#endif
}



uniform sampler2D tex0;
uniform sampler2D shadow_tex;
uniform sampler2D shadow_tex2;
uniform sampler2D tex_def;

uniform sampler2D tex_sand;
uniform sampler2D tex_rock;
uniform sampler2D tex_grass;

uniform sampler2D bump4_tex;
uniform sampler2D bump6_tex;

uniform sampler2D bolder_tex;

uniform sampler2D deltaAdd_tex;
uniform sampler2D deltaMult_tex;

uniform mat4 ProjectionMatrix;
uniform mat4 osg_ViewMatrix;
uniform mat4 light_mv_matrix;

uniform int id;

in vec3 p;
in vec4 ex_shadowCoord;
in vec3 n;
in vec2 uv;
in float ex_lightdistance;

in vec3 out_normal;
in vec3 lightPos;
in vec3 eyeW;
in vec3 positionW;

in vec2 ex_texCoord;

in vec4 fragC;

out vec4 out_color;
out vec4 out_color2;

float chebyshevUpperBound(vec2 moments, float distance)
{	
	if (distance <= moments.x)	return 1.0;				
	
	float variance = moments.y - (moments.x*moments.x);	
	variance = clamp(variance, 0, 100000);
	float d = distance - moments.x;
	float p_max = (variance / (variance + d*d));
	float min1 = 0.1;
	float max1 = 1.0;
	p_max = clamp((p_max-min1)/(max1-min1), 0.0, 1.0);	
	return p_max;
}
	
void main(void)
{
	vec4 ShadowCoordPostW = ex_shadowCoord;	
	ShadowCoordPostW.xy = ShadowCoordPostW.xy / ex_shadowCoord.w;
	ShadowCoordPostW.xy = ShadowCoordPostW.xy * 0.5 + 0.5;	
	 
	vec4 shadow_texture = texture(shadow_tex, ShadowCoordPostW.xy);
	float shadow = chebyshevUpperBound(shadow_texture.st, ShadowCoordPostW.z-0.005);		
		
	vec4 shadow_texture2 = texture(shadow_tex2, ShadowCoordPostW.xy);
	float shadow2 = chebyshevUpperBound(shadow_texture2.st, ShadowCoordPostW.z-0.05);	
	
	if(ShadowCoordPostW.s < 0.0 || ShadowCoordPostW.t < 0.0 || ShadowCoordPostW.s > 1.0 || ShadowCoordPostW.t > 1.0)
	{
		shadow = 1.0;	
		shadow2 = 1.0;	
	}	
		
	vec3 WCP = (inverse(osg_ViewMatrix) * vec4(0.0,0.0,0.0,1.0)).xyz;		
    vec3 WSD = normalize( (inverse(light_mv_matrix) * vec4(0.0,0.0,0.0,1.0)).xyz);
    vec3 v = normalize(p - WCP);

    vec3 fn = normalize(n);
	//fn.z = sqrt(max(0.0, 1.0 - dot(fn.xy, fn.xy)));

    vec3 sunL;
    vec3 skyE;
    sunRadianceAndSkyIrradiance(p, fn, WSD, sunL, skyE);

    vec3 reflectance; //textureTile(orthoSampler, uv).rgb * 0.2;
	vec4 layerdef = texture(tex_def , uv);
	vec4 color2= texture(tex0, uv);		
	vec4 tgrass = texture(tex_grass, uv*vec2(100.0,100.0));
	color2 += texture(tex_grass,  uv*vec2(80.0,80.0))  * (1.0-clamp(layerdef.b+layerdef.r+layerdef.g/*+layerdef.a*/, 0.0, 1.0)); 	
	//color2 += texture(bolder_tex, uv*vec2(10.0,10.0))  * layerdef.a; 
	color2 += texture(tex_rock,  uv*vec2(2.0*80.0,2.0*80.0))   * layerdef.b; 		
	color2 += tgrass * layerdef.g * 1.5; 	
	color2 += texture(tex_sand,  uv*vec2(80.0,80.0))   * layerdef.r; 			
	
	
	vec4 bump4 = texture(bump4_tex, ex_texCoord.xy*vec2(100,100));
	vec4 bump6 = texture(bump6_tex, ex_texCoord.xy*vec2(100,100));
	bump4.xyz = 2.0*bump4.xyz-1.0;
	bump6.xyz = 2.0*bump6.xyz-1.0;
	
	vec3 N = bump4.xyz;		
	N	= mix(bump6.xyz, N, 1.0-layerdef.b);			
	N = normalize(N);
	
	//vec3 out_normal = Normal;
	mat3 normal_rotation_matrix;
	normal_rotation_matrix[2]= out_normal;
	normal_rotation_matrix[1]=normalize(cross(vec3(-1.0,0.0,0.0), out_normal));
	normal_rotation_matrix[0]=normalize(cross(normal_rotation_matrix[1],normal_rotation_matrix[2]));
	N= normalize(normal_rotation_matrix * N);	
	
	vec2 texCoord = gl_FragCoord.st / vec2(800.0, 600.0);	

	
	vec3 E = normalize(eyeW-positionW);
	vec3 L = normalize(lightPos-positionW);

	vec3 deltaAdds = texture(deltaAdd_tex, texCoord).xyz;
	float deltaZ = abs(deltaAdds.g);
	float deltaN = abs(deltaAdds.r);
	
	vec2 deltaMult = texture(deltaMult_tex, texCoord).st;
	float zmin = abs(300000.0-abs(deltaMult.r));
	float zmax = abs(deltaMult.g);
	
	float z_ground = abs(fragC.z);
	
	float l = clamp((deltaZ-z_ground*deltaN), 0.0, z_ground-zmin);//z_ground););		
	l=0;
	//color2 += vec4(0.01);		
			
	//color2.xyz = vec3(1.0);
	vec4 color = vec4(0.0,0.0,0.0,1.0);	
	float NdotL = max(dot(N, L), 0.0);
	if (NdotL > 0.0) 
	{			
		color += color2 * NdotL;
		vec3 R = normalize(reflect(L, N));
	
		float EdotR = max(dot(normalize(-E).xyz, R), 0.0);
		//color += vec4(1.0) * pow(EdotR, 10.0);
	}			
	
	vec3 groundColor = clamp(color.xyz,vec3(0.0),vec3(1.0));
		
    vec3 extinction;
    vec3 inscatter = inScattering(WCP, p, WSD, extinction, l*1.0);    
	
	vec3 finalColor = groundColor * extinction + inscatter;		
	out_color.xyz = finalColor;	
	//out_color.xyz = vec3(clamp(N.xyz, 0.1, 1.0));										 	
	//out_color.xyz *= min(vec3(clamp(clamp(1.0-(shadow_texture.z+0.0),0.0,1.0), 0.0, 1.0)), vec3(clamp((shadow), 0.0, 1.0)))+0.1;										 
		
	out_color.xyz*=4.0;
	if(out_color.x != out_color.x) out_color.x = 2.0;
	if(out_color.y != out_color.y) out_color.y = 2.0;
	if(out_color.z != out_color.z) out_color.z = 2.0;
	out_color.xyz *= shadow2;										 	
	out_color.xyz *= 1.0-clamp(shadow_texture.z*1.0, 0.1, 1.0);										 	
	out_color.xyz = clamp(out_color.xyz, vec3(0.0), vec3(4.0));

	out_color.w = 1.0;
	out_color2 = vec4(id, 0.0, 0.0, 1.0);
	
}
