#version 330 core
#extension GL_EXT_geometry_shader4 : enable
layout (points) in;
layout (triangle_strip, max_vertices = 4) out;

uniform mat4 project;
uniform mat4 viewMatrix;
uniform mat4  viewMatrixInv;
uniform vec3 un_light_pos;

uniform vec3 v3InvWavelength;	// 1 / pow(wavelength, 4) for the red, green, and blue channels
uniform float fOuterRadius;		// The outer (atmosphere) radius
uniform float fOuterRadius2;	// fOuterRadius^2
uniform float fInnerRadius;		// The inner (planetary) radius
uniform float fInnerRadius2;	// fInnerRadius^2
uniform float fKrESun;			// Kr * ESun
uniform float fKmESun;			// Km * ESun
uniform float fKr4PI;			// Kr * 4 * PI
uniform float fKm4PI;			// Km * 4 * PI
uniform float fScale;			// 1 / (fOuterRadius - fInnerRadius)
uniform float fScaleDepth;		// The scale depth (i.e. the altitude at which the atmosphere's average density is found)
uniform float fScaleOverScaleDepth;	// fScale / fScaleDepth

uniform float g;
uniform float g2;

const int nSamples = 8;
const float fSamples = 8.0;

vec3 CalcMie(vec3 front_color, vec3 secondary_color, vec3 v3Direction )
{
	vec3 lightDir = normalize(un_light_pos);

	float fCos = dot(lightDir, v3Direction) / length(v3Direction);
	float fMiePhase = 1.5 * ((1.0 - g2) / (2.0 + g2)) * (1.0 + fCos*fCos) / pow(1.0 + g2 - 2.0*g*fCos, 1.5);
	vec3 color = clamp(front_color + secondary_color*fMiePhase,0.0, 3.0);	
	
	//color = vec3(1.0) - clamp(exp(color*-1.0),0.1,1.0);
	
	return color;	
}


float scale(float fCos)
{
	float x = 1.0 - fCos;
	return fScaleDepth * exp(-0.00287 + x*(0.459 + x*(3.83 + x*(-6.80 + x*5.25))));
}

void CalcSunColors(in vec4 vertex, out vec3 secondary_color, out vec3 front_color, out vec3 v3Direction)
{
	vec3 v3CameraPos = (viewMatrixInv * vec4(0.0, 0.0, 0.0, 1.0)).xyz;
	//vec3 v3CameraPos = vec3(0.0, 0.0, 0.0);

	vec3 vecCamera = v3CameraPos;
	vecCamera /= 1000000.0*fInnerRadius;
	vecCamera.z += fInnerRadius;
	
	float fCameraHeight = length(vecCamera);
	float fCameraHeight2 = pow(fCameraHeight, 2.f);	
		
	vec3 lightPos = normalize(un_light_pos);

	// Get the ray from the camera to the vertex, and its length (which is the far point of the ray passing through the atmosphere)
	vec3 v3Pos = vertex.xyz/1000000.0*fInnerRadius; //HACK 50000 gehrt eigentlich
	v3Pos.z += fInnerRadius;
	vec3 v3Ray = v3Pos - vecCamera;
	float fFar = length(v3Ray);
	v3Ray /= fFar;

	// Calculate the ray's starting position, then calculate its scattering offset
	vec3 v3Start = vecCamera;
	float fHeight = length(v3Start);
	float fDepth = exp(fScaleOverScaleDepth * (fInnerRadius - fCameraHeight));
	float fStartAngle = dot(v3Ray, v3Start) / fHeight;
	float fStartOffset = fDepth*scale(fStartAngle);

	// Initialize the scattering loop variables
	float fSampleLength = fFar / fSamples;
	float fScaledLength = fSampleLength * fScale;
	vec3 v3SampleRay = v3Ray * fSampleLength;
	vec3 v3SamplePoint = v3Start + (v3SampleRay * vec3(0.5));

	vec3 v3FrontColor = vec3(0.0);
	vec3 v3Attenuate = vec3(1.0);
	
	if(v3Pos.z < vecCamera.z)
	{		
		float fDepth = exp(1.0/fScaleDepth *(fInnerRadius - fCameraHeight));		// exp(fScaleOverScaleDepth * (InnerRadius - fCameraHeight));//
		float fLightAngle = dot(lightPos.xyz, v3Pos) / length(v3Pos);
		float fCameraAngle = dot(-v3Ray, v3Pos) / length(v3Pos);
		float fCameraScale = scale(fCameraAngle);
		float fLightScale = scale(fLightAngle);
		float fCameraOffset = fDepth*fCameraScale;		

		// Initialize the scattering loop variables
		float fSampleLength = fFar / fSamples;
		float fScaledLength = fSampleLength * fScale;
		vec3 v3SampleRay = v3Ray * fSampleLength;	
		vec3 v3SamplePoint = v3Start + v3SampleRay * 0.5;

		float fHeight = length(v3SamplePoint);												//OK
		float fStartAngle = dot(v3Ray, v3Start) / fHeight;
		float fStartOffset = fDepth*scale(fStartAngle);
			
		// Now loop through the sample rays	
		for(int i=0; i<nSamples; i++)
		{
			float fHeight = length(v3SamplePoint);												//OK
			float fDepth = exp(1.0/fScaleDepth * (fInnerRadius - fHeight));					//OK		
			
			float fScatter = fDepth*(scale(fLightAngle) + scale(fCameraAngle)) - fCameraOffset;
			
			v3Attenuate = exp(-fScatter * (v3InvWavelength * fKr4PI + fKm4PI));				//OK				
			v3FrontColor += v3Attenuate* (fDepth * fScaledLength);		
			v3SamplePoint += v3SampleRay;
		}
	}
	else
	{
		// Calculate the ray's starting position, then calculate its scattering offset
		vec3 v3Start = vecCamera;
		float fHeight = length(v3Start);
		float fDepth = exp(1.0/fScaleDepth * (fInnerRadius - fCameraHeight));
		float fStartAngle = dot(v3Ray, v3Start) / fHeight;
		float fStartOffset = fDepth*scale(fStartAngle);

		// Initialize the scattering loop variables
		float fSampleLength = fFar / fSamples;
		float fScaledLength = fSampleLength * fScale;
		vec3 v3SampleRay = v3Ray * fSampleLength;		
		vec3 v3SamplePoint = v3Start + (v3SampleRay * 0.5);

		// Now loop through the sample rays	
		for(int i=0; i<nSamples; i++)
		{
			float fHeight = length(v3SamplePoint);													//OK
			float fDepth = exp(1.0/fScaleDepth * (fInnerRadius - fHeight));					//OK
			
			float fLightAngle = dot(lightPos.xyz, v3SamplePoint.xyz) / fHeight;								//OK
			float fCameraAngle = dot(v3Ray, v3SamplePoint) / fHeight;								//OK
			float fScatter = (fDepth*(scale(fLightAngle) - scale(fCameraAngle))) + fStartOffset;	//OK
			
			v3Attenuate = exp(-fScatter * (v3InvWavelength * fKr4PI + fKm4PI));				//OK				
			v3FrontColor += v3Attenuate* (fDepth * fScaledLength);		
			v3SamplePoint += v3SampleRay;
		}
	}

	// Finally, scale the Mie and Rayleigh colors and set up the varying variables for the pixel shader
	secondary_color = v3FrontColor * fKmESun;
	front_color = v3FrontColor * (v3InvWavelength * fKrESun);
	v3Direction = vecCamera - v3Pos;
}


uniform mat4 viewHalfAngleMatrix;
uniform mat4 invViewHalfAngleMatrix;	

uniform mat4 project_light;
uniform mat4 viewMatrix_light;

uniform mat3 un_dir_colors;

uniform float un_timeOfDay;
uniform float un_fading;

uniform vec3 un_skyLight_h0;
uniform vec3 un_skyLight_h1;
uniform vec3 un_skyLight_t0;
uniform vec3 un_skyLight_t1;

uniform mat4 modelViewMatrix_light;
uniform mat4 viewMatrixInv_light;

uniform vec3 un_sunLight;

in vec4 ex_vd_size[];
in vec4 ex_vd_center[];
in vec3 ex_vd_bb_min[];
in vec3 ex_vd_bb_max[];
in vec2 ex_vd_density[];

out vec4 ex_texCoord;
out vec4 ex_ambient;
out vec4 ex_diffuse;
out vec3 ex_mie;
out vec4 ex_position;
out vec3 ex_normal;
out vec2 ex_density;


vec3 ambientColor(float fragHeight, vec3 mie)
{
	vec3 cloudCenterPosition = (ex_vd_bb_max[0]  + ex_vd_bb_min[0]) / 2.0;
	float cloudHeight = ex_vd_bb_max[0].z-ex_vd_bb_min[0].z;	
	
	float timeOfDay = 0.2;
	float layerBrightness = 0.1;
	
	//mie.xyz = vec3(0.5);
	vec3 skyLight_h0 = max(vec3(0.2, 0.2, 0.2), mie.xyz); //clamp(mie.xyz-vec3(0.3),0.0,1.0);
	vec3 skyLight_h1 = clamp(mie.xyz+vec3(0.4), vec3(0.0), vec3(1.0));

	vec3 sunPosition = normalize(un_light_pos);

	//h=0 -> bottom, h=1 -> top
	float h = 1.0-(((cloudCenterPosition.z + cloudHeight/2.0)-fragHeight) / cloudHeight);
	float transferFunc = clamp(h, 0.01, 1.0);
		
	vec3 inCloudScattering = skyLight_h1*(transferFunc) + skyLight_h0*(1.0-transferFunc);
	//inCloudScattering = inCloudScattering*(1.0-layerBrightness);// + vec3(1.0, 1.0, 1.0)*layerBrightness;
	vec3 skyLight0 = un_skyLight_h0;
	vec3 skyLight1 = un_skyLight_h1;

	//vec3 ambientColor = (timeOfDay*skyLight0 + (1.0-timeOfDay)*skyLight1)*inCloudScattering;
	vec3 ambientColor = inCloudScattering;
	

	return ambientColor;// * (1.0+layerBrightness*0.5);	
}

vec3 directionalColor(vec4 worldpos, vec3 cloudCenterPosition, vec3 mie)
{	
	vec3 mie2 = mie;
	vec3 cmin = (un_dir_colors[0]*0.7)+(0.3*clamp(mie2.xyz-vec3(0.5, 0.5, 0.5), 0.0, 1.0));
	vec3 cmed = (un_dir_colors[1]*0.7)+(0.3*clamp(mie2.xyz-vec3(0.2, 0.2, 0.2), 0.0, 1.0));
	vec3 cmax = (un_dir_colors[2]*0.9)+(0.1*clamp(mie2.xyz-vec3(0.0, 0.0, 0.0), 0.0, 1.0));
	
	cmin = vec3(0.1,0.1,0.1);
	cmed = vec3(0.3,0.3,0.3);
	cmax = vec3(0.7,0.7,0.7);

	float time_of_day = un_timeOfDay;
	
	vec3 c_t0 = vec3(1.0,1.0,1.0);//*normalize(un_sunLight);
	vec3 c_t1 = vec3(1.3,1.3,1.3);//*normalize(un_sunLight);

	vec3 sunPosition = un_light_pos;
	vec3 n1 = normalize(worldpos.xyz-cloudCenterPosition);
	vec3 n2 = normalize(un_light_pos);
	float dotres = clamp(dot(n1,n2),0.0, 1.0);

	vec3 color;
	if (dotres < 0.0)
	{
		dotres += 1.0;
		color = ((1.0-dotres)*cmin) + (dotres*cmed);
	}
	else
	{
		color = ((1.0-dotres)*cmed) + (dotres*cmax);
	}
	
	return color* ( (1.0-time_of_day) * c_t0 + (1.0-(1.0-time_of_day))*c_t1);	
}

float CalcTransparency(vec4 pos)
{
	vec3 cloudCenterPosition = (ex_vd_bb_max[0]  + ex_vd_bb_min[0]) / 2.0;

	float time = un_fading;
	float radius = length(ex_vd_bb_max[0]-cloudCenterPosition);	
	float length = length(pos.xyz-cloudCenterPosition);	
	
	if(time < -0.5 || time > 0.5)
	{
		time = abs(time);

		if(length < (radius/2.0))
		{
			return (1.0-time);
		}
		else return 0.0;
	}	
	else
	{
		time = abs(time);

		if(length < (radius/2.0))
		{
			return ((1.0-time));		
		}
		else
		{
			return (0.5-time)/0.5;		
		}
	}	
}

void main()                                                   
{
vec3 cloudCenterPosition = (ex_vd_bb_max[0]  + ex_vd_bb_min[0]) / 2.0;
mat4 model = viewMatrixInv_light * modelViewMatrix_light;

	float cbeta = cos(ex_vd_center[0].w);	float sbeta = sin(ex_vd_center[0].w);
	mat2 rot = mat2(1.0, 0.0, 0.0, 1.0); //mat2(cbeta, -sbeta, sbeta, cbeta);
		
	vec3 pos = (viewMatrix*model*gl_PositionIn[0]).xyz; // eye space

	// rotation
	vec2 s = vec2(1.0, 1.0);
//	s = vec2(0.2, 0.2);
	float x1 = ex_vd_size[0].x*s.x; float y1 = ex_vd_size[0].y*s.y; vec2 t1 = rot*vec2(-x1,-y1); vec2 t2 = rot*vec2(-x1, y1); vec2 t3 = rot*vec2( x1,-y1); vec2 t4 = rot*vec2( x1, y1);
		
	vec4 v1 = vec4(pos.x+t1.x, pos.y+t1.y, pos.z, 1.0);
	vec4 v2 = vec4(pos.x+t2.x, pos.y+t2.y, pos.z, 1.0);
	vec4 v3 = vec4(pos.x+t3.x, pos.y+t3.y, pos.z, 1.0);
	vec4 v4 = vec4(pos.x+t4.x, pos.y+t4.y, pos.z, 1.0);
 
	mat4 transformation = project*viewMatrix;
		
	vec4 p = viewMatrixInv*v1;
	ex_texCoord = vec4(0.0, 0.0, ex_vd_size[0].z, ex_vd_size[0].w);	
	
	vec3 front_color;
	vec3 secondary_color;
	vec3 v3Direction;
	CalcSunColors(p, secondary_color, front_color, v3Direction);
	ex_mie = CalcMie(front_color, secondary_color, v3Direction).xyz;	
	ex_ambient = vec4(ambientColor(p.z, ex_mie), CalcTransparency(p));
	ex_diffuse.xyz = directionalColor(p, ex_vd_center[0].xyz, ex_mie);
	ex_diffuse.w = clamp((p.z-ex_vd_bb_min[0].z) /(ex_vd_bb_max[0].z-ex_vd_bb_min[0].z),0.0,1.0);	
	ex_position = p;
	ex_normal = normalize(p.xyz - cloudCenterPosition.xyz).xyz;//+0.5*normalize(p.xyz - ex_vd_center[0].xyz);
	ex_density = ex_vd_density[0];
	gl_Position = transformation * p;
	EmitVertex();
	
	p = viewMatrixInv*v2;
	ex_texCoord = vec4(0.0, 1.0, ex_vd_size[0].z, ex_vd_size[0].w);		
	CalcSunColors(p, secondary_color, front_color, v3Direction);
	ex_mie = CalcMie(front_color, secondary_color, v3Direction).xyz;	
	ex_ambient = vec4(ambientColor(p.z, ex_mie), CalcTransparency(p));
	ex_diffuse.xyz = min(directionalColor(p, ex_vd_center[0].xyz, ex_mie), directionalColor(p, cloudCenterPosition, ex_mie));		
	ex_diffuse.w = clamp((p.z-ex_vd_bb_min[0].z) /(ex_vd_bb_max[0].z-ex_vd_bb_min[0].z),0.0,1.0);	
	ex_position = p;
	ex_normal = normalize(p.xyz - cloudCenterPosition.xyz).xyz;//+0.5*normalize(p.xyz - ex_vd_center[0].xyz);
	ex_density = ex_vd_density[0];
	gl_Position = transformation * p;
	EmitVertex();

	p = viewMatrixInv*v3;
	ex_texCoord = vec4(1.0, 0.0, ex_vd_size[0].z, ex_vd_size[0].w);      			
	CalcSunColors(p, secondary_color, front_color, v3Direction);
	ex_mie = CalcMie(front_color, secondary_color, v3Direction).xyz;	
	ex_ambient = vec4(ambientColor(p.z, ex_mie), CalcTransparency(p));
	ex_diffuse.xyz = directionalColor(p, ex_vd_center[0].xyz, ex_mie);
	ex_diffuse.w = clamp((p.z-ex_vd_bb_min[0].z) /(ex_vd_bb_max[0].z-ex_vd_bb_min[0].z),0.0,1.0);	
	ex_position = p;
	ex_normal = normalize(p.xyz - cloudCenterPosition.xyz).xyz;//+0.5*normalize(p.xyz - ex_vd_center[0].xyz);
	ex_density = ex_vd_density[0];
	gl_Position = transformation * p;
	EmitVertex();

	p = viewMatrixInv*v4;
	ex_texCoord = vec4(1.0, 1.0, ex_vd_size[0].z, ex_vd_size[0].w);	
	CalcSunColors(p, secondary_color, front_color, v3Direction);
	ex_mie = CalcMie(front_color, secondary_color, v3Direction).xyz;	
	ex_ambient = vec4(ambientColor(p.z, ex_mie), CalcTransparency(p));
	ex_diffuse.xyz = directionalColor(p, ex_vd_center[0].xyz, ex_mie);
	ex_diffuse.w = clamp((p.z-ex_vd_bb_min[0].z) /(ex_vd_bb_max[0].z-ex_vd_bb_min[0].z),0.0,1.0);	
	
	ex_position = p;	
	ex_normal = normalize(p.xyz - cloudCenterPosition.xyz).xyz;//+0.5*normalize(p.xyz - ex_vd_center[0].xyz);
	ex_density = ex_vd_density[0];
	gl_Position =  transformation * p;
	EmitVertex();
}  