#version 330 core

uniform mat4 ProjectionMatrix;
uniform mat4 ViewMatrix;
uniform mat4 ModelMatrix;
uniform mat4 light_proj_matrix;
uniform mat4 light_mv_matrix;

uniform mat4 osg_ViewMatrix;


layout (location = 0) in vec4 in_position;
layout (location = 1) in vec4 in_texCoord;
layout (location = 2) in vec4 in_normal;

out float ex_lightdistance;
out vec2 ex_texCoord;
out vec4 ex_shadowCoord;

out vec4 ambientColor;
out vec4 diffuseColor;
out vec4 specularColor;

out vec3 normalEye;
out vec3 lightDir;
out vec3 positionEye;




uniform vec3 v3LightPos;		// The direction vector to the light source
uniform vec3 v3InvWavelength;	// 1 / pow(wavelength, 4) for the red, green, and blue channels
uniform float fOuterRadius;		// The outer (atmosphere) radius
uniform float fOuterRadius2;	// fOuterRadius^2
uniform float fInnerRadius;		// The inner (planetary) radius
uniform float fInnerRadius2;	// fInnerRadius^2
uniform float fKrESun;			// Kr * ESun
uniform float fKmESun;			// Km * ESun
uniform float fKr4PI;			// Kr * 4 * PI
uniform float fKm4PI;			// Km * 4 * PI
uniform float fScale;			// 1 / (fOuterRadius - fInnerRadius)
uniform float fScaleDepth;		// The scale depth (i.e. the altitude at which the atmosphere's average density is found)
uniform float fScaleOverScaleDepth;	// fScale / fScaleDepth


const int nSamples = 30;
const float fSamples = 30.0;

out vec3 lightVec;
out vec3 eyeVec;

out vec3 ex_secondary_color;
out vec3 ex_front_color;
out vec3 ex_v3Direction;

//Ground to space
float PI = 3.147;
float SkydomeRadius = 100000.0;
float InnerRadius = 1.0;
float OuterRadius = 1.25;
float InnerRadius2=pow(InnerRadius,2.0);
float OuterRadius2=pow(OuterRadius,2.0);
float ESun = 10.0;
float Km = 0.0075;
float Kr = Km/2.0;
float KrESun =  Kr * ESun;
float KmESun =  Km * ESun;
float Kr4PI = Kr * 4.0 * PI;
float Km4PI = Km * 4.0 * PI;
float Scale = 1.0/(OuterRadius - InnerRadius);	
float ScaleDepth = (OuterRadius - InnerRadius) / 2.0f;
float ScaleOverScaleDepth = Scale / ScaleDepth;

float scale(float fCos)
{
	float x = 1.0 - fCos;
	return ScaleDepth * 1.0 * exp(-0.00287 + x*(0.459 + x*(3.83 + x*(-6.80 + x*5.25))));
}

float getNearIntersection(vec3 v3Pos, vec3 v3Ray, float fDistance2, float fRadius2)
{
   float B = 2.0 * dot(v3Pos, v3Ray);
   float C = fDistance2 - fRadius2;
   float fDet = max(0.0, B*B - 4.0 * C);
   return 0.5 * (-B - sqrt(fDet));
}

void CalcSunColors(in vec4 vertex, out vec3 secondary_color, out vec3 front_color, out vec3 v3Direction)
{	
	vec4 lightPosition = inverse(light_mv_matrix) * vec4(0.0,0.0,0.0,1.0);

	vec3 v3CameraPos = (inverse(osg_ViewMatrix) * vec4(0.0, 0.0, 0.0, 1.0)).xyz;		
	vec3 vecCamera = v3CameraPos;		
	vecCamera /= SkydomeRadius;	
	vecCamera *= InnerRadius;	
	vecCamera.z += InnerRadius;	
	
	float fCameraHeight = length(vecCamera);
	float fCameraHeight2 = pow(fCameraHeight, 2.f);	
	
	vec4 lightP = vec4(lightPosition.xyz, 1.0);
	vec3 lightPos = ((lightP)).xyz;		
	lightPos = normalize(lightPos);
	
	
	// Get the ray from the camera to the vertex, and its length (which is the far point of the ray passing through the atmosphere)
	vec3 v3Pos = vertex.xyz;	
	vec3 Position = v3Pos;
	v3Pos /= SkydomeRadius;		
	v3Pos *= InnerRadius;	
	v3Pos.z += InnerRadius;	
	
	
	vec3 v3Ray = v3Pos-vecCamera;
	float fFar = length(v3Ray);
	v3Ray /= fFar;

	float near = 0.0; //getNearIntersection(v3Pos, v3Ray, length(v3Ray), InnerRadius2);
	
	// Calculate the ray's starting position, then calculate its scattering offset
	vec3 v3Start = vecCamera;// + (v3Ray * near);		
	fFar -= near;

	vec3 v3FrontColor = vec3(0.0);
	vec3 v3Attenuate = vec3(1.0);
		
	if(vecCamera.z > v3Pos.z)
	{
		float fDepth = exp((InnerRadius - fCameraHeight) / ScaleDepth);		// exp(fScaleOverScaleDepth * (InnerRadius - fCameraHeight));//
		float fLightAngle = dot(lightPos.xyz, v3Pos) / length(v3Pos);
		float fCameraAngle = dot(-v3Ray, v3Pos) / length(v3Pos);
		float fCameraScale = scale(fCameraAngle);
		float fLightScale = scale(fLightAngle);
		float fCameraOffset = fDepth*fCameraScale;
		float fTemp = (fLightScale + fCameraScale);

		// Initialize the scattering loop variables
		float fSampleLength = fFar / fSamples;
		float fScaledLength = fSampleLength * Scale;
		vec3 v3SampleRay = v3Ray * fSampleLength;	
		vec3 v3SamplePoint = v3Start + v3SampleRay * 0.5;

		float fHeight = length(v3SamplePoint);												//OK
		float fStartAngle = dot(-v3Ray, v3Start) / fHeight;
		float fStartOffset = fDepth*scale(fStartAngle);
		
		// Now loop through the sample rays
		
		for(int i=0; i<nSamples; i++)
		{
			float fHeight = length(v3SamplePoint);												//OK
			float fDepth = exp(ScaleOverScaleDepth * (InnerRadius - fHeight));					//OK		
			float fScatter = fDepth*fTemp - fCameraOffset;
			v3Attenuate = exp(-fScatter * (v3InvWavelength * Kr4PI + Km4PI));				//OK				
			v3FrontColor += v3Attenuate* (fDepth * fScaledLength);		
			v3SamplePoint += v3SampleRay;

		}
		
		secondary_color.xyz = v3Attenuate+0.41;	
		//secondary_color.xyz = v3FrontColor;
		front_color.rgb = v3FrontColor * (v3InvWavelength  * KrESun+ KmESun);
	}
	else
	{
		// Calculate the ray's starting position, then calculate its scattering offset
		float fHeight = length(v3Start);
		float fDepth = exp(ScaleOverScaleDepth * (InnerRadius - fCameraHeight));
		float fStartAngle = dot(v3Ray, v3Start) / fHeight;
		float fStartOffset = fDepth*scale(fStartAngle);

		// Initialize the scattering loop variables
		float fSampleLength = fFar / fSamples;
		float fScaledLength = fSampleLength * Scale;
		vec3 v3SampleRay = v3Ray * fSampleLength;
		vec3 v3SamplePoint = v3Start + (v3SampleRay * 0.5);

		// Now loop through the sample rays
		for(int i=0; i<nSamples; i++)
		{
			float fHeight = length(v3SamplePoint);
			float fDepth = exp(ScaleOverScaleDepth * (InnerRadius - fHeight));
			float fLightAngle = dot(lightPos, v3SamplePoint) / fHeight;
			float fCameraAngle = dot(v3Ray, v3SamplePoint) / fHeight;
			float fScatter = (fStartOffset + fDepth*(scale(fLightAngle) - scale(fCameraAngle)));
			v3Attenuate = exp(-fScatter * (v3InvWavelength * Kr4PI + Km4PI));
			v3FrontColor += (v3Attenuate * vec3(fDepth * fScaledLength));
			v3SamplePoint += v3SampleRay;
		}
		
		//secondary_color.xyz = v3FrontColor;
		secondary_color.xyz = v3Attenuate+0.41;			
		front_color.rgb = v3FrontColor * (v3InvWavelength  * KrESun+ KmESun);
	}
	
	

	// Finally, scale the Mie and Rayleigh colors and set up the varying variables for the pixel shader		
	v3Direction = -vecCamera+v3Pos;	
}

void main(void)
{ 
	vec4 lightPosition= (inverse(light_mv_matrix) * vec4(0.0,0.0,0.0,1.0));
	vec4 worldPos = ModelMatrix * in_position;

	vec4 lightAmbientColor=vec4(0.05);
	vec4 lightDiffuseColor=vec4(0.9);
	vec4 lightSpecularColor=vec4(1.0);
	vec4 materialAmbientColor=vec4(0.05);
	vec4 materialDiffuseColor=vec4(0.80);
	vec4 materialSpecularColor=vec4(1.0);	
	ambientColor = materialAmbientColor * lightAmbientColor;
	diffuseColor = materialDiffuseColor * lightDiffuseColor;
	specularColor = materialSpecularColor * lightSpecularColor;
	
	mat4 mvpMatrix = ProjectionMatrix * ViewMatrix;
	positionEye = worldPos.xyz;
	
	normalEye = normalize((inverse(transpose(ModelMatrix)) * vec4(in_normal.xyz, 0.0)).xyz);
	lightDir = normalize((lightPosition - worldPos).xyz);

	ex_shadowCoord = light_proj_matrix * light_mv_matrix * worldPos;
	ex_texCoord = in_texCoord.xy;
	gl_Position = mvpMatrix * worldPos;
	
	ex_lightdistance = length((lightPosition - worldPos).xyz);
	
	
	vec3 secondary_color;
	vec3 front_color;
	vec3 v3Direction;
	CalcSunColors(worldPos, secondary_color, front_color, v3Direction);
	
	ex_secondary_color = secondary_color;
	ex_front_color = front_color;
	ex_v3Direction = v3Direction;
}
