#pragma once
#include "CloudScene.h"
#include <osg/Depth>

namespace osgCloudyDay
{
	/**
	 * Class to create a HUD texture. 
	 * Mix the cloud, goodray and scene texture
	 * Convert the scene to luminance, which is used for HDR mapping.
	 * This class is parent for calculation of the average luminance, glare and star-effect as well as the final hdr mapping and color graining effect.
	 */
	class HUD
	{
	public:
		/**
		 * Constructor
		 * @param debug_tex texture to debug the scene
		 */
		HUD(osg::ref_ptr<osg::Texture> debug_tex);
		/**
		 * Constructor
		 */
		HUD(void);
		/**
		 * Deconstructor
		 */
		~HUD(void);

		/**
		 * Initialize the scene
		 */
		static void Initialize();

		/**
		 * Create the geometry for rendering
		 */
		void CreateGeometry();

		/**
		 * Setup the uniform
		 * @param states states of the geode
		 */
		virtual void SetupUniform(osg::ref_ptr<osg::StateSet> states);
		/**
		 * Setup the textures
		 * @param states states of the geode
		 */
		virtual void SetupTextures(osg::ref_ptr<osg::StateSet> states);

		/**
		 * Create the camera for the post-processing effect
		 */
		virtual void CreateCamera();

		/**
		 * Sets the scene texture
		 * @param tex scene texture
		 */
		void SetSceneTexture(osg::ref_ptr<osg::Texture2D> tex);
		/**
		 * Sets the godray texture
		 * @param tex godray texture
		 */
		void SetGodrayTexture(osg::ref_ptr<osg::Texture2D> tex);
		/**
		 * Sets the cloud texture
		 * @param tex cloud texture
		 */
		void SetCloudTexture(osg::ref_ptr<osg::Texture2D> tex);
	
		/**
		 * Returns the post-processing camera
		 * @return post-processing camera
		 */
		osg::ref_ptr<osg::Camera> GetCamera();	

	protected:
		osg::ref_ptr<osg::Texture2D> m_scenetex;
		osg::ref_ptr<osg::Texture2D> m_goodraytex;
		osg::ref_ptr<osg::Texture2D> m_cloudtex;

		osg::ref_ptr<osg::Program> sh_blur_linear;	
		osg::ref_ptr<osg::Uniform> uniform_blur_texsize;	

		osg::ref_ptr<osg::Uniform> uniform_avg_texsize;
		osg::ref_ptr<osg::Program> sh_lum;

		osg::ref_ptr<osg::Camera> cam_hud;	
		osg::ref_ptr<osg::StateSet> nodessP;
		osg::ref_ptr<osg::Program> planeProg;	
		osg::ref_ptr<osg::Geode> plane;

		int renderorder;

		osg::ref_ptr<osg::Texture> m_debug_tex;

		//CALCULATION OF LUMMINANCE
		static osg::ref_ptr<osg::Texture2D> fbo_lum0_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_lum1_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_lum2_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_lum3_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_lum4_texture;
		static osg::FrameBufferObject* fbo_Lum0;
		static osg::FrameBufferObject* fbo_Lum1;
		static osg::FrameBufferObject* fbo_Lum2;
		static osg::FrameBufferObject* fbo_Lum3;
		static osg::FrameBufferObject* fbo_Lum4;	


		//POST PROCESSING: STAR EFFECT AND GLARE EFFECT

		static osg::ref_ptr<osg::Texture2D> fbo_blurV1_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_blurV2_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_blurV3_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_blurV4_texture;
	
		static osg::ref_ptr<osg::Texture2D> fbo_blurH1_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_blurH2_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_blurH3_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_blurH4_texture;
	
		static osg::ref_ptr<osg::Texture2D> fbo_star1_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_star2_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_star3_texture;
		static osg::ref_ptr<osg::Texture2D> fbo_star4_texture;	
	
		static osg::FrameBufferObject* fbo_V1_blur;
		static osg::FrameBufferObject* fbo_H1_blur;
		static osg::FrameBufferObject* fbo_V2_blur;
		static osg::FrameBufferObject* fbo_H2_blur;
		static osg::FrameBufferObject* fbo_V3_blur;
		static osg::FrameBufferObject* fbo_H3_blur;
		static osg::FrameBufferObject* fbo_V4_blur;
		static osg::FrameBufferObject* fbo_H4_blur;

		static osg::ref_ptr<osg::Texture2D> fbo_glare_texture;
	};

	/**
	 * Class manages the uniforms of luminance caluclation
	 */
	class Luminance : public osg::Geometry
	{
	public:
		/**
		 * Constructor
		 * @param lum_prog average luminance caluclation program
		 * @param _fbo_scene_texture scene texture
		 * @param _fbo_Lum0 reduced size framebuffer 256x256
		 * @param _fbo_Lum1 reduced size framebuffer 64x64
		 * @param _fbo_Lum2 reduced size framebuffer 16x16
		 * @param _fbo_Lum3 reduced size framebuffer 4x4
		 * @param _fbo_Lum4 reduced size framebuffer 1x1 
		 * @param _fbo_lum0 reduced size texture 256x256
		 * @param _fbo_lum1 reduced size texture 64x64
		 * @param _fbo_lum2 reduced size texture 16x16
		 * @param _fbo_lum3 reduced size texture 4x4
		 * @param _fbo_lum4 reduced size texture 1x1
		 */
		Luminance(  osg::ref_ptr<osg::Program> lum_prog,
						 osg::ref_ptr<osg::Texture2D> _fbo_scene_texture,
						 osg::ref_ptr<osg::FrameBufferObject> _fbo_Lum0,
						 osg::ref_ptr<osg::FrameBufferObject> _fbo_Lum1,
						 osg::ref_ptr<osg::FrameBufferObject> _fbo_Lum2,
						 osg::ref_ptr<osg::FrameBufferObject> _fbo_Lum3,
						 osg::ref_ptr<osg::FrameBufferObject> _fbo_Lum4,
						 osg::ref_ptr<osg::Texture2D> _fbo_lum0,
						 osg::ref_ptr<osg::Texture2D> _fbo_lum1,
						 osg::ref_ptr<osg::Texture2D> _fbo_lum2,
						 osg::ref_ptr<osg::Texture2D> _fbo_lum3,
						 osg::ref_ptr<osg::Texture2D> _fbo_lum4)
		{
			sh_blur_linear_vert1 = (lum_prog);
			
			fbo_scene_texture = _fbo_scene_texture;

			fbo_Lum0 = _fbo_Lum0;
			fbo_Lum1 = _fbo_Lum1;
			fbo_Lum2 = _fbo_Lum2;
			fbo_Lum3 = _fbo_Lum3,
			fbo_Lum4 = _fbo_Lum4;
			fbo_lum0_texture = _fbo_lum0;
			fbo_lum1_texture = _fbo_lum1;
			fbo_lum2_texture = _fbo_lum2;
			fbo_lum3_texture = _fbo_lum3;
			fbo_lum4_texture = _fbo_lum4;
		}
		/**
		 * Deconstructor
		 */
		~Luminance(void)
		{

		}

		/**
		 * Updates the uniforms at runtime
		 */
		void drawImplementation(osg::RenderInfo& renderInfo) const
		{	
			if (_internalOptimizedGeometry.valid())
			{
				_internalOptimizedGeometry->drawImplementation(renderInfo);
				return;
			}

			osg::State& state = *renderInfo.getState();

			bool checkForGLErrors = state.getCheckForGLErrors()==osg::State::ONCE_PER_ATTRIBUTE;
			if (checkForGLErrors) state.checkGLErrors("start of Geometry::drawImplementation()");
    
			bool useFastPath = areFastPathsUsed();
			// useFastPath = false;

			bool usingVertexBufferObjects = _useVertexBufferObjects && state.isVertexBufferObjectSupported();
			bool handleVertexAttributes = !_vertexAttribList.empty();

			osg::ArrayDispatchers& arrayDispatchers = state.getArrayDispatchers();

			arrayDispatchers.reset();
			arrayDispatchers.setUseVertexAttribAlias(useFastPath && state.getUseVertexAttributeAliasing());
			arrayDispatchers.setUseGLBeginEndAdapter(!useFastPath);

			arrayDispatchers.activateNormalArray(_normalData.binding, _normalData.array.get(), _normalData.indices.get());
			arrayDispatchers.activateColorArray(_colorData.binding, _colorData.array.get(), _colorData.indices.get());
			arrayDispatchers.activateSecondaryColorArray(_secondaryColorData.binding, _secondaryColorData.array.get(), _secondaryColorData.indices.get());
			arrayDispatchers.activateFogCoordArray(_fogCoordData.binding, _fogCoordData.array.get(), _fogCoordData.indices.get());

			if (handleVertexAttributes)
			{
				for(unsigned int unit=0;unit<_vertexAttribList.size();++unit)
				{
					arrayDispatchers.activateVertexAttribArray(_vertexAttribList[unit].binding, unit, _vertexAttribList[unit].array.get(), _vertexAttribList[unit].indices.get());
				}
			}

			// dispatch any attributes that are bound overall
			arrayDispatchers.dispatch(BIND_OVERALL,0);

			state.lazyDisablingOfVertexAttributes();

			if (useFastPath)
			{
				// set up arrays
				if( _vertexData.array.valid() )
					state.setVertexPointer(_vertexData.array.get());

				if (_normalData.binding==BIND_PER_VERTEX && _normalData.array.valid())
					state.setNormalPointer(_normalData.array.get());

				if (_colorData.binding==BIND_PER_VERTEX && _colorData.array.valid())
					state.setColorPointer(_colorData.array.get());

				if (_secondaryColorData.binding==BIND_PER_VERTEX && _secondaryColorData.array.valid())
					state.setSecondaryColorPointer(_secondaryColorData.array.get());

				if (_fogCoordData.binding==BIND_PER_VERTEX && _fogCoordData.array.valid())
					state.setFogCoordPointer(_fogCoordData.array.get());

				for(unsigned int unit=0;unit<_texCoordList.size();++unit)
				{
					const osg::Array* array = _texCoordList[unit].array.get();
					if (array) state.setTexCoordPointer(unit,array);
				}

				if( handleVertexAttributes )
				{
					for(unsigned int index = 0; index < _vertexAttribList.size(); ++index )
					{
						const osg::Array* array = _vertexAttribList[index].array.get();
						const AttributeBinding ab = _vertexAttribList[index].binding;
						if( ab == BIND_PER_VERTEX && array )
						{
							state.setVertexAttribPointer( index, array, _vertexAttribList[index].normalize );
						}
					}
				}
			}
			else
			{
				for(unsigned int unit=0;unit<_texCoordList.size();++unit)
				{
					arrayDispatchers.activateTexCoordArray(BIND_PER_VERTEX, unit, _texCoordList[unit].array.get(), _texCoordList[unit].indices.get());
				}

				arrayDispatchers.activateVertexArray(BIND_PER_VERTEX, _vertexData.array.get(), _vertexData.indices.get());
			}

			state.applyDisablingOfVertexAttributes();

			bool bindPerPrimitiveSetActive = arrayDispatchers.active(BIND_PER_PRIMITIVE_SET);
			bool bindPerPrimitiveActive = arrayDispatchers.active(BIND_PER_PRIMITIVE);			

			if (checkForGLErrors) state.checkGLErrors("Geometry::drawImplementation() after vertex arrays setup.");
		
	
			osg::GL2Extensions* extension = osg::GL2Extensions::Get(0,true);

			glClearColor(0.f, 0.f, 0.f, 0.f);
			///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
			//
			// draw the primitives themselves.
			//	
			
			
			fbo_Lum0->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
			glViewport(0,0,256, 256);		
			glClear( GL_COLOR_BUFFER_BIT);

			fbo_Lum1->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
			glViewport(0,0,64, 64);		
			glClear( GL_COLOR_BUFFER_BIT);

			fbo_Lum2->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
			glViewport(0,0,16, 16);		
			glClear( GL_COLOR_BUFFER_BIT);
			
			fbo_Lum3->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
			glViewport(0,0,4, 4);		
			glClear( GL_COLOR_BUFFER_BIT);

			fbo_Lum4->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
			glViewport(0,0,1, 1);		
			glClear( GL_COLOR_BUFFER_BIT);
			

			glViewport(0,0,osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());		
		
			for(unsigned int primitiveSetNum=0; primitiveSetNum!= _primitives.size() ; ++primitiveSetNum)
			{			
				// dispatch any attributes that are bound per primitive
				if (bindPerPrimitiveSetActive) 				
					arrayDispatchers.dispatch(BIND_PER_PRIMITIVE_SET, primitiveSetNum);
				
				const osg::PrimitiveSet* primitiveset = _primitives[primitiveSetNum].get();									
												
				glDisable(GL_BLEND);
				glDisable(GL_DEPTH_TEST);								
				glDisable(GL_CULL_FACE);

				osg::State& state = *renderInfo.getState();

				osg::Depth* depth2 = new osg::Depth(osg::Depth::ALWAYS, 0.0, 1.0, false);	//MIT DEPTH => FEHLER	
				depth2->apply(state);
				
							
				glViewport(0, 0, 256, 256);				
				sh_blur_linear_vert1->apply(state);
				state.applyTextureAttribute(0, fbo_scene_texture);
				fbo_Lum0->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
				uniform_avg_texsize->set(256.f);
				uniform_avg_texsize->apply(extension, state.getUniformLocation("texsize"));
				primitiveset->draw(state, usingVertexBufferObjects);				

				glViewport(0, 0, 64, 64);				
				sh_blur_linear_vert1->apply(state);
				state.applyTextureAttribute(0, fbo_lum0_texture);
				fbo_Lum1->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
				uniform_avg_texsize->set(64.f);
				uniform_avg_texsize->apply(extension, state.getUniformLocation("texsize"));
				primitiveset->draw(state, usingVertexBufferObjects); 													

				glViewport(0, 0, 16, 16);				
				sh_blur_linear_vert1->apply(state);
				state.applyTextureAttribute(0, fbo_lum1_texture);				
				fbo_Lum2->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
				uniform_avg_texsize->set(16.f);
				uniform_avg_texsize->apply(extension, state.getUniformLocation("texsize"));
				primitiveset->draw(state, usingVertexBufferObjects); 													

				glViewport(0, 0, 4, 4);				
				sh_blur_linear_vert1->apply(state);
				state.applyTextureAttribute(0, fbo_lum2_texture);
				fbo_Lum3->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
				uniform_avg_texsize->set(4.f);
				uniform_avg_texsize->apply(extension, state.getUniformLocation("texsize"));
				primitiveset->draw(state, usingVertexBufferObjects); 													

				glViewport(0, 0, 1, 1);				
				sh_blur_linear_vert1->apply(state);
				state.applyTextureAttribute(0, fbo_lum3_texture);
				fbo_Lum4->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
				uniform_avg_texsize->set(1.f);
				uniform_avg_texsize->apply(extension, state.getUniformLocation("texsize"));
				primitiveset->draw(state, usingVertexBufferObjects); 																	
								
				glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());
				glEnable(GL_DEPTH_TEST);						
			}

			// unbind the VBO's if any are used.
			state.unbindVertexBufferObject();
			state.unbindElementBufferObject();
	
			if (checkForGLErrors) state.checkGLErrors("end of Geometry::drawImplementation().");
		}		

	public:
		osg::ref_ptr<osg::Uniform> uniform_avg_texsize;

	protected:
		osg::ref_ptr<osg::Program> sh_blur_linear_vert1;		
		osg::ref_ptr<osg::FrameBufferObject> fbo_Lum0;
		osg::ref_ptr<osg::FrameBufferObject> fbo_Lum1;
		osg::ref_ptr<osg::FrameBufferObject> fbo_Lum2;
		osg::ref_ptr<osg::FrameBufferObject> fbo_Lum3;
		osg::ref_ptr<osg::FrameBufferObject> fbo_Lum4;
		osg::ref_ptr<osg::Texture2D> fbo_lum0_texture;
		osg::ref_ptr<osg::Texture2D> fbo_lum1_texture;
		osg::ref_ptr<osg::Texture2D> fbo_lum2_texture;
		osg::ref_ptr<osg::Texture2D> fbo_lum3_texture;
		osg::ref_ptr<osg::Texture2D> fbo_lum4_texture;

	
		osg::ref_ptr<osg::Texture2D> fbo_scene_texture;
	};

	/**
	 * Class manages the uniforms of blur and star effect
	 */
	class BlurGeometry : public osg::Geometry
	{
	public:
		BlurGeometry(  osg::ref_ptr<osg::Program> prog_vert1, int mode,
			osg::ref_ptr<osg::Texture2D> _fbo_glare_texture,
			osg::ref_ptr<osg::Uniform> _uniform_blur_texsize,

			osg::ref_ptr<osg::FrameBufferObject> _fbo_V1_blur, osg::ref_ptr<osg::FrameBufferObject> _fbo_H1_blur,
			osg::ref_ptr<osg::FrameBufferObject> _fbo_V2_blur, osg::ref_ptr<osg::FrameBufferObject> _fbo_H2_blur,
			osg::ref_ptr<osg::FrameBufferObject> _fbo_V3_blur, osg::ref_ptr<osg::FrameBufferObject> _fbo_H3_blur,
			osg::ref_ptr<osg::FrameBufferObject> _fbo_V4_blur, osg::ref_ptr<osg::FrameBufferObject> _fbo_H4_blur,
			
			osg::ref_ptr<osg::Texture2D> _fbo_blurV1_texture, osg::ref_ptr<osg::Texture2D> _fbo_blurH1_texture,
			osg::ref_ptr<osg::Texture2D> _fbo_blurV2_texture, osg::ref_ptr<osg::Texture2D> _fbo_blurH2_texture,
			osg::ref_ptr<osg::Texture2D> _fbo_blurV3_texture, osg::ref_ptr<osg::Texture2D> _fbo_blurH3_texture,
			osg::ref_ptr<osg::Texture2D> _fbo_blurV4_texture, osg::ref_ptr<osg::Texture2D> _fbo_blurH4_texture
			)
		{
			sh_blur_linear_vert1 = (prog_vert1);
			m_mode = mode;

			fbo_glare_texture = _fbo_glare_texture;
			uniform_blur_texsize = _uniform_blur_texsize;

			fbo_V1_blur = _fbo_V1_blur; fbo_H1_blur = _fbo_H1_blur;
			fbo_V2_blur = _fbo_V2_blur; fbo_H2_blur = _fbo_H2_blur;
			fbo_V3_blur = _fbo_V3_blur; fbo_H3_blur = _fbo_H3_blur;
			fbo_V4_blur = _fbo_V4_blur; fbo_H4_blur = _fbo_H4_blur;

			fbo_blurV1_texture = _fbo_blurV1_texture; fbo_blurH4_texture = _fbo_blurH1_texture;
			fbo_blurV2_texture = _fbo_blurV2_texture; fbo_blurH4_texture = _fbo_blurH2_texture;
			fbo_blurV3_texture = _fbo_blurV3_texture; fbo_blurH4_texture = _fbo_blurH3_texture;
			fbo_blurV4_texture = _fbo_blurV4_texture; fbo_blurH4_texture = _fbo_blurH4_texture;
		}
		/**
		 * Deconstructor
		 */
		~BlurGeometry(void)
		{

		}

		/**
		 * Updates the uniforms at runtime
		 */
		void drawImplementation(osg::RenderInfo& renderInfo) const
		{	
			if (_internalOptimizedGeometry.valid())
			{
				_internalOptimizedGeometry->drawImplementation(renderInfo);
				return;
			}

			osg::State& state = *renderInfo.getState();

			bool checkForGLErrors = state.getCheckForGLErrors()==osg::State::ONCE_PER_ATTRIBUTE;
			if (checkForGLErrors) state.checkGLErrors("start of Geometry::drawImplementation()");
    
			bool useFastPath = areFastPathsUsed();
			// useFastPath = false;

			bool usingVertexBufferObjects = _useVertexBufferObjects && state.isVertexBufferObjectSupported();
			bool handleVertexAttributes = !_vertexAttribList.empty();

			osg::ArrayDispatchers& arrayDispatchers = state.getArrayDispatchers();

			arrayDispatchers.reset();
			arrayDispatchers.setUseVertexAttribAlias(useFastPath && state.getUseVertexAttributeAliasing());
			arrayDispatchers.setUseGLBeginEndAdapter(!useFastPath);

			arrayDispatchers.activateNormalArray(_normalData.binding, _normalData.array.get(), _normalData.indices.get());
			arrayDispatchers.activateColorArray(_colorData.binding, _colorData.array.get(), _colorData.indices.get());
			arrayDispatchers.activateSecondaryColorArray(_secondaryColorData.binding, _secondaryColorData.array.get(), _secondaryColorData.indices.get());
			arrayDispatchers.activateFogCoordArray(_fogCoordData.binding, _fogCoordData.array.get(), _fogCoordData.indices.get());

			if (handleVertexAttributes)
			{
				for(unsigned int unit=0;unit<_vertexAttribList.size();++unit)
				{
					arrayDispatchers.activateVertexAttribArray(_vertexAttribList[unit].binding, unit, _vertexAttribList[unit].array.get(), _vertexAttribList[unit].indices.get());
				}
			}

			// dispatch any attributes that are bound overall
			arrayDispatchers.dispatch(BIND_OVERALL,0);

			state.lazyDisablingOfVertexAttributes();

			if (useFastPath)
			{
				// set up arrays
				if( _vertexData.array.valid() )
					state.setVertexPointer(_vertexData.array.get());

				if (_normalData.binding==BIND_PER_VERTEX && _normalData.array.valid())
					state.setNormalPointer(_normalData.array.get());

				if (_colorData.binding==BIND_PER_VERTEX && _colorData.array.valid())
					state.setColorPointer(_colorData.array.get());

				if (_secondaryColorData.binding==BIND_PER_VERTEX && _secondaryColorData.array.valid())
					state.setSecondaryColorPointer(_secondaryColorData.array.get());

				if (_fogCoordData.binding==BIND_PER_VERTEX && _fogCoordData.array.valid())
					state.setFogCoordPointer(_fogCoordData.array.get());

				for(unsigned int unit=0;unit<_texCoordList.size();++unit)
				{
					const osg::Array* array = _texCoordList[unit].array.get();
					if (array) state.setTexCoordPointer(unit,array);
				}

				if( handleVertexAttributes )
				{
					for(unsigned int index = 0; index < _vertexAttribList.size(); ++index )
					{
						const osg::Array* array = _vertexAttribList[index].array.get();
						const AttributeBinding ab = _vertexAttribList[index].binding;
						if( ab == BIND_PER_VERTEX && array )
						{
							state.setVertexAttribPointer( index, array, _vertexAttribList[index].normalize );
						}
					}
				}
			}
			else
			{
				for(unsigned int unit=0;unit<_texCoordList.size();++unit)
				{
					arrayDispatchers.activateTexCoordArray(BIND_PER_VERTEX, unit, _texCoordList[unit].array.get(), _texCoordList[unit].indices.get());
				}

				arrayDispatchers.activateVertexArray(BIND_PER_VERTEX, _vertexData.array.get(), _vertexData.indices.get());
			}

			state.applyDisablingOfVertexAttributes();

			bool bindPerPrimitiveSetActive = arrayDispatchers.active(BIND_PER_PRIMITIVE_SET);
			bool bindPerPrimitiveActive = arrayDispatchers.active(BIND_PER_PRIMITIVE);			

			if (checkForGLErrors) state.checkGLErrors("Geometry::drawImplementation() after vertex arrays setup.");
		
	
			osg::GL2Extensions* extension = osg::GL2Extensions::Get(0,true);

			glClearColor(0.f, 0.1f, 0.f, 1.f);
			///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
			//
			// draw the primitives themselves.
			//	
			
			if(m_mode == 0)
			{
				fbo_V1_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glViewport(0,0,osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());		
				glClear( GL_COLOR_BUFFER_BIT);

				fbo_H1_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glClear( GL_COLOR_BUFFER_BIT);


				fbo_V2_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glViewport(0,0,osgCloudyDay::Scene::GetWidth()/2, osgCloudyDay::Scene::GetHeight()/2);		
				glClear( GL_COLOR_BUFFER_BIT);

				fbo_H2_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glClear( GL_COLOR_BUFFER_BIT);


				fbo_V3_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glViewport(0,0,osgCloudyDay::Scene::GetWidth()/4, osgCloudyDay::Scene::GetHeight()/4);		
				glClear( GL_COLOR_BUFFER_BIT);

				fbo_H3_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);					
				glClear( GL_COLOR_BUFFER_BIT);


				fbo_V4_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glViewport(0,0,osgCloudyDay::Scene::GetWidth()/8, osgCloudyDay::Scene::GetHeight()/8);		
				glClear( GL_COLOR_BUFFER_BIT);

				fbo_H4_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glClear( GL_COLOR_BUFFER_BIT);			
			}


			glViewport(0,0,osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());		
		
			for(unsigned int primitiveSetNum=0; primitiveSetNum!= _primitives.size() ; ++primitiveSetNum)
			{			
				// dispatch any attributes that are bound per primitive
				if (bindPerPrimitiveSetActive) 				
					arrayDispatchers.dispatch(BIND_PER_PRIMITIVE_SET, primitiveSetNum);
				
				const osg::PrimitiveSet* primitiveset = _primitives[primitiveSetNum].get();									
												
				glDisable(GL_BLEND);
				glDisable(GL_DEPTH_TEST);								
				glDisable(GL_CULL_FACE);

				osg::State& state = *renderInfo.getState();

				osg::Depth* depth2 = new osg::Depth(osg::Depth::ALWAYS, 0.0, 1.0, false);	//MIT DEPTH => FEHLER	
				depth2->apply(state);
				
				if(m_mode == 0)
				{					
					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_glare_texture);
					fbo_V1_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetWidth()));
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects);				

					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth()/2, osgCloudyDay::Scene::GetHeight()/2);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_glare_texture);
					fbo_V2_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetWidth())/2.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 													

					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth()/4, osgCloudyDay::Scene::GetHeight()/4);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_glare_texture);
					fbo_V3_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetWidth())/4.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 													

					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth()/8, osgCloudyDay::Scene::GetHeight()/8);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_glare_texture);
					fbo_V4_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetWidth())/8.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 																	
				}
				else
				{	
					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_blurV1_texture);					
					state.applyTextureAttribute(1, fbo_glare_texture);
					fbo_H1_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetHeight()));
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 													
									
					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth()/2, osgCloudyDay::Scene::GetHeight()/2);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_blurV2_texture);
					state.applyTextureAttribute(1, fbo_glare_texture);
					fbo_H2_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetHeight())/2.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 												

					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth()/4, osgCloudyDay::Scene::GetHeight()/4);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, fbo_blurV3_texture);
					state.applyTextureAttribute(1, fbo_glare_texture);
					fbo_H3_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetHeight())/4.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 																	
				
					glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth()/8, osgCloudyDay::Scene::GetHeight()/8);				
					sh_blur_linear_vert1->apply(state);		
					state.applyTextureAttribute(0, fbo_blurV4_texture);
					state.applyTextureAttribute(1, fbo_glare_texture);
					fbo_H4_blur->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set((float)(osgCloudyDay::Scene::GetHeight())/8.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 													
				}
				
				glViewport(0.f, 0.f, osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());
				glEnable(GL_DEPTH_TEST);						
			}

			// unbind the VBO's if any are used.
			state.unbindVertexBufferObject();
			state.unbindElementBufferObject();
	
			if (checkForGLErrors) state.checkGLErrors("end of Geometry::drawImplementation().");
		}
				
		int m_mode;

	private:
		osg::ref_ptr<osg::Program> sh_blur_linear_vert1;

		osg::ref_ptr<osg::FrameBufferObject> fbo_V1_blur; osg::ref_ptr<osg::FrameBufferObject> fbo_H1_blur;
		osg::ref_ptr<osg::FrameBufferObject> fbo_V2_blur; osg::ref_ptr<osg::FrameBufferObject> fbo_H2_blur;
		osg::ref_ptr<osg::FrameBufferObject> fbo_V3_blur; osg::ref_ptr<osg::FrameBufferObject> fbo_H3_blur;
		osg::ref_ptr<osg::FrameBufferObject> fbo_V4_blur; osg::ref_ptr<osg::FrameBufferObject> fbo_H4_blur;

		osg::ref_ptr<osg::Texture2D> fbo_blurV1_texture; osg::ref_ptr<osg::Texture2D> fbo_blurH1_texture;
		osg::ref_ptr<osg::Texture2D> fbo_blurV2_texture; osg::ref_ptr<osg::Texture2D> fbo_blurH2_texture;
		osg::ref_ptr<osg::Texture2D> fbo_blurV3_texture; osg::ref_ptr<osg::Texture2D> fbo_blurH3_texture;
		osg::ref_ptr<osg::Texture2D> fbo_blurV4_texture; osg::ref_ptr<osg::Texture2D> fbo_blurH4_texture;

		osg::ref_ptr<osg::Texture2D> fbo_glare_texture;
		osg::ref_ptr<osg::Uniform> uniform_blur_texsize;
	};

	/**
	 * Class manages the uniforms of viewer hud
	 */
	class FBODrawCallback : public osg::Drawable::DrawCallback
	{
	public:
		/**
		 * Constructor
		 * @param texture texture scene
		 * @param texture2 texture cloud
		 */
		FBODrawCallback(osg::Texture* texture, osg::Texture* texture2) 
			: osg::Drawable::DrawCallback()
			, _texture(texture)
			, _texture2(texture2)
		{}
		/**
		 * Deconstructor
		 */
		~FBODrawCallback()
		{}

		/**
		 * Updates the uniforms at runtime
		 */
		void drawImplementation(osg::RenderInfo& ri,const osg::Drawable* dr) const
		{       
			//std::cout << "draw HUD" << std::endl;
			//osg::FBOExtensions::instance(0,true)->__glewGenerateMipmap(GL_TEXTURE_2D);		

			osg::FBOExtensions* fbo_ext = osg::FBOExtensions::instance(ri.getContextID(),true);
			ri.getState()->setActiveTextureUnit(0);
			ri.getState()->applyTextureAttribute(0, _texture);        
			//_texture->setUseHardwareMipMapGeneration(true);
			_texture->getTextureObject(ri.getContextID())->bind();
			//fbo_ext->__glewGenerateMipmap(_texture->getTextureTarget());
		
			ri.getState()->setActiveTextureUnit(1);
			ri.getState()->applyTextureAttribute(1, _texture2);
			//_texture2->setUseHardwareMipMapGeneration(true);
			_texture2->getTextureObject(ri.getContextID())->bind();
			//fbo_ext->__glewGenerateMipmap(_texture2->getTextureTarget());	
		
			dr->drawImplementation(ri);

		}
		osg::Texture* _texture;
		osg::Texture* _texture2;
	};

	/**
	 * Class manages the uniforms of viewer hud
	 */
	class ViewerCallbackHUD : public osg::NodeCallback
	{
	public:
		/**
		 * Constructor
		 */
		ViewerCallbackHUD(void)
		{

		}
		/**
		 * Deconstructor
		 */
		~ViewerCallbackHUD(){}

		/**
		 * Updates the uniforms at runtime
		 */
		virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
		{
			osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);	   
			osg::ref_ptr<osg::Geode> geometry = dynamic_cast<osg::Geode*> (node);		

			if(geometry && cv)
			{
				osg::RefMatrixd* m = cv->getModelViewMatrix();
				osg::Matrixd viewMatrix = Scene::GetViewMatrix_View();
				osg::Matrixd invViewMatrix = osg::Matrixd::inverse(Scene::GetViewMatrix_View());
				osg::Matrixd worldmat = invViewMatrix * (*m);
				osg::Matrixd invWorldMatrix = osg::Matrixd::inverse(worldmat);

				geometry->getOrCreateStateSet()->getUniform("ProjectionMatrix")->set(Scene::GetProjectionMatrix_View());
				geometry->getOrCreateStateSet()->getUniform("ModelViewMatrix")->set(viewMatrix*worldmat);
				geometry->getOrCreateStateSet()->getUniform("ViewMatrix")->set(viewMatrix);	
				if(Scene::m_skydome != 0) 
				{
					osg::Vec3 lightpos = Scene::m_skydome->GetLightPosition();
					geometry->getOrCreateStateSet()->getUniform("un_lightPos")->set(lightpos);			
				}
			
			}
			traverse(node, nv);
			
		}
	};

	class UniformBlurTexsize : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		UniformBlurTexsize() {}
		/**
		 * Deconstructor
		 */
		~UniformBlurTexsize() {}

		/**
		 * Updates the uniforms at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( (float)(osgCloudyDay::Scene::GetWidth()));}
	};
}