#pragma once
#include "Cloud2DState.h"
#include "ObjectModel.h"
#include "Ground.h"
#include "HUD.h"
#include "PostProcess.h"
#include "AtmosphereMie.h"
#include "AtmosphereHimmel.h"
#include "TerrainMIE.h"
#include "TerrainHimmel.h"
#include "TerrainGeometry.h"
#include "ReflectionCamera.h"
#include "CloudCreator.h"
#include "Atmosphereprecompute.h"

#include <osg/PolygonMode>
#include <osg/ShapeDrawable>
#include <osg/Depth>
#include <osg/Program>
#include <osg/Shader>
#include <osg/Uniform>

namespace osgCloudyDay
{
	/**
	 * Class, which updates the 2D clouds at runtime
	 */
	class PerlinNoiseCallback : public osg::NodeCallback
	{
	public:
		/**
		 * Constructor
		 */
		PerlinNoiseCallback(void)
		{

		}

		/**
		 * Set the uniforms at runtime
		 */
		virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
		{
			osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);	   
			osg::ref_ptr<osg::Geode> geometry = dynamic_cast<osg::Geode*> (node);

			if(geometry && cv)
			{
				osg::RefMatrixd* m = cv->getModelViewMatrix();
				osg::Matrixd viewMatrix = cv->getCurrentCamera()->getViewMatrix();
				osg::Matrixd invViewMatrix = osg::Matrixd::inverse(viewMatrix);
				osg::Matrixd worldmat = invViewMatrix * (*m);
				osg::Matrixd invWorldMatrix = osg::Matrixd::inverse(worldmat);
	
				geometry->getOrCreateStateSet()->getUniform("ProjectionMatrix")->set(cv->getCurrentCamera()->getProjectionMatrix());
				geometry->getOrCreateStateSet()->getUniform("ViewMatrix")->set(cv->getCurrentCamera()->getViewMatrix());
				geometry->getOrCreateStateSet()->getUniform("ModelMatrix")->set(worldmat);
				geometry->getOrCreateStateSet()->getUniform("NormalMatrix")->set(invWorldMatrix);
				geometry->getOrCreateStateSet()->getUniform("LightMatrix")->set(Scene::GetLightCamera()->getViewMatrix());			
			}
			traverse(node, nv);
		}
	};

	/**
	 * Class to create a 2d cloud
	 */
	class Create2DCloud
	{
	public:
		/**
		 * Constructor
		 */
		Create2DCloud(void);
		/**
		 * Deconstructor
		 */
		~Create2DCloud(void);

		/**
		 * Initialize the 2D clouds.
		 * @aram cloud2dstates  configuration of 2D clouds
		 */
		void Initialize(std::vector<Cloud2DState> cloud2dstates);
		/**
		 * Results a geode to render it from the viewer
		 * @return geode
		 */
		osg::Node* GetNode();
		/**
		 * Results a geode to render it from the lightsource
		 * @return geode
		 */
		osg::Node* GetNodeLight();
		/**
		 * Results a geode to render it for reflections
		 * @param at index of the camera
		 * @return geode
		 */
		osg::Node* GetReflectionCamera(int at);

	protected:
		std::vector<osg::ref_ptr<osg::Geode>> m_geodes;
		std::vector<osg::ref_ptr<osg::Geode>> m_light_geodes;

		osg::Group* m_node_scene;
		osg::Group* m_node_light;
		osg::Group** m_reflection;
	};
}