#pragma once
#include "CloudGeometry.h"
#include "CloudLayerUpdate.h"

#include "Skydome.h"
#include "Fog.h"
#include "WangCloud.h"
#include "CloudLayerUpdate.h"
#include "CloudLayerState.h"

#define SHADOW_MAPPING

namespace osgCloudyDay
{
	/**
	 * Class, updates the project uniform at runtime
	 */
	class project : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		project() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( Scene::GetProjectionMatrix_View() );}
	};
	/**
	 * Class, updates the viewMatrix uniform at runtime
	 */
	class viewMatrix : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		viewMatrix() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( Scene::GetViewCamera()->getViewMatrix() );}
	};
	/**
	 * Class, updates the modelViewMatrix uniform at runtime
	 */
	class modelViewMatrix : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		modelViewMatrix() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( Scene::GetViewCamera()->getViewMatrix() );}
	};
	/**
	 * Class, updates the viewMatrixInv uniform at runtime
	 */
	class viewMatrixInv : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		viewMatrixInv() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( osg::Matrix::inverse(Scene::GetViewCamera()->getViewMatrix()));}
	};
	/**
	 * Class, updates the project_light uniform at runtime
	 */
	class project_light : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		project_light() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( Scene::GetProjectionMatrix_Light() );}
	};
	/**
	 * Class, updates the viewMatrix_light uniform at runtime
	 */
	class viewMatrix_light : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		viewMatrix_light() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( Scene::GetLightCamera()->getViewMatrix() );}
	};
	/**
	 * Class, updates the modelViewMatrix_light uniform at runtime
	 */
	class modelViewMatrix_light : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		modelViewMatrix_light() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( Scene::GetLightCamera()->getViewMatrix() );}
	};
	/**
	 * Class, updates the viewMatrixInv_light uniform at runtime
	 */
	class viewMatrixInv_light : public osg::Uniform::Callback
	{
	public:
		/**
		 * Constructor
		 */
		viewMatrixInv_light() {}
		/**
		 * Updates uniform at runtime
		 */
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( osg::Matrix::inverse(Scene::GetProjectionMatrix_Light()) );}
	};

	/**
	 * Class, which is used to manage the 3D clouds
	 */
	class CloudScene : public osg::Referenced
	{
	public:
		/**
		 * Cloud types
		 */
		enum CloudType
		{
			CT_Stratus,
			CT_AltStratus,
			CT_StratoCumulusGenerator,
			CT_Cumulus,
			CT_Nimbostratus,
			CT_Cumolonimbus,
			CT_AltCumulusGenerator,
			CT_Quantity
		};

		enum WC_Uniform
		{
			WCU_LightPos,
			WCU_TimeOfDay,
			WCU_Fading,
			WCU_Density,
			WCU_AmbientLight_H0,
			WCU_AmbientLight_H1,
			WCU_AmbientLight_T0,
			WCU_AmbientLight_T1,
			WCU_DirectionalColor,
			WCU_SunLightColor,
			quantity
		};

	public:
		/**
		 * Constructor
		 */
		CloudScene();
		//CloudScene(int m_category, osg::ref_ptr<osg::Vec3Array> vertices, osg::ref_ptr<osg::Vec4Array> rotation, osg::ref_ptr<osg::Vec4Array> center, osg::ref_ptr<osg::Vec4Array> ids, osg::ref_ptr<osg::Vec4Array> boxcenters, osg::ref_ptr<osg::Vec3Array> ambientlight_h, osg::ref_ptr<osg::Vec3Array> ambientlight_t, osg::ref_ptr<osg::Vec3Array> diffuselight);

		/**
		 * Deconstructor
		 */
		~CloudScene(void);

		/**
		 * Inserts a cloud
		 * @param cloud cloud object
		 * @param layerID lalyer id
		 */
		osg::ref_ptr<osg::Geometry> InsertCloud(WangCloud* cloud, int layerID);
		
		/**
		 * Adds a clouds
		 * @param cloud cloud object
		 * @param layerID lalyer id
		 */
		void AddCloud(WangCloud* cloud, int layerID);
		/**
		 * Updates the clouds at runtime
		 * @param cloud cloud object
		 * @param at lalyer id
		 */
		void UpdateCloud(WangCloud* cloud, int at);

		/**
		 * Adds a layer 
		 * @param layerID layerID
		 * @param layer layer
		 */
		void AddLayer(int layerID, int layer);
		/**
		 * Returns the type of a layer
		 * @param layer id of a layer
		 * @return type of a layer
		 */
		int GetType(int layer);
		/**
		 * Returns the number of layers
		 * @return number of layers
		 */
		int GetNumberOfLayers();
	
		/**
		 * Creates the clouds
		 */
		void Create();
		/**
		 * Method to delete the states
		 */
		static void DeleteStates();


		CloudLayerUpdate* cloud_layer_update;

		/**
		 * Adds a fog
		 * @param fog fog object
		 */
		void SetFog(Fog* fog);

		/**
		 * Sets the OpenGL states
		 */
		void Setup();

		/**
		 * Sets OpenGL States
		 */
		void SetupAttributes(osg::ref_ptr<osg::StateSet> nodess4);

		/**
		 * LoadingTexture loads the cloud texture
		 */
		static void LoadingTexture();
		/**
		 * Loads the shader
		 */
		static void LoadingShader();

		/**
		 * Method to set the states
		 * @param state states of the cloud layer
		 */
		static void SetStates(CloudLayerState* state);
		/**
		 * Method, which returns the states of the cloud layer
		 * @return cloud layer states
		 */
		static CloudLayerState* GetStates();

		/**
		 * This method updates the uniforms of the shader
		 * @param type uniform, which should updated
		 */
		void UpdateUniform(int type);

		/**
		 * Returns the number of clouds
		 * @return number of clousd
		 */
		int GetNumberOfClouds();
		/**
		 * Returns the nubmer of particles
		 * @return number of particles
		 */
		unsigned int GetNumberOfParticles();

		/**
		 * Initialize Method
		 */
		static void Initialize();


		static float timeOfDay;
		static float fading;
		static float dens;
		static osg::Vec3 ambientLight_h0;
		static osg::Vec3 ambientLight_h1;
		static osg::Vec3 ambientLight_t0;
		static osg::Vec3 ambientLight_t1;
		static osg::Matrix3 directionalColors;
		static float time;
		static osg::Vec3 sunLightColor;

		static osg::ref_ptr<osg::Program> cloudShadowMapProg;
		static osg::ref_ptr<osg::Program> cloudBlurHoriProg;
		static osg::ref_ptr<osg::Program> cloudBlurVertProg;
		static osg::ref_ptr<osg::Program> cloudBlurLinearHoriProg;
		static osg::ref_ptr<osg::Program> cloudBlurLinearVertProg;
		static osg::ref_ptr<osg::Program> cloudProg; 
		static osg::ref_ptr<osg::Program> cloudShadowProg;
		static osg::ref_ptr<osg::Texture2D> tex_clouds;
	
		static osg::FrameBufferObject* fbo_viewer;
		static osg::FrameBufferObject* fbo_light;


		osg::ref_ptr<osg::Uniform> project_lightuniform;
		osg::ref_ptr<osg::Uniform> modelViewMatrix_lightuniform;
		osg::ref_ptr<osg::Uniform> viewMatrix_lightuniform;
		osg::ref_ptr<osg::Uniform> viewMatrixInv_lightuniform;
		osg::ref_ptr<osg::Uniform> projectuniform;
		osg::ref_ptr<osg::Uniform> modelViewMatrixuniform;
		osg::ref_ptr<osg::Uniform> viewMatrixuniform;
		osg::ref_ptr<osg::Uniform> viewMatrixInvuniform;

		osg::ref_ptr<osg::Geode> quadBillBoard;
		osg::ref_ptr<osg::StateSet> nodess4;

		osg::ref_ptr<osg::Vec3Array> m_cloudcenter;

		osg::ref_ptr<osg::IntArray> m_start_indices;

		bool setup_attribute;
		osg::ref_ptr<osg::Geode> geode;

	#ifdef SHADOW_MAPPING
		osg::ref_ptr<osgCloudyDay::CloudGeometry> geometry;
	#else
		osg::ref_ptr<osg::Geometry> geometry;
	#endif

		/**
		 * Method to reomove the screen billboard 
		 */
		void RemoveScreenBillBoard();	

		/**
		 * Method calculates the position
		 * @param m_type  cloud type
		 * @return positon
		 */
		osg::Vec3 CalculatePosition(int m_type);

		int numClouds;
		
		/**
		 * Calcualtes the bounding boxes of the cloud
		 * @param middlePoint location of the sphere
		 * @param radius radius of the sphere
		 * @param vpr vpr
		 * @return geometry with the bounding spheres
		 */
		osg::Geometry* createSphere(osg::Vec3 middlePoint, float radius, int vpr);
		/**
		 * Calcualtes the bounding boxes of the cloud
		 * @param cloudID cloud id
		 * @return geometry with the calculates bounding boxes
		 */
		osg::ref_ptr<osg::Geometry> CreateBBForCloud(int cloudID);	

		std::vector<WangCloud*> m_clouds;
		std::vector<int> cloudstartindex;
		std::vector<int> cloudendindex;

		/**
		 * Returns a layer
		 * @return layer
		 */
		unsigned int GetLayer(int index);

		static bool m_backtofront;
		static bool m_backtofront_old;	



		/**
		 * Method to create a cloud camera
		 */
		static void CreateCloudCamera();

		/**
		 * Method, which returns a cloud camera
		 * @return cloud camera
		 */
		static osg::ref_ptr<osg::Camera> GetCloudCamera();
		

		static osg::ref_ptr<osg::Texture2D> fbo_light_texture;	//SHADOW		
		static osg::ref_ptr<osg::Texture2D> fbo_cloud_texture;	//CLOUD TEXTURE
		
		static osg::FrameBufferObject* fbo_cloud_viewer;		//CLOUD FRAMEBUFFER
		static osg::FrameBufferObject* fbo_shadow;
		static osg::FrameBufferObject* fbo_vert_linear_blur;
		static osg::FrameBufferObject* fbo_hori_linear_blur;
		static osg::FrameBufferObject* fbo_vert_blur;
		static osg::FrameBufferObject* fbo_hori_blur;
		static osg::FrameBufferObject* fbo_hori_blur2;	



	protected:		
		static osg::ref_ptr<osg::Camera> m_lightCloudCamera;	

		/**
		 * Adds indices to the index array
		 * @param index indices to add
		 * @param start offfset, where to add
		 * @param count number of indices to add
		 */
		void AddDrawElementsPoint(std::vector<unsigned int> index, unsigned int start, unsigned int count, int k);

		/**
		 * Insert a billboard at the end of the object
		 */
		void InsertScreenBillBoard();
		/*
		 * Random generator
		 * @return random value
		 */
		inline float frand();
		/**
		 * Get a random number, which follows a Gaussian function
		 * @param m mean value of the gaussian function
		 * @param v variance value of the gaussian function
		 * @return random number, distributed after a Gaussian function
		 */
		osg::Vec2 GetGaussianDistributedRandomNumber(osg::Vec2 m, float v);
		/**
		 * Method to generate a random sequence
		 * @param num_elements number of random values
		 */	
		osg::Vec2 m_numOfClouds;
	
		unsigned int m_num_indices;	

		std::vector<unsigned int> m_layerToNumberCloud;
		std::vector<unsigned int> m_layerToCloudType;
		std::vector<unsigned int> m_layerToCloudLayerID;
		osg::ref_ptr<osg::IntArray> m_startLayerIndex;

		osg::ref_ptr<osg::Vec3Array> m_vertices;
		osg::ref_ptr<osg::Vec3Array> m_bb_min;
		osg::ref_ptr<osg::Vec3Array> m_bb_max;
		osg::ref_ptr<osg::Vec4Array> m_rotation;
		osg::ref_ptr<osg::Vec4Array> m_center;
		osg::ref_ptr<osg::Vec4Array> m_ids;
		osg::ref_ptr<osg::Vec4Array> m_box_centers;	
		osg::ref_ptr<osg::Vec4Array> m_color;	
		
		osg::ref_ptr<osg::IntArray> m_numPrimitiveArrays;	

		std::map<int, std::string> m_uniforms;

		osg::ref_ptr<osg::Vec3Array> m_ambientlight_h;
		osg::ref_ptr<osg::Vec3Array> m_ambientlight_t;
		osg::ref_ptr<osg::Vec3Array> m_diffuselight;

		static CloudLayerState* m_CloudLayerState;	

		Fog* m_fog;

		
	};
}