#pragma once
#include <iostream>
#include <algorithm>
#include <string>
#include <vector>
#include <map>
#include <osg/Vec2>
#include <osg/Vec3>
#include <osg/Vec4>
#include <osg/Referenced>

namespace osgCloudyDay
{
	/**
	 * States of a 3D cloud layer
	 */
	class CloudLayerState
	{
	public:
		/**
		 * Constructor
		 */
		CloudLayerState(void);
		/**
		 * Deconstructor
		 */
		~CloudLayerState(void);

		/**
		 * This method adds a cloud layer
		 * @param layer layer id
		 * @param type type of the cloud
		 */
		void AddLayer(unsigned int layer, unsigned int type);
		/**
		 * This method returns all cloud layers
		 * @param layer layer id
		 * @return all cloud layers
		 */
		std::vector<unsigned int> GetLayers();

		/**
		 * This method sets the type of the cloud layer
		 * @param layer layer id
		 * @param type type of the cloud
		 */
		void SetType(unsigned int layer, unsigned int type);
		/**
		 * This method returns the type of the cloud layer
		 * @param layer layer id
		 * @return type of the cloud layer
		 */
		int	GetType(unsigned int layer);

		/**
		 * This method sets the number of clots of the cloud layer
		 * @param layer layer id
		 * @param clots number of clots
		 */
		void setClot(unsigned int cloud_type, unsigned int clot);
		/**
		 * This method returns the number of clots
		 * @param layer layer id
		 * @return number of clots
		 */
		int getClot(unsigned int cloud_type);

		/**
		 * This method sets the variance of the cloud clots
		 * @param layer layer id
		 * @param variance variance of the clots
		 */
		void setVariance(unsigned int cloud_type, float v);
		/**
		 * This method returns the variance of the cloud layer
		 * @param layer layer id
		 * @return variance of the cloud layer
		 */
		float getVariance(unsigned int cloud_type);

		/**
		 * This method sets the measured overcast of the cloud layer
		 * @param layer layer id
		 * @param overcast measured overcast
		 */
		void setMeasureOvercast(unsigned int cloud_type, float overcast);
		/**
		 * This method returns the measured overcast of the cloud layer
		 * @param layer layer id
		 * @return measured overcast of the cloud layer
		 */
		float getMeasureOvercast(unsigned int cloud_type);		

		/**
		 * This method sets color of the cloud layer
		 * @param layer layer id
		 * @param color color of the cloud layer
		 */
		void setColor(unsigned int cloud_type, osg::Vec4f v);
		/**
		 * This method returns the color of the cloud layer
		 * @param layer layer id
		 * @return color of the cloud layer
		 */
		osg::Vec4f getColor(unsigned int cloud_type);

		/**
		 * This method sets a threshold. We add clouds until the measured overcast is below the threshold.
		 * @param layer layer id
		 * @param overcast theshold
		 */
		void setOvercast(unsigned int cloud_type, float overcast);
		/**
		 * This method returns the threshold until clouds are added to a cloud layer
		 * @param layer layer id
		 * @return threshold
		 */
		float getOvercast(unsigned int cloud_type);		

		/**
		 * This method sets the height of the cloud layer
		 * @param layer layer id
		 * @param height height of the cloud layer
		 */
		void setHeight(unsigned int cloud_type, float height);
		/**
		 * This method returns the height of the cloud layer
		 * @param layer layer id
		 * @return height of the cloud layer
		 */
		float getHeight(unsigned int cloud_type);		

		/**
		 * This method sets the height of the cloud layer
		 * @param layer layer id
		 * @return number of 
		 */
		unsigned int GetEstimationOfNumberOfCloudsOfLayer(unsigned int cloud_type);

		/**
		 * This method sets the position of the cloud layer
		 * @param layer layer id
		 * @param middlepoint position of the cloud layer
		 */
		void setMiddlePoint(unsigned int cloud_type, osg::Vec3 middlepoint);
		/**
		 * This method returns the position of the cloud layer
		 * @param layer layer id
		 * @return position of the cloud layer
		 */
		osg::Vec3 getMiddlePoint(unsigned int cloud_type);		

		/**
		 * this method sets the size of the cloud layer
		 * @param layer layer id
		 * @param size size of the cloud layer
		 */
		void setSize(unsigned int cloud_type, osg::Vec3 size);
		/**
		 * This method returns the size of the cloud layer
		 * @param layer layer id
		 * @return size of the cloud layer
		 */
		osg::Vec3 getSize(unsigned int cloud_type);		

		/**
		 * This method returns the number of clouds
		 * @param layer layer id
		 * @return number of clouds
		 */
		unsigned int getNumClouds(unsigned int cloud_type);
		/**
		 * this method sets the number of clouds
		 * @param layer layer id
		 * @param num_clouds number of clouds
		 */
		void setNumClouds(unsigned int cloud_type, unsigned int num_clouds);

	private:
		std::vector<unsigned int> m_keys;

		std::map<unsigned int, int> m_type;

		std::map<unsigned int, float> m_overcast;
		std::map<unsigned int, float> m_overcast_measure;
		std::map<unsigned int, float> m_height;
		std::map<unsigned int, unsigned int> m_numClouds;

		std::map<unsigned int, osg::Vec3> m_middlepoint;
		std::map<unsigned int, osg::Vec3> m_size;
		std::map<unsigned int, osg::Vec4> m_color;

		std::map<unsigned int, unsigned int> m_clot;
		std::map<unsigned int, float> m_var;
	};
}
