#pragma once
#include <osg/Node>
#include <osgDB/ReadFile> 

#include <string>
#include <vector>

namespace osgCloudyDay
{
	/**
	 * Method, which calculates a ray with a triangle
	 * @param r ray
	 * @param T triangle
	 * @param I intersection point (if exists)
	 * @return -1, if triangle is degenerated, 0 disjoint, 1 intersect, 2 same plane
	 */
	//http://www.softsurfer.com/Archive/algorithm_0105/algorithm_0105.htm#intersect_RayTriangle%28%29
	//    Input:  a ray R, and a triangle T
	//    Output: *I = intersection point (when it exists)
	//    Return: -1 = triangle is degenerate (a segment or point)
	//             0 = disjoint (no intersect)
	//             1 = intersect in unique point I1
	//             2 = are in the same plane
	inline int intersect_RayTriangle( osg::Vec3* R, osg::Vec3* T, osg::Vec3& I )
	{
		osg::Vec3    u, v, n;             // triangle vectors
		osg::Vec3    dir, w0, w;          // ray vectors
		float     r, a, b;             // params to calc ray-plane intersect

		// get triangle edge vectors and plane normal
		u = T[1] - T[0];
		v = T[2] - T[0];
		n = u ^ v;             // cross product
		if (n == osg::Vec3())            // triangle is degenerate
			return -1;                 // do not deal with this case

		dir = R[1] - R[0];             // ray direction vector
		w0 = R[0] - T[0];
		a = -n * w0;
		b =  n * dir;

		if (fabs(b) < 0.00000001) {     // ray is parallel to triangle plane
			if (a == 0)  return 2;      // ray lies in triangle plane
			else return 0;             // ray disjoint from plane
		}

		// get intersect point of ray with triangle plane
		r = a / b;
		if (r < 0.0)                   // ray goes away from triangle
			return 0;                  // => no intersect
		// for a segment, also test if (r > 1.0) => no intersect

		I = R[0] + (dir * r);           // intersect point of ray and plane

		// is I inside T?
		float    uu, uv, vv, wu, wv, D;
		uu = u * u;
		uv = u * v;
		vv = v * v;
		w = I - T[0];
		wu = w * u;
		wv = w * v;
		D = uv * uv - uu * vv;

		// get and test parametric coords
		float s, t;
		s = (uv * wv - vv * wu) / D;
		if (s < 0.0 || s > 1.0)        // I is outside T
			return 0;
		t = (uv * wu - uu * wv) / D;
		if (t < 0.0 || (s + t) > 1.0)  // I is outside T
			return 0;

		return 1;                      // I is in T
	}

	/**
	 * Helperclass to create a cloud which uses a controlmesh.
	 * @author Michael Beham
	 */
	class CloudCreateVolume
	{
	public:
		/**
		 * Standard constructor
		 */
		CloudCreateVolume(void);
		/**
		 * Standard deconstructor
		 */
		~CloudCreateVolume(void);

		/**
		 * This method loads a mesh, which is used as control mesh.
		 * @param path path to the mesh
		 * @return mesh- node
		 */
		osg::Node* LoadingVolume(std::string path);

		/**
		 * This method perform an in- out- test at a specific point.
		 * @param point, which should checked. 
		 * @return true if the point is inside the mesh. Otherwise false.
		 */
		bool PointInside(osg::Vec3 point);

		/**
		 * AABB-Boundingbox of the mesh
		 */
		std::vector<std::pair<osg::Vec3, osg::Vec3>> m_aabb_boundbox;

	private:
		osg::ref_ptr<osg::Node> object;
	};
}