#pragma once
#include "Scene.h"
#include "CloudGenerator.h"
#include "CloudScene.h"

#include <stdlib.h>
#include <string>
#include <stdio.h>
#include <iostream>
#include <cstdlib>
#include <math.h>

namespace osgCloudyDay
{
	class CelluarAutomataGenerator : public osgCloudyDay::CloudGenerator
	{		
	public:
		/**
		 * Constructor
		 * @param width width of the simulation space
		 * @param depth depth of the simulation space
		 * @param height height of the simulation space
		 * @param type cloud type
		 * @param color color of the cloud
		 */
		CelluarAutomataGenerator(unsigned int width, unsigned int depth, unsigned int height, int type, osg::Vec4 color);
		/**
		 * Deconstructor
		 */
		~CelluarAutomataGenerator(void);

		/**
		 * Method simulates the cloud
		 */
		void Perform(void);
		/**
		 * Method sets the cloud to render it
		 */
		bool Update(void);

	protected:
		/**
		 * Initialize creates the boxes and sprites which are needed to create a Cumulus cloud.
		 */
		void Initialize();

		int m_type;

		/**
		 * Method adds an ellipsoid
		 * @param radius radius of th ellipsoid
		 * @param position position of the ellipsoid
		 * @param lifetime lifetime of the ellipsoid
		 */
		void AddEllipsoide(osg::Vec3 radius, osg::Vec3 position, float lifetime);

		/**
		 * Returns the density of a cell
		 * @param i x-coordinate of the cell
		 * @param j y-coordinate of the cell 
		 * @param k z-coordinate of the cell
		 * @return density value
		 */
		float GetCellDensity( int i, int j, int k );
		/**
		 * Returns the point density of the cloud at a specific position
		 * @param Point position of the cloud
		 * @return density value
		 */
		float GetPointDensity(osg::Vec3 Point);	

	private:
		/**
		 * Method defines the initializion values
		 */
		void DefineInitValues();
		/**
		 * Method interpolates the density value of the cloud
		 * @param t time-value
		 */
		void Interpolate(float& t);

		/**
		 * Method returns the index of the cell
		 * @param i x-coordinate of the cell
		 * @param j y-coordinate of the cell
		 * @param k z-coordinate of the cell
		 * @return index of the cell
		 */
		inline int Index(int i, int j, int k);
		/**
		 * Checks, if the cell is within the space
		 * @return true, if it is in space. Otherwise false.
		 */
		inline bool IsCellInSpace(int x, int y, int z);

		/**
		 * Method updates the ellipsoids
		 */
		void UpdateEllisoides();
		/**
		 * Tests if the position is inside or outside of the volume
		 * @return returns true, if it is outside of the volume, otherwise false
		 */
		bool IsOutside(osg::Vec3 pos);
		/**
		 * Simulates the growing of the cloud
		 */
		void Grow();
		/**
		 * Simulates the extrinction of the cloud
		 */
		void Extrinction();
		/**
		 * Intialize a cell
		 * @param x x coordinate of the cell
		 * @param y y coordinate of the cell
		 * @param z z coordinate of the cell
		 * @param fProbSeed probability value to initalize the cell
		 */
		void InitCell( int x, int y, int z, float fProbSeed );
		/**
		 * Shapes the volume
		 */
		void ShapeVolume();
		/**
		 * Setup the arrays
		 */
		void SetupArrays();
		/**
		 * Calculates the density
		 */
		void CalcDensity();
		/**
		 * Simulates vapor supply
		 */
		void SupplyVapor();
		/**
		 * Simulates wind advection
		 */
		void AdvectionWind();

		float* m_current_density;
		float** m_density2;

		unsigned int** m_hum;
		unsigned int** m_act;
		unsigned int** m_cld;
		unsigned int* m_fact;

		float* m_phum;
		float* m_pext;
		float* m_pact;	

		int m_depth;
		int m_width;
		int m_height;
		float f_depth;
		float f_width;
		float f_height;

		unsigned int t;
		unsigned int tdens;

		unsigned int m_elapsedSteps;

		osg::Vec3 bb_min;
		osg::Vec3 bb_max;

		osg::Vec3 windVector;
	

		osg::ref_ptr<osg::Vec3Array> m_positions;
		osg::ref_ptr<osg::Vec3Array> m_radius;
		osg::ref_ptr<osg::UIntArray> m_lifetimes;

		float m_time;

		osg::Vec3* randomPos;
		osg::Vec3 windDirection;
	};
}