#pragma once
#include "CloudScene.h"
#include <osg/Depth>

namespace osgCloudyDay
{
	/**
	 * Class to blur the shadow map
	 */
	class BlurShadow
	{
	public:
		/**
		 * Constructor
		 */
		BlurShadow();
		/**
		 * Deconstructor
		 */
		~BlurShadow(void);

		/**
		 * Initialize method
		 */
		void Initialize();
		/**
		 * Setups the geometry
		 * @param rendering blurring direction
		 */
		void CreateGeometry(int rendering);

		/**
		 * Creates the camera
		 */
		virtual void CreateCamera();
	
		/**
		 * Returns the camera
		 * @return camera object
		 */
		osg::ref_ptr<osg::Camera> GetCamera();	

	protected:
		osg::ref_ptr<osg::Program> sh_blur_linear_vert1;
		osg::ref_ptr<osg::Program> sh_blur_linear_hori1;
		osg::ref_ptr<osg::Uniform> uniform_blur_texsize2;
		osg::ref_ptr<osg::Uniform> uniform_blur_texsize1;

		osg::ref_ptr<osg::Camera> cam_hud;	
		osg::ref_ptr<osg::StateSet> nodessP;
		osg::ref_ptr<osg::Program> planeProg;		

		int rendering;
	};

	
	class BlurShadowGeometry : public osg::Geometry
	{
	public:
		BlurShadowGeometry(  osg::ref_ptr<osg::Program> prog_vert1, osg::ref_ptr<osg::Uniform> _uniform_blur_texsize, int mode)
		{
			sh_blur_linear_vert1 = (prog_vert1);
			uniform_blur_texsize = _uniform_blur_texsize;
			m_mode = mode;
		}

		~BlurShadowGeometry(void)
		{

		}

		void drawImplementation(osg::RenderInfo& renderInfo) const
		{	
			if (_internalOptimizedGeometry.valid())
			{
				_internalOptimizedGeometry->drawImplementation(renderInfo);
				return;
			}

			osg::State& state = *renderInfo.getState();

			bool checkForGLErrors = state.getCheckForGLErrors()==osg::State::ONCE_PER_ATTRIBUTE;
			if (checkForGLErrors) state.checkGLErrors("start of Geometry::drawImplementation()");
    
			bool useFastPath = areFastPathsUsed();
			// useFastPath = false;

			bool usingVertexBufferObjects = _useVertexBufferObjects && state.isVertexBufferObjectSupported();
			bool handleVertexAttributes = !_vertexAttribList.empty();

			osg::ArrayDispatchers& arrayDispatchers = state.getArrayDispatchers();

			arrayDispatchers.reset();
			arrayDispatchers.setUseVertexAttribAlias(useFastPath && state.getUseVertexAttributeAliasing());
			arrayDispatchers.setUseGLBeginEndAdapter(!useFastPath);

			arrayDispatchers.activateNormalArray(_normalData.binding, _normalData.array.get(), _normalData.indices.get());
			arrayDispatchers.activateColorArray(_colorData.binding, _colorData.array.get(), _colorData.indices.get());
			arrayDispatchers.activateSecondaryColorArray(_secondaryColorData.binding, _secondaryColorData.array.get(), _secondaryColorData.indices.get());
			arrayDispatchers.activateFogCoordArray(_fogCoordData.binding, _fogCoordData.array.get(), _fogCoordData.indices.get());

			if (handleVertexAttributes)
			{
				for(unsigned int unit=0;unit<_vertexAttribList.size();++unit)
				{
					arrayDispatchers.activateVertexAttribArray(_vertexAttribList[unit].binding, unit, _vertexAttribList[unit].array.get(), _vertexAttribList[unit].indices.get());
				}
			}

			// dispatch any attributes that are bound overall
			arrayDispatchers.dispatch(BIND_OVERALL,0);

			state.lazyDisablingOfVertexAttributes();

			if (useFastPath)
			{
				// set up arrays
				if( _vertexData.array.valid() )
					state.setVertexPointer(_vertexData.array.get());

				if (_normalData.binding==BIND_PER_VERTEX && _normalData.array.valid())
					state.setNormalPointer(_normalData.array.get());

				if (_colorData.binding==BIND_PER_VERTEX && _colorData.array.valid())
					state.setColorPointer(_colorData.array.get());

				if (_secondaryColorData.binding==BIND_PER_VERTEX && _secondaryColorData.array.valid())
					state.setSecondaryColorPointer(_secondaryColorData.array.get());

				if (_fogCoordData.binding==BIND_PER_VERTEX && _fogCoordData.array.valid())
					state.setFogCoordPointer(_fogCoordData.array.get());

				for(unsigned int unit=0;unit<_texCoordList.size();++unit)
				{
					const osg::Array* array = _texCoordList[unit].array.get();
					if (array) state.setTexCoordPointer(unit,array);
				}

				if( handleVertexAttributes )
				{
					for(unsigned int index = 0; index < _vertexAttribList.size(); ++index )
					{
						const osg::Array* array = _vertexAttribList[index].array.get();
						const AttributeBinding ab = _vertexAttribList[index].binding;
						if( ab == BIND_PER_VERTEX && array )
						{
							state.setVertexAttribPointer( index, array, _vertexAttribList[index].normalize );
						}
					}
				}
			}
			else
			{
				for(unsigned int unit=0;unit<_texCoordList.size();++unit)
				{
					arrayDispatchers.activateTexCoordArray(BIND_PER_VERTEX, unit, _texCoordList[unit].array.get(), _texCoordList[unit].indices.get());
				}

				arrayDispatchers.activateVertexArray(BIND_PER_VERTEX, _vertexData.array.get(), _vertexData.indices.get());
			}

			state.applyDisablingOfVertexAttributes();

			bool bindPerPrimitiveSetActive = arrayDispatchers.active(BIND_PER_PRIMITIVE_SET);
			bool bindPerPrimitiveActive = arrayDispatchers.active(BIND_PER_PRIMITIVE);			

			if (checkForGLErrors) state.checkGLErrors("Geometry::drawImplementation() after vertex arrays setup.");
		
	
			osg::GL2Extensions* extension = osg::GL2Extensions::Get(0,true);

			glClearColor(0.f, 0.1f, 0.f, 1.f);
			///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
			//
			// draw the primitives themselves.
			//	
			
			if(m_mode == 0)
			{
				glViewport(0,0,1024, 1024);		
				Scene::fbo_vert_linear_blur2->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);						
				glClear( GL_COLOR_BUFFER_BIT);				
			}
			else
			{
				Scene::fbo_hori_linear_blur2->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);		
				glClear( GL_COLOR_BUFFER_BIT);
			}	
		
			for(unsigned int primitiveSetNum=0; primitiveSetNum!= _primitives.size() ; ++primitiveSetNum)
			{			
				// dispatch any attributes that are bound per primitive
				if (bindPerPrimitiveSetActive) 				
					arrayDispatchers.dispatch(BIND_PER_PRIMITIVE_SET, primitiveSetNum);
				
				const osg::PrimitiveSet* primitiveset = _primitives[primitiveSetNum].get();									
												
				glDisable(GL_BLEND);
				glDisable(GL_DEPTH_TEST);								
				glDisable(GL_CULL_FACE);

				osg::State& state = *renderInfo.getState();

				osg::Depth* depth2 = new osg::Depth(osg::Depth::ALWAYS, 0.0, 1.0, false);	//MIT DEPTH => FEHLER	
				depth2->apply(state);
				
				if(m_mode == 0)
				{					
					glViewport(0.f, 0.f, 1024, 1024);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, Scene::GetShadowTexture());
					Scene::fbo_vert_linear_blur2->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set(1024.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects);									
				}
				else
				{	
					glViewport(0.f, 0.f, 1024, 1024);				
					sh_blur_linear_vert1->apply(state);
					state.applyTextureAttribute(0, Scene::GetBlurTexture());
					Scene::fbo_hori_linear_blur2->apply(state, osg::FrameBufferObject::READ_DRAW_FRAMEBUFFER);								
					uniform_blur_texsize->set(1024.f);
					uniform_blur_texsize->apply(extension, state.getUniformLocation("texsize"));
					primitiveset->draw(state, usingVertexBufferObjects); 																										
				}
								
				glEnable(GL_DEPTH_TEST);						
			}
						
			glViewport(0,0, osgCloudyDay::Scene::GetWidth(), osgCloudyDay::Scene::GetHeight());	

			// unbind the VBO's if any are used.
			state.unbindVertexBufferObject();
			state.unbindElementBufferObject();
	
			if (checkForGLErrors) state.checkGLErrors("end of Geometry::drawImplementation().");
		}

		osg::ref_ptr<osg::Program> sh_blur_linear_vert1;
		osg::ref_ptr<osg::Uniform> uniform_blur_texsize;
		int m_mode;
	};

	class UniformBlurShadowTexsize : public osg::Uniform::Callback
	{
	public:
		UniformBlurShadowTexsize() {}
		void operator()( osg::Uniform* uniform, osg::NodeVisitor* nv ){	uniform->set( (float)(osgCloudyDay::Scene::GetWidth()));}
	};

	class BlurCullCallback : public osg::NodeCallback
	{
	public:
		BlurCullCallback(void) : iteration(0)
		{

		}

		virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
		{
			osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);	   
			osg::ref_ptr<osg::Geode> geometry = dynamic_cast<osg::Geode*> (node);		
	
		
			if(geometry && cv && false)
			{
				/*//Hack for Shader
				switch(iteration)
				{
				case 1:	geometry->getOrCreateStateSet()->setAttribute(HUD::sh_blur_linear_vert1);
						iteration = 0;
					break;
				case 0:	geometry->getOrCreateStateSet()->setAttribute(HUD::sh_blur_linear_hori1);				
						iteration = 1;
					break;
				}*/
			}
			traverse(node, nv);
			
		}

		int iteration;
	};
}