#pragma once
#include "Scene.h"
#include "Fog.h"
#include <osg/PolygonMode>
#include <osg/ShapeDrawable>

namespace osgCloudyDay
{
	/**
	 * Drawing Callback for atmosphere nodes
	 */
	class AthmosphereCallback : public osg::NodeCallback
	{
	public:
		/**
		 * Constructor
		 */
		AthmosphereCallback(void)
		{

		}

		/**
		 * Sets the uniforms for the atmosphere
		 */
		virtual void operator()(osg::Node* node, osg::NodeVisitor* nv)
		{
			osgUtil::CullVisitor* cv = dynamic_cast<osgUtil::CullVisitor*>(nv);	   
			osg::ref_ptr<osg::Geode> geometry = dynamic_cast<osg::Geode*> (node);

			if(geometry && cv)
			{
				osg::Matrixd view(cv->getCurrentCamera()->getViewMatrix());
				osg::Matrixd invViewMatrix = osg::Matrixd::inverse(view);
				osg::Matrixd modelview(*cv->getModelViewMatrix());
				osg::Matrixd model = modelview*invViewMatrix;
				osg::Matrixd proj(cv->getCurrentCamera()->getProjectionMatrix());
		
				geometry->getOrCreateStateSet()->getUniform("ModelMatrix")->set(model);
				geometry->getOrCreateStateSet()->getUniform("ViewMatrix")->set(view);
				geometry->getOrCreateStateSet()->getUniform("ProjectionMatrix")->set(proj);		
			
				geometry->getOrCreateStateSet()->getUniform("light_mv_matrix")->set(Scene::GetLightCamera()->getViewMatrix());			
			}
			traverse(node, nv);
		}
	};

	/**
	 * Abstract parent class to create a atmosphere
	 **/
	class Atmosphere
	{
	public:
		/**
		 * Atmoshpere constructor
		 */
		Atmosphere(void);
		/**
		* Atmosphere deconstructor
		*/
		virtual ~Atmosphere(void)=0;

		/**
		 * Abstract function to create a atmosphere
		 */
		virtual void Initialize() = 0;

		/**
		 * Function, which returns the node of the atmosphere
		 * @return Node, which saves the atmosphere
		 */
		osg::ref_ptr<osg::Geode> GetNode();		
		
		/**
		 * Set function for fog
		 * @param Fog class, which saves the fog.
		 */
		void SetFog(Fog* fog);

	protected:				
		Fog* m_fog;

		osg::ref_ptr<osg::StateSet> nodessSkydome;
		osg::ref_ptr<osg::Geode> athmosphere;
	};
}