#include <wx/wx.h>
#include <wx/button.h>

#include "vuSpectral.h"
#include "vuColourRGBa.h"
#include "vuMatrix.h"
#include "vuArcBall.h"
#include "vuPreviewWin.h"

//----------------------------------------------------------------------------
//------------------------- The vuSpectral event table -----------------------
//----------------------------------------------------------------------------

enum
{
    idCANVAS,
    idRENDER,
    idLOADSETUP,
    idLGTINT,
    idLGTCOL,
    idLIGHTPOS,
};

BEGIN_EVENT_TABLE(vuSpectral, vuBasicUtility)
    EVT_COMMAND_SCROLL(idLGTINT, vuSpectral::OnSlideLight)
    EVT_COMMAND_SCROLL(idLGTCOL, vuSpectral::OnSlideLight)
    EVT_CHECKBOX(vuSpectral::idDOSPECULAR, vuSpectral::handleGUIevent)
    EVT_CHECKBOX(vuSpectral::idDRAWPREV, vuSpectral::handleGUIevent)
    EVT_BUTTON  (idRENDER, vuSpectral::OnButtonRender)
    EVT_BUTTON  (idLOADSETUP, vuSpectral::OnButtonLoadSetup)
    EVT_BUTTON  (idLIGHTPOS, vuSpectral::OnButtonLightPos)
END_EVENT_TABLE();

//----------------------------------------------------------------------------
//------------------------- The constructor ----------------------------------
//----------------------------------------------------------------------------

vuSpectral::vuSpectral() : m_TFunc(32), m_TFuncDlg(this, m_TFunc),
			   m_LightDial(this, m_TFunc)
{
    m_Data = NULL;
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vuSpectral::~vuSpectral()
{
    if (m_Data != 0) delete m_Data;
}

//----------------------------------------------------------------------------
//------------------------- public: getFileType() ----------------------------
//----------------------------------------------------------------------------

const char* vuSpectral::getFileType()
{
    return "11121";
}

//----------------------------------------------------------------------------
//------------------------- public: init() -----------------------------------
//----------------------------------------------------------------------------

bool vuSpectral::init(const char* DataFile)
{
    //Set up the window
    SetTitle("Spectral Volume Rendering");
    CreateStatusBar();


    //Create a volume data instance.
    m_Data = new vu1112112;
    m_Data->setFileName(DataFile);

    //Set the transfer function for the data.
    m_TFunc.addOpacity(1,0.01);
    m_TFunc.addOpacity(11,0.01);
    m_TFunc.addOpacity(12,1);
    m_TFunc.addOpacity(32,1);
    m_TFunc.addColour(1,vuColourRGBa(0.f));
    m_TFunc.addColour(11,vuColourRGBa(1.f));
    m_TFunc.addColour(32,vuColourRGBa(0.f,0.f,0.6f));
    m_TFunc.setOpacitySmoothing(0);
    m_TFunc.setColourSmoothing(0);
    m_TFunc.generateFunction();
    m_Data->setTransferFunc(m_TFunc);

    //Read in the data.
    bool success = m_Data->read();
    if (success)
    {
        m_glCanvas->SetSize(512,512);
        Fit();
		m_Preview->attachCamera(&m_Data->getCamera());
		m_Preview->setCubeSize(m_Data->getDim1Size(),m_Data->getDim2Size(),m_Data->getDim3Size());
    }
    else
    {
        wxMessageDialog dlg(this,m_Data->getErrorMessage(),"vuSpectralVR",wxOK);
        dlg.ShowModal();
    }

    return success;
}

//----------------------------------------------------------------------------
//------------------------- public: addRight() -------------------------------
//----------------------------------------------------------------------------

void vuSpectral::addRight(wxSizer *sizer)
{
  //Add some control elements
    sizer->Add( new wxButton(this, idRENDER, "Render"),
		0,           // make horizontally unstretchable
		wxALL,       // make border all around (implicit top alignment)
		10 );        // set border width to 10
    sizer->Add( new wxButton(this, idLIGHTPOS, "Place Light"),
		0, wxALL, 10 );

    wxCheckBox *CBdoSpec = new wxCheckBox(this,idDOSPECULAR,"do specular");
    CBdoSpec->SetValue(false);
    if(m_Data) m_Data->setDoSpecular(CBdoSpec->GetValue());
    wxCheckBox *CBdrawPrev = new wxCheckBox(this,idDRAWPREV,"draw preview");
    CBdrawPrev->SetValue(true);
    if(m_Data) m_Data->setDrawPreview(CBdrawPrev->GetValue());
    sizer->Add( CBdrawPrev,0,wxALL|wxALIGN_LEFT,1);
    sizer->Add( CBdoSpec,0,wxALL|wxALIGN_LEFT,1);
    
    m_Preview = new vuPreviewWin(this,200,200);
    sizer->Add( m_Preview, 
		1,           // make horizontally stretchable
		wxALL,       // make border all around (implicit top alignment)
		10 );        // set border width to 10
    
}

//----------------------------------------------------------------------------
//------------------------- public: OnSlideLight() -----------------
//----------------------------------------------------------------------------

void vuSpectral::OnSlideLight( wxScrollEvent& event)
{
    // int w = m_LightColour->GetValue();
    //int l = m_LightIntensity->GetValue();
  m_glCanvas->redraw();
}

//----------------------------------------------------------------------------
//------------------------- public: OnButtonRender() -------------------------
//----------------------------------------------------------------------------

void vuSpectral::OnButtonRender( wxCommandEvent& event)
{
  m_DrawPreview = false;
  m_Data->doRefresh();
  m_glCanvas->redraw();
}


//----------------------------------------------------------------------------
//------------------------- public: OnButtonLoadSetup() ----------------------
//----------------------------------------------------------------------------

void vuSpectral::OnButtonLoadSetup( wxCommandEvent& event)
{
  wxFileDialog fd(this,"Choose a file","","","*.txt");
  if(fd.ShowModal() == wxID_OK)
    {

      cout<<"loading... "<<flush;
      try {
	if(m_Data->load_scene(fd.GetPath()))
	  cout<<"successful."<<endl;
	else
	  cout<<"failed."<<endl;
      } catch (char *msg) {
	printf("%s\n",msg);
      }
    }
}

void vuSpectral::notifyDataChanged()
{
    m_glCanvas->redraw();
}

//----------------------------------------------------------------------------
//------------------------- protected: glInit() ------------------------------
//----------------------------------------------------------------------------

bool vuSpectral::glInit(void)
{
    if (m_Data == 0) return false;
  /*
    glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

    glEnable(GL_LIGHTING);
    glEnable(GL_LIGHT0);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
  */
    glEnable(GL_DEPTH_TEST);
    glDepthFunc(GL_LESS);
    m_Data->initOpenGL();

    return true;
};

//----------------------------------------------------------------------------
//------------------------- protected: onRender() ----------------------------
//----------------------------------------------------------------------------

//#define SHOWBOOL(a) ((a)?"true":"false")
void vuSpectral::glRender()
{
  bool preview = false;
  if(m_TFuncDlg.isUpdated() || m_LightDial.isUpdated()) {
    if(!m_TFuncDlg.isUpdated())
	m_TFuncDlg.updateSliders();
    else m_LightDial.updateSliders();
    m_TFuncDlg.unsetUpdated();
    vuColour31a light(m_TFunc.getLight());
    //light /= vuColourXYZa(light)[1]; no constant light intensity
    m_Data->setLight(light);
  } else preview = m_DrawPreview;
  m_Data->setTransferFunc(m_TFunc);	// better once too often :-|

  glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glOrtho(0,m_glCanvas->getWidth(),
	  0,m_glCanvas->getHeight(),-1,1);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
	  
  wxStopWatch watch;
  watch.Start();
	  
  if(preview)
      m_Data->preview();
  else
      m_Data->render();

  watch.Pause();
  SetStatusText(wxString("Render Time: ") + vuString(watch.Time()).c_str() + "ms");
  m_DrawPreview = true;
};

//----------------------------------------------------------------------------
//------------------------- protected: glResize() ----------------------------
//----------------------------------------------------------------------------

void vuSpectral::glResize()
{
  m_Data->setImageSize(m_glCanvas->getWidth(),m_glCanvas->getHeight());
}

//----------------------------------------------------------------------------
//------------------------- protected: glOnMouse() ---------------------------
//----------------------------------------------------------------------------

void vuSpectral::onMouse(wxMouseEvent &ev)
{
  if(!m_Data) return;

  if (ev.LeftDClick()) {
    m_TFuncDlg.Show(true);
  }
  else if (ev.RightDClick()) {
    m_LightDial.Show(true);
  }
}

vuCamera* vuSpectral::getCamera ()
{
	return m_Data->getCameraPtr ();
}

vuImage* vuSpectral::getCurrentImage ()

{
	return m_Data->getImage ();
//	return NULL;
}

void vuSpectral::DrawAgain()

{
	m_Data->doRefresh();
	m_glCanvas->redraw();
}

void vuSpectral::DrawFromImage ()

{
//	m_Data->displayFromImage ();
//	m_Data->render ();
	m_glCanvas->redraw ();
}

void vuSpectral::handleGUIevent(wxCommandEvent& ev)
{
    switch(ev.GetId()) {
	case idDOSPECULAR:
	    m_Data->setDoSpecular(ev.IsChecked());
	    break;
	case idDRAWPREV:
	    m_Data->setDrawPreview(ev.IsChecked());
	    break;
    }
}

void vuSpectral::OnButtonLightPos(wxCommandEvent& ev) {
    vuVector v(m_Data->getCenter());
    v-=getCamera()->getPosition();
    v.makeUnit();
    v*=-1.0f;
    m_Data->setLightDir(v);
}

vu1 *vuSpectral::getVolume()
{
  return (vu1*)m_Data;
}
