#include "vuShearWarp.h"
#include "../../vuDrawTools.h"
#include "../../wxUIElements/vuTransferDialog.h"
#include <wx/wx.h>
#include <wx/button.h>
#include <wx/choice.h>
#include <wx/gdicmn.h>
#include <wx/string.h>

//----------------------------------------------------------------------------
//-- The vuShearWarp event table                                           ---
//----------------------------------------------------------------------------
enum
  {
    idTRANSFERFUNCTION,
    idSPECULAR,
    idPROJECTION,
    idAPPLYVIEWING,
    idSCROLL,
    idWARP_OPENGL,
    idFAST_CLASSIFICATION
  };

BEGIN_EVENT_TABLE(vuShearWarp, vuBasicUtility)
  EVT_BUTTON     (idTRANSFERFUNCTION, vuShearWarp::OnButtonTransferFunction)
  EVT_BUTTON     (idAPPLYVIEWING, vuShearWarp::OnButtonApplyViewing)
  EVT_CHECKBOX   (idSPECULAR, vuShearWarp::OnCheckBoxSpecular)
  EVT_RADIOBOX   (idPROJECTION, vuShearWarp::OnRadioBoxProjection)
  EVT_CHECKBOX   (idFAST_CLASSIFICATION, vuShearWarp::OnCheckBoxFastClassification)
  EVT_CHECKBOX   (idWARP_OPENGL, vuShearWarp::OnCheckBoxWarpOpenGL)
  EVT_COMMAND_SCROLL (idSCROLL, vuShearWarp::OnScrollPerspective)
  END_EVENT_TABLE()

  //----------------------------------------------------------------------------
  //-- Constructor                                                           ---
  //----------------------------------------------------------------------------
  vuShearWarp::vuShearWarp()
{
  m_Data = 0;
  m_ViewScale = 1.0f;
}

//----------------------------------------------------------------------------
//-- Destructor                                                            ---
//----------------------------------------------------------------------------
vuShearWarp::~vuShearWarp()
{
  if (m_Data != 0) delete m_Data;
}

void vuShearWarp::DrawAgain() {
}

void vuShearWarp::DrawFromImage() {
}

vuCamera *vuShearWarp::getCamera() {
  return new vuCamera();
}

vuImage *vuShearWarp::getCurrentImage() {
  return new vuImage();
}

//----------------------------------------------------------------------------
//-- Returns the file type processed by this class                         ---
//----------------------------------------------------------------------------
const char* vuShearWarp::getFileType()
{
  return "11121";
}

//----------------------------------------------------------------------------
//-- Initialization                                                        ---
//----------------------------------------------------------------------------
bool vuShearWarp::init(const char* DataFile)
{
  //Set up the window
  int x = 0;
  int y = 0;
  int z = 0;
  vuString dimX_str;
  vuString dimY_str;
  vuString dimZ_str;

  CreateStatusBar();

  useOpenGL(true);

  //Create a volume data instance.
  m_Data = new vu111211A;
  m_Data->setFileName(DataFile);

  //Set the transfer function for the data.
  m_TFunc.addOpacity(64,0.0);
  m_TFunc.addOpacity(65,1.0);
  m_TFunc.addOpacity(140,1.0);
  m_TFunc.addOpacity(249,1.0);
  m_TFunc.addOpacity(250,0.0);
  /*
    m_TFunc.addColour(0,0,0,0);
    m_TFunc.addColour(250,0.0,0.0,1.0);
  */
  m_TFunc.setOpacitySmoothing(0);
  m_TFunc.setColourSmoothing(0);
  m_TFunc.generateFunction();

  m_Data->setTransferFunc(m_TFunc);

  //Read in the data.
  bool success = m_Data->read();
  if (success) {
    m_Data->getDimensions(x, y, z);

    dimX_str = x;
    dimY_str = y;
    dimZ_str = z;

    char fileName[400];
    char totalFileName[400];
    strcpy(totalFileName,m_Data->getFileName());
    int start = 0;
    int end = 0;

    // find the end of the String
    for ( ; totalFileName[end] != '\0'; ++end);
    start = end;

    // find the last '/' in the string
    for ( ; ((start > 0) && (totalFileName[start] != '/')); --start);

    // copy the file name without the path name
    int i = 0;
    for (i = 0; i < end-start; ++i) {
      fileName[i] = totalFileName[start+i+1];
    }
    fileName[i] = '\0';

    // construct the window-title
    char buffer[200] = "vuVolume: Shear Warp (Regular) - ";
    char *seperate = " x ";
    char *title;
    title = strcat(buffer,fileName);
    title = strcat(title,": ");
    title = strcat(title,dimX_str);
    title = strcat(title,seperate);
    title = strcat(title,dimY_str);
    title = strcat(title,seperate);
    title = strcat(title,dimZ_str);

    // set the window title
    SetTitle(title);

    m_glCanvas->SetSize(725,512);
    glResize();
    Fit();
  }
  else {
    wxMessageDialog dlg(this,m_Data->getErrorMessage(),"vuShearWarp",wxOK);
    dlg.ShowModal();
  }

  return success;
};


//----------------------------------------------------------------------------
//-- Add Left                                                              ---
//----------------------------------------------------------------------------
void vuShearWarp::addLeft(wxSizer *sizer) {
  //========================================================================
  //Add the elements for the general settings
  //========================================================================
  //Box around the general settings
  wxBoxSizer *BoxGeneralSettings = new wxStaticBoxSizer(
							new wxStaticBox(this,
									401,
									" general settings "),
							wxVERTICAL);

  //Check Box for turning on/off specular light
  specularCheckBox = new wxCheckBox(this, idSPECULAR, "Specular Light");
  BoxGeneralSettings->Add( specularCheckBox, 0, wxALL | wxGROW, 2);

  //Check Box for turning on/off fast fast classification
  fastClassCheckBox = new wxCheckBox(this, idFAST_CLASSIFICATION, "Fast Classification");
  fastClassCheckBox->SetValue(0);
  BoxGeneralSettings->Add(fastClassCheckBox, 0, wxALL, 2);

  BoxGeneralSettings->Add(
			  new wxCheckBox(this, 
					 idWARP_OPENGL, "use OpenGL for ortho warping"), 
			  0, wxALL, 2);

  sizer->Add( BoxGeneralSettings, 0, wxEXPAND | wxALL, 10 );

  //========================================================================
  // Add wxRadioBox for choosing between orthogonal 
  // and perspective projection
  //========================================================================
  //Radio Box for choosing between orthogonal and perspective viewing
  wxString *choices = new wxString[2];
  choices[0] = "Orthogonal";
  choices[1] = "Perspective";
  projectionRadioBox = new wxRadioBox(this, 
				      idPROJECTION, 
				      "Viewing", 
				      wxDefaultPosition,
				      wxDefaultSize, 2, choices,
				      2, wxRA_SPECIFY_ROWS, 
				      wxDefaultValidator, "radioBox");
  sizer->Add(projectionRadioBox, 0, wxLEFT | wxRIGHT | wxEXPAND, 10);

  //========================================================================
  //Add ScrollBar for choosing distance between eye and projection plane
  //========================================================================
  wxBoxSizer *BoxPerspectiveSettings = new wxBoxSizer(wxHORIZONTAL);
  wxScrollBar *ScrollPerspective = new wxScrollBar(this,
						   idSCROLL,
						   wxPoint(40,40),
						   wxSize(100,20),
						   wxHORIZONTAL,
						   wxDefaultValidator,
						   "ScrollbarPerspective");
  ScrollPerspective->SetScrollbar(0,20,120,120,TRUE);


  BoxPerspectiveSettings->Add(new wxStaticText(this, 300, "Perspective   "));
  BoxPerspectiveSettings->Add(ScrollPerspective);
  sizer->Add(BoxPerspectiveSettings, 0, wxEXPAND | wxALL, 15);

  //========================================================================
  //Add elements for choosing specific viewing view- and up-vectors
  //========================================================================
  //Text fields for entering a specific viewing-position
  viewXTextCtrl = new wxTextCtrl(this, 201, "");
  viewYTextCtrl = new wxTextCtrl(this, 202, "");
  viewZTextCtrl = new wxTextCtrl(this, 203, "");
  upXTextCtrl = new wxTextCtrl(this, 211, "");
  upYTextCtrl = new wxTextCtrl(this, 212, "");
  upZTextCtrl = new wxTextCtrl(this, 213, "");

  //Box around the input of the camera position
  wxBoxSizer *BoxNewCameraSettings = new wxStaticBoxSizer(
							  new wxStaticBox(this,
									  401,
									  " define camera position "),
							  wxVERTICAL);

  //BoxSizer for input of look-at-vector
  wxBoxSizer *BoxViewX = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *BoxViewY = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *BoxViewZ = new wxBoxSizer(wxHORIZONTAL);

  //BoxSizer for input of up-vector
  wxBoxSizer *BoxUpX = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *BoxUpY = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *BoxUpZ = new wxBoxSizer(wxHORIZONTAL);

  BoxNewCameraSettings->Add(
			    new wxStaticText(this, 102, "new look-at-vector: "), 
			    0, wxEXPAND | wxALL, 5);

  BoxViewX->Add(new wxStaticText(this, 103, "X: "), 
		0, wxALIGN_CENTER | wxLEFT, 25);
  BoxViewX->Add(viewXTextCtrl, 0, wxALIGN_CENTER | wxALL, 0);
  BoxNewCameraSettings->Add(BoxViewX,0,wxALL,0);

  BoxViewY->Add(new wxStaticText(this, 104, "Y: "), 0, 
		wxALIGN_CENTER | wxLEFT, 25);
  BoxViewY->Add(viewYTextCtrl, 0, wxALIGN_CENTER | wxALL, 0);
  BoxNewCameraSettings->Add(BoxViewY,0,wxALL,0);

  BoxViewZ->Add(new wxStaticText(this, 105, "Z: "), 0, 
		wxALIGN_CENTER | wxLEFT, 25);
  BoxViewZ->Add(viewZTextCtrl, 0, wxALIGN_CENTER | wxALL, 0);
  BoxNewCameraSettings->Add(BoxViewZ,0,wxALL,0);

  BoxNewCameraSettings->Add(
			    new wxStaticText(this, 106, "new up-vector: "), 
			    0, wxLEFT, 15);

  BoxUpX->Add(new wxStaticText(this, 107, "X: "), 0, 
	      wxALIGN_CENTER | wxLEFT, 25);
  BoxUpX->Add(upXTextCtrl, 0, wxALIGN_CENTER | wxALL, 0);
  BoxNewCameraSettings->Add(BoxUpX,0,wxALL,0);

  BoxUpY->Add(new wxStaticText(this, 108, "Y: "), 0, 
	      wxALIGN_CENTER | wxLEFT, 25);
  BoxUpY->Add(upYTextCtrl, 0, wxALIGN_CENTER | wxALL, 0);
  BoxNewCameraSettings->Add(BoxUpY,0,wxALL,0);

  BoxUpZ->Add(new wxStaticText(this, 109, "Z: "), 0, 
	      wxALIGN_CENTER | wxLEFT, 25);
  BoxUpZ->Add(upZTextCtrl, 0, wxALIGN_CENTER | wxALL, 0);
  BoxNewCameraSettings->Add(BoxUpZ,0,wxALL,0);

  applyViewingButton = new wxButton(this, idAPPLYVIEWING, "Apply Viewing");
  BoxNewCameraSettings->Add(applyViewingButton, 0, wxEXPAND | wxALL, 5);

  sizer->Add(BoxNewCameraSettings, 0, wxALL | wxEXPAND ,10);

  //========================================================================
  //Add button to open transfer-function editor
  //========================================================================
  sizer->Add(new wxButton(this, idTRANSFERFUNCTION, 
			  "Edit Transferfunction..."),
	     0, wxALL | wxEXPAND, 10);
}

//----------------------------------------------------------------------------
//-- Add Bottom                                                            ---
//----------------------------------------------------------------------------
void vuShearWarp::addBottom(wxSizer *sizer)
{
}

//----------------------------------------------------------------------------
//-- Init OpenGL                                                           ---
//----------------------------------------------------------------------------
bool vuShearWarp::glInit(void)
{
  if (m_Data == 0) return false;

  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);

  m_Data->initOpenGL();

  //vuDrawTools::clearBG();

  return true;
}

//----------------------------------------------------------------------------
//-- Calls method render() of the data-object                              ---
//----------------------------------------------------------------------------
void vuShearWarp::glRender()
{
  wxStopWatch watch;
  watch.Start();

  m_Data->setViewVectors(m_Camera.getLookAtVector(),
			 m_Camera.getUpVector(),
			 m_Camera.getRightVector());
  m_Data->render();

  SetStatusText(wxString("RenderTime: ") + 
		vuString(watch.Time()).c_str() + "ms");

}

//----------------------------------------------------------------------------
//-- Resizes OpenGL-Viewport                                               ---
//----------------------------------------------------------------------------
void vuShearWarp::glResize()
{
  //Resize viewport.
  glViewport(0,0,(GLint)m_glCanvas->getWidth(),(GLint)m_glCanvas->getHeight());

  m_Data->setCanvasSize(m_glCanvas->getWidth(),m_glCanvas->getHeight());
  glInit();
}

//----------------------------------------------------------------------------
//-- Check Box "Fast Classification" selected                              ---
//----------------------------------------------------------------------------
void vuShearWarp::OnCheckBoxFastClassification( wxCommandEvent& event)
{
  if (event.IsChecked()) {
    if (projectionRadioBox->GetSelection() == 1) {
      wxMessageDialog dlg(this,"Fast Classification with perspective viewing not implemented yet!","vuShearWarp",wxOK);
      dlg.ShowModal();
      fastClassCheckBox->SetValue(false);
    } else {
      m_Data->setFastClassification(1);
    }
  } else {
    m_Data->setFastClassification(0);
  }
}

//----------------------------------------------------------------------------
//-- Check Box "Warp with OpenGL" selected                                 ---
//----------------------------------------------------------------------------
void vuShearWarp::OnCheckBoxWarpOpenGL( wxCommandEvent& event)
{
  if (event.IsChecked()) {
    m_Data->setOrthogonalWarpOpenGL(1);
  } else {
    m_Data->setOrthogonalWarpOpenGL(0);
  }
}

//----------------------------------------------------------------------------
//-- Scrollbar for distance between eye and projection plane changed       ---
//----------------------------------------------------------------------------
#if wxMINOR_VERSION < 5
void vuShearWarp::OnScrollPerspective( wxCommandEvent& event) {
#else
void vuShearWarp::OnScrollPerspective( wxScrollEvent& event) {
#endif
  m_Data->setEyeDistance(m_Data->getMinEyeDistance() + 
			 (event.GetInt() * m_Data->getMaxSize() / 50));
  m_Data->render();
  m_glCanvas->redraw();
}

//----------------------------------------------------------------------------
//-- Check Box "Specular light" selected                                   ---
//----------------------------------------------------------------------------
void vuShearWarp::OnCheckBoxSpecular( wxCommandEvent& event)
{
  if (event.IsChecked()) {
    m_Data->setSpecular(1);
  } else {
    m_Data->setSpecular(0);
  }
}

//----------------------------------------------------------------------------
//-- Button "Apply Viewing" pressed                                        ---
//----------------------------------------------------------------------------
void vuShearWarp::OnButtonApplyViewing( wxCommandEvent& event) 
{
  // get the coordinates of the new viewing vector as string
  wxString viewXStr = viewXTextCtrl->GetValue();
  wxString viewYStr = viewYTextCtrl->GetValue();
  wxString viewZStr = viewZTextCtrl->GetValue();

  // get the coordinates of the new up vector as string
  wxString upXStr = upXTextCtrl->GetValue();
  wxString upYStr = upYTextCtrl->GetValue();
  wxString upZStr = upZTextCtrl->GetValue();

  bool success = true;

  // get the coordinates of the new viewing vector as double
  double viewXDouble = 0.0;
  double viewYDouble = 0.0;
  double viewZDouble = 0.0;
  if (viewXStr.ToDouble(&viewXDouble) == 0) success = false;
  if (viewYStr.ToDouble(&viewYDouble) == 0) success = false;
  if (viewZStr.ToDouble(&viewZDouble) == 0) success = false;

  // get the coordinates of the new up vector as double
  double upXDouble = 0.0;
  double upYDouble = 0.0;
  double upZDouble = 0.0;
  if (upXStr.ToDouble(&upXDouble) == 0) success = false;
  if (upYStr.ToDouble(&upYDouble) == 0) success = false;
  if (upZStr.ToDouble(&upZDouble) == 0) success = false;

  if (success) {
    vuVector newLookAtVector = vuVector((float)viewXDouble,
					(float)viewYDouble,
					(float)viewZDouble);

    vuVector newUpVector = vuVector((float)upXDouble,
				    (float)upYDouble,
				    (float)upZDouble);

    vuVector newRightVector = newLookAtVector.cross(newUpVector);
    newUpVector = newRightVector.cross(newLookAtVector);

    newLookAtVector.makeUnit();
    newUpVector.makeUnit();

    m_Camera.setLookAtVector(newLookAtVector);
    m_Camera.setUpVector(newUpVector);

    m_Data->setViewVectors(m_Camera.getLookAtVector(),
			   m_Camera.getUpVector(),
			   m_Camera.getRightVector());

    m_Data->render();
    m_glCanvas->redraw();

  } else {
    wxMessageDialog dlg(this,
			"The coordinates have to be numbers!",
			"vuShearWarp",wxOK);
    dlg.ShowModal();
  }
}

//----------------------------------------------------------------------------
//-- Open transfer-function-editor                                         ---
//----------------------------------------------------------------------------
void vuShearWarp::OnButtonTransferFunction( wxCommandEvent& event)
{
  //Pop up the transfer function editor
  vuTransferDialog dlg(this,m_TFunc);

  if (dlg.ShowModal() == wxID_OK) {
    m_TFunc = dlg.getTransferFunc();
    m_Data->setTransferFunc(m_TFunc);
    m_Data->runlengthEncode();
    m_Data->render();
    glResize();
    m_glCanvas->redraw();
  }
}


//----------------------------------------------------------------------------
//-- Radio Box "Projection" selected                                       ---
//----------------------------------------------------------------------------
void vuShearWarp::OnRadioBoxProjection(wxCommandEvent &event) 
{
  if ((fastClassCheckBox->GetValue() == true) && (event.GetSelection() == 1)) {
    wxMessageDialog dlg(this,
          "Fast Classification with perspective viewing not implemented yet!",
			"vuShearWarp",
			wxOK);
    dlg.ShowModal();
    projectionRadioBox->SetSelection(0);
    m_Data->render();
    m_glCanvas->redraw();
  }
  else {
    m_Data->setViewing(event.GetSelection());
    m_Data->render();
    m_glCanvas->redraw();
  }
}

//----------------------------------------------------------------------------
//-- Mouse event                                                           ---
//----------------------------------------------------------------------------
void vuShearWarp::glOnMouse(wxMouseEvent &ev)
{
  if (ev.LeftDown() || ev.RightDown()) {
    //Store the click position.
    m_x = (int) ev.GetX();
    m_y = (int) ev.GetY();
  }
  else if (ev.LeftIsDown() && ev.Moving()) {
    //Rotate the volume
    vuVector t = m_Camera.getPosition();
    float d = t.norm();

    m_Camera.translateXYZ(0.0f, 0.0f, d);
    m_Camera.rotateAboutUp((ev.GetX() - m_x));
    m_Camera.rotateAboutRight(m_y - ev.GetY());

    m_Camera.translateXYZ(0.0f, 0.0f, -d);
    glResize();
    m_glCanvas->redraw();

    //Store the click position.
    m_x = (int) ev.GetX();
    m_y = (int) ev.GetY();
  }
  else if (ev.RightIsDown() && ev.Moving()) {
    //Zoom the volume.
    m_ViewScale -= ((float)ev.GetY() - m_y)/500.0f;
    m_Data->zoom(m_ViewScale);
    m_glCanvas->redraw();

    //Store the click position.
    m_x = (int) ev.GetX();
    m_y = (int) ev.GetY();
  }
  else if (ev.LeftDClick()) {
    //Pop up the transfer function editor
    vuTransferDialog dlg(this,m_TFunc);

    if (dlg.ShowModal() == wxID_OK) {
      m_TFunc = dlg.getTransferFunc();
      m_Data->setTransferFunc(m_TFunc);
      if (m_Data->getFastClassification() == 1) {   // fast classification
	m_Data->classify();
      }
      else {                                      // runlength encoding
	m_Data->runlengthEncode();
      }
      glResize();
      m_glCanvas->redraw();
    }
  }
}













