/**
 * @file	vuMarchingCubes.h
 *
 * GUI part of the Marching Cubes Engine.
 *
 * \todo this text!
 *
 * Because this class is, what the user will see when (s)he invokes the marching cubes utility, I
 * defined this class to be the entry point of the documentation. All the neccessary general
 * information about this utility can be found in this file. In all other parts of the utility you
 * will only find spezial documentation of this part. Please view the
 * <a href="changelog.html">ChangeLog</a> for activities on the project.
 *
 *
 * @author	Markus Trenkwalder
 * @date	08. Jul 2004, 22:36:49
 * @version	0.0.1
 *
 * $Id: vuMarchingCubes.h,v 1.36 2004/11/24 17:13:41 maxx Exp $
 *
 */

// {{{ The Marching Cubes Engine
/**
 * \page MCoverview The Marching Cubes Engine
 *
 * <h1>The algorithm</h1>
 *
 * The marching cubes algorithm is defined on regular three dimentional grids. Therefor this engine
 * can only work on data type below the Volume\Regular\Unimodal\3d tree in this project. The
 * algorithm is threshold based and tries to calculate an exact surface defined by this threshold.
 *
 * <h2>The mechanism</h2>
 * 
 * The algorytm visites every cube in the volume. A cube is defined by eight neighboured data values
 * which form the edges of the cube. The edges are numbered in a defined way.
 *
 * In the first step each edge is tested against the threshold and for every edge that is inside the
 * object, i.e the data value of the edge is higher than the threshold, a corresponding bit in an
 * edge type integer is set to zero. All other bits are set to one. The result of this step is a
 * number which defines the cube type. According to the fact that a cube has eight edges the type
 * integer is eight bit long and therefore there can only exist 256 different types of cubes. If
 * symetry and rotation is taken into account the number of different cubes can be reduced to XXX.
 * For each of these XXX cube types a triangle setup can be defined. These triangle setups with
 * their rotated and mirrored relatives can now be and written to a lookup table of the size 256
 * (see MarchingCubesBase::_marchingCubesTable). Depending on the actual type the corresponding
 * triangles are written to a triangle list and this list is then drawn.
 *
 * <h2>Some semi intelligence</h2>
 *
 * In order to speed up the surface calculation a mechanism is included in the marching cubes
 * algorithm to reuse already calculated triangle vertices. This mechanism also has several other
 * benefits. First this mechanism can be used to calculate triangle strips which in turn speed up
 * the rendering (in this project OpenGL is used for rendering). Second this information can be used
 * to calculate correct normal vectors and third smoothing algorithms can also take advantage of the
 * additional information of this mechanism.
 *
 * \code
 *               o-------------o-------------o
 *              /|            /|            /|
 *             / |           / |           / |
 *            /  |          /  |          /  |
 *           /   |         /   |         /   |
 *          o-------------o-------------o    |
 *         /|    |       /:    |       /|    |
 *        / |    |      / :    |      / |    |
 *       /  |(x,y,z-1)-/--:(x+1,y,z+1/--|----o
 *      /   |   /     /1=5=3=7/     /   |   /
 *     o-------------o-------------o    |  /
 *     |    | /      |    : /      |    | /
 *     |    |/       |    :/       |    |/
 *     | (x,y,z)-----|(x+1,y,z)----|----o
 *     |   /         |   /         |   /     y
 *     |  /          |  /          |  /      |
 *     | /           | /           | /       o-- x
 *     |/            |/            |/       /
 *     o-------------o-------------o       z
 * \endcode
 *
 * The idea behind this mechanism is to try to map corresponding edges. One such edge ist drawn in
 * dots in the above image. This edge has the number 1 in the cube at the left front, it has the
 * number 7 in the left backward cube and so on. This mapping is done by a mapping table
 * (MarchingCubesBase::_edgeMapping). This table is described in detail in \ref MCedgeMapping.
 *
 * But this mapping table is not the only part of the mechanism. There also exists a bunch of
 * classes which saves the vertices in such way that they can serve more than one purpose. First the
 * vertices should be saved so that they can be found fast and easily. Another requirement is to
 * provide data structures that can directly be accessed by OpenGL. Other purposes have already been
 * mentioned above. The classes also provide connectivity information between vertices. For more
 * information about this data structure see \ref MCdata.
 *
 * \todo Description of the vertex remembering
 *
 * <h1>The Design</h1>
 *
 * The marching cubes engine consists of three components. Its design is derived from the
 * model-view-controller pattern. How this designe is realized can be seen in the following image.
 *
 * \image html MCstructure.png "Implementation of the MVC"
 *
 * In terms of the MVC pattern the class vuMarchingCubes and it's parts (widgets) hold the view and
 * the controller and the class MarchingCubesBase and the data spezial classes vu1112xxx hold the
 * model.
 *
 * To be more specific, vuMarchingCubes implements all the code to view and to controll the creation
 * of the object. It also provides a user interface to the marching cubes browser, a little tool
 * which shows all possible cube configurations. The widget classes are implemented to work mostly
 * autonomyouse and give their feedback through so called callback classes which are described
 * later. The view and controller classes are collected in the files vuMarchingCubes.h and
 * vuMarchingCubes.cpp.
 *
 * MarchingCubesBase holds the data array and implements all the algorithms neccessary for the
 * marching cubes engine itself. All the data specific classes as for example vu112211 are derived
 * from MarchingCubesBase. In this way I could avoid to implement the algorithms several times and
 * can provide a uniqe interface to the controller independent from the data type. MarchingCubesBase
 * resides in the files MarchingCubesBase.h and MarchingCubesBase.cpp. All the spezial files can be
 * found in the Volume subtree of this project.
 *
 * <h2>Widgets</h2>
 *
 * Marching cubes widgets are parts of the GUI which serve spezial needs. There is for example a
 * widget that shows a key value pair. All widgets are derived from vuMCWidget which holds a wxSizer
 * and is itself derived from wxPanel. All widgets are designed also to hold data they present and
 * therefore provide an interface to access this data from the parent (which is in most cases
 * vuMarchingCubes). For more information on the marching cubes widgets idea see the class
 * vuMCWidget.
 *
 * <h2>Callback classes</h2>
 *
 * Callbacks in the marching cubes engine are realized trough the decorator pattern, This pattern
 * decorates a class with a defined interface provided to the caller. In this engine we have a
 * unique interface defined in vuMCDecorator which is at the one hand a abstract base class to this
 * mechanism and on the other hand defined as a function object. So all derived classes should also
 * be callback classes. Each widget that want's to send a message back to it's parent can do this
 * only if the parent object has provided widge with a callback object of the type vuMCDecorator and
 * when the parent should be informed about anything, it executes the function object.
 *
 * This mechanism is very flexibe. If the action that should take place, when the callback occures,
 * should be changed, the widget has to simply be given a callback object of another type.
 *
 * <h3>Callback objects and memory management</h3>
 *
 * A Callback object should hold a reference to the object that should be called. The callback
 * object is only a decorator to this referenced object and therefor the callback object is not and
 * can not be responsible for deleting the referenced object.
 *
 * For the caller this looks different. With retreaving a callback object it also retreaves response
 * for this object and therefor has to delete it at the end of it's live cycle.
 *
 * <h2>Some thoughts on (the) heap</h2>
 * 
 * In this design I follow the rule: a class want's an object, it takes also it's controll. What
 * does that mean. Let's take as an example the above described callback mechanism. A widget want's
 * to call back it's parent so it requests a callback object. The parent creates the object and
 * present's it to the widget. By taking the object from the parent, the widget has also takes over
 * it's controll and hase therefore to destroy it at the end of the widgets live cycle. wxWidgets
 * also fallows this rule and therefore non of the widgets is destroyed in vuMarchingCubes.
 *
 * The only problem is to define when an object is given to another class. My definition of this
 * situation is when the given object isn't interfaced (i.e. none of it's member functions is
 * called) in class which gives the object away. In all other cases the object remains in the
 * controll of the class which created it and this class has to destroy it.
 * 
 */
// }}}


// {{{ Edge mapping table
/**
 * \page MCedgeMapping Edge mapping table
 *
 * <h1>Overview</h1>
 *
 * To understand the following diagrams the cube configuration used in this engine is shown in the
 * following image.
 *
 * \image html cubedef.png "Cube definition"
 *
 * <h1>The twelve edges</h1>
 *
 * This section shows the twelve edges and it's neighbouring cubes.
 *
 * <h2>Edge 0 of source cube</h2>
 *
 * \image html edge00.png "Edge 0"
 *
 * <h2>Edge 1 of source cube</h2>
 *
 * \image html edge01.png "Edge 1"
 *
 * <h2>Edge 2 of source cube</h2>
 *
 * \image html edge02.png "Edge 2"
 *
 * <h2>Edge 3 of source cube</h2>
 *
 * \image html edge03.png "Edge 3"
 *
 * <h2>Edge 4 of source cube</h2>
 *
 * \image html edge04.png "Edge 4"
 *
 * <h2>Edge 5 of source cube</h2>
 *
 * \image html edge05.png "Edge 5"
 *
 * <h2>Edge 6 of source cube</h2>
 *
 * \image html edge06.png "Edge 6"
 *
 * <h2>Edge 7 of source cube</h2>
 *
 * \image html edge07.png "Edge 7"
 *
 * <h2>Edge 8 of source cube</h2>
 *
 * \image html edge08.png "Edge 8"
 *
 * <h2>Edge 9 of source cube</h2>
 *
 * \image html edge09.png "Edge 9"
 *
 * <h2>Edge 10 of source cube</h2>
 *
 * \image html edge10.png "Edge 10"
 *
 * <h2>Edge 11 of source cube</h2>
 *
 * \image html edge11.png "Edge 11"
 *
 */
// }}}


#ifndef _vuMarchingCubesUtility_h_
#define _vuMarchingCubesUtility_h_

#include <wx/textctrl.h>
#include <wx/checkbox.h>
#include <wx/button.h>
#include <wx/notebook.h>
#include <wx/artprov.h>
#include <wx/colour.h>
#include <wx/colordlg.h>
#include <wx/cmndata.h>
#include <wx/imaglist.h>
#include <wx/dcclient.h>

#include <math.h>
#include <stdlib.h>
#include <map>

#include "../vuBasicUtility.h"
#include "../../wxUIElements/vuTransferDialog.h"
#include "Volume/Regular/Unimodal/3d/2B/Threshold/MarchingCubes/MarchingCubes.h"
#include "vuTFunc/vuTFDesign.h"
#include "vuVector.h"


// {{{ Decorator base:


/**
 * Abstract base class for the callback decorator framework in the Marching cubes engine.
 * This class is overwritten by classes that do concrete work in vuMarchingCubes. The idea is to
 * define a single entry point for classes (such as the widget classes) which do not know anything
 * about vuMarchingCubes. Therefor this class only defines the entry point.
 * \attention	A class that retreaves a callback object to call it's parent class must delete the
 *		callback object at it's end of livecycle.
 */
template<class t>
class vuMCDecorator // {{{
{

	public:

		//! The access method for decorators
		virtual void operator()(t val) = 0;

}; /* }}} vuMCDecorator */


// }}} Decorator base


// {{{ Widgets:


//! Maximum number of widgets we do want to allow.
#define VU_MC_MAX_WIDGETS 100


/**
 * Base class of all Marching Cubes spezial widgets. The class remembers the parent widget but does
 * not create an widget. It also provides the _sizer member which should be filled by descendend
 * classes.
 *
 * The idea of this class structure is to create the needed wxWindows elements in a hirarchical way.
 * All classes behave with the paradigma that wxWindows manages all widgets, so non of the classes
 * here takes care about removing widgets it created.
 *
 * <strong>Widget structure</strong>
 *
 * This base class as described above only remembers the parent widget for later use. The next level
 * in this hirarchy should create a sizer and override Sizer(). Any successive class then creates
 * widgets with _parent as it's parent.
 *
 * <strong>Usage</strong>
 *
 * First create an object of any class in the third level of this hirarchy. Than use Sizer() to add
 * it to the sizer of the window.
 *
 * \todo	Think about creating a wxStaticBoxSizer (how should we provide the text?) Maybe by
 *		using multiple constructors?
 */
class vuMCWidget : public wxPanel // {{{
{

	public:

		enum {
			typeBox,
			typeStaticBox,
			typeGrid
		};

	private:

		int _sizerType;

		wxSizer* _sizer;

	public:

		//! Hiding the default constructor
		vuMCWidget() {}
		vuMCWidget(wxWindow* p, int orient=wxVERTICAL, int type=typeBox, int hgap=5, int cols=4, int vgap=0);
		//! The destructor does not take care of wxWidgets.
		virtual ~vuMCWidget();

		wxSizer* Sizer();

	public:
		DECLARE_DYNAMIC_CLASS(vuMCWidget);
		DECLARE_EVENT_TABLE();

}; /* }}} vuMCWidget */


/**
 * Displays a key-value-pair. This class creates two labels side by side. The left side holds the
 * key, and the right side holds the value.
 */
class vuMCKeyValue : public vuMCWidget // {{{
{

	private:

		//! Holds the value.
		GLfloat _value;
		//! We keep this value to be able to enable and disable it.
		wxStaticText* _key;
		//! This widget shows the value.
		wxStaticText* _valueText;

	protected:

		//! The number formating text.
		const char* _format;

	private:

	public:

		//! Hiding the default constructor.
		vuMCKeyValue() {}
		vuMCKeyValue(wxWindow* p, const char* key, const char* format, GLfloat val=0);
		//! Destructor
		virtual ~vuMCKeyValue();

		//! Value setter that takes a number.
		virtual void operator=(GLfloat v);
		virtual void operator+=(GLfloat v);
		virtual void operator-=(GLfloat v);
		//! Value setter taking the label text directly.
		void Label(wxString txt);

		//! Value getter.
		inline operator GLfloat();

	public:
		DECLARE_DYNAMIC_CLASS(vuMCKeyValue);
		DECLARE_EVENT_TABLE();

}; /* }}} vuMCKeyValue */


#define VU_MCKV_MAX_SIZE 8

/**
 * Displays a key-value-pair where the value is a vector. This class makes use of the functionallity
 * already implemented in vuMCKeyValue.
 *
 * Internally the class calculates the string to be displayed and passes it to the label in
 * vuMCKeyValue via vuMCKeyValue::Label().
 *
 * Several operators are implemented to set the values of the vector. This is realized by
 * implementing a state machine (member _pos). operator[] sets up the state machine while all other
 * operators (the manipulating operators, such as operator=) us this state.
 */
class vuMCKeyVector : public vuMCKeyValue // {{{
{
	private:

		GLfloat _vector[VU_MCKV_MAX_SIZE];

		//! Actual size of the vector.
		int _size;
		/**
		 * State for the operator[]. It is initialized with -1 which
		 * means "no position". 
		 */
		int _pos;

		wxString _mylabel;

		//! Construct the label from _vector.
		void _updateLabel();

	public:

		//! wxWindows needs a default constructor so we define an empty one.
		vuMCKeyVector() {}
		//! My default constructor.
		vuMCKeyVector(wxWindow* p, const char* key, int size, const char* format);
		//! Destructor
		virtual ~vuMCKeyVector();

		//! Vector value setter.
		void operator()(int n, GLfloat val);
		GLfloat Get(int i);

		//! Vector getter.
		operator const GLfloat*();

		//! Imcrement component \p n of the vector by \p val.
		void Inc(int n, GLfloat val);

		/**
		 * Selects a element of the vector. This operator also sets up
		 * the state machine (_pos) which other operaters use to
		 * manipulate the vector.
		 */
		vuMCKeyVector& operator[](int n);

		//! Set component _pos of the vector to \p v.
		virtual void operator=(GLfloat v);

		//! Increment component _pos of the vector by \p val.
		void operator+=(GLfloat val);
		//! Decrement component _pos of the vector by \p val.
		void operator-=(GLfloat val);

	public:
		DECLARE_DYNAMIC_CLASS(vuMCKeyVector);
		DECLARE_EVENT_TABLE();

}; // }}} vuMCKeyVector


//! Maximum number of bits a bitfield can have.
#define VU_MCBF_MAX_SIZE 32


/**
 * This widget provides GUI elementes to manipulate a bitfield. The maximum size of the bitfield is
 * defined by VU_MCBF_MAX_SIZE. The widget presents a checkbox for every bit and a slider to
 * manipulate the bitfield and a label to display it's value. The slider updates the checkboxes and
 * vice versa and the both widgets update the label.
 */
class vuMCBitField : public vuMCWidget // {{{
{
	public:

		enum {
			idMCBFCheckboxes = 13000,	// we need more than one
			idMCBFSlider = idMCBFCheckboxes+VU_MCBF_MAX_SIZE,
			idMCBFLast
		};

	private:

		//! Manipulation callback.
		vuMCDecorator<int>* _callback;
		//! Part of the title befor the value.
		const char* _titleString;
		//! This label combines the title and the value of the slider.
		wxStaticText* _title;
		//! Actual size of the bit field.
		int _size;
		//! The bitfield checkboxes.
		wxCheckBox* _checkboxes[VU_MCBF_MAX_SIZE];
		/**
		 * The Bitfield slider. This slider is initialized to a range
		 * from zero to the maximum integer number that could be
		 * generated with the number of bits defined by size.
		 */
		wxSlider* _slider;
		/**
		 * The bit that should be manipulated. This member is
		 * initialised with -1 and operater[] sets it to a specified
		 * value (see also vuMCKeyVector).
		 */
		int _pos;

		void _setTitle(int val);
		//! Sets _checkboxes and _title to \p val.
		void _setCheckboxes(int val);

	public:

		//! Default constructor
		vuMCBitField() {}
		//! Constructs the bitfield with 
		vuMCBitField(wxWindow* p, const char* title, vuMCDecorator<int>* cb, int size, int init);
		//! Destructor
		virtual ~vuMCBitField();

		//! Bitfield setter by one number.
		void operator=(int val);
		//! Bitfield bit setter.
		void operator()(int n, bool val);
		//! Bitfield getter.
		operator int();
		//! As in vuMCKeyVector this operator only sets the internal state.
		vuMCBitField& operator[](int n);
		//! Depending on the internal state, a bit is set to \p val.
		void operator=(bool val);

		//! Processes the checkbox events.
		void OnCheckboxClicked(wxCommandEvent& event);
		//! Processes changes of the slider Position.
		void OnScroll(wxScrollEvent& event);

	public:
		DECLARE_DYNAMIC_CLASS(vuMCBitField);
		DECLARE_EVENT_TABLE();

}; /* }}} vuMCBitField */


/**
 * This is the maximum value for the slider. It is also used to calculate the actual color.
 */
#define VU_MCGC_SLIDER_MAX	100


/**
 * This class provides widgets to change a gray color. The Color is calcualted by taking the color
 * white and multiplicating it with a normalized value of a slider.
 *
 * \todo	have a look on the callback.
 */
class vuMCGrayColor : public vuMCWidget // {{{
{

	public:

		enum {
			idMCGCSlider = vuMCBitField::idMCBFLast,
			idMCGCLast
		};

	private:

		//! The color given to a OpenGL command.
		float _glColor[4];
		//! This label is remembered only to be able to disable it.
		wxStaticText* _title;
		//! Display's the actual color.
		wxPanel* _color;
		//! Shows the actual color value.
		wxStaticText* _colorText;
		//! We save this pointer to enable and disable it.
		wxSlider* _slider;
		//! This callback is called, whenever the slider is moved.
		vuMCDecorator<float>* _callback;

		void _setColor(float col);

	public:

		//! Default constructor
		vuMCGrayColor() {}
		vuMCGrayColor(wxWindow* p, const char* title,
			      vuMCDecorator<float>* cb, float color);
		//! Also hide the estructor
		virtual ~vuMCGrayColor();

		//! Returns the OpenGL color (ie. a float[4] with alpha=0).
		operator float*();

		void OnScroll(wxScrollEvent& event);

	public:
		DECLARE_DYNAMIC_CLASS(vuMCGrayColor);
		DECLARE_EVENT_TABLE();

}; /* }}} vuMCGrayColor */


#define VU_MCRGB_SLIDER_MAX 100

/**
 * Class provides a Button and a slider to manipulate a color. This class is nearly the same as
 * vuMCGrayColor but it handles RGB colors and therefore has a button to change the RGB value of the
 * color. At the moment, the alpha chanel isnt used. Instead we multiply the alpha value from
 * _slider directly to the RGB-values of the color.
 */
class vuMCRGBColor : public vuMCWidget // {{{
{

	public:

		enum {
			idMCRGBSlider = vuMCGrayColor::idMCGCLast,
			idMCRGBButton,
			idMCRGBLast
		};

	private:

		GLfloat _glColor[4];

		wxStaticText* _title;
		wxButton* _color;
		wxSlider* _slider;
		wxStaticText* _colorText;

		vuMCDecorator<float>* _callback;


		void _setColor(wxColour& col);
		void _setColor();
		void _setText();

		void _OnScroll(wxCommandEvent& event);

	public:

		//! Default constructor
		vuMCRGBColor() {}
		vuMCRGBColor(wxWindow* p, const char* title,
			     vuMCDecorator<float>* cb, wxColour c, float a);
		//! Destructor
		virtual ~vuMCRGBColor();

		//! When the button is clicket, a the color should be set by a dialog.
		void operator=(wxColour& c);
		//! Returns the color as OpenGL color.
		operator GLfloat*();
		vuVector tovuVector();

		GLfloat Red();
		GLfloat Green();
		GLfloat Blue();
		GLfloat Alpha();

		void OnButton(wxCommandEvent& event);
		void OnScroll(wxScrollEvent& event);

	public:
		DECLARE_DYNAMIC_CLASS(vuMCRGBColor);
		DECLARE_EVENT_TABLE();

}; // }}} vuMCRGBColor


#define VU_MCS_SLIDER_MAX        100
#define VU_MCS_CONTINUOUS_UPDATE true
#define VU_MCS_END_UPDATE        false

/**
 * The widget provides a slider and to labels to display a title an the actual value. The label
 * which holds the value is updated automatically by the slider and by class updates.
 */
class vuMCSlider : public vuMCWidget // {{{
{

	public:

		enum {
			idMCSSlider = vuMCRGBColor::idMCRGBLast,
			idMCSLast
		};

	private:

		float _min;
		float _max;
		int _slMax;

		wxStaticText* _title;
		const char* _format;
		wxStaticText* _value;
		wxSlider* _slider;

		vuMCDecorator<float>* _callback;

		bool _updateMode;

		void _setText(float val);
		void _setValue(float val);
		float _getValue();

	public:

		//! Default constructor
		vuMCSlider() {}
		vuMCSlider(wxWindow* p, const char* title, const char* format,
			   vuMCDecorator<float>* cb, float min, float max, float val,
			   int slMax = VU_MCS_SLIDER_MAX, bool cont = VU_MCS_CONTINUOUS_UPDATE);
		//! Destructor
		virtual ~vuMCSlider();

		//! Value getter.
		operator float();
		//! Value setter.
		void operator=(float val);

		//! _min getter.
		float Min();
		//! _min setter.
		void Min(float min);

		//! _max getter.
		float Max();
		//! _max setter.
		void Max(float max);

		//! _format getter.
		const char* Format();
		//! _format setter.
		void Format(const char* f);

		//! _updateMode getter. A value of true means, 
		bool UpdateMode();
		//! _updateMode setter.
		void UpdateMode(bool mode);

		void OnScroll(wxScrollEvent& event);

	public:
		DECLARE_DYNAMIC_CLASS(vuMCSlider);
		DECLARE_EVENT_TABLE();

}; // }}} vuMCSlider


class vuMCHistogram;


/**
 * This watermark indicates the number of data updates between two subsequent GUI updates.
 */
#define HC_UPDATE_WATERMARK 500


/**
 * This GUI element is not for common use in vuMarchingCubes. As a consequence of a bug in wxGTK-2.4
 * we had to move this class out from vuMCHistogram. This class provides the DC where the histogram
 * is actually drawn. The class also provides functionallity to controll the GUI-update behaviour.
 */
class _HistoCanvas : public wxPanel // {{{
{

	private:

		/**
		 * This variable will be set to the lowest index in the map _histo. See description of
		 * Inc for more information.
		 */
		int _min;
		/**
		 * This variable will be set to the highest index in the map _histo. See description of
		 * Inc for more information.
		 */
		int _max;
		/**
		 * This value holds the biggest value in the map _histo. It is needed to speed up the
		 * histogram drawing.
		 */
		int _vmax;
		//! Holds the density in y-direction (values per pixel).
		float _vpp;
		//! Holds the density in x-direction (pixel per element).
		float _ppe;
		//! The map holds the histogram values that will be painted.
		map<int, int> _histo;
		//! We want to inform the parent if, the mouse is over a element in the histogram.
		vuMCHistogram* _callback;
		//! This variable counts GUI updates.
		int _updateCounter;
		//! If this value is true, GUI updates take place only when _updateCounter reaches
		//! HC_UPDATE_WATERMARK.
		bool _delayedUpdate;
		//! The actual index, where the mouse is on. This is updated when the mouse moves.
		int _idx;
		//! The selected index.
		int _selected;
		//! The color used to paint the selected index.
		wxColour _selColor;

	public:

		//! Default constructor
		_HistoCanvas();
		_HistoCanvas(vuMCHistogram* parent, wxWindowID id = -1, const wxPoint& pos = wxDefaultPosition,
			     const wxSize& size = wxDefaultSize);
		virtual ~_HistoCanvas();

		/**
		 * Resets the histogram to it's start value.
		 */
		void Reset();
		/**
		 * Increments the value at position \p n. Incrementing is a little bit of a problem
		 * because we do not know how the map intitializes the elements. So we implement the
		 * following algorithm:
		 * \li first of all we can rely on the int variables _max and _min which represent the
		 *     biggest and the smallest element in the map.
		 * \li At the beginning we initialize this values with -1 so we get to a defined state.
		 * \li The first element int the map that is changed will find _min and _max out of
		 *     range and set both to the actual index and initialize the value itself to zero.
		 * \li Every change to any of the two variables _min and _max initializes the range
		 *     from the old value of the respective variable to the new one with zeros.
		 *
		 * This ensures, no element in the map is reset to zero by accident.
		 */
		void Inc(int n, int val);

		void SetSelColor(wxColour &c);

		void Select(int idx);

		void DelayedUpdate(bool m);
		bool DelayedUpdate();
		
		void OnPaint(wxPaintEvent& event);

		void OnMotion(wxMouseEvent& event);

		void OnMouse(wxMouseEvent& event);

		DECLARE_DYNAMIC_CLASS(_HistoCanvas);
		DECLARE_EVENT_TABLE();

}; // }}} _HistoCanvas


/**
 * This widget shows a historgram and displayes the value of a bar under the mouse cursor. The
 * widget also provides a checkbox to change the GUI update behaviour.
 */
class vuMCHistogram : public vuMCWidget // {{{
{

	public:

		enum {
			idMCHPaint = vuMCSlider::idMCSLast,
			idCbxDelayedUpdate,
			idClkHistogram,
			idMCHLast
		};

	private:

		//! Checkbox which controlls the widget repainting.
		wxCheckBox   *_delayedUpdate;
		//! We remember the parent for the OnPaint event handler.
		wxWindow     *_parent;
		//! This widget plots the histogram.
		_HistoCanvas *_histoDC;
		//! If the mouse is over a bar in the histogram, it's
		//! value is shown here.
		wxStaticText *_desc;

		vuMCDecorator<int>* _callback;

	public:

		//! Default constructor
		vuMCHistogram() {}

		vuMCHistogram(wxWindow* p, wxSize size, vuMCDecorator<int>* cb = NULL);
		//! Destructor
		virtual ~vuMCHistogram();

		void Clear();

		void Flush();

		void Inc(int i, int val=1);

		void SetSelColor(vuVector c);
		void SetSelColor(wxColour &c);

		void Select(int type);

		void OnTextChange(wxString txt);

		void OnClicked(int idx);

		void OnDelayedUpdateChecked(wxCommandEvent &event);

		DECLARE_DYNAMIC_CLASS(vuMCHistogram);
		DECLARE_EVENT_TABLE();

}; // }}} vuMCHistogram


// }}} Widgets


// {{{ GUI:

class vuMCMCBProgressCallback;

class vuMCMCBHistoCallback;

#define MC_DEFAULT_LIGHT1_ENABLED	false
#define MC_DEFAULT_FPS_WATERMARK	10
#define MC_DEFAULT_DRAW_FRAME		true
#define MC_DEFAULT_DRAW_NORMALS		false

/**
 * A utility window for the marching cubes algorithm on regular unimodal grids.
 *
 * This class provides the frontend for the marching cubes algorithm. It does all the rendering
 * while the data classes (such as vu1112211) do the data reading and scanning and provide the
 * vertices and triangles. This stands in opposition to the paradigma of all other "tools" in this
 * framework.
 *
 * The class also provides a "slider" to change the threshold for the raycaster, provides
 * information about the engine state and let's the user change some settings, such as several
 * colors and the behaviour of the controller. It also provides a interface to the cube browser,
 * which is implemented in MarchingCubesBase.
 *
 * For more information about the structure of the engine see the general documentation of this
 * file.
 *
 * \todo	Refactor documentation.
 *
 * \bug		Changing the cube size does not update the cube browser.
 */
class vuMarchingCubes : public vuBasicUtility // {{{
{
private:
	friend class vuMCMCBProgressCallback;
	friend class vuMCMCBHistoCallback;

public:
	enum {
		idMCFirst = vuMCHistogram::idMCHLast,
		idMCCbxCubeEnabled,
		idMCCbxLight1Enabled,
		idMCCbxDrawFrame,
		idMCCbxDrawLines,
		idMCRbxRotate,
		idMCCbxCullLinesFrontBack,
		idMCCbxUpdateMode,
		idMCCbxMeshStatisitcs,
		idMCCbxDrawNormals,
		idMCBtnUpdate
	};

public:
	//! Constructor.
	vuMarchingCubes();
	//! Destructor.
	virtual ~vuMarchingCubes();

	//!Returns the type of vu Datafile that the utility works with.
	static const char* getFileType();

public:
	//! Initialize the tool.
	bool init(const char* DataFile);
	//! This method adds controls to the top of the window.
	virtual void addTop(wxSizer *sizer);
	//! This method adds controls to the right of the window.
	virtual void addRight(wxSizer *sizer);
	//! Adds the Info page to the notebook on the right side.
	wxPanel* createInfoNotebook(wxNotebook* nb);
	wxPanel* createSettingsNotebook(wxNotebook* nb);
	wxPanel* createGLNotebook(wxNotebook* nb);
	wxPanel* createHelpNotebook(wxNotebook* nb);

	//! Abstract function needs to be overwritten.
        virtual vuCamera* getCamera ();
	//! Abstract function needs to be overwritten.
	void setCamera(vuCamera* cam);
	//! Abstract function needs to be overwritten.
        virtual vuImage* getCurrentImage ();
	//! Abstract function needs to be overwritten.
        virtual void DrawFromImage ();
	//! Abstract function needs to be overwritten.
        virtual void DrawAgain ();

protected:

	//! returns a pointer to m_Data
	vu1 *vuMarchingCubes::getVolume();

	//! Initialize OpenGL.
	bool glInit();
	//! Called, when the window is resized.
	void glResize();
	//! Called, when anything in the window is changed (object rotated, or
	//! other).
	void glRender();

private:
	//! The raycaster owned by the utility.
	vu1*               m_Data;
	/**
	 * Shortcut the the MarchingCubesBase methods. This variable is set,
	 * when the data is created.
	 * \todo	How can we make sure, that m_base is valid when
	 *		function are called?
	 */
	MarchingCubesBase *m_base;

	//! Global data used by several classes in the marching cubes engine.
	MCGlobalData      m_gd;

	//! Images for the Page headers on m_notebook.
	wxImageList       *m_imageList;

	//! This slider holds the threshold for the marching cubes alogrithm.
	vuMCSlider        *m_thres;
#if defined LINUX
	//! Needed because of the incompletnes of linux wxGTK.
	wxButton          *m_update;
#endif
	//! Indicates the number of cubes total.
	vuMCKeyValue      *m_cubesTotal;
	//! Cubes of type 0 are handled separatly so the historgram does show anything meaningful.
	vuMCKeyValue      *m_cube0;
	//! Cubes of type 255 are also handled separatly.
	vuMCKeyValue      *m_cube255;
	//! Indicates the number of cubes of other types.
	vuMCKeyValue      *m_cubeOther;
	//! This widget shows the number of occurrences of all cube types.
	vuMCHistogram     *m_cubeHist;
	vuMCKeyVector     *m_size;
	//! Holds the Rotation angle of the object.
	vuMCKeyVector     *m_angle;
	//! Holds and shows the position of the opbject.
	vuMCKeyVector     *m_pos;
	vuMCKeyValue      *m_zoom;
	//! Holds an displays the position of light 0.
	vuMCKeyVector     *m_light0Pos;
	//! Light 1 can be enabled.
	wxCheckBox        *m_light1Enabled;
	//! Holds an displays the position of light 1.
	vuMCKeyVector     *m_light1Pos;
	vuMCKeyValue      *m_numTriangles;
	vuMCKeyValue      *m_numVertices;
	vuMCKeyValue      *m_numIndices;
	//! Indicates the memory usage of the complete mesh.
	vuMCKeyValue      *m_meshSize;

	//! This label should be set to explain the status shonw in the m_status.
	wxStaticText      *m_statusText;
	wxGauge           *m_status;

	//! Light Color.
	vuMCRGBColor      *m_backColor;
	vuMCRGBColor      *m_lightColor;
	vuMCGrayColor     *m_lightDiffuse;
	vuMCGrayColor     *m_lightSpecular;
	vuMCGrayColor     *m_lightShiny;
	vuMCRGBColor      *m_frameColor;
	vuMCRGBColor      *m_lineColor;
	vuMCRGBColor      *m_selectColor;
	vuMCRGBColor      *m_objectColor;
	vuMCGrayColor     *m_objectDiffuse;
	vuMCGrayColor     *m_objectSpecular;
	vuMCGrayColor     *m_objectShiny;

	vuMCSlider        *m_mouseSpeed;
	wxCheckBox        *m_drawFrame;
	wxCheckBox        *m_cullBack;
	wxCheckBox        *m_cullFront;
	wxCheckBox        *m_linesBack;
	wxCheckBox        *m_linesFront;
	vuMCSlider        *m_frameRadius;
	wxRadioBox        *m_rotateX;
	wxRadioBox        *m_rotateY;
	wxRadioBox        *m_rotateZ;
	wxCheckBox        *m_updateMode;
	wxCheckBox        *m_doMeshStatistics;
	wxCheckBox        *m_cubeEnabled;
	//! Manipulates and displayes the cube setup.
	vuMCBitField      *m_cube;
	wxCheckBox        *m_drawNormals;
	vuMCSlider        *m_drawNormalsSize;
	vuMCSlider        *m_cubeSizeX;
	vuMCSlider        *m_cubeSizeY;
	vuMCSlider        *m_cubeSizeZ;

	int                m_fpsTime;
	int                m_fpsCount;
	vuMCSlider        *m_fpsWatermark;
	float              m_fps;
	float              m_avgFps;

	GLdouble x1, y1, z1;
	GLdouble x2, y2, z2;

protected:

	//! We do keyboard handling for our own.
	void OnChar(wxKeyEvent& event);
	//! Handles mouse event
	void glOnMouse(wxMouseEvent &event);
	/**
	 * The function handles mouse drags with the left mouse button pressed.
	 * When used with modifier keys different modes are realized. The
	 * following list describes the actions depending on the pressed
	 * modifier key:
	 * \li \b None	rotates the object arround the x and y axis,
	 * \li \b Shift	panes the object in x and y direction,
	 * \li \b Ctrl	moves light 0 in x and z direction.
	 * \li \b Atl	moves light 1 in x and z direction.
	 */
	void onMouseLeftMoving(wxMouseEvent &event);
	void onMouseRightMoving(wxMouseEvent &event);
	void onMouseDown(wxMouseEvent &event);
	void onMouseUp(wxMouseEvent &event);
	/**
	 * On a double click the cube under the "cursor" will be highlighted.
	 *
	 * \bug	There is a bug in the selection of the startin position. Maybe we change this code
	 *	to some simpler one and think more about the stepping.
	 */
	void onDblClick(wxMouseEvent &event);

	void OnLight1Checked(wxCommandEvent &event);
	void OnCubeEnabledChecked(wxCommandEvent &event);
	void OnFrameChecked(wxCommandEvent &event);
	void OnRotateChecked(wxCommandEvent &event);
	void OnCullLinesFrontBackChecked(wxCommandEvent &event);
	void OnUpdateModeChecked(wxCommandEvent &event);
#ifdef LINUX
	void OnUpdateClicked(wxCommandEvent &event);
#endif
	void OnDrawNormalsChecked(wxCommandEvent &event);

public:

	//! Redraws the gl context.
	void redraw();
	//! called, wenn the gl geometry is changed.
	void resize();
	//! Called, when the cube config is changed by the respective widgets.
	void OnCubeChange(int val);

	void setBackground();

	//! Updates the progress bar in the right side tab.
	void OnProgress(float v);
	//! This function is called from a decorator whenever the frame setting is changed.
	void OnChangeFrame();
	void OnThresChange();
	void OnNormalLengthChange(float l);
	void OnSetObjectColor();
	void OnSetLineColor();
	void OnCubeSizeChange();
	void OnSelectType(int n);

private:

	//! Rotate the object in x an y direction.
	void rotate(float ax, float ay);

	//! Format the float value \a val to a \code %0.3f \endcode string.
	inline wxString format(double val) { return wxString::Format("%0.3f", val); }

	void doThresChange();
	void doResetThres();

	/**
	 * Takes a \a point and a volume \a size and returns, if the point is outside this volume.
	 * The volume is considered to be double the size given in \a size.
	 */
	inline int outside(vuVector &point, double *size);

public:

	DECLARE_EVENT_TABLE()

}; // }}}

/**
 * Implementation of the MarchingCubesBase progress callback class. See MCBProgressCallback for more information.
 */
class vuMCMCBProgressCallback : public MCBProgressCallback // {{{
{
	private:

		vuMarchingCubes *_mc;

	public:

		vuMCMCBProgressCallback(vuMarchingCubes *mc);

		virtual void SetText(const char *txt);
		virtual void Start();
		virtual void Update(int val);
		virtual void End();

}; // }}} vuMCMCBProgressCallback

class vuMCMCBHistoCallback : public MCBHistoCallback // {{{
{

	private:

		vuMarchingCubes *_mc;

	public:

		vuMCMCBHistoCallback(vuMarchingCubes *mc);

		virtual void Clear();
		virtual void Inc(int idx, int val = 1);
		virtual void Flush();

}; // }}} vuMCMCBHistoCallback

// }}} GUI


// {{{ vuMarchingCubes Decorators:


class vuMCCubeChange : public vuMCDecorator<int> // {{{
{

	private:

		vuMarchingCubes* _mc;

	public:

		//! Default constructor
		vuMCCubeChange(vuMarchingCubes* mc);

		//! Implements the entry point as described in vuMCDecorator.
		virtual void operator()(int c);

}; /* }}} vuMCCubeChange */


class vuMCRender : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Hiding the default constructor
		vuMCRender() {}

	public:

		//! Default constructor
		vuMCRender(vuMarchingCubes* mc);

		//! Implements the decorator function object operator.
		virtual void operator()(float);

}; // }}} vuMCRender


class vuMCResize : public vuMCDecorator<int> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Hiding the default constructor
		vuMCResize() {}

	public:

		//! Default constructor
		vuMCResize(vuMarchingCubes* mc);

		//! Implements the decorator function object operator.
		virtual void operator()(int);

}; // }}} vuMCResize


class vuMCBackgroundChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Hiding the default constructor
		vuMCBackgroundChange() {}

	public:

		//! Default constructor
		vuMCBackgroundChange(vuMarchingCubes* mc);

		//! Implements the decorator function object operator.
		virtual void operator()(float);

}; // }}} vuMCBackgroundChange


class vuMCProgress : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCProgress() {}

	public:

		vuMCProgress(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCProgress


class vuMCFrameChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCFrameChange() {}

	public:

		vuMCFrameChange(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCFrameChange


class vuMCThresChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCThresChange();

	public:

		vuMCThresChange(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCThresChange


class vuMCNormalLengthChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCNormalLengthChange();

	public:

		vuMCNormalLengthChange(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCNormalLengthChange


class vuMCObjectColorChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCObjectColorChange();

	public:

		vuMCObjectColorChange(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCObjectColorChange


class vuMCLineColorChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCLineColorChange();

	public:

		vuMCLineColorChange(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCLineColorChange


class vuMCCubeSizeChange : public vuMCDecorator<float> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Default constructor
		vuMCCubeSizeChange();

	public:

		vuMCCubeSizeChange(vuMarchingCubes* mc);

		virtual void operator()(float);

}; // }}} vuMCCubeSizeChange


class vuMCSelectType : public vuMCDecorator<int> // {{{
{

	private:

		vuMarchingCubes* _mc;

		//! Hiding the default constructor
		vuMCSelectType() {}

	public:

		//! Default constructor
		vuMCSelectType(vuMarchingCubes* mc);

		//! Implements the decorator function object operator.
		virtual void operator()(int);

}; // }}} vuMCSelectType


// }}} vuMarchingCubes Decorators


#endif


// vim:fdm=marker:fdc=3:tw=100
