/**
 * @file	vuMarchingCubes.cpp
 *
 * Implementation of the Marching Cubes GUI class.
 *
 * @author	Markus Trenkwalder
 * @date	08. Jul 2004, 23:58:27
 * @version	0.0.1
 *
 * $Id: vuMarchingCubes.cpp,v 1.38 2004/11/24 17:13:41 maxx Exp $
 */
#include "vuMarchingCubes.h"

#include <fstream.h>
#include <assert.h>

#include "vuColourRGBa.h"
#include "vuPreviewWin.h"
#include "vuMatrix.h"

#include "../../wxUIElements/vuTransferDialog.h"

// Actual Mod: 7
#define DEBUG_MODE 7
//#define DEBUG_CONST
//#define DEBUG_DEST
#include "General/vuMarchingCubes/debug.h"

#include "gl-24x24.xpm"


// {{{ class vuMCWidget:

IMPLEMENT_DYNAMIC_CLASS(vuMCWidget, wxPanel);

BEGIN_EVENT_TABLE(vuMCWidget, wxPanel) // {{{
END_EVENT_TABLE(); // }}}

vuMCWidget::vuMCWidget(wxWindow* p, int orient, int type, int cols, int hgap, int vgap) // {{{
: wxPanel(p, -1)
{
	DEBUGC("vuMCWidget::vuMCWidget(wxWindow* p="<<p<<", int orient="<<orient<<", int type="<<type<<", "
	       <<"int cols="<<cols<<", int hgap="<<hgap<<", int vgap="<<vgap<<")\n");
	_sizer = 0;
	switch (type) {
		case typeBox:
			_sizer = new wxBoxSizer(orient);
			break;
		case typeStaticBox:
			_sizer = new wxStaticBoxSizer(new wxStaticBox(this, -1, _T("No title!")), orient);
			break;
		case typeGrid:
			_sizer = new wxGridSizer(cols, vgap, hgap);
			break;
	}
	SetSizer(_sizer);
} // }}} vuMCWidget::vuMCWidget(wxWindow* p, int orient, int type, int cols, int hgap, int vgap)

/**
 * The sizer inhere should be destroyed but can not, because than the application crashes :(.
 */
vuMCWidget::~vuMCWidget() // {{{
{
	DEBUGC("virtual vuMCWidget::~vuMCWidget()\n");
	DEBUG4("_sizer = "<<_sizer<<"\n");
	//delete _sizer;
} /* }}} vuMCWidget::~vuMCWidget() */

wxSizer* vuMCWidget::Sizer() // {{{
{
	DEBUG0("wxSizer* vuMCWidget::Sizer()\n");
	return _sizer;
} // }}} wxSizer* vuMCWidget::Sizer()

// }}} vuMCWidget


// {{{ class vuMCKeyValue:

IMPLEMENT_DYNAMIC_CLASS(vuMCKeyValue, vuMCWidget);

BEGIN_EVENT_TABLE(vuMCKeyValue, vuMCWidget) // {{{
END_EVENT_TABLE(); // }}}

/**
 * Two labels will be created. The first displayes the key, the other the
 * value. The second one has to be remembered in _value for later use.
 */
vuMCKeyValue::vuMCKeyValue(wxWindow* p, const char* key, const char* format, GLfloat val) // {{{
: vuMCWidget(p, wxHORIZONTAL), _value(0), _format(format)
{
	DEBUGC("vuMCKeyValue<t>::vuMCKeyValue(wxWindow* p="<<p<<", "
	       <<"const char* key="<<key<<", const char* format="<<format<<")\n");
	_key = new wxStaticText(this, -1, key, wxDefaultPosition, wxDefaultSize, wxALIGN_RIGHT);
	Sizer()->Add(_key, 1, wxGROW, 5);
	_valueText = new wxStaticText(this, -1, wxString::Format(format, 0.0f));
	Sizer()->Add(_valueText, 1, wxGROW, 5);
	Fit();
} /* }}} vuMCKeyValue::vuMCKeyValue(wxWindow* p, const char* key, const char* format) */

vuMCKeyValue::~vuMCKeyValue() // {{{
{
	DEBUGD("virtual vuMCKeyValue::~vuMCKeyValue()\n");
	delete _valueText;
	delete _key;
} /* }}} vuMCKeyValue::~vuMCKeyValue() */

void vuMCKeyValue::operator=(GLfloat v) // {{{
{
	DEBUG0("void vuMCKeyValue::operator=(GLfloat v="<<v<<")\n");
	DEBUG3("v = "<<v<<"\n");
	_value = v;
	_valueText->SetLabel(wxString::Format(_format, _value));
} /* }}} void vuMCKeyValue::operator=(GLfloat v) */

void vuMCKeyValue::operator+=(GLfloat v) // {{{
{
	DEBUG0("void vuMCKeyValue::operator+=(GLfloat v="<<v<<")\n");
	_value += v;
	_valueText->SetLabel(wxString::Format(_format, _value));
} /* }}} void vuMCKeyValue::operator+=(GLfloat v) */

void vuMCKeyValue::operator-=(GLfloat v) // {{{
{
	DEBUG0("void vuMCKeyValue::operator-=(GLfloat v="<<v<<")\n");
	_value -= v;
	_valueText->SetLabel(wxString::Format(_format, _value));
} /* }}} void vuMCKeyValue::operator-=(GLfloat v) */

void vuMCKeyValue::Label(wxString txt) // {{{
{
	DEBUG0("void vuMCKeyValue::Label(wxString txt="<<txt<<")\n");
	_valueText->SetLabel(txt);
} /* }}} void vuMCKeyValue::Label(wxString txt) */

inline vuMCKeyValue::operator GLfloat() // {{{
{
	DEBUG0("vuMCKeyValue::operator GLfloat()\n");
	return _value;
} /* }}} vuMCKeyValue::operator GLfloat() */

// }}} vuMCKeyValue


// {{{ class vuMCKeyVector:

IMPLEMENT_DYNAMIC_CLASS(vuMCKeyVector, vuMCKeyValue);

BEGIN_EVENT_TABLE(vuMCKeyVector, vuMCKeyValue) // {{{
END_EVENT_TABLE(); // }}}

vuMCKeyVector::vuMCKeyVector(wxWindow* p, const char* key, int size, const char* format) // {{{
: vuMCKeyValue(p, key, format), _size(size), _pos(-1)
{
	DEBUGC("vuMCKeyVector::vuMCKeyVector(wxWindow* p="<<p<<", "
	       <<"const char* key="<<key<<", int size="<<size<<", "
	       <<"const char* format="<<format<<")\n");
	for(int i = 0; i < size; ++i) _vector[i] = 0.0f;
	_updateLabel();
	Fit();
} /* }}} vuMCKeyVector::vuMCKeyVector(wxWindow* p, const char* key, const char* format) */

vuMCKeyVector::~vuMCKeyVector() // {{{
{
	DEBUGD("virtual vuMCKeyVector::~vuMCKeyVector()\n");
} /* }}} vuMCKeyVector::~vuMCKeyVector() */

void vuMCKeyVector::_updateLabel() // {{{
{
	DEBUG0("wxString vuMCKeyVector::_updateLabel()\n");
	wxString label = "[";
	label << wxString::Format(_format, _vector[0]);
	for(int i=1; i<_size; ++i) label << ", " << wxString::Format(_format, _vector[i]);
	label << "]";
	DEBUG1("label = "<<label<<"\n");
	Label(label);
} /* }}} wxString vuMCKeyVector::_updateLabel() */

void vuMCKeyVector::operator()(int n, GLfloat val) // {{{
{
	DEBUG0("void vuMCKeyVector<t, size>::operator()(int n="<<n<<", GLfloat val"<<val<<")\n");
	assert(0 <= n && n < _size);
	_vector[n] = val;
	_updateLabel();
} /* }}} void vuMCKeyVector::operator()(int n, GLfloat val) */

GLfloat vuMCKeyVector::Get(int pos) // {{{
{
	DEBUG0("GLfloat vuMCKeyVector::Get(int pos)\n");
	assert(0 <= pos && pos < _size);
	return _vector[pos];
} // }}} t vuMCKeyVector::Get(int pos)

vuMCKeyVector::operator const GLfloat*() // {{{
{
	DEBUG0("vuMCKeyVector::operator const GLfloat*()\n");
	return _vector;
} /* }}} vuMCKeyVector::operator const GLfloat*(int n) */

void vuMCKeyVector::Inc(int n, GLfloat val) // {{{
{
	DEBUG0("void vuMCKeyVector::Inc(int n="<<n<<", GLfloat val="<<val<<")\n");
	assert(0 <= n && n < _size);
	_vector[n] += val;
	_updateLabel();
} /* }}} void vuMCKeyVector::Inc(int n, GLfloat val) */

/**
 * We do care about correctnes of \p n here so we can trust on it later.
 */
vuMCKeyVector& vuMCKeyVector::operator[](int n) // {{{
{
	DEBUG0("vuMCKeyVector& vuMCKeyVector::operator[](int n="<<n<<")\n");
	assert(0 <= n && n < _size);
	_pos = n;
	return *this;
} /* }}} vuMCKeyVector& vuMCKeyVector::operator[](int n) */

void vuMCKeyVector::operator=(GLfloat val) // {{{
{
	DEBUG0("void vuMCKeyVector::operator=(GLfloat val="<<val<<")\n");
	assert(_pos != -1);
	_vector[_pos] = val;
	_updateLabel();
} /* }}} void vuMCKeyVector::operator=(GLfloat val) */

void vuMCKeyVector::operator+=(GLfloat val) // {{{
{
	DEBUG0("void vuMCKeyVector::operator+=(GLfloat val = "<<val<<")\n");
	assert(_pos != -1);
	_vector[_pos] += val;
	_updateLabel();
} /* }}} void vuMCKeyVector::operator+=(GLfloat val) */

void vuMCKeyVector::operator-=(GLfloat val) // {{{
{
	DEBUG0("void vuMCKeyVector::operator-=(GLfloat val = "<<val<<")\n");
	assert(_pos != -1);
	_vector[_pos] -= val;
	_updateLabel();
} /* }}} void vuMCKeyVector::operator-=(GLfloat val) */

// }}} vuMCKeyVector


// {{{ class vuMCBitField:

IMPLEMENT_DYNAMIC_CLASS(vuMCBitField, vuMCWidget);

BEGIN_EVENT_TABLE(vuMCBitField, vuMCWidget) // {{{
	EVT_COMMAND_RANGE (vuMCBitField::idMCBFCheckboxes, vuMCBitField::idMCBFCheckboxes+(VU_MCBF_MAX_SIZE),
			   wxEVT_COMMAND_CHECKBOX_CLICKED, vuMCBitField::OnCheckboxClicked)
	EVT_COMMAND_SCROLL(vuMCBitField::idMCBFSlider,     vuMCBitField::OnScroll)
END_EVENT_TABLE(); // }}}

vuMCBitField::vuMCBitField(wxWindow* p, const char* title, vuMCDecorator<int>* cb, int size, int init) // {{{
: vuMCWidget(p), _callback(cb), _titleString(title), _size(size), _pos(-1)
{
	DEBUGC("vuMCBitField::vuMCBitField(wxWindow* p="<<p<<", const char* title="<<title<<", "
	       <<"vuMCDecorator<int>* dec="<<cb<<", int size="<<size<<", int init="<<init<<")\n");
	assert(size <= VU_MCBF_MAX_SIZE);
	//! First line is the label.
	_title = new wxStaticText(this, -1, wxString(""));
	_setTitle(init);
	Sizer()->Add(_title, 0, wxGROW, 5);
	wxSizer* sz = new wxGridSizer(4, 0, 5);
	Sizer()->Add(sz, 0, wxGROW, 5);
	for(int i = 0; i < VU_MCBF_MAX_SIZE; ++i) _checkboxes[i] = 0;
	for(int i = 0; i < size; ++i) {
		_checkboxes[i] = new wxCheckBox(this, idMCBFCheckboxes+i, wxString("")<<i);
		_checkboxes[i]->SetValue((1<<i)&init);
		sz->Add(_checkboxes[i], 0, wxGROW, 5);
	}
	_slider = new wxSlider(this, idMCBFSlider, init, 0, (1<<size)-1);
	Sizer()->Add(_slider, 0, wxGROW, 5);
	Fit();
} // }}} vuMCBitField::vuMCBitField(wxWindow* p, vuMCDecorator<int>* dec, int size, int init)

vuMCBitField::~vuMCBitField() // {{{
{
	DEBUGD("virtual vuMCBitField::~vuMCBitField()\n");
	if (_callback) delete _callback;
	delete _slider;
	for(int i = 0; i < VU_MCBF_MAX_SIZE; ++i)
		if (_checkboxes[i]) delete _checkboxes[i];
	delete _title;
} // }}} vuMCBitField::~vuMCBitField()

void vuMCBitField::_setTitle(int val) // {{{
{
	DEBUG0("void vuMCBitField::_setTitle(int val)\n");
	_title->SetLabel(wxString(_titleString)<<" ("<<val<<"):");
} // }}} void vuMCBitField::_setTitle(int val)

void vuMCBitField::_setCheckboxes(int val) // {{{
{
	DEBUG0("void vuMCBitField::_setCheckboxes(int val)\n");
	for(int i = 0; i < _size; ++i) {
		_checkboxes[i]->SetValue((1<<i) & val);
	}
} // }}} void vuMCBitField<size>::_setCheckboxes(int val)

void vuMCBitField::operator=(int val) // {{{
{
	DEBUG0("void vuMCBitField::operator=(int val)\n");
	_setTitle(val);
	_setCheckboxes(val);
	_slider->SetValue(val);
} // }}} void vuMCBitField<size>::operator=(int val)

void vuMCBitField::operator()(int n, bool val) // {{{
{
	DEBUG0("void vuMCBitField::operator()(int n="<<n<<", bool val="<<val<<")\n");
	assert(0 <= n && n < _size);
	int bit = 1<<n;
	int bitfield = _slider->GetValue();
	if (val && !(bit&bitfield)) {
		//! Change the bit, if val is true and the bit is not set.
		this->operator = (bitfield+bit);
	} else if (!val && bit&bitfield) {
		//! Change the bit, if val is false and the bit is set.
		this->operator = (bitfield-bit);
	}
} // }}} void vuMCBitField::operator()(int n, bool val)

vuMCBitField::operator int() // {{{
{
	DEBUG0("int vuMCBitField::operator()()\n");
	return _slider->GetValue();
} // }}} int vuMCBitField<size>::operator()()

vuMCBitField& vuMCBitField::operator[](int n) // {{{
{
	DEBUG0("vuMCBitField& vuMCBitField::operator[](int n)\n");
	assert(0 <= n && n < _size);
	_pos = n;
	return *this;
} /* }}} vuMCBitField& vuMCBitField::operator[](int n) */

void vuMCBitField::operator=(bool val) // {{{
{
	DEBUG0("void vuMCBitField::operator=(bool val)\n");
	assert(_pos != -1);
	int bit = 1<<_pos;
	int bitfield = _slider->GetValue();
	if (val && !(bit&bitfield)) {
		//! Change the bit, if val is true and the bit is not set.
		this->operator = (bitfield-bit);
	} else if (!val && bit&bitfield) {
		//! Change the bit, if val is false and the bit is set.
		this->operator = (bitfield-bit);
	}
	//! In all other cases do nothing.
} // }}} void vuMCBitField<size>::operator=(bool val)

void vuMCBitField::OnCheckboxClicked(wxCommandEvent& event) // {{{
{
	DEBUG0("void vuMCBitField::OnCheckboxClicked(wxCommandEvent& event)\n");
	int pos = event.GetId() - idMCBFCheckboxes;
	DEBUG1("pos = "<<pos<<"\n");
	assert(0 <= pos && pos < _size);
	DEBUG1("_checkboxes["<<pos<<"] = "<<_checkboxes[pos]<<"\n");
	(*this)(pos, _checkboxes[pos]->GetValue());
	if (_callback) (*_callback)(_slider->GetValue());
} // }}} void vuMCBitField<size>::OnCheckboxClicked(wxCommandEvent& event)

void vuMCBitField::OnScroll(wxScrollEvent& event) // {{{
{
	DEBUG0("void vuMCBitField::OnScroll(wxScrollEvent& event)\n");
	DEBUG1("_slider = "<<_slider<<"\n");
	_setTitle(_slider->GetValue());
	_setCheckboxes(_slider->GetValue());
	if (_callback) (*_callback)(_slider->GetValue());
} // }}} void vuMCBitField<size>::OnScroll(wxScrollEvent& event)

// }}} vuMCBitField


// {{{ class vuMCGrayColor:

IMPLEMENT_DYNAMIC_CLASS(vuMCGrayColor, vuMCWidget);

BEGIN_EVENT_TABLE(vuMCGrayColor, vuMCWidget) // {{{
	EVT_COMMAND_SCROLL(vuMCGrayColor::idMCGCSlider, vuMCGrayColor::OnScroll)
END_EVENT_TABLE(); // }}}

vuMCGrayColor::vuMCGrayColor(wxWindow* p, const char* title, vuMCDecorator<float>* cb, float color) // {{{
: vuMCWidget(p), _callback(cb)
{
	DEBUGC("vuMCGrayColor::vuMCGrayColor(wxWindow* p="<<p<<", const char* title="<<title<<", "
	       <<"vuMCDecorator<float>* cb="<<cb<<", float color="<<color<<")\n");
	_title = new wxStaticText(this, -1, wxString(title));
	Sizer()->Add(_title, 0, 0, 5);
	wxSizer* sz = new wxBoxSizer(wxHORIZONTAL);
	Sizer()->Add(sz, 1, wxGROW, 0);
	_color = new wxPanel(this, -1, wxDefaultPosition, wxSize(20, 20));
	sz->Add(_color, 0, 0, 5);
	_slider = new wxSlider(this, idMCGCSlider, (int)((float)VU_MCGC_SLIDER_MAX*color),
			       0, VU_MCGC_SLIDER_MAX);
	sz->Add(_slider, 1, wxGROW, 5);
	_colorText = new wxStaticText(this, -1, wxString("0.00"));
	sz->Add(_colorText, 0, 0, 5);
	_setColor(color);
	Fit();
} // }}} vuMCGrayColor::vuMCGrayColor(wxWindow* p, const char* title, vuMCDecorator<float>* cb, float color)

vuMCGrayColor::~vuMCGrayColor() // {{{
{
	DEBUGD("virtual vuMCGrayColor::~vuMCGrayColor()\n");
	if (_callback) delete _callback;
	delete _colorText;
	delete _slider;
	delete _color;
	delete _title;
} // }}} vuMCGrayColor::~vuMCGrayColor()

void vuMCGrayColor::_setColor(float col) // {{{
{
	DEBUG0("void vuMCGrayColor::_setColor(float col="<<col<<")\n");
	assert(0.0f <= col && col <= 1.0f);
	_glColor[0] = _glColor[1] = _glColor[2] = col;
	_glColor[3] = 1.0;
	_color->SetBackgroundColour(wxColour((unsigned char)(255.0*col),
					     (unsigned char)(255.0*col),
					     (unsigned char)(255.0*col)));
	_color->Refresh();
	_colorText->SetLabel(wxString::Format("%0.2f", col));
		//! We do not take care about the widget is existing already
		//! because we create it in the constructor.
} // }}} void vuMCGrayColor::_setColor(float col)

vuMCGrayColor::operator float*() // {{{
{
	DEBUG0("vuMCGrayColor::operator float*()\n");
	return _glColor;
} // }}} vuMCGrayColor::operator float*()

void vuMCGrayColor::OnScroll(wxScrollEvent& event) // {{{
{
	DEBUG0("void vuMCGrayColor::OnScroll(wxScrollEvent& event)\n");
	_setColor((float)event.GetPosition()/(float)VU_MCGC_SLIDER_MAX);
	(*_callback)(0);
} // }}} void vuMCGrayColor::OnScroll(wxCommandEvent& event)

// }}} vuMCGrayColor


// {{{ class vuMCRGBColor:

IMPLEMENT_DYNAMIC_CLASS(vuMCRGBColor, vuMCWidget);

BEGIN_EVENT_TABLE(vuMCRGBColor, vuMCWidget) // {{{
	EVT_COMMAND_SCROLL(vuMCRGBColor::idMCRGBSlider, vuMCRGBColor::OnScroll)
	EVT_BUTTON        (vuMCRGBColor::idMCRGBButton, vuMCRGBColor::OnButton)
END_EVENT_TABLE(); // }}}

vuMCRGBColor::vuMCRGBColor(wxWindow* p, const char* title, // {{{
			   vuMCDecorator<float>* cb, wxColour c, float a)
: vuMCWidget(p), _callback(cb)
{
	DEBUGC("vuMCRGBColor::vuMCRGBColor(wxWindow* p="<<p<<", const char* title="<<title<<", "
	       <<"vuMCDecorator<float>* cb="<<cb<<", wxColour c="<<&c<<", float a="<<a<<")\n");
	_title = new wxStaticText(this, -1, wxString(title));
	Sizer()->Add(_title, 0, 0, 5);
	wxSizer* sz = new wxBoxSizer(wxHORIZONTAL);
	Sizer()->Add(sz, 1, wxGROW, 0);
	_color = new wxButton(this, idMCRGBButton, _T(""), wxDefaultPosition, wxSize(20, 20));
	sz->Add(_color, 0, 0, 5);
	_slider = new wxSlider(this, idMCRGBSlider, (int)(a*(float)VU_MCRGB_SLIDER_MAX), 0, VU_MCRGB_SLIDER_MAX);
	sz->Add(_slider, 1, wxGROW, 5);
	_colorText = new wxStaticText(this, -1, wxString("0.00"));
	sz->Add(_colorText, 0, 0, 5);
	_setColor(c);
	_setText();
	Fit();
} // vuMCRGBColor::vuMCRGBColor(wxWindow* p, const char* title,
  // }}}			vuMCDecorator<float>* cb, wxColour c, float a)

vuMCRGBColor::~vuMCRGBColor() // {{{
{
	DEBUGD("virtual vuMCRGBColor::~vuMCRGBColor()\n");
	if (_callback) delete _callback;
	delete _colorText;
	delete _slider;
	delete _color;
	delete _title;
} // }}} vuMCRGBColor::~vuMCRGBColor()

void vuMCRGBColor::_setColor(wxColour& col) // {{{
{
	DEBUG0("void vuMCRGBColor::_setColor(wxColour& col)\n");
	_color->SetBackgroundColour(col);
	_glColor[0] = ((float)col.Red()/255.0f)*((float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX);
	_glColor[1] = ((float)col.Green()/255.0f)*((float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX);
	_glColor[2] = ((float)col.Blue()/255.0f)*((float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX);
	_glColor[3] = 1.0;
} // }}} void vuMCRGBColor::_setColor(wxColour& col)

void vuMCRGBColor::_setColor() // {{{
{
	DEBUG0("void vuMCRGBColor::_setColor()\n");
	wxColour col = _color->GetBackgroundColour();
	_glColor[0] = ((float)col.Red()/255.0f)*((float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX);
	_glColor[1] = ((float)col.Green()/255.0f)*((float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX);
	_glColor[2] = ((float)col.Blue()/255.0f)*((float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX);
	_glColor[3] = 1.0;
} // }}} void vuMCRGBColor::_setColor(wxColour& col)

void vuMCRGBColor::_setText() // {{{
{
	DEBUG0("void vuMCRGBColor::_setText()\n");
	_colorText->SetLabel(wxString::Format("%0.2f", (float)_slider->GetValue()/(float)VU_MCRGB_SLIDER_MAX));
} // }}} void vuMCRGBColor::_setText()

void vuMCRGBColor::operator=(wxColour& c) // {{{
{
	DEBUG0("void vuMCRGBColor::operator=(wxColour& c)\n");
	_setColor(c);
} // }}} void vuMCRGBColor::operator=(wxColour& c)

vuMCRGBColor::operator GLfloat*() // {{{
{
	DEBUG0("vuMCRGBColor::operator GLfloat*()\n");
	return _glColor;
} // }}} vuMCRGBColor::operator GLfloat*()

vuVector vuMCRGBColor::tovuVector() // {{{
{
	DEBUG0("vuVector vuMCRGBColor::tovuVector()\n");
	return vuVector(_glColor[0], _glColor[1], _glColor[2]);
} // }}} vuVector vuMCRGBColor::tovuVector()

void vuMCRGBColor::OnButton(wxCommandEvent& event) // {{{
{
	DEBUG0("void vuMCRGBColor::OnButton(wxCommandEvent& event)\n");
	wxColourData cd;
	cd.SetColour(_color->GetBackgroundColour());
	wxColourDialog dialog(this, &cd);
	dialog.ShowModal();
	cd = dialog.GetColourData();
	wxColour c = cd.GetColour();
	_color->SetBackgroundColour(c);
	_color->Refresh();
	_setColor();
	(*_callback)(0);
} // }}} void vuMCRGBColor::OnButton(wxCommandEvent& event)

void vuMCRGBColor::OnScroll(wxScrollEvent& event) // {{{
{
	DEBUG0("void vuMCRGBColor::OnScroll(wxScrollEvent& event)\n");
	_setColor();
	_setText();
	(*_callback)(0);
} // }}} void vuMCRGBColor::OnScroll(wxScrollEvent& event)

GLfloat vuMCRGBColor::Red() // {{{
{
	DEBUG0("GLfloat vuMCRGBColor::Red()\n");
	return _glColor[0];
} // }}} GLfloat vuMCRGBColor::Red()

GLfloat vuMCRGBColor::Green() // {{{
{
	DEBUG0("GLfloat vuMCRGBColor::Green()\n");
	return _glColor[1];
} // }}} GLfloat vuMCRGBColor::Green()

GLfloat vuMCRGBColor::Blue() // {{{
{
	DEBUG0("GLfloat vuMCRGBColor::Blue()\n");
	return _glColor[2];
} // }}} GLfloat vuMCRGBColor::Blue()

GLfloat vuMCRGBColor::Alpha() // {{{
{
	DEBUG0("GLfloat vuMCRGBColor::Alpha()\n");
	return _glColor[3];
} // }}} GLfloat vuMCRGBColor::Alpha()

// }}} vuMCRGBColor


// {{{ class vuMCSlider:

IMPLEMENT_DYNAMIC_CLASS(vuMCSlider, vuMCWidget);

BEGIN_EVENT_TABLE(vuMCSlider, vuMCWidget) // {{{
	EVT_COMMAND_SCROLL(vuMCSlider::idMCSSlider, vuMCSlider::OnScroll)
END_EVENT_TABLE(); // }}}

vuMCSlider::vuMCSlider(wxWindow* p, const char* title, const char *format, // {{{
		       vuMCDecorator<float>* cb, float min, float max, float val, int slMax, bool cont)
: vuMCWidget(p, wxHORIZONTAL), _min(min), _max(max), _slMax(slMax), _format(format), _callback(cb), _updateMode(cont)
{
	DEBUGC("vuMCSlider::vuMCSlider(wxWindow* p="<<p<<", const char* title="<<title<<", "
	       <<"const char* format="<<format<<", vuMCDecorator<float>* cb="<<cb<<", "
	       <<"float min="<<min<<", float max="<<max<<", float val="<<val<<", int slMax="<<slMax<<", bool cont)\n");
	assert(val >= _min);
	DEBUG6("title = "<<title<<", _min = "<<_min<<", _max = "<<_max<<", _slMax = "<<_slMax
	       <<", val = "<<val<<", position = "<<(int)((float)(val-min)*(float)slMax/(float)(max-min))<<"\n");
	_title = new wxStaticText(this, -1, wxString(title));
	Sizer()->Add(_title, 0, 0, 5);
	_slider = new wxSlider(this, idMCSSlider, 0, 0, slMax);
	Sizer()->Add(_slider, 1, wxGROW, 5);
	_value = new wxStaticText(this, -1, wxString::Format(format, max), wxDefaultPosition, wxDefaultSize, wxALIGN_RIGHT);
	Sizer()->Add(_value, 0, 0, 5);
	_setText(val);
	_setValue(val);
	Fit();
} // vuMCSlider::vuMCSlider(wxWindow* p, const char* title, const char* format, vuMCDecorator<float>* cb,
  // }}}		    float min, float max, float val, int slMax, bool cont)

vuMCSlider::~vuMCSlider() // {{{
{
	DEBUGD("virtual vuMCSlider::~vuMCSlider()\n");
	if (_callback) delete _callback;
	delete _value;
	delete _slider;
	delete _title;
} // }}} virtual vuMCSlider::~vuMCSlider()

void vuMCSlider::_setText(float val) // {{{
{
	DEBUG0("void vuMCSlider::_setText(float val)\n");
	_value->SetLabel(wxString::Format(_format, val));
} // }}} void vuMCSlider::_setText(float val)

void vuMCSlider::_setValue(float val) // {{{
{
	DEBUG0("void vuMCSlider::_setValue(float val)\n");
	DEBUG6("setting slider for '"<<_title->GetLabel()<<"' to: "<<val<<", pos: "<<(int)((val-_min)*(float)_slMax/(_max-_min))<<"\n");
	assert(_slider != 0);
	_slider->SetValue((int)((val-_min)*(float)_slMax/(_max-_min)));
} // }}} void vuMCSlider::_setValue(float val)

float vuMCSlider::_getValue() // {{{
{
	DEBUG0("float vuMCSlider::_getValue()\n");
	return _min+((float)_slider->GetValue()*(_max-_min)/(float)_slMax);
} // }}} float vuMCSlider::_getValue()

vuMCSlider::operator float() // {{{
{
	DEBUG0("vuMCSlider::operator float()\n");
	return _getValue();
} // }}} vuMCSlider::operator float()

void vuMCSlider::operator=(float val) // {{{
{
	DEBUG0("void vuMCSlider::operator=(float val="<<val<<")\n");
	_setText(val);
	_setValue(val);
} // }}} void vuMCSlider::operator=(float val)

float vuMCSlider::Min() // {{{
{
	DEBUG0("float vuMCSlider::Min()\n");
	return _min;
} // }}} float vuMCSlider::Min()

void vuMCSlider::Min(float min) // {{{
{
	DEBUG0("void vuMCSlider::Min(float min="<<min<<")\n");
	assert(min < _max);
	float val = _getValue();
	_min = min;
	if (val < _min) {
		_setValue(_min);
		_setText(_min);
	} else {
		_setValue(val);
	}
} // }}} void vuMCSlider::Min(float min)

float vuMCSlider::Max() // {{{
{
	DEBUG0("float vuMCSlider::Max()\n");
	return _max;
} // }}} float vuMCSlider::Max()

void vuMCSlider::Max(float max) // {{{
{
	DEBUG0("void vuMCSlider::Max(float max)\n");
	assert(_min < max);
	float val = _getValue();
	_max = max;
	if (_max < val) {
		_setValue(_max);
		_setText(_max);
	} else {
		_setValue(val);
	}
} // }}} void vuMCSlider::Max(float max)

const char* vuMCSlider::Format() // {{{
{
	DEBUG0("const char* vuMCSlider::Format()\n");
	return _format;
} // }}} const char* vuMCSlider::Format()

void vuMCSlider::Format(const char* f) // {{{
{
	DEBUG0("void vuMCSlider::Format(const char* f)\n");
	_format = f;
} // }}} void vuMCSlider::Format(const char* f)

bool vuMCSlider::UpdateMode() // {{{
{
	DEBUG0("bool vuMCSlider::UpdateMode()\n");
	return _updateMode;
} // }}} bool vuMCSlider::UpdateMode()

void vuMCSlider::UpdateMode(bool mode) // {{{
{
	DEBUG0("void vuMCSlider::UpdateMode(bool mode)\n");
	_updateMode = mode;
} // }}} void vuMCSlider::UpdateMode(bool mode)

void vuMCSlider::OnScroll(wxScrollEvent& event) // {{{
{
	DEBUG0("static void vuMCSlider::OnScroll(wxScrollEvent& event)\n");
	wxEventType eventType = event.GetEventType();
	float val = _getValue();
	_setText(val);
	if (_updateMode == VU_MCS_END_UPDATE && eventType != wxEVT_SCROLL_ENDSCROLL) return;
	DEBUG2("Doing the update\n");
	if (_callback) (*_callback)(val);
} // }}} static void vuMCSlider::OnScroll(wxScrollEvent& event)

// }}} vuMCSlider


// {{{ class vuMCHistogram:

IMPLEMENT_DYNAMIC_CLASS(_HistoCanvas, wxPanel)

BEGIN_EVENT_TABLE(_HistoCanvas, wxPanel) // {{{
	EVT_PAINT (_HistoCanvas::OnPaint)
	EVT_MOTION(_HistoCanvas::OnMotion)
	EVT_MOUSE_EVENTS(_HistoCanvas::OnMouse)
END_EVENT_TABLE(); // }}}

_HistoCanvas::_HistoCanvas() // {{{
: _selected(-1)
{
	DEBUGC("_HistoCanvas::_HistoCanvas()\n");
} // }}} _HistoCanvas::_HistoCanvas()

_HistoCanvas::_HistoCanvas(vuMCHistogram* parent, wxWindowID id, // {{{
			   const wxPoint& pos, const wxSize& size)
: wxPanel(parent, id, pos, size), _min(-1), _max(-1), _vmax(0), _vpp(0.0), _ppe(0.0),
  _callback(parent), _updateCounter(0), _delayedUpdate(true), _idx(-1)
{
	DEBUGC("_HistoCanvas::_HistoCanvas(vuMCHistogram* parent, "
	       <<"wxWindowID id, const wxPoint& pos, const wxSize& size)\n");
} // _HistoCanvas::_HistoCanvas(vuMCHistogram* parent, wxWindowID id,
  // }}}			const wxPoint& pos, const wxSize& size)

_HistoCanvas::~_HistoCanvas() // {{{
{
	DEBUGD("_HistoCanvas::~_HistoCanvas()\n");
} // }}} _HistoCanvas::~_HistoCanvas()

void _HistoCanvas::Reset() // {{{
{
	DEBUG0("void _HistoCanvas::Reset()\n");
	_min = -1;
	_max = -1;
	_vmax = 0;
	_histo.clear();
	_updateCounter = 0;
	Refresh();
} // }}} void _HistoCanvas::Reset()

void _HistoCanvas::Inc(int n, int val) // {{{
{
	DEBUG0("void _HistoCanvas::Inc(int n="<<n<<", int val="<<val<<")\n");
	if (_min==-1 && _max==-1) {
		DEBUG2("Initial change at "<<n<<" ...\n");
		_min = n;
		_max = n;
		_histo[n] = 0;
	} else if (n > _max) {
		DEBUG2("Moving _max from "<<_max<<" to "<<n<<" ...\n");
		for(int i = _max+1; i <= n; ++i) _histo[i] = 0;
		_max = n;
	} else if (n < _min) {
		DEBUG2("Movint _min from "<<_min<<" to "<<n<<" ... \n");
		for(int i = n; i < _min; ++i) _histo[i] = 0;
		_min = n;
	}
	_histo[n] += val;
	if (_histo[n] > _vmax) {
		_vmax = _histo[n];
		DEBUG2("Increasing _vmax to "<<_vmax<<" ... \n");
	}
	if (_delayedUpdate) {
		++_updateCounter;
		if (_updateCounter < HC_UPDATE_WATERMARK) return;
		_updateCounter = 0;
		Refresh();
	}
} // }}} void _HistoCanvas::Inc(int n, int val)

void _HistoCanvas::SetSelColor(wxColour &c) // {{{
{
	DEBUG0("void _HistoCanvas::SetSelColor(wxColour &c)\n");
	float max = (c.Red()>c.Blue()) ? c.Red() : c.Blue();
	max = (max>c.Green()) ? max : c.Green();
	_selColor = wxColour(int((float)c.Red()*255.0/max),
			     int((float)c.Green()*255.0/max),
			     int((float)c.Blue()*255.0/max));
} // }}} void _HistoCanvas::SetSelColor(wxColour &c)

void _HistoCanvas::Select(int idx) // {{{
{
	DEBUG0("void _HistoCanvas::Select(int type)\n");
	_selected = idx;
} // }}} void _HistoCanvas::Select(int type)

void _HistoCanvas::DelayedUpdate(bool m) // {{{
{
	DEBUG0("void _HistoCanvas::DelayedUpdate(bool m)\n");
	_delayedUpdate = m;
	_updateCounter = 0;
	DEBUG3("_delayedUpdate = "<<_delayedUpdate<<"\n");
} // }}} void _HistoCanvas::DelayedUpdate(bool m)

bool _HistoCanvas::DelayedUpdate() // {{{
{
	DEBUG0("bool _HistoCanvas::DelayedUpdate()\n");
	return _delayedUpdate;
} // }}} bool _HistoCanvas::DelayedUpdate()

void _HistoCanvas::OnPaint(wxPaintEvent& event) // {{{
{
	DEBUG0("void _HistoCanvas::OnPaint(wxPaintEvent& event)\n");
	//! This is how we work:
	//! \li First we set the device up for drawing.
	DEBUG3("Painting ...\n");
	wxPaintDC dc(this);
	PrepareDC(dc);
	dc.SetBackground(wxBrush(wxColour(255, 255, 255), wxSOLID));
	//! \li We clear it.
	dc.Clear();
	//! \li and set the description.
	if (_callback) {
		wxString txt("min: ");
		txt<<_min<<", max: "<<_max;
		_callback->OnTextChange(txt);
	}
	//! \li If the history is empty, we exit here.
	if (_min==-1 || _max==-1) return;
	wxPen pen;
	pen.SetColour(wxColour(50, 50, 50));
	wxBrush brush;
	brush.SetColour(wxColour(50, 50, 50));
	wxPen spen;
	spen.SetColour(_selColor);
	wxBrush sbrush;
	sbrush.SetColour(_selColor);
	dc.SetPen(pen);
	dc.SetBrush(brush);
	//! \li Next we calculate the geometry.
	int width;
	int height;
	GetClientSize(&width, &height);
	_vpp = (float)height / (float)_vmax;
	_ppe = (float)(_max-_min+1) / (float)width;
	DEBUG2("_ppe = "<<_ppe<<", _vpp = "<<_vpp<<"\n");
	if (_ppe < 1.0) {
		//! \li and than we either draw blocks, if there are less blocks than horizontal
		//!     pixels, or
		float bw = 1.0f/_ppe;
		DEBUG2("bw = "<<bw<<", width = "<<width<<", calc width = "<<bw*(_max-_min)<<"\n");
		float pos = 0.0;
		for(int i=_min; i<=_max; ++i) {
			int h = (int)(_histo[i]*_vpp);
			if (i == _selected) {
				dc.SetPen(spen);
				dc.SetBrush(sbrush);
			}
			dc.DrawRectangle((int)pos, height-h, (int)(bw+0.99f), h);
			pos += bw;
			dc.SetPen(pen);
			dc.SetBrush(brush);
		}
	} else {
		//! \li lines otherwise. In this case we calculate a average over all elements at
		//!     this pixel.
		int idx = _min;
		for(int i=1; i<=width; ++i) {
			int val = 0;
			int cnt = 0;
			while(idx<=(_min+_ppe*i) && idx<=_max) {
				cnt++;
				val += _histo[idx++];
				if (i == _selected) {
					dc.SetPen(spen);
					dc.SetBrush(sbrush);
				}
			}
			int h = (int)(((float)val/(float)cnt)*_vpp);
			dc.DrawLine(i, height-h, i, height);
			dc.SetPen(pen);
			dc.SetBrush(brush);
		}
	}
} // }}} void _HistoCanvas::OnPaint(wxPaintEvent& event)

void _HistoCanvas::OnMotion(wxMouseEvent& event) // {{{
{
	DEBUG0("void _HistoCanvas::OnMotion(wxMouseEvent& event)\n");
	wxString txt("min: ");
	txt<<_min<<", max: "<<_max<<", ";
	DEBUG2("x = "<<event.m_x<<", y = "<<event.m_y<<", _ppe = "<<_ppe<<"\n");
	float idx;
	if (_ppe < 1.0) {
		idx = (float)_min+(float)event.m_x*_ppe;
		DEBUG2("element = "<<idx<<"\n");
		txt<<"["<<(int)idx<<"]="<<_histo[(int)idx];
	} else {
		idx = (float)_min+((float)event.m_x-0.99)*_ppe;
		DEBUG2("first element = "<<idx<<"\n");
		while(idx<=(_min+_ppe*(float)event.m_x) && idx<=_max) {
			txt<<"["<<(int)idx<<"]="<<_histo[(int)idx]<<" ";
			idx += 1;
		}
		DEBUG2("last element = "<<_idx<<"\n");
	}
	_idx = (int)idx;
	if (_callback) _callback->OnTextChange(txt);
} // }}} void _HistoCanvas::OnMotion(wxMouseEvent& event)

void _HistoCanvas::OnMouse(wxMouseEvent& event) // {{{
{
	DEBUG0("void _HistoCanvas::OnMouse(wxMouseEvent& event)\n");
	if (event.LeftUp()) {
		_selected = _idx;
		_callback->OnClicked(_idx);
	}
} // }}} void _HistoCanvas::OnMouse(wxMouseEvent& event)

IMPLEMENT_DYNAMIC_CLASS(vuMCHistogram, vuMCWidget)

BEGIN_EVENT_TABLE(vuMCHistogram, vuMCWidget) // {{{
	EVT_CHECKBOX(vuMCHistogram::idCbxDelayedUpdate, vuMCHistogram::OnDelayedUpdateChecked)
	//EVT_COMMAND_LEFT_CLICK(vuMCHistogram::idClkHistogram, vuMCHistogram::OnHistogramClick)
	EVT_TOOL(vuMCHistogram::idClkHistogram, vuMCHistogram::OnClicked)
END_EVENT_TABLE(); // }}}

vuMCHistogram::vuMCHistogram(wxWindow* p, wxSize size, vuMCDecorator<int>* cb) // {{{
: vuMCWidget(p), _parent(p), _callback(cb)
{
	DEBUGC("vuMCHistogram::vuMCHistogram(wxWindow* p, wxSize size, vuMCDecorator<int>* cb)\n");
	_delayedUpdate = new wxCheckBox(this, idCbxDelayedUpdate, "Delayed update");
	Sizer()->Add(_delayedUpdate, 0, wxGROW, 5);
	_histoDC = new _HistoCanvas(this, idClkHistogram, wxDefaultPosition, size);
	Sizer()->Add(_histoDC, 1, wxGROW, 5);
	_delayedUpdate->SetValue(_histoDC->DelayedUpdate());
	_histoDC->SetHelpText("Nasenhelptext");
	_desc = new wxStaticText(this, -1, _T("No elements"));
	Sizer()->Add(_desc, 0, wxGROW, 5);
	Fit();
} // }}} vuMCHistogram::vuMCHistogram(wxWindow* p, wxSize size, vuMCDecorator<int>* cb)

vuMCHistogram::~vuMCHistogram() // {{{
{
	DEBUGD("virtual vuMCHistogram::~vuMCHistogram()\n");
	delete _desc;
	delete _histoDC;
	delete _delayedUpdate;
} // }}} vuMCHistogram::~vuMCHistogram()

void vuMCHistogram::Clear() // {{{
{
	DEBUG0("void vuMCHistogram::Clear()\n");
	_histoDC->Reset();
} // }}} void vuMCHistogram::Clear()

void vuMCHistogram::Flush() // {{{
{
	DEBUG0("void vuMCHistogram::Flush()\n");
	wxPaintEvent ev;
	_histoDC->OnPaint(ev);
	//_histoDC->Update();
} // }}} void vuMCHistogram::Flush()

void vuMCHistogram::Inc(int i, int val) // {{{
{
	DEBUG0("void vuMCHistogram::Inc(int i="<<i<<", int val="<<val<<")\n");
	_histoDC->Inc(i, val);
} // }}} void vuMCHistogram::Inc(int i, int val)

void vuMCHistogram::SetSelColor(vuVector c) // {{{
{
	DEBUG0("void vuMCHistogram::SetSelColor(vuVector c)\n");
	wxColour col(int(255.0*c[0]), int(255.0*c[1]), int(255.0*c[2]));
	_histoDC->SetSelColor(col);
} // }}} void vuMCHistogram::SetSelColor(vuVector c)

void vuMCHistogram::SetSelColor(wxColour &c) // {{{
{
	DEBUG0("void vuMCHistogram::SetSelColor(wxColour &c)\n");
	_histoDC->SetSelColor(c);
} // }}} void vuMCHistogram::SetSelColor(wxColour &c)

void vuMCHistogram::Select(int type) // {{{
{
	DEBUG0("void vuMCHistogram::Select(int type)\n");
	_histoDC->Select(type);
	Flush();
} // }}} void vuMCHistogram::Select(int type)

void vuMCHistogram::OnTextChange(wxString txt) // {{{
{
	DEBUG0("void vuMCHistogram::OnTextChange(wxString txt)\n");
	_desc->SetLabel(txt);
} // }}} void vuMCHistogram::OnTextChange(wxString txt)

void vuMCHistogram::OnClicked(int idx) // {{{
{
	DEBUG0("void vuMCHistogram::OnClicked(wxCommandEvent &event)\n");
	if (0 < idx && idx < 255) {
		DEBUG5("idx = "<<idx<<"\n");
		_histoDC->Select(idx);
		Flush();
		if (_callback) (*_callback)(idx);
	}
} // }}} void vuMCHistogram::OnClicked(wxCommandEvent &event)

void vuMCHistogram::OnDelayedUpdateChecked(wxCommandEvent &event) // {{{
{
	DEBUG0("void vuMCHistogram::OnDelayedUpdateChecked(wxCommandEvent &event)\n");
	_histoDC->DelayedUpdate(_delayedUpdate->IsChecked());
} // }}} void vuMCHistogram::OnDelayedUpdateChecked(wxCommandEvent &event)

// }}} vuMCHistogram

// >>>
// {{{ class vuMarchingCubes:

//----------------------------------------------------------------------------
//------------------------- The wxWindows event table ------------------------
//----------------------------------------------------------------------------
BEGIN_EVENT_TABLE(vuMarchingCubes, vuBasicUtility) // {{{
	EVT_CHECKBOX (vuMarchingCubes::idMCCbxCubeEnabled,   vuMarchingCubes::OnCubeEnabledChecked)
	EVT_CHECKBOX (vuMarchingCubes::idMCCbxLight1Enabled, vuMarchingCubes::OnLight1Checked)
	EVT_CHECKBOX (vuMarchingCubes::idMCCbxDrawFrame,     vuMarchingCubes::OnFrameChecked)
	EVT_RADIOBOX (vuMarchingCubes::idMCRbxRotate,        vuMarchingCubes::OnRotateChecked)
	EVT_CHECKBOX (vuMarchingCubes::idMCCbxCullLinesFrontBack,
		      					     vuMarchingCubes::OnCullLinesFrontBackChecked)
	EVT_CHECKBOX (vuMarchingCubes::idMCCbxUpdateMode,    vuMarchingCubes::OnUpdateModeChecked)
#ifdef LINUX
	EVT_BUTTON   (vuMarchingCubes::idMCBtnUpdate,        vuMarchingCubes::OnUpdateClicked)
#endif
	EVT_CHECKBOX (vuMarchingCubes::idMCCbxDrawNormals,   vuMarchingCubes::OnDrawNormalsChecked)
END_EVENT_TABLE(); // }}}

//----------------------------------------------------------------------------
//------------------------- The constructor ----------------------------------
//----------------------------------------------------------------------------
vuMarchingCubes::vuMarchingCubes() // {{{
: m_Data(0),
  m_imageList(0),
  m_thres(0),
#if defined(LINUX)
  m_update(0),
#endif
  m_cubesTotal(0),
  m_cube0(0),
  m_cube255(0),
  m_cubeOther(0),
  m_cubeHist(0),
  m_size(0),
  m_angle(0),
  m_pos(0),
  m_zoom(0),
  m_light0Pos(0),
  m_light1Enabled(0),
  m_light1Pos(0),
  m_numTriangles(0),
  m_numVertices(0),
  m_numIndices(0),
  m_statusText(0),
  m_status(0),
  m_backColor(0),
  m_lightColor(0),
  m_lightDiffuse(0),
  m_lightSpecular(0),
  m_lightShiny(0),
  m_frameColor(0),
  m_lineColor(0),
  m_selectColor(0),
  m_objectColor(0),
  m_objectDiffuse(0),
  m_objectSpecular(0),
  m_objectShiny(0),
  m_mouseSpeed(0),
  m_drawFrame(0),
  m_cullBack(0),
  m_cullFront(0),
  m_linesBack(0),
  m_linesFront(0),
  m_frameRadius(0),
  m_rotateX(0),
  m_rotateY(0),
  m_rotateZ(0),
  m_updateMode(0),
  m_cubeEnabled(0),
  m_cube(0),
  m_drawNormals(0),
  m_drawNormalsSize(0),
  m_cubeSizeX(0),
  m_cubeSizeY(0),
  m_cubeSizeZ(0),
  m_fpsTime(0),
  m_fpsCount(0),
  m_fpsWatermark(0),
  m_fps(1.0),
  m_avgFps(1.0),
  x1(0.0), y1(0.0), z1(0.0),
  x2(1.0), y2(1.0), z2(1.0)
{
	DEBUGC("vuMarchingCubes::vuMarchingCubes()\n");
	m_Data = NULL;
} // }}}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------
vuMarchingCubes::~vuMarchingCubes() // {{{
{
	DEBUGD("virtual vuMarchingCubes::~vuMarchingCubes()\n");
	if (m_Data)		delete m_Data;
	if (m_imageList)	delete m_imageList;
	if (m_thres)		delete m_thres;
#if defined(LINUX)
	if (m_update)		delete m_update;
#endif
	if (m_cubesTotal)	delete m_cubesTotal;
	if (m_cube0)		delete m_cube0;
	if (m_cube255)		delete m_cube255;
	if (m_cubeOther)	delete m_cubeOther;
	if (m_cubeHist)		delete m_cubeHist;
	if (m_size)		delete m_size;
	if (m_angle)		delete m_angle;
	if (m_pos)		delete m_pos;
	if (m_zoom)		delete m_zoom;
	if (m_light0Pos)	delete m_light0Pos;
	if (m_light1Enabled)	delete m_light1Enabled;
	if (m_light1Pos)	delete m_light1Pos;
	if (m_numTriangles)	delete m_numTriangles;
	if (m_numVertices)	delete m_numVertices;
	if (m_numIndices)	delete m_numIndices;
	if (m_statusText)	delete m_statusText;
	if (m_status)		delete m_status;
	if (m_backColor)	delete m_backColor;
	if (m_lightColor)	delete m_lightColor;
	if (m_lightDiffuse)	delete m_lightDiffuse;
	if (m_lightSpecular)	delete m_lightSpecular;
	if (m_lightShiny)	delete m_lightShiny;
	if (m_frameColor)	delete m_frameColor;
	if (m_lineColor)	delete m_lineColor;
	if (m_selectColor)	delete m_selectColor;
	if (m_objectColor)	delete m_objectColor;
	if (m_objectDiffuse)	delete m_objectDiffuse;
	if (m_objectSpecular)	delete m_objectSpecular;
	if (m_objectShiny)	delete m_objectShiny;
	if (m_mouseSpeed)	delete m_mouseSpeed;
	if (m_drawFrame)	delete m_drawFrame;
	if (m_cullBack)		delete m_cullBack;
	if (m_cullFront)	delete m_cullFront;
	if (m_linesBack)	delete m_linesBack;
	if (m_linesFront)	delete m_linesFront;
	if (m_frameRadius)	delete m_frameRadius;
	if (m_rotateX)		delete m_rotateX;
	if (m_rotateY)		delete m_rotateY;
	if (m_rotateZ)		delete m_rotateZ;
	if (m_updateMode)	delete m_updateMode;
	if (m_cubeEnabled)	delete m_cubeEnabled;
	if (m_cube)		delete m_cube;
	if (m_drawNormals)	delete m_drawNormals;
	if (m_drawNormalsSize)	delete m_drawNormalsSize;
	if (m_cubeSizeX)	delete m_cubeSizeX;
	if (m_cubeSizeY)	delete m_cubeSizeY;
	if (m_cubeSizeZ)	delete m_cubeSizeZ;
	if (m_fpsWatermark)	delete m_fpsWatermark;
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: getFileType() ----------------------------
//----------------------------------------------------------------------------
const char* vuMarchingCubes::getFileType() // {{{
{
	DEBUG0("const char* vuMarchingCubes::getFileType()\n");
	return "1112211 1112212";
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: init() -----------------------------------
//----------------------------------------------------------------------------
bool vuMarchingCubes::init(const char* DataFile) // {{{
{
	DEBUG0("bool vuMarchingCubes::init(" << DataFile << ")\n");
	assert(m_size != NULL);
	SetEvtHandlerEnabled(true);

	//Set up the window
	SetTitle("RGB Marching Cubes");
	CreateStatusBar();

	//Read in the data.
	m_Data = new vu1112211;
	m_base = (MarchingCubesBase*)((vu1112211*)m_Data);
	m_base->setGlobalData(&m_gd);
	m_base->SetProgressCallback(new vuMCMCBProgressCallback(this));
	m_base->SetHistoCallback(new vuMCMCBHistoCallback(this));
	m_Data->setFileName(DataFile);
	bool success = m_Data->read();
	if (!success) return false;
	m_base->marchCubes();
	doThresChange();
	DEBUG1("m_base->Size1() = "<<m_base->Size1()<<"\n");
	DEBUG1("m_base->Size2() = "<<m_base->Size2()<<"\n");
	DEBUG1("m_base->Size3() = "<<m_base->Size3()<<"\n");

	(*m_size)[0] = m_base->Size1();
	(*m_size)[1] = m_base->Size2();
	(*m_size)[2] = m_base->Size3();
	(*m_pos)[2] = -m_base->maxSize()*2;
	(*m_light0Pos)[0] = -m_base->maxSize();
	(*m_light0Pos)[1] =  m_base->maxSize();
	(*m_light0Pos)[2] =  m_base->maxSize();
	(*m_light1Pos)[0] =  m_base->maxSize();
	(*m_light1Pos)[1] =  m_base->maxSize();
	(*m_light1Pos)[2] =  m_base->maxSize();
	m_frameRadius->Min(0.0);
	m_frameRadius->Max((float)m_base->maxSize() / 200.0);
	m_frameRadius->Min((float)m_base->maxSize() / 1000.0);
	*m_frameRadius = m_gd.frameRadius;
	m_frameRadius->Format("%0.3f");
	*m_cubesTotal = float((m_base->Size1()-1)*(m_base->Size2()-1)*(m_base->Size3()-1));
	*m_cubeOther = (float)*m_cubesTotal;
	m_drawNormals->SetValue(m_gd.drawNormals);
	m_drawNormalsSize->Min(1.0);
	m_drawNormalsSize->Max((float)m_base->maxSize() / 2.0);
	*m_drawNormalsSize = m_gd.normalsLength;
	*m_meshSize = m_gd.meshSize;
	m_cubeHist->SetSelColor(m_selectColor->tovuVector());

	return success;
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: getCamera() ------------------------------
//----------------------------------------------------------------------------
vuCamera* vuMarchingCubes::getCamera() // {{{
{
	DEBUG0("vuCamera* vuMarchingCubes::getCamera()\n");
	return 0;	// we have no camera!
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: setCamera() ------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::setCamera(vuCamera* cam) // {{{
{
	DEBUG0("void vuMarchingCubes::setCamera(vuCamera* cam)\n");
	//m_Data->setCamera(cam);
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: getCurrentImage() ------------------------
//----------------------------------------------------------------------------
vuImage* vuMarchingCubes::getCurrentImage() // {{{
{
	DEBUG0("vuImage* vuMarchingCubes::getCurrentImage()\n");
	return NULL;
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: DrawFromImage() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::DrawFromImage() // {{{
{
	DEBUG0("void vuMarchingCubes::DrawFromImage()\n");
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: getCamera() ------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::DrawAgain() // {{{
{
	DEBUG0("void vuMarchingCubes::DrawAgain()\n");
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: addTop() ---------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::addTop(wxSizer *sizer) // {{{
{
	DEBUG0("void vuMarchingCubes::addTop(" << sizer << ")\n");
	m_thres = new vuMCSlider(this, "Threshold: ", "%5.0f", new vuMCThresChange(this),
				 m_gd.valueMin, m_gd.valueMax, m_gd.thres,
				 4095, VU_MCS_END_UPDATE);
	sizer->Add(m_thres, 1, wxALL, 5);
#if defined LINUX
	m_update = new wxButton(this, idMCBtnUpdate, "update");
	m_update->Enable(!m_thres->UpdateMode());
	sizer->Add(m_update, 0, wxGROW, 5);
#endif
	SetAutoLayout(TRUE);
	sizer->Fit(this);
} // }}}

//----------------------------------------------------------------------------
//------------------------- public: addRight() -------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::addRight(wxSizer *sizer) // {{{
{
	DEBUG0("void vuMarchingCubes::addRight(" << sizer << ")\n");

	// First let's create some icons for the tabs:
	m_imageList = new wxImageList(16, 16);
	wxSize size(16, 16);
	m_imageList->Add(wxArtProvider::GetIcon(wxART_INFORMATION, wxART_OTHER, size));
	m_imageList->Add(wxArtProvider::GetIcon(wxART_QUESTION, wxART_OTHER, size));
	m_imageList->Add(wxBitmap(opengl_logo_xpm));

	// Notebook setup:
	wxNotebook* notebook = new wxNotebook(this, -1);
	sizer->Add(new wxNotebookSizer(notebook), 1, wxGROW, 5);
	notebook->SetImageList(m_imageList);

	// Adding the info page to the notebook:
	notebook->AddPage(createInfoNotebook(notebook), _T("Info"), TRUE, 0);

	// Adding the settings page to the notebook:
	notebook->AddPage(createSettingsNotebook(notebook), _T("Settings"), FALSE, 1);

	// Adding the OpenGL page to the notebook:
	notebook->AddPage(createGLNotebook(notebook), _T("GL"), FALSE, 2);

	// Adding the Help page to the notebook:
	notebook->AddPage(createHelpNotebook(notebook), _T("Help"), FALSE, 0);

	sizer->SetMinSize(250, 10);
} // }}}

//----------------------------------------------------------------------------
//------------------------- protected: createInfoNotebook() ------------------
//----------------------------------------------------------------------------
wxPanel* vuMarchingCubes::createInfoNotebook(wxNotebook* nb) // {{{
{
	DEBUG0("wxPanel* vuMarchingCubes::createInfoNotebook(wxNotebook* nb)\n");
	wxPanel*     page;
	wxSizer*     szPage;
	wxSizer*     szSB;
	//wxGridSizer* szGrid;

	// Setup:
	page = new wxPanel(nb);
	szPage = new wxBoxSizer(wxVERTICAL);
	page->SetSizer(szPage);

	// Cube Counter:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Cube histogram:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_cubesTotal = new vuMCKeyValue(page, "Cubes total: ", "%1.0f");
	(*m_cubesTotal) = 0.0f;
	szSB->Add(m_cubesTotal, 0, wxGROW, 5);
	m_cube0 = new vuMCKeyValue(page, "Cubes of type 0: ", "%1.0f");
	(*m_cube0) = 0.0f;
	szSB->Add(m_cube0, 0, wxGROW, 5);
	m_cube255 = new vuMCKeyValue(page, "Cubes of type 255: ", "%1.0f");
	(*m_cube255) = 0.0f;
	szSB->Add(m_cube255, 0, wxGROW, 5);
	m_cubeOther = new vuMCKeyValue(page, "Cubes of other types: ", "%1.0f");
	(*m_cubeOther) = 0.0f;
	szSB->Add(m_cubeOther, 0, wxGROW, 5);
	m_cubeHist = new vuMCHistogram(page, wxSize(200, 100), new vuMCSelectType(this));
	szSB->Add(m_cubeHist, 0, wxGROW, 5);

	// States:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Internal States:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_size = new vuMCKeyVector(page, "Data Size: ", 3, "%1.0f");
	(*m_size)[0] = 0.0;
	(*m_size)[1] = 0.0;
	(*m_size)[2] = 0.0;
	szSB->Add(m_size, 0, wxGROW, 5);
	m_angle = new vuMCKeyVector(page, "Angle: ", 3, "%0.1f");
	szSB->Add(m_angle, 0, wxGROW, 5);
	m_pos = new vuMCKeyVector(page, "Position: ", 4, "%0.1f");
	(*m_pos)[3] = 1.0;	// Last element must be one.
	szSB->Add(m_pos, 0, wxGROW, 5);
	m_zoom = new vuMCKeyValue(page, "Zoom: ", "%0.3f");
	(*m_zoom) = 0.5f;
	szSB->Add(m_zoom, 0, wxGROW, 5);
	m_light0Pos = new vuMCKeyVector(page, "Pos Light 0: ", 4, "%0.1f");
	(*m_light0Pos)[3] = 1.0;
	szSB->Add(m_light0Pos, 0, wxGROW, 5);
	m_light1Pos = new vuMCKeyVector(page, "Pos Light 1: ", 4, "%0.1f");
	(*m_light1Pos)[3] = 1.0;
	szSB->Add(m_light1Pos, 0, wxGROW, 5);
	m_light1Pos->Enable(MC_DEFAULT_LIGHT1_ENABLED);
	m_numTriangles = new vuMCKeyValue(page, "Number of Triangles: ", "%1.0f");
	(*m_numTriangles) = 0.0;
	szSB->Add(m_numTriangles, 0, wxGROW, 5);
	m_numVertices = new vuMCKeyValue(page, "Number of Vertices: ", "%1.0f");
	(*m_numVertices) = 0.0;
	szSB->Add(m_numVertices, 0, wxGROW, 5);
	m_numIndices = new vuMCKeyValue(page, "Number of Indices: ", "%1.0f");
	(*m_numIndices) = 0.0;
	szSB->Add(m_numIndices, 0, wxGROW, 5);
	m_meshSize = new vuMCKeyValue(page, "Memory usage: ", "%1.0f");
	szSB->Add(m_meshSize, 0, wxGROW, 5);

	// States:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Progress:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_statusText = new wxStaticText(page, -1, "No status:");
	m_statusText->Disable();
	szSB->Add(m_statusText, 0, wxGROW, 5);
	m_status = new wxGauge(page, -1, 100, wxDefaultPosition, wxSize(-1, 10));
	m_status->Disable();
	szSB->Add(m_status, 0, wxGROW, 5);

	return page;
} /* }}} wxPanel* vuMarchingCubes::createInfoNotebook(wxNotebook* nb) */

//----------------------------------------------------------------------------
//------------------------- protected: createSettingsNotebook() --------------------
//----------------------------------------------------------------------------
wxPanel* vuMarchingCubes::createSettingsNotebook(wxNotebook* nb) // {{{
{
	wxPanel*     page;
	wxSizer*     szPage;
	wxSizer*     szSB;
	wxString     choices[3];

	// Page Setup:
	page = new wxPanel(nb);
	szPage = new wxBoxSizer(wxVERTICAL);
	page->SetSizer(szPage);

	// Mousing:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Mouse:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	//m_mouseSpeed = new vuMCSlider(page, "Speed ", "%3.0f" , new vuMCRender(this), 1.0, 100.0, 85.0);
	m_mouseSpeed = new vuMCSlider(page, "Speed ", "%3.0f" , NULL, 1.0, 100.0, 85.0);
	szSB->Add(m_mouseSpeed, 0, wxGROW, 20);

	// Frame:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Frame:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	//   Should frame be drawn:
	m_drawFrame = new wxCheckBox(page, idMCCbxDrawFrame, "Draw Frame");
	m_drawFrame->SetValue(MC_DEFAULT_DRAW_FRAME);
	szSB->Add(m_drawFrame, 0, wxGROW, 5);
	//   Frame radius:
	m_frameRadius = new vuMCSlider(page, "Radius: ", "%1.2f",
				       new vuMCFrameChange(this), 1.0, 100.0, 10.0);
	m_frameRadius->Enable(MC_DEFAULT_DRAW_FRAME);
	szSB->Add(m_frameRadius, 0, wxGROW, 5);

	// Rotate Object:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Rotate Object:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	choices[0] = wxString("-90");
	choices[1] = wxString("0");
	choices[2] = wxString("90");
	m_rotateX = new wxRadioBox(page, idMCRbxRotate, "Rotate arround x-axis:",
				   wxDefaultPosition, wxDefaultSize,
				   3, choices, 3, wxRA_SPECIFY_COLS);
	m_rotateX->SetSelection(1);
	szSB->Add(m_rotateX, 1, wxGROW, 5);
	m_rotateY = new wxRadioBox(page, idMCRbxRotate, "Rotate arround y-axis:",
				   wxDefaultPosition, wxDefaultSize,
				   3, choices, 3, wxRA_SPECIFY_COLS);
	m_rotateY->SetSelection(1);
	szSB->Add(m_rotateY, 1, wxGROW, 5);
	m_rotateZ = new wxRadioBox(page, idMCRbxRotate, "Rotate arround z-axis:",
				   wxDefaultPosition, wxDefaultSize,
				   3, choices, 3, wxRA_SPECIFY_COLS);
	m_rotateZ->SetSelection(1);
	szSB->Add(m_rotateZ, 1, wxGROW, 5);

	// Other:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Other:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_cullBack = new wxCheckBox(page, idMCCbxCullLinesFrontBack, "Cull Back Faces");
	m_cullBack->SetValue(false);
	szSB->Add(m_cullBack, 0, wxGROW, 5);
	m_cullFront = new wxCheckBox(page, idMCCbxCullLinesFrontBack, "Cull Front Faces");
	m_cullFront->SetValue(false);
	szSB->Add(m_cullFront, 0, wxGROW, 5);
	m_linesBack = new wxCheckBox(page, idMCCbxCullLinesFrontBack, "Lines on Back Faces");
	m_linesBack->SetValue(false);
	szSB->Add(m_linesBack, 0, wxGROW, 5);
	m_linesFront = new wxCheckBox(page, idMCCbxCullLinesFrontBack, "Lines on Front Faces");
	m_linesFront->SetValue(false);
	szSB->Add(m_linesFront, 0, wxGROW, 5);
	m_fpsWatermark = new vuMCSlider(page, "FPS wm: ", "%3.0f", NULL, 1.0, 1000.0,
					(float)MC_DEFAULT_FPS_WATERMARK, 1000);
	szSB->Add(m_fpsWatermark, 0, wxGROW, 5);
	m_updateMode = new wxCheckBox(page, idMCCbxUpdateMode, "Continuous Update on threshold change");
	m_updateMode->SetValue(m_thres->UpdateMode());
	szSB->Add(m_updateMode, 0, wxGROW, 5);
	m_doMeshStatistics = new wxCheckBox(page, idMCCbxMeshStatisitcs,
					    "Do memory statistics on the mesh (slower)");
	m_doMeshStatistics->SetValue(m_gd.doMeshStatistics);
	szSB->Add(m_doMeshStatistics, 0, wxGROW, 5);

	// Cube Browser:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Cube Setup Browser:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_cubeEnabled = new wxCheckBox(page, idMCCbxCubeEnabled, "Enable Cube Browser");
	m_cubeEnabled->SetValue(m_gd.cubeBrowser);
	szSB->Add(m_cubeEnabled, 0, 0, 5);
	m_cube = new vuMCBitField(page, "Cube", new vuMCCubeChange(this), 8, 255);
	m_cube->Enable(m_gd.cubeBrowser);
	szSB->Add(m_cube, 0, wxGROW, 5);

	// Drawing Normals:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Drawing Normals:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_drawNormals = new wxCheckBox(page, idMCCbxDrawNormals, "Draw Normals");
	m_drawNormals->SetValue(m_gd.drawNormals);
	szSB->Add(m_drawNormals, 0, 0, 5);
	m_drawNormalsSize = new vuMCSlider(page, "Length: ", "%2.1f", new vuMCNormalLengthChange(this), 0.1, 5.0, m_gd.normalsLength);
	m_drawNormalsSize->Enable(m_gd.drawNormals);
	szSB->Add(m_drawNormalsSize, 0, wxGROW, 5);

	// Cube Size:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Cube size:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_cubeSizeX = new vuMCSlider(page, "Size X: ", "%2.1f", new vuMCCubeSizeChange(this), 0.1, 5.0, 1.0);
	szSB->Add(m_cubeSizeX, 0, wxGROW, 5);
	m_cubeSizeY = new vuMCSlider(page, "Size Y: ", "%2.1f", new vuMCCubeSizeChange(this), 0.1, 5.0, 1.0);
	szSB->Add(m_cubeSizeY, 0, wxGROW, 5);
	m_cubeSizeZ = new vuMCSlider(page, "Size Z: ", "%2.1f", new vuMCCubeSizeChange(this), 0.1, 5.0, 1.0);
	szSB->Add(m_cubeSizeZ, 0, wxGROW, 5);

	// fire it back:
	return page;
} /* }}} wxPanel* vuMarchingCubes::createSettingsNotebook(wxNotebook* nb) */

//----------------------------------------------------------------------------
//------------------------- protected: createGLNotebook() --------------------
//----------------------------------------------------------------------------
wxPanel* vuMarchingCubes::createGLNotebook(wxNotebook* nb) // {{{
{
	wxPanel*    page;
	wxSizer*    szPage;
	wxSizer*    szSB;

	// Page setup:
	page = new wxPanel(nb);
	szPage = new wxBoxSizer(wxVERTICAL);
	page->SetSizer(szPage);

	// Some Colors:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Light:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_light1Enabled = new wxCheckBox(page, idMCCbxLight1Enabled, "Enable Light 1");
	m_light1Enabled->SetValue(MC_DEFAULT_LIGHT1_ENABLED);
	szSB->Add(m_light1Enabled, 0, wxGROW, 5);
	m_backColor = new vuMCRGBColor(page, "Background:", new vuMCBackgroundChange(this),
				       wxColour(255, 255, 255), 0.45);
	szSB->Add(m_backColor, 0, wxGROW, 5);
	setBackground();
	m_lightColor = new vuMCRGBColor(page, "Color:", new vuMCRender(this),
					wxColour(255, 255, 255), 1.0);
	szSB->Add(m_lightColor, 0, wxGROW, 5);
	m_lightDiffuse = new vuMCGrayColor(page, "Diffuse:", new vuMCRender(this), 0.4);
	szSB->Add(m_lightDiffuse, 0, wxGROW, 5);
	m_lightSpecular = new vuMCGrayColor(page, "Specular:", new vuMCRender(this), 0.6);
	szSB->Add(m_lightSpecular, 0, wxGROW, 5);
	m_lightShiny = new vuMCGrayColor(page, "Shinyness:", new vuMCRender(this), 0.8);
	szSB->Add(m_lightShiny, 0, wxGROW, 5);

	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Objects:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	m_frameColor = new vuMCRGBColor(page, "Frame:", new vuMCRender(this),
					wxColour(128, 64, 64), 0.8);
	szSB->Add(m_frameColor, 0, wxGROW, 5);
	m_lineColor = new vuMCRGBColor(page, "Line:", new vuMCLineColorChange(this),
				       wxColour(0, 128, 0), 0.4);
	m_gd.lineColor = m_lineColor->tovuVector();
	szSB->Add(m_lineColor, 0, wxGROW, 5);
	m_selectColor = new vuMCRGBColor(page, "Selection:", new vuMCObjectColorChange(this),
				       wxColour(0, 0, 128), 0.4);
	szSB->Add(m_selectColor, 0, wxGROW, 5);
	m_objectColor = new vuMCRGBColor(page, "Object:", new vuMCObjectColorChange(this),
					 wxColour(128, 128, 0), 0.4);
	for(int i = 0; i < 256; ++i) {
		m_gd.objectColor[i] = m_objectColor->tovuVector();
	}
	szSB->Add(m_objectColor, 0, wxGROW, 5);
	m_objectDiffuse = new vuMCGrayColor(page, "Diffuse:", new vuMCRender(this), 0.8);
	szSB->Add(m_objectDiffuse, 0, wxGROW, 5);
	m_objectSpecular = new vuMCGrayColor(page, "Specular:", new vuMCRender(this), 0.9);
	szSB->Add(m_objectSpecular, 0, wxGROW, 5);
	m_objectShiny = new vuMCGrayColor(page, "Shinyness:", new vuMCRender(this), 0.85);
	szSB->Add(m_objectShiny, 0, wxGROW, 5);

	// Fire it back:
	return page;
} /* }}} wxPanel* vuMarchingCubes::createGLNotebook(wxNotebook* nb) */

//----------------------------------------------------------------------------
//------------------------- protected: createHelpNotebook() --------------------
//----------------------------------------------------------------------------
wxPanel* vuMarchingCubes::createHelpNotebook(wxNotebook* nb) // {{{
{
	wxPanel*      page;
	wxSizer*      szPage;
	wxSizer*      szSB;
	wxStaticText* text;

	// Page setup:
	page = new wxPanel(nb);
	szPage = new wxBoxSizer(wxVERTICAL);
	page->SetSizer(szPage);

	// Moving the Object:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Moving the object:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	text = new wxStaticText(page, -1, "Left mouse: Rotate object");
	szSB->Add(text, 0, wxGROW, 5);
	text = new wxStaticText(page, -1, "Shift left mouse: Move object along the x- and y-axis");
	szSB->Add(text, 0, wxGROW, 5);
	text = new wxStaticText(page, -1, "Ctrl left mouse: Move object along the z-axis");
	szSB->Add(text, 0, wxGROW, 5);

	// Other movements:
	szSB = new wxStaticBoxSizer(new wxStaticBox(page, -1, _T("Other movements:")), wxVERTICAL);
	szPage->Add(szSB, 0, wxGROW, 5);
	text = new wxStaticText(page, -1, "Alt left mouse: Zoom (down = in, up = out)");
	szSB->Add(text, 0, wxGROW, 5);
	text = new wxStaticText(page, -1, "Ctrl right mouse: Move light 0 along the x- and z-axis");
	szSB->Add(text, 0, wxGROW, 5);
	text = new wxStaticText(page, -1, "Alt right mouse: Move light 1 along the x- and z-axis");
	szSB->Add(text, 0, wxGROW, 5);

	// Fire it back:
	return page;
} /* }}} wxPanel* vuMarchingCubes::createGLNotebook(wxNotebook* nb) */

//----------------------------------------------------------------------------
//------------------------- protected: getVolume() ---------------------------
//----------------------------------------------------------------------------
vu1 *vuMarchingCubes::getVolume() // {{{
{
	DEBUG0("vu1 *vuMarchingCubes::getVolume()\n");
	return (vu1 *)m_Data;
	//return NULL;
} // }}}

//----------------------------------------------------------------------------
//------------------------- protected: glInit() ------------------------------
//----------------------------------------------------------------------------
bool vuMarchingCubes::glInit() // {{{
{
	DEBUG0("bool vuMarchingCubes::glInit(void)\n");
	if (m_Data == 0) return false;
	assert(m_size != NULL);
	glClearColor(m_backColor->Red(), m_backColor->Green(), m_backColor->Blue(), m_backColor->Alpha());

	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glShadeModel(GL_SMOOTH);
	glEnable(GL_LINE_SMOOTH);
	glEnable(GL_DEPTH_TEST);
	//glEnable(GL_COLOR_MATERIAL);

	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);
	glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);

	glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
	glLightModeli(GL_LIGHT_MODEL_LOCAL_VIEWER, GL_TRUE);
	//glDepthFunc(GL_LESS);
	//m_Data->initOpenGL();
	
	return true;
}; // }}}

//----------------------------------------------------------------------------
//------------------------- protected: glResize() ----------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::glResize() // {{{
{
	DEBUG0("void vuMarchingCubes::glResize()\n");

	GLint w = (GLint)m_glCanvas->getWidth();
	GLint h = (GLint)m_glCanvas->getHeight();
	float ratio = (float)w/(float)h;
	float maxCS = (m_gd.cubeSize[0]>m_gd.cubeSize[1]) ? m_gd.cubeSize[0] : m_gd.cubeSize[1];
	maxCS = (maxCS>m_gd.cubeSize[2]) ? maxCS : m_gd.cubeSize[2];
	float size = (float)m_base->maxSize()*maxCS;
	float zoom = *m_zoom;
	DEBUG7("size = "<<size<<", zoom = "<<zoom<<"\n");
	glViewport(0, 0, w, h);

	glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
	if (ratio > 1.0) {
		glFrustum(-zoom*ratio, zoom*ratio,
			  -zoom, zoom,
			  1.0, size*10.0);
	} else {
		glFrustum(-zoom, zoom,
			  -zoom/ratio, zoom/ratio,
			  1.0, size*10.0);
	}

        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
} /* }}} void vuMarchingCubes::glResize(void) */

//----------------------------------------------------------------------------
//------------------------- public: glRender() -------------------------------
//----------------------------------------------------------------------------
/**
 * The functions does the actual rendering. It first resets the color and
 * modelview matrix, than rotates the object and at least calls the objects
 * rendering code, i.e. m_data->glRender().
 */
void vuMarchingCubes::glRender() // {{{
{
	DEBUG0("void vuMarchingCubes::glRender()\n");
	wxStopWatch watch;
	watch.Start();

	DEBUGGL("================ Begin frame ================\n");
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();

	//! Light settings could be moved to a extra function which is called
	//! whenever a light is changed.
	glLightfv(GL_LIGHT0, GL_POSITION,  *m_light0Pos);
	glLightfv(GL_LIGHT0, GL_AMBIENT,   *m_lightColor);
	glLightfv(GL_LIGHT0, GL_DIFFUSE,   *m_lightDiffuse);
	glLightfv(GL_LIGHT0, GL_SPECULAR,  *m_lightSpecular);
	glLightfv(GL_LIGHT0, GL_SHININESS, *m_lightShiny);
	if (m_light1Enabled->IsChecked()) {
		glLightfv(GL_LIGHT1, GL_POSITION,  *m_light1Pos);
		glLightfv(GL_LIGHT1, GL_AMBIENT,   *m_lightColor);
		glLightfv(GL_LIGHT1, GL_DIFFUSE,   *m_lightDiffuse);
		glLightfv(GL_LIGHT1, GL_SPECULAR,  *m_lightSpecular);
		glLightfv(GL_LIGHT1, GL_SHININESS, *m_lightShiny);
	}
	glTranslatef(m_pos->Get(0), m_pos->Get(1), m_pos->Get(2));
	glRotatef(m_angle->Get(0), 1.0, 0.0, 0.0);
	glRotatef(m_angle->Get(1), 0.0, 1.0, 0.0);
	if (m_rotateX->GetSelection() == 0) {
		glRotatef(-90.0, 1.0, 0.0, 0.0);
	} else if (m_rotateX->GetSelection() == 2) {
		glRotatef(90.0, 1.0, 0.0, 0.0);
	}
	if (m_rotateY->GetSelection() == 0) {
		glRotatef(-90.0, 0.0, 1.0, 0.0);
	} else if (m_rotateY->GetSelection() == 2) {
		glRotatef(90.0, 0.0, 1.0, 0.0);
	}
	if (m_rotateZ->GetSelection() == 0) {
		glRotatef(-90.0, 0.0, 0.0, 1.0);
	} else if (m_rotateZ->GetSelection() == 2) {
		glRotatef(90.0, 0.0, 0.0, 1.0);
	}

	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,   *m_objectDiffuse);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR,  *m_objectSpecular);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, *m_objectShiny);

	if (m_drawFrame->IsChecked()) {
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, *m_frameColor);
		m_base->glRenderFrame();
	}
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, *m_objectColor);
	m_base->glRender();

#if DEBUG_LEVEL <= 7
	GLfloat matWhite[4] = { 1.0, 1.0, 1.0, 1.0 };
	glBegin(GL_LINES);
		DEBUG7("Drawin line from: "<<x1<<", "<<y1<<", "<<z1<<"\n");
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, *m_backColor);
		glVertex3d(x1 * *m_cubeSizeX, y1 * *m_cubeSizeY, z1 * *m_cubeSizeZ);
		DEBUG7("to: "<<x2<<", "<<y2<<", "<<z2<<"\n");
		glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, matWhite);
		glVertex3d(x2 * *m_cubeSizeX, y2 * *m_cubeSizeY, z2 * *m_cubeSizeZ);
	glEnd();
#endif

	DEBUGGL("================ End frame ================\n");

	//watch.Pause();
	int time = watch.Time();
	DEBUG5("time used: "<<m_fpsTime<<"\n");
	if (m_gd.drawObject) {
		m_fpsTime += time;
		m_fpsCount++;
		DEBUG2("m_fpsCount = "<<m_fpsCount<<"\n");
		if (m_fpsCount >= (*m_fpsWatermark)) {
			DEBUG2("calcing fps ...\n");
			if (m_fpsTime != 0) m_avgFps = (float)m_fpsCount*1000.0f/(float)m_fpsTime;
			m_fpsCount = 0;
			m_fpsTime = 0;
		}
	}
	m_fps = (time>0) ? 1000.0/(float)time : 1000;
	SetStatusText(wxString("")<<"avg: "<<format(m_avgFps)<<" ("
		      <<(int)(*m_fpsWatermark)-m_fpsCount<<") / fps "<<format(m_fps)<<" fps");
} // }}}

//----------------------------------------------------------------------------
//------------------------- protected: OnChar() ------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnChar(wxKeyEvent& event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnChar(wxKeyEvent& event)\n");
	DEBUG7("char code = " << (int)event.GetKeyCode() << "\n");
	switch(event.GetKeyCode()) {
		case 13:
			m_cubeHist->Flush();
			break;
		case 27:
			exit(0);
			break;
	}
} /* }}} void vuMarchingCubes::OnChar(wxKeyEvent& event) */

//----------------------------------------------------------------------------
//------------------------- protected: onMouse() -----------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::glOnMouse(wxMouseEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::onMouse(wxMouseEvent &event)\n");
	if (event.LeftDClick())
		onDblClick(event);
	else if (event.LeftDown() || event.RightDown())
		onMouseDown(event);
	//else if (event.LeftIsDown() && event.Moving())
	else if (event.LeftIsDown())
		onMouseLeftMoving(event);
	//else if (event.RightIsDown() && event.Moving())
	else if (event.RightIsDown())
		onMouseRightMoving(event);
	else if (event.LeftUp() || event.RightUp())
		onMouseUp(event);
} // }}}

//----------------------------------------------------------------------------
//------------------------- protected: onMouseLeftMoving() -------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::onMouseLeftMoving(wxMouseEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::onMouseLeftMoving(wxMouseEvent &event)\n");
	float difX = ((float)event.GetX()-(float)m_MouseX);
	float difY = ((float)event.GetY()-(float)m_MouseY);
	storeMousePosition(event);
	DEBUG1("difX = "<<difX<<", difY = "<<difY<<"\n");
#ifdef _WIN32
	if (fabs(difX) > 10.0 || fabs(difY) > 10.0) return;
#endif
	difX *= (*m_mouseSpeed)/100.0f;
	difY *= (*m_mouseSpeed)/100.0f;
	float size = m_base->maxSize();
	if (event.ShiftDown()) {
		(*m_pos)[0] += (difX*size/100.0f);
		(*m_pos)[1] -= (difY*size/100.0f);
		m_glCanvas->redraw();
	} else if (event.ControlDown()) {
		(*m_pos)[2] += (difY*size/100.0f);
		m_glCanvas->redraw();
	} else if (event.AltDown()) {
		(*m_zoom) -= difY/100.0;
		glResize();
		m_glCanvas->redraw();
	} else {
		(*m_angle)[0] += difY;
		if (m_angle->Get(0) >  180.0f) (*m_angle)[0]  =  180.0f;
		if (m_angle->Get(0) < -180.0f) (*m_angle)[0]  = -180.0f;
		(*m_angle)[1] += difX;
		if (m_angle->Get(1) >  360.0f) (*m_angle)[1] -=  360.0f;
		if (m_angle->Get(1) <    0.0f) (*m_angle)[1] +=  360.0f;
		//m_cubeHist->Inc((int)(m_angle->Get(1)*10.0f));
		//m_cubeHist->Inc((int)(m_angle->Get(1)));
		m_glCanvas->redraw();
	}
} /* }}} void vuMarchingCubes::onMouseLeftMoving(wxMouseEvent &event) */

//----------------------------------------------------------------------------
//------------------------- protected: onMouseRightMoving() ------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::onMouseRightMoving(wxMouseEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::onMouseRightMoving(wxMouseEvent &event)\n");
	float difX = ((float)event.GetX()-(float)m_MouseX)*(*m_mouseSpeed)/100.0f;
	float difY = ((float)event.GetY()-(float)m_MouseY)*(*m_mouseSpeed)/100.0f;
	storeMousePosition(event);
	float size = m_base->maxSize();
	if (event.ShiftDown()) {
	} else if (event.ControlDown()) {
		(*m_light0Pos)[0] += difX*size/100.0f;
		(*m_light0Pos)[2] += difY*size/100.0f;
		m_glCanvas->redraw();
	} else if (event.AltDown()) {
		if (m_light1Enabled->IsChecked()) {
			(*m_light1Pos)[0] += difX*size/100.0f;
			(*m_light1Pos)[2] += difY*size/100.0f;
			m_glCanvas->redraw();
		}
	}
} /* }}} void vuMarchingCubes::onMouseRightMoving(wxMouseEvent &event) */

//----------------------------------------------------------------------------
//------------------------- protected: onMouseDown() -------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::onMouseDown(wxMouseEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::onMouseDown(wxMouseEvent &event)\n");
	storeMousePosition(event);
	if (m_fps < m_gd.fpsThres && m_base) m_gd.drawObject = false;
} // }}} void vuMarchingCubes::onMouseDown(wxMouseEvent &event)

//----------------------------------------------------------------------------
//------------------------- protected: onMouseUp() ---------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::onMouseUp(wxMouseEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::onMouseUp(wxMouseEvent &event)\n");
	if (m_base) m_gd.drawObject = true;
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::onMouseUp(wxMouseEvent &event)

//----------------------------------------------------------------------------
//------------------------- protected: onDblClick() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::onDblClick(wxMouseEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::onDblClick(wxMouseEvent &event)\n");
	DEBUG7("Double clicked the window\n");
	GLdouble modelview[16];
	GLdouble projection[16];
	GLint viewport[16];
	glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	glGetIntegerv(GL_VIEWPORT, viewport);
	gluProject(0.0, 0.0, 0.0, modelview, projection, viewport, &x1, &y1, &z1);
	DEBUG7("projected (0.0, 0.0, 0.0) to ("<<x1<<", "<<y1<<", "<<z1<<")\n");
	GLdouble winX = event.m_x;
	GLdouble winY = m_glCanvas->getHeight()-event.m_y;
	GLdouble winZ = z1;
	gluUnProject(winX, winY, 0.0, modelview, projection, viewport, &x1, &y1, &z1);
	DEBUG7("unprojected ("<<winX<<", "<<winY<<", 0.0) to ("<<x1<<", "<<y1<<", "<<z1
	       <<"), i.e. eye point.\n");
	gluUnProject(winX, winY, winZ, modelview, projection, viewport, &x2, &y2, &z2);
	DEBUG7("unprojected ("<<winX<<", "<<winY<<", "<<winZ<<") to ("<<x2<<", "<<y2<<", "<<z2
	       <<"), i.e. unprojected area through the center.\n");

	vuVector start(x2 / *m_cubeSizeX, y2 / *m_cubeSizeY, z2 / *m_cubeSizeZ);
	vuVector eye(x1 / *m_cubeSizeX, y1 / *m_cubeSizeY, z1 / *m_cubeSizeZ);
	vuVector viewdir = (eye-start).makeUnit();
	double size2[3] = {
		(double)m_base->Size1() / 2.0,
		(double)m_base->Size2() / 2.0,
		(double)m_base->Size3() / 2.0
	};

	double max;
	int dir;
	if (fabs(viewdir[0])>fabs(viewdir[1])) {
		max = fabs(viewdir[0]);
		dir = 0;
	} else {
		max = fabs(viewdir[1]);
		dir = 1;
	}
	if (fabs(viewdir[2])>max) {
		max = fabs(viewdir[2]);
		dir = 2;
	}
	DEBUG7("Calculated direction: "<<dir<<"\n");
	int cnt = 0;
	vuVector p = start;
	while (dir >= 0 && cnt < 3) {
		DEBUG7("We are outside the borders in direction "<<dir
		       <<", vector: "<<p[0]<<", "<<p[1]<<", "<<p[2]<<"\n");
		double theSize = (viewdir[dir]>0) ? size2[dir] : -size2[dir];
		double length = (theSize-p[dir])/viewdir[dir];
		DEBUG7("distance to the border in main direction ("<<dir<<"): "<<length<<"\n");
		p = p + viewdir*length;
		x1 = p[0]; y1 = p[1]; z1 = p[2];
		dir = outside(p, size2);
		cnt++;
	}
	int mySize[3] = { m_base->Size1(), m_base->Size2(), m_base->Size3() };
	double size[3] = { (double)mySize[0]-1.0, (double)mySize[1]-1.0, (double)mySize[2]-1.0 };
	int count = m_base->maxSize();
	vuVector step = viewdir*(-1.0/max);
	//! Move the volume out of the center,
	p += vuVector(size2[0], size2[1], size2[2]);
	if (outside(p, size)!=-1) p+= step;
	DEBUG7("start point: ("<<p[0]<<", "<<p[1]<<", "<<p[2]<<")\n");
	//! and step inside a cube.
	p += (step*0.001);
	float value = m_base->valueAt(p[0], p[1], p[2]);
	bool below = value < *m_thres;
	DEBUG7("stepping vector: ("<<step[0]<<", "<<step[1]<<", "<<step[2]<<")\n");
	DEBUG7("start point: ("<<p[0]<<", "<<p[1]<<", "<<p[2]<<")\n");
	DEBUG7("value = "<<value<<"\n");
	while (fabs(*m_thres-value)>0.001 && count--) {
		if (outside(p, size)!=-1) {
			// As soon as we are outside the first time, the challange is over!
			DEBUG7("We are outside, so we stop the algorithm!\n");
			return;
		}
		DEBUG7("bevor estimation: below = "<<below<<", step = "
		       <<step[0]<<", "<<step[1]<<", "<<step[2]<<"\n");
		if (below && (value>*m_thres)) {
			step *= -0.5;
			below = false;
		} else if (!below && (value<*m_thres)) {
			step *= -0.5;
			below = true;
		}
		DEBUG7("after estimation: below = "<<below<<", step = "
		       <<step[0]<<", "<<step[1]<<", "<<step[2]<<"\n");
		p += step;
		value = m_base->valueAt(p[0], p[1], p[2]);
		DEBUG7("value = "<<value<<", p = "<<p[0]<<", "<<p[1]<<", "<<p[2]
		       <<", dist = "<<fabs(*m_thres-value)<<"\n");
		DEBUG7("---------------------------------------\n");
	}
	DEBUG7("New position is: "<<p[0]<<", "<<p[1]<<", "<<p[2]<<"\n");
	x2 = (p[0]-size2[0]); y2 = (p[1]-size2[1]); z2 = (p[2]-size2[2]);
	if (outside(p, size) == -1) {
		int x = int(p[0]);
		int y = int(p[1]);
		int z = int(p[2]);
		int type = m_base->getCubeType(x, y, z);
		if (type != 0 && type != 255) {
			m_cubeHist->Select(type);
			OnSelectType(type);
			m_glCanvas->redraw();
			return;
		}
	}
} // }}} void vuMarchingCubes::onDblClick(wxMouseEvent &event)

//----------------------------------------------------------------------------
//------------------------- protected: OnLight1Checked() ---------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnLight1Checked(wxCommandEvent& event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnLight1Checked(wxCommandEvent& event)\n");
	wxCheckBox* cb = (wxCheckBox*)event.GetEventObject();
	m_light1Pos->Enable(cb->IsChecked());
	if (cb->IsChecked()) {
		glEnable(GL_LIGHT1);
	} else {
		glDisable(GL_LIGHT1);
	}
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnLight1Checked(wxCommandEvent& event)

//----------------------------------------------------------------------------
//------------------------- protected: OnRotateChecked() ---------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnRotateChecked(wxCommandEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnRotateChecked(wxCommandEvent &event)\n");
	DEBUG4("rotating ...\n");
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnRotateChecked(wxCommandEvent &event)

//----------------------------------------------------------------------------
//------------------------- protected: OnCubeEnabledChecked() ----------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnCubeEnabledChecked(wxCommandEvent& event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnCubeEnabledChecked(wxCommandEvent& event)\n");
	wxCheckBox* cb = (wxCheckBox*)event.GetEventObject();
	m_cube->Enable(cb->IsChecked());
	m_gd.cubeBrowser = cb->IsChecked();
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnCubeEnabledChecked(wxCommandEvent& event)

//----------------------------------------------------------------------------
//------------------------- protected: OnFrameChecked() ---------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnFrameChecked(wxCommandEvent& event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnFrameChecked(wxCommandEvent& event)\n");
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnFrameChecked(wxCommandEvent& event)

//----------------------------------------------------------------------------
//------------------------- protected: OnCullLinesBackFrontFacesChecked() ------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnCullLinesFrontBackChecked(wxCommandEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnCullLinesFrontBackChecked(wxCommandEvent &event)\n");
	assert(m_base != NULL);
	DEBUG4("setting lines and culling ...\n");
	m_gd.cullMode = m_gd.linesMode = 0;
	if (m_cullBack->IsChecked()) m_gd.cullMode = GL_BACK;
	if (m_cullFront->IsChecked()) m_gd.cullMode = GL_FRONT;
	if (m_cullBack->IsChecked() && m_cullFront->IsChecked()) m_gd.cullMode = GL_FRONT_AND_BACK;
	if (m_linesBack->IsChecked()) m_gd.linesMode = GL_BACK;
	if (m_linesFront->IsChecked()) m_gd.linesMode = GL_FRONT;
	if (m_linesBack->IsChecked() && m_linesFront->IsChecked()) m_gd.linesMode = GL_FRONT_AND_BACK;
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnCullLinesFrontBackChecked(wxCommandEvent &event)

//----------------------------------------------------------------------------
//------------------------- protected: OnUpdateModeChecked() -----------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnUpdateModeChecked(wxCommandEvent& event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnUpdateModeChecked(wxCommandEvent& event)\n");
	if (m_thres) m_thres->UpdateMode(m_updateMode->GetValue());
#ifdef LINUX
	m_update->Enable(!m_thres->UpdateMode());
#endif
} // }}} void vuMarchingCubes::OnUpdateModeChecked(wxCommandEvent& event)

//----------------------------------------------------------------------------
//------------------------- public: redraw() ---------------------------------
//----------------------------------------------------------------------------
#ifdef LINUX
void vuMarchingCubes::OnUpdateClicked(wxCommandEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnUpdateClicked(wxCommandEvent &event)\n");
	OnThresChange();
} // }}} void vuMarchingCubes::OnUpdateClicked(wxCommandEvent &event)
#endif

//----------------------------------------------------------------------------
//------------------------- public: OnDrawNormalsChecked() -------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnDrawNormalsChecked(wxCommandEvent &event) // {{{
{
	DEBUG0("void vuMarchingCubes::OnDrawNormalsChecked(wxCommandEvent &event)\n");
	m_gd.drawNormals = m_drawNormals->GetValue();
	m_drawNormalsSize->Enable(m_drawNormals->GetValue());
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnDrawNormalsChecked(wxCommandEvent &event)

//----------------------------------------------------------------------------
//------------------------- public: redraw() ---------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::redraw() // {{{
{
	DEBUG0("void vuMarchingCubes::redraw()\n");
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::redraw()

//----------------------------------------------------------------------------
//------------------------- public: resize() ---------------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::resize() // {{{
{
	DEBUG0("void vuMarchingCubes::resize()\n");
	glResize();
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::resize()

//----------------------------------------------------------------------------
//------------------------- public: OnCubeChange() ---------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnCubeChange(int val) // {{{
{
	DEBUG0("void vuMarchingCubes::OnCubeChange(int val)\n");
	m_gd.objectColor[m_gd.cube] = m_objectColor->tovuVector();
	m_gd.isObjectColor[m_gd.cube] = true;
	m_gd.cube = *m_cube;
	m_gd.objectColor[m_gd.cube] = m_selectColor->tovuVector();
	m_gd.isObjectColor[m_gd.cube] = false;
	m_cubeHist->Select(m_gd.cube);
	m_cubeHist->Flush();
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnCubeChange(int val)

//----------------------------------------------------------------------------
//------------------------- public: setBackground() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::setBackground() // {{{
{
	DEBUG0("void vuMarchingCubes::setBackground()\n");
	glClearColor(m_backColor->Red(), m_backColor->Green(), m_backColor->Blue(), m_backColor->Alpha());
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::setBackground()

//----------------------------------------------------------------------------
//------------------------- public: OnProgress() -----------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnProgress(float v) // {{{
{
	DEBUG0("void vuMarchingCubes::OnProgress(float v="<<v<<")\n");
	if (0.0 <= v && v <= 100.0) {
		m_statusText->Enable();
		m_status->Enable();
		m_status->SetValue((int)v);
	} else {
		m_statusText->Disable();
		m_status->Disable();
	}
} // }}} void vuMarchingCubes::OnProgress(float v)

//----------------------------------------------------------------------------
//------------------------- public: OnChangeFrame() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnChangeFrame() // {{{
{
	DEBUG0("void vuMarchingCubes::OnChangeFrame()\n");
	m_gd.frameRadius = (*m_frameRadius);
	DEBUG1("r = "<<m_gd.frameRadius<<"\n");
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnChangeFrame()

//----------------------------------------------------------------------------
//------------------------- public: OnThresChange() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnThresChange() // {{{
{
	DEBUG0("void vuMarchingCubes::OnThresChange()\n");
	m_gd.thres = *m_thres;
	doResetThres();
	m_base->marchCubes();
	doThresChange();
	m_cubeHist->Flush();
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnThresChange()

//----------------------------------------------------------------------------
//------------------------- public: OnThresChange() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnNormalLengthChange(float l) // {{{
{
	DEBUG0("void vuMarchingCubes::OnNormalLengthChange(float l)\n");
	m_gd.normalsLength = l;
	m_gd.compileNormals = true;
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnNormalLengthChange(float l)

//----------------------------------------------------------------------------
//------------------------- public: OnThresChange() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnSetObjectColor() // {{{
{
	DEBUG0("void vuMarchingCubes::OnSetObjectColor()\n");
	for(int i = 0; i < 256; ++i) {
		if (m_gd.isObjectColor[i]) {
			m_gd.objectColor[i] = m_objectColor->tovuVector();
		} else {
			m_gd.objectColor[i] = m_selectColor->tovuVector();
			m_cubeHist->SetSelColor(m_gd.objectColor[i]);
		}
	}
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnSetObjectColor()

//----------------------------------------------------------------------------
//------------------------- public: OnThresChange() --------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnSetLineColor() // {{{
{
	DEBUG0("void vuMarchingCubes::OnSetLineColor()\n");
	m_gd.lineColor = m_lineColor->tovuVector();
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnSetLineColor()

//----------------------------------------------------------------------------
//------------------------- public: OnCubeSizeChange() -----------------------
//----------------------------------------------------------------------------
//! \todo Think on code regarding a rescan (extra button?).
void vuMarchingCubes::OnCubeSizeChange() // {{{
{
	DEBUG0("void vuMarchingCubes::OnCubeSizeChange()\n");
	m_gd.cubeSize[0] = *m_cubeSizeX;
	m_gd.cubeSize[1] = *m_cubeSizeY;
	m_gd.cubeSize[2] = *m_cubeSizeZ;
	m_base->Origin((float)m_base->Size1()*m_gd.cubeSize[0]/-2.0,
		       (float)m_base->Size2()*m_gd.cubeSize[1]/-2.0,
		       (float)m_base->Size3()*m_gd.cubeSize[2]/-2.0);
	glResize();
	m_gd.compileData = true;
	m_gd.compileNormals = true;
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnCubeSizeChange()

//----------------------------------------------------------------------------
//------------------------- public: OnSelectType() ---------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::OnSelectType(int n) // {{{
{
	DEBUG0("void vuMarchingCubes::OnSelectType(int n)\n");
	DEBUG5("Setting color for type = "<<n<<"\n");
	assert(0 < n && n < 255);
	int type = *m_cube;
	if (0 < type && type < 255) {
		m_gd.objectColor[type] = m_objectColor->tovuVector();
		m_gd.isObjectColor[type] = true;
	}
	*m_cube = n;
	m_gd.cube = n;
	m_gd.objectColor[n] = m_selectColor->tovuVector();
	m_gd.isObjectColor[n] = false;
	m_glCanvas->redraw();
} // }}} void vuMarchingCubes::OnSelectType(int n)

//----------------------------------------------------------------------------
//------------------------- protected: rotate() ------------------------------
//----------------------------------------------------------------------------
/**
 * Function relies on the fact, that there is one matrix on the modelview stack.
 */
void vuMarchingCubes::rotate(float ax, float ay) // {{{
{
	glPopMatrix();
	glRotatef(ax, 1.0, 0.0, 0.0);
	glRotatef(ay, 0.0, 1.0, 0.0);
	glPushMatrix();
	glTranslatef(0.0, 0.0, 0.0);
} /* }}} void vuMarchingCubes::rotate(float ax, float ay) */

//----------------------------------------------------------------------------
//------------------------- protected: doThresChage() ------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::doThresChange() // {{{
{
	DEBUG0("void vuMarchingCubes::doThresChange()\n");
	if (m_thres) {
		DEBUG3("thres min = "<<m_gd.valueMin<<", thres max = "<<m_gd.valueMax<<"\n");
		m_thres->Min(m_gd.valueMin);
		m_thres->Max(m_gd.valueMax);
	}
	if (m_numTriangles) {
		DEBUG3("number of triangles: "<<m_base->numTriangles()<<"\n");
		*m_numTriangles = m_base->numTriangles();
	}
	if (m_numVertices) {
		DEBUG3("number of vertices: "<<m_base->numVertices()<<"\n");
		*m_numVertices = m_base->numVertices();
	}
	if (m_numIndices) {
		DEBUG3("number of indices: "<<m_base->numIndices()<<"\n");
		*m_numIndices = m_base->numIndices();
	}
	if (m_meshSize) {
		*m_meshSize = m_gd.meshSize;
	}
} // }}} void vuMarchingCubes::doThresChange()

//----------------------------------------------------------------------------
//------------------------- protected: doThresChage() ------------------------
//----------------------------------------------------------------------------
void vuMarchingCubes::doResetThres() // {{{
{
	DEBUG0("void vuMarchingCubes::doResetThres()\n");
	if (m_numTriangles) {
		*m_numTriangles = 0.0;
	}
	if (m_numVertices) {
		*m_numVertices = 0.0;
	}
	if (m_numIndices) {
		*m_numIndices = 0.0;
	}
} // }}} void vuMarchingCubes::doResetThres()

//----------------------------------------------------------------------------
//------------------------- protected: doThresChage() ------------------------
//----------------------------------------------------------------------------
inline int vuMarchingCubes::outside(vuVector &point, double *size) // {{{
{
	DEBUG0("inline int vuMarchingCubes::outside(vuVector &point, double *size)\n");
	int dir = -1;
	for(int i = 0; i <3; ++i) {
		if (fabs(point[i]) > size[i]) {
			dir = i;
			break;
		}
	}
	return dir;
} // }}} inline int vuMarchingCubes::outside(vuVector &point, double *size)

// }}}
// <<<

// {{{ class vuMCMCBProgressCallback:

vuMCMCBProgressCallback::vuMCMCBProgressCallback(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCMCBProgressCallback::vuMCMCBProgressCallback(vuMarchingCubes* mc)\n");
} // }}} vuMCMCBProgressCallback::vuMCMCBProgressCallback(vuMarchingCubes* mc)

void vuMCMCBProgressCallback::SetText(const char* txt) // {{{
{
	DEBUG0("void vuMCMCBProgressCallback::SetText(const char* txt)\n");
	assert(_mc != NULL);
	_mc->m_statusText->SetLabel(txt);
} // }}} void vuMCMCBProgressCallback::SetText(const char* txt)

void vuMCMCBProgressCallback::Start() // {{{
{
	DEBUG0("void vuMCMCBProgressCallback::Start()\n");
	assert(_mc != NULL);
	_mc->m_statusText->Enable();
	_mc->m_status->Enable();
} // }}} void vuMCMCBProgressCallback::Start()

void vuMCMCBProgressCallback::Update(int val) // {{{
{
	DEBUG0("void vuMCMCBProgressCallback::Update(int val)\n");
	assert(_mc != NULL);
	_mc->m_status->SetValue(val);
} // }}} void vuMCMCBProgressCallback::Update(int val)

//! We leave the status text as it is so the user can see the last action that took place.
void vuMCMCBProgressCallback::End() // {{{
{
	DEBUG0("void vuMCMCBProgressCallback::End()\n");
	assert(_mc != NULL);
	_mc->m_status->SetValue(0);
	_mc->m_statusText->Disable();
	_mc->m_status->Disable();
} // }}} void vuMCMCBProgressCallback::End()

// }}} vuMCMCBProgressCallback


// {{{ class vuMCMCBHistoCallback:

vuMCMCBHistoCallback::vuMCMCBHistoCallback(vuMarchingCubes *mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCMCBHistoCallback::vuMCMCBHistoCallback(vuMarchingCubes *mc)\n");
} // }}} vuMCMCBHistoCallback::vuMCMCBHistoCallback(vuMarchingCubes *mc)

void vuMCMCBHistoCallback::Clear() // {{{
{
	DEBUG0("void vuMCMCBHistoCallback::Clear()\n");
	*(_mc->m_cube0) = 0.0;
	*(_mc->m_cube255) = 0.0;
	*(_mc->m_cubeOther) = (float)*(_mc->m_cubesTotal);
	_mc->m_cubeHist->Clear();
} // }}} void vuMCMCBHistoCallback::Clear()

void vuMCMCBHistoCallback::Inc(int idx, int val) // {{{
{
	DEBUG0("void vuMCMCBHistoCallback::Inc(int idx="<<idx<<", int val="<<val<<")\n");
	assert(_mc != NULL);
	if (idx == 0 || idx == 255) {
		*(_mc->m_cubeOther) -= (float)val;
		*(_mc->m_meshSize) = _mc->m_gd.meshSize;
	}
	if (idx == 0)   {
		*(_mc->m_cube0) += (float)val;
		return;
	}
	if (idx == 255) {
		*(_mc->m_cube255) += (float)val;
		return;
	}
	_mc->m_cubeHist->Inc(idx, val);
} // }}} void vuMCMCBHistoCallback::Inc(int idx, int val = 1)

void vuMCMCBHistoCallback::Flush() // {{{
{
	DEBUG0("void vuMCMCBHistoCallback::Flush()\n");
	_mc->m_cubeHist->Flush();
} // }}} void vuMCMCBHistoCallback::Flush()

// }}} vuMCMCBHistoCallback


// {{{ class vuMCCubeDecorator:

vuMCCubeChange::vuMCCubeChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCCubeChange::vuMCCubeChange(vuMarchingCubes* mc="<<mc<<")\n");
} /* }}} vuMCCubeChange::vuMCCubeChange(vuMarchingCubes* mc) */

void vuMCCubeChange::operator()(int c) // {{{
{
	DEBUG0("void vuMCCubeChange::operator()(int c="<<c<<")\n");
	_mc->OnCubeChange(c);
} /* }}} void vuMCCubeChange::operator()(int c) */

// }}} vuMCCubeDecorator


// {{{ class vuMCRender:

vuMCRender::vuMCRender(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCRender::vuMCRender(vuMarchingCubes* mc)\n");
} // }}} vuMCRender::vuMCRender(vuMarchingCubes* mc)

void vuMCRender::operator()(float) // {{{
{
	DEBUG0("virtual void vuMCRender::operator()(float)\n");
	_mc->redraw();
} // }}} virtual void vuMCRender::operator()(float)

// }}} vuMCRender


// {{{ class vuMCResize:

vuMCResize::vuMCResize(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCResize::vuMCResize(vuMarchingCubes* mc)\n");
} // }}} vuMCResize::vuMCResize(vuMarchingCubes* mc)

void vuMCResize::operator()(int) // {{{
{
	DEBUG0("virtual void vuMCResize::operator()(int)\n");
	_mc->resize();
} // }}} virtual void vuMCResize::operator()(int)

// }}} vuMCResize


// {{{ class vuMCBackgroundChange:

vuMCBackgroundChange::vuMCBackgroundChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCBackgroundChange::vuMCBackgroundChange(vuMarchingCubes* mc)\n");
} // }}} vuMCBackgroundChange::vuMCBackgroundChange(vuMarchingCubes* mc)

void vuMCBackgroundChange::operator()(float) // {{{
{
	DEBUG0("virtual void vuMCBackgroundChange::operator()(float)\n");
	_mc->setBackground();
	_mc->redraw();
} // }}} virtual void vuMCBackgroundChange::operator()(float)

// }}} vuMCBackgroundChange


// {{{ class vuMCProgress:

vuMCProgress::vuMCProgress(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCProgress::vuMCProgress(vuMarchingCubes* mc)\n");
} // }}} vuMCProgress::vuMCProgress(vuMarchingCubes* mc)

void vuMCProgress::operator()(float v) // {{{
{
	DEBUG0("void vuMCProgress::operator()(float)\n");
	_mc->OnProgress(v);
} // }}} void vuMCProgress::operator()(float)

// }}} vuMCProgress


// {{{ class vuMCFrameChange:

vuMCFrameChange::vuMCFrameChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCFrameChange::vuMCFrameChange(vuMarchingCubes* mc)\n");
} // }}} vuMCFrameChange::vuMCFrameChange(vuMarchingCubes* mc)

void vuMCFrameChange::operator()(float) // {{{
{
	DEBUG0("void vuMCFrameChange::operator()(float)\n");
	_mc->OnChangeFrame();
} // }}} void vuMCFrameChange::operator()(float)

// }}} vuMCFrameChange


// {{{ class vuMCThresChange:

vuMCThresChange::vuMCThresChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCThresChange::vuMCThresChange(vuMarchingCubes* mc)\n");
} // }}} vuMCThresChange::vuMCThresChange(vuMarchingCubes* mc)

void vuMCThresChange::operator()(float) // {{{
{
	DEBUG0("void vuMCThresChange::operator()(float)\n");
	_mc->OnThresChange();
} // }}} void vuMCThresChange::operator()(float)

// }}} vuMCThresChange


// {{{ class vuMCNormalLengthChange:

vuMCNormalLengthChange::vuMCNormalLengthChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCNormalLengthChange::vuMCNormalLengthChange(vuMarchingCubes* mc)\n");
} // }}} vuMCNormalLengthChange::vuMCNormalLengthChange(vuMarchingCubes* mc)

void vuMCNormalLengthChange::operator()(float l) // {{{
{
	DEBUG0("void vuMCNormalLengthChange::operator()(float l)\n");
	_mc->OnNormalLengthChange(l);
	_mc->redraw();
} // }}} void vuMCNormalLengthChange::operator()(float l)

// }}} vuMCNormalLengthChange


// {{{ class vuMCObjectColorChange:

vuMCObjectColorChange::vuMCObjectColorChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCObjectColorChange::vuMCObjectColorChange(vuMarchingCubes* mc)\n");
} // }}} vuMCObjectColorChange::vuMCObjectColorChange(vuMarchingCubes* mc)

void vuMCObjectColorChange::operator()(float l) // {{{
{
	DEBUG0("void vuMCObjectColorChange::operator()(float l)\n");
	_mc->OnSetObjectColor();
} // }}} void vuMCObjectColorChange::operator()(float l)

// }}} vuMCObjectColorChange


// {{{ class vuMCLineColorChange:

vuMCLineColorChange::vuMCLineColorChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCLineColorChange::vuMCLineColorChange(vuMarchingCubes* mc)\n");
} // }}} vuMCLineColorChange::vuMCLineColorChange(vuMarchingCubes* mc)

void vuMCLineColorChange::operator()(float l) // {{{
{
	DEBUG0("void vuMCLineColorChange::operator()(float l)\n");
	_mc->OnSetLineColor();
} // }}} void vuMCLineColorChange::operator()(float l)

// }}} vuMCLineColorChange


// {{{ class vuMCCubeSizeChange:

vuMCCubeSizeChange::vuMCCubeSizeChange(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCCubeSizeChange::vuMCCubeSizeChange(vuMarchingCubes* mc)\n");
} // }}} vuMCCubeSizeChange::vuMCCubeSizeChange(vuMarchingCubes* mc)

void vuMCCubeSizeChange::operator()(float l) // {{{
{
	DEBUG0("void vuMCCubeSizeChange::operator()(float l)\n");
	_mc->OnCubeSizeChange();
} // }}} void vuMCCubeSizeChange::operator()(float l)

// }}} vuMCCubeSizeChange


// {{{ class vuMCSelectType:

vuMCSelectType::vuMCSelectType(vuMarchingCubes* mc) // {{{
: _mc(mc)
{
	DEBUGC("vuMCSelectType::vuMCSelectType(vuMarchingCubes* mc)\n");
} // }}} vuMCSelectType::vuMCSelectType(vuMarchingCubes* mc)

void vuMCSelectType::operator()(int n) // {{{
{
	DEBUG0("virtual void vuMCSelectType::operator()(int n)\n");
	_mc->OnSelectType(n);
} // }}} virtual void vuMCSelectType::operator()(int n)

// }}} vuMCSelectType


// vim:fdm=marker:fdc=3:tw=100
