#include "vuSimpleFBR.h"
#include "Volume/Lightfield/Unimodal/Spheric/vuSphericFilterFactory.h"
#include "General/vuLightfield/vuSphericViewFilterFactory.h"
#include "../../../wxUIElements/vuSimpleProgressHandler.h"

#include <wx/checklst.h>

#define MAX_SLICE_FILTER_SIZE 6

template <int S, class T>
vuSimpleFBR<S,T>::vuSimpleFBR()
{
  m_Data          = NULL;
  m_FourierFilter = NULL;
  m_SliceFilter   = NULL;
  m_Image         = NULL;
  m_Interactive   = NULL;

  m_CHOICEfilter               = NULL;
  m_TEXTfilterHint             = NULL;
  m_CHOICEfourierFilter        = NULL;
  m_CHOICEsliceFilter          = NULL;
  m_CHOICEsliceFilterKind      = NULL;
  m_CHOICEsliceFilterSize      = NULL;
  m_CHOICErenderMethod         = NULL;
  m_CHOICElowPass              = NULL;
  m_CHECKBOXinteractiveReconst = NULL;
  m_CHECKBOXchannels           = NULL;
  m_SLIDERimageScale           = NULL;
  m_BUTTONfitScale             = NULL;
  m_BUTTONpreprocess           = NULL;
  m_BUTTONreset                = NULL;
  m_BUTTONfill                 = NULL;
  m_BUTTONsave                 = NULL;
  m_SIZERpreprocess            = NULL;
  m_SIZERhint                  = NULL;

  m_FBRsubViewer               = NULL;
}

template <int S, class T>
vuSimpleFBR<S,T>::~vuSimpleFBR()
{
  CHECKNDELETE(m_Data);
  CHECKNDELETE(m_FourierFilter);
  CHECKNDELETE(m_SliceFilter);
  CHECKNDELETE(m_Image);
  CHECKNDELETE(m_FBRsubViewer);
  CHECKNDELETE(m_Interactive);

  m_CHOICEfilter               = NULL;
  m_TEXTfilterHint             = NULL;
  m_CHOICEfourierFilter        = NULL;
  m_CHOICEsliceFilter          = NULL;
  m_CHOICEsliceFilterKind      = NULL;
  m_CHOICEsliceFilterSize      = NULL;
  m_CHOICElowPass              = NULL;
  m_CHOICErenderMethod         = NULL;
  m_CHECKBOXinteractiveReconst = NULL;
  m_CHECKBOXchannels           = NULL;
  m_SLIDERimageScale           = NULL;
  m_BUTTONfitScale             = NULL;
  m_BUTTONpreprocess           = NULL;
  m_BUTTONreset                = NULL;
  m_BUTTONfill                 = NULL;
  m_BUTTONsave                 = NULL;
  m_SIZERpreprocess            = NULL;
  m_SIZERhint                  = NULL;

  m_FBRsubViewer               = NULL;
}

template <int S, class T>
const char *vuSimpleFBR<S,T>::_titleString()
{
  return "Simple Fourier Based Renderer";
}


template <int S, class T>
bool vuSimpleFBR<S,T>::init(const char* DataFile)
{
  SetEvtHandlerEnabled(true);

  //Set up the window
  SetTitle(_titleString());
  CreateStatusBar();

  // m_Data is instantiated in template instances
  m_Data->setFileName(DataFile);

  //Create Interactive Helper
  m_Interactive = new vuSphericInteractive<S,T>;

  //Create FBR subView
  m_FBRsubViewer = new vuFBRSubViewer<S,T>(this);
  m_FBRsubViewer->init(false);
  m_FBRsubViewer->setInteractive(m_Interactive);
  
  //Read in the data.
  bool success = m_Data->read();
  if (success) {
    m_glCanvas->SetSize(720,512);
    vuString str;
    str  = _titleString();
    str += ": [";
    str += m_Data->getWidth();
    str += "x";
    str += m_Data->getHeight();
    str += "] x ";
    str += m_Data->getNumberOfViews();
    str += " Views <";
    str += S;
    str += " ";
    str += m_Data->_typeName();
    str += ">";
    SetTitle(str.c_str());
    Fit();
    m_Data->setViewFilter(vuSphericViewFilterFactory<S,T>::getFilter("Fourier"));
    m_Interactive->setVolume(m_Data);
  }
  else {
    wxMessageDialog dlg(this,m_Data->getErrorMessage(),	"vuSimpleFBR",wxOK);
    dlg.ShowModal();
  }
  return success;
}


/* ------------------------------------------------------------------------ */
/* --- add some ui elements ----------------------------------------------- */
/* ------------------------------------------------------------------------ */


template <int S, class T>
void vuSimpleFBR<S,T>::addRight(wxSizer *sizer)
{
  wxBoxSizer *verSpacer = new wxBoxSizer(wxVERTICAL);
  wxBoxSizer *horSpacer = new wxBoxSizer(wxHORIZONTAL);

  // --- preprocessing interface -------------------------------------------

  wxStaticBox *myBox = new wxStaticBox(this, -1, "Preprocessing",
				       wxDefaultPosition,
				       wxSize(-1,-1),wxCAPTION);
  wxStaticBoxSizer *mySizer = new wxStaticBoxSizer(myBox, wxVERTICAL);

  // view filter
  {  
    m_CHOICEfilter = 
      new wxChoice(this, idFILTER, wxDefaultPosition, wxSize(96,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "Filter");

    m_CHOICEfilter->Append("Z-Axis");
    m_CHOICEfilter->Append("XZ-Axis");
    m_CHOICEfilter->Append("XYZ-Axis");
    m_CHOICEfilter->Append("Nearest");
    m_CHOICEfilter->Append("Fit Angle");
    m_CHOICEfilter->Append("All");
    m_CHOICEfilter->SetSelection(0);
  }

  // view filter hint
  {
    m_TEXTfilterHint = new wxTextCtrl(this,idFILTERHINT, "1",
				      wxDefaultPosition,
				      wxSize(30,18),
				      wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB|
				      wxALIGN_RIGHT,
				      wxTextValidator(wxFILTER_NUMERIC));
  }
    
  // reconstruction filter
  {
    m_CHOICEsliceFilter = 
      new wxChoice(this, idSLICEFILTER, wxDefaultPosition, wxSize(130,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "SliceFilter");

    dword    count        = 0;
    vuString *filterNames = NULL;
    
    vuSliceFilter::getFilterNames(filterNames, count);
    for (dword i=0; i< count; i++) {
      m_CHOICEsliceFilter->Append(filterNames[i].c_str());
    }
    m_CHOICEsliceFilter->SetSelection(0);
  }

  // reconstruction filter
  {
    m_CHOICEsliceFilterKind = 
      new wxChoice(this, idSLICEFILTERKIND, wxDefaultPosition, wxSize(130,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "SliceFilterKind");

    m_CHOICEsliceFilterKind->Append("separable");
    m_CHOICEsliceFilterKind->Append("spheric");
    m_CHOICEsliceFilterKind->SetSelection(0);
  }


  // filter size of reconstruction filter
  {
    m_CHOICEsliceFilterSize = 
      new wxChoice(this, idSLICEFILTERSIZE, wxDefaultPosition, wxSize(130,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "SliceFilterSize");

    for (dword i=1; i <= MAX_SLICE_FILTER_SIZE; i++) {
      m_CHOICEsliceFilterSize->Append(vuString(i).c_str());
    }
    m_CHOICEsliceFilterSize->SetSelection(0);
  }

  // lowPass factor

  {
    m_CHOICElowPass = 
      new wxChoice(this, idLOWPASS, wxDefaultPosition, wxSize(130,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "LowPassFactor");
    m_CHOICElowPass->Append("100 %");
    m_CHOICElowPass->Append(" 75 %");
    m_CHOICElowPass->Append(" 50 %");
    m_CHOICElowPass->Append(" 25 %");
    m_CHOICElowPass->Append(" 15 %");
    m_CHOICElowPass->Append(" 10 %");
    m_CHOICElowPass->Append("  5 %");
    m_CHOICElowPass->Append("  2 %");
    m_CHOICElowPass->Append("  1 %");
    m_CHOICElowPass->SetSelection(0);
  }
  
  // interactive reconstruction checkbox
  {
    m_CHECKBOXinteractiveReconst =
      new wxCheckBox(this, idINTERACT_RECONST, "", wxDefaultPosition,
		     wxSize(15,15));
  }
  
  wxFlexGridSizer *tmp;

  tmp = new wxFlexGridSizer(2,5,5);
  tmp->Add(80,5);
  tmp->Add(0,5);
  tmp->Add(new wxStaticText(this, -1, "Views:"), 0, wxALIGN_RIGHT);
  
  {
    m_SIZERhint = new wxBoxSizer(wxHORIZONTAL);
    
    tmp->Add(m_SIZERhint);

    _filterHintTextUpdate();
  }
  

  tmp->Add(new wxStaticText(this, -1, "Filter:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHOICEsliceFilter,0,wxALL|wxALIGN_LEFT,1);
  tmp->Add(new wxStaticText(this, -1, "Filter Type:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHOICEsliceFilterKind,0,wxALL|wxALIGN_LEFT,1);
  tmp->Add(new wxStaticText(this, -1, "Filter Size:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHOICEsliceFilterSize,0,wxALL|wxALIGN_LEFT,1);
  tmp->Add(new wxStaticText(this, -1, "Low Pass:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHOICElowPass,0,wxALL|wxALIGN_LEFT,1);

  tmp->Add(new wxStaticText(this, -1, "Interactive:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHECKBOXinteractiveReconst);

  tmp->Add(80,5);
  tmp->Add(0, 5);

  m_BUTTONpreprocess = new wxButton(this, idPREPROCESS,
				    "        Preprocess        ");
  m_BUTTONreset      = new wxButton(this, idRESET, "   Start  ");
  m_BUTTONfill       = new wxButton(this, idFILL,  "   Fill   ");

  m_BUTTONreset->Show(false);
  m_BUTTONfill->Show(false);

  mySizer->Add(tmp);
  {
    m_SIZERpreprocess = new wxBoxSizer(wxHORIZONTAL);

    m_SIZERpreprocess->Add(m_BUTTONpreprocess);
    m_SIZERpreprocess->Layout();
    m_SIZERpreprocess->Fit(this);

    mySizer->Add(m_SIZERpreprocess , 0, wxALIGN_CENTER);
  }
  mySizer->Add(0,5);

  verSpacer->Add(mySizer,0,wxEXPAND, 5);


  // --- rendering interface ------------------------------------------------

  myBox   = new wxStaticBox(this, -1, "Rendering", wxDefaultPosition,
			    wxSize(-1,-1),wxCAPTION);
  mySizer = new wxStaticBoxSizer(myBox, wxVERTICAL);

  tmp = new wxFlexGridSizer(2,5,5);

  // interpolation filter for slicing
  {
    dword    count        = 0;
    vuString *filterNames = NULL;
    
    m_CHOICEfourierFilter = 
      new wxChoice(this, idFOURIERFILTER, wxDefaultPosition, wxSize(130,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "SliceFilter");

    vuTorstensFourierFilter::getFilterNames(filterNames, count);
    for (dword i=0; i< count; i++) {
      if (filterNames[i].hasPrefix("d0"))
	m_CHOICEfourierFilter->Append(filterNames[i].c_str());
    }
    m_CHOICEfourierFilter->SetSelection(0);
  }

  // render method choice
  {
    m_CHOICErenderMethod =
      new wxChoice(this, idRENDERMETHOD, wxDefaultPosition, wxSize(130,20),
		   0, NULL, wxCAPTION, wxDefaultValidator, "RenderMethod");

    m_CHOICErenderMethod->Append("Spatial");
    m_CHOICErenderMethod->Append("Freq. Amplitude ");
    m_CHOICErenderMethod->Append("Freq. Phase");
    m_CHOICErenderMethod->Append("Freq. Real");
    m_CHOICErenderMethod->Append("Freq. Imaginary");

    m_CHOICErenderMethod->SetSelection(0);
  }

  // image scale factor
  {
    m_SLIDERimageScale =
      new wxSlider(this, idIMAGESCALE, 1, 1, 10000, wxDefaultPosition,
		   wxSize(106,16), wxSL_HORIZONTAL, 
		   wxDefaultValidator, "imageScaleSlider");
    m_SLIDERimageScale->SetValue(2550);
  }

  // image scale fit button
  {
    m_BUTTONfitScale = new wxButton(this,idFITSCALE,"fit",
				    wxPoint(0,0),wxSize(20,16));
  }
  wxBoxSizer *scaleSizer = new wxBoxSizer(wxHORIZONTAL);
  scaleSizer->Add(m_SLIDERimageScale);
  scaleSizer->Add(2,0);
  scaleSizer->Add(m_BUTTONfitScale);
  
  tmp->Add(80,5);
  tmp->Add(0, 5);
  tmp->Add(new wxStaticText(this, -1, "Filter:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHOICEfourierFilter,0,wxALL|wxALIGN_LEFT,1);

  tmp->Add(new wxStaticText(this, -1, "Method:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_CHOICErenderMethod,0,wxALL|wxALIGN_LEFT,1);

  _addChannelControls(tmp); // add r g b checkboxes (only for 3B)

  tmp->Add(new wxStaticText(this,-1,"Scale:"),0,wxALIGN_RIGHT|wxALIGN_BOTTOM);
  tmp->Add(scaleSizer,0,wxALL|wxALIGN_LEFT,1);

  tmp->Add(new wxStaticText(this, -1, "Snap To View:"), 0, wxALIGN_RIGHT);
  tmp->Add(new wxButton(this,idSNAP2VIEW,"snap", wxDefaultPosition,
			wxSize(130,18)), 0, wxALL|wxALIGN_LEFT,1);

  tmp->Add(new wxStaticText(this, -1, "Sub Window:"), 0, wxALIGN_RIGHT);
  tmp->Add(new wxButton(this,idSHOWSUBWINDOW,"show", wxDefaultPosition,
			wxSize(130,18)), 0, wxALL|wxALIGN_LEFT,1);
  
  tmp->Add(0,3);
  tmp->Add(0,3);

  mySizer->Add(tmp);

  verSpacer->Add(0,15);
  verSpacer->Add(mySizer);

  // --- misc  -------------------------------------------

  myBox   = new wxStaticBox(this, -1, "Misc",wxDefaultPosition,
			    wxSize(-1,-1),wxCAPTION);
  mySizer = new wxStaticBoxSizer(myBox, wxVERTICAL);

  tmp = new wxFlexGridSizer(2,5,5);

  m_BUTTONsave = new wxButton(this, idSAVE, "save",
			      wxDefaultPosition,wxSize(130,20));

  tmp->Add(80,1);
  tmp->Add(0, 1);

  tmp->Add(new wxStaticText(this, -1, "Save:"), 0, wxALIGN_RIGHT);
  tmp->Add(m_BUTTONsave);

  tmp->Add(0,3);
  tmp->Add(0,3);

  mySizer->Add(tmp);

  verSpacer->Add(0,15);
  verSpacer->Add(mySizer);

  // ------------------------------------------------------------------------
  
  horSpacer->Add(10,0);
  horSpacer->Add(verSpacer);
  horSpacer->Add(10,0);

  sizer->Add(horSpacer,0,wxEXPAND, 5);
}


template <int S, class T>
bool vuSimpleFBR<S,T>::glInit()
{
  if (m_Data == NULL) return false;
  m_Data->initOpenGL();
  m_Data->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
  return true;
}

template <int S, class T>
void vuSimpleFBR<S,T>::glRender()
{
  _interactiveReconstruction(); // reconstruct interactively (if enabled)

  _renderImage();  // render image and store the result in m_Image

  _displayImage(); // display m_Image

  m_FBRsubViewer->redraw();
}


template <int S, class T>
void vuSimpleFBR<S,T>::glResize()
{
  m_Data->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
}

template <int S, class T>
void vuSimpleFBR<S,T>::onMouse(wxMouseEvent &ev)
{
  // here is the place for additional mouse bevaviour.
}

template <int S, class T>
void vuSimpleFBR<S,T>::onKeyboard(wxKeyEvent& event)
{
}

template <int S, class T>
vu1 *vuSimpleFBR<S,T>::getVolume()
{
  return (vu1 *)m_Data;
}

/* --------------------------------------------------------------------- */
/* --- some GUI callbacks ---------------------------------------------- */
/* --------------------------------------------------------------------- */

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonPreprocess(wxCommandEvent& event)
{
  bool isPreprocess = false;

  _updateFourierFilter();
  isPreprocess = _updateSliceFilter();
  isPreprocess = _updateViewFilter(isPreprocess);

  if (isPreprocess) {
    vuSimpleProgressHandler handler(this);
    m_Data->preprocess(&handler);
  }
  else
    cout << "No preprocessing neccessary!" << endl;

  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonReset(wxCommandEvent& event)
{
  if (m_BUTTONreset->GetLabel() == "Start")
    m_BUTTONreset->SetLabel("Reset");

  _resetInteractivePreprocessing();
  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonFill(wxCommandEvent& event)
{
  vuSphericViewFilter<S,T> *filter = m_Data->getViewFilter();
  int   idx;
  dword cnt = filter->getNumberOfViews();
  
  for (dword i=0; i<cnt; i++) {
    idx = m_Data->getIndexOfView(filter->getView(i));
    if (idx <0) continue;
    if (!m_Interactive->isVisited(idx)) {
      m_Interactive->setIsVisited(idx, true);
      _addViewWithIndex(idx); //, i);
    }
  }
  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnChoiceRenderMethod(wxCommandEvent& event)
{
  if (m_CHOICErenderMethod->GetSelection() == 0) { // spatial domain
    m_SLIDERimageScale->SetValue(255*10);
  }
  else { // frequency domain
    m_SLIDERimageScale->SetValue(1);
  }
  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnChoiceFilter(wxCommandEvent& event)
{
  _filterHintTextUpdate();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnChoiceFourierFilter(wxCommandEvent& event)
{
  vuTorstensFourierFilter *tmp = m_FourierFilter;
  vuString name = m_CHOICEfourierFilter->GetStringSelection().c_str();

  if (m_FourierFilter) {
    if (m_FourierFilter->getFilterName() == name) return;
  }

  m_FourierFilter = vuTorstensFourierFilter::getFilter(name);

  if (m_FourierFilter) {
    getFourierViewFilter()->setFourierFilter(m_FourierFilter);
    CHECKNDELETE(tmp);
  }
  else
    m_FourierFilter = tmp;

  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();  
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonSnap2View(wxCommandEvent& event)
{
  vuSphericViewFilter<S,T> *subFilter = m_Data->getViewFilter();
  dword                    numOfViews = subFilter->getNumberOfViews();

  if (numOfViews > 0) {
    vuSphericView<S,T>** views = subFilter->getViews();
    vuCamera          *camera  = m_Data->getCameraPtr();
    dword             *idxList = NULL;
    dword             count    = 1;
    vuVector          lookAt   = camera->getLookAtVector();
    vuVector          up       = camera->getUpVector();
    vuVector          right    = camera->getRightVector();

    lookAt.makeUnit();
    vuSphericFilter<S,T>::getNearestViews(idxList, count, lookAt * -1,
					  numOfViews, views);    

    if (count != 1) return;

    vuVector lookFrom = views[idxList[0]]->getLookFrom();

    // check opposite view direction as well...
    {
      CHECKNDELETE(idxList);
      vuVector looookAt = lookAt * -1;
      
      looookAt.makeUnit();
      vuSphericFilter<S,T>::getNearestViews(idxList, count, looookAt * -1,
					    numOfViews, views);

      if (count != 1) return;

      vuVector looookFrom = views[idxList[0]]->getLookFrom();
      looookFrom.makeUnit();

      if (fabs(looookFrom.dot(looookAt)) > fabs(lookFrom.dot(lookAt))) {
	lookFrom = looookFrom;
      }
    }
    bool isFromBack;
    
    if (lookAt.dot(lookFrom) < 0) {
      isFromBack = true;
      lookAt = lookFrom * -1;
    }
    else {
      isFromBack = false;
      lookAt = lookFrom;
    }
    
    vuFourierVolume<S>::calcViewVectors(lookAt, up, right);
    
    camera->setLookAtVector(lookAt);
    camera->setUpVector(up);
    camera->setRightVector(right);
    camera->init();

    CHECKNDELETE(idxList);
  }
  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonFitScale(wxCommandEvent& event)
{
  if (m_Image == NULL) return;

  float minValue =  100000000000.0f;
  float maxValue = -100000000000.0f;

  for (int i=0; i<S; i++) {
    float minVal, maxVal;

    m_Image->getMinAndMaxValue(minVal, maxVal, i);

    if (minValue > minVal) minValue = minVal;
    if (maxValue < maxVal) maxValue = maxVal;
  }

  m_SLIDERimageScale->SetValue((int)(maxValue*10));

  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonShowSubWindow(wxCommandEvent& event)
{
  _updateFourierFilter();
  m_FBRsubViewer->Show(true);
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnButtonSave(wxCommandEvent& event)
{
  wxFileDialog *saveDialog = NULL;

  saveDialog = new wxFileDialog(this, "Save Reconstructed Volume",
				"", // default dir
				"", // default file
				"Fourier Volume (*.fvr)| *.fvr|"
				"Spatial Volume (*.vud)| *.vud", //wildcard
				wxHIDE_READONLY|wxSAVE|wxOVERWRITE_PROMPT);
  if (saveDialog->ShowModal() == wxID_OK) { // OK pressed
    if (saveDialog->GetFilterIndex() == 0) { // save fourier volume (*.fvr)
      const char *fileName = saveDialog->GetPath();
      CHECKNDELETE(saveDialog);
      m_glCanvas->redraw();
      Refresh();

      vuSimpleProgressHandler handle(this, "Writing Frequency Volume To Disk");
      getFourierViewFilter()->writeFourierToFile(fileName, &handle);
    }
    else { // save spatial volume (*.vud)
      getFourierViewFilter()->writeSpatialVolume(saveDialog->GetPath());
      CHECKNDELETE(saveDialog);
    }
  }
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnCheckboxInteractiveReconst(wxCommandEvent& event)
{
  if (m_CHECKBOXinteractiveReconst->GetValue()) {
    m_SIZERpreprocess->Remove(0);
    m_SIZERpreprocess->Add(m_BUTTONfill);
    m_SIZERpreprocess->Add(m_BUTTONreset);
    m_BUTTONpreprocess->Show(false);
    m_BUTTONreset->Show(true);
    m_BUTTONfill->Show(true);
    m_BUTTONreset->SetLabel("Start");
    m_SIZERpreprocess->Layout();
    m_SIZERpreprocess->Fit(this);
  }
  else {
    getFourierViewFilter()->setNoInteractiveMode();
    m_SIZERpreprocess->Remove(1);
    m_SIZERpreprocess->Remove(0);
    m_SIZERpreprocess->Add(m_BUTTONpreprocess);
    m_BUTTONpreprocess->Show(true);
    m_BUTTONreset->Show(false);
    m_BUTTONfill->Show(false);
    m_SIZERpreprocess->Layout();
    m_SIZERpreprocess->Fit(this);
  }
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnCheckboxChannels(wxCommandEvent& event)
{
  m_glCanvas->redraw();
}

template <int S, class T>
void vuSimpleFBR<S,T>::OnSliderImageScale(wxScrollEvent& event)
{
  m_glCanvas->redraw();
}


template <int S, class T>
vuCamera* vuSimpleFBR<S,T>::getCamera()
{
  return m_Data->getCameraPtr ();
}

template <int S, class T>
vuImage* vuSimpleFBR<S,T>::getCurrentImage()
{
  return NULL;
}


template <int S, class T>
void vuSimpleFBR<S,T>::DrawFromImage()
{
  m_glCanvas->redraw ();
}

template <int S, class T>
void vuSimpleFBR<S,T>::DrawAgain()
{
}

/* --------------------------------------------------------------------- */
/* --- some private functions ------------------------------------------ */
/* --------------------------------------------------------------------- */

template <int S, class T>
vuSphVwFlt_Fourier<S,T> *vuSimpleFBR<S,T>::getFourierViewFilter()
{
  return (vuSphVwFlt_Fourier<S,T> *)(m_Data->getViewFilter());
}

template <int S, class T>
bool vuSimpleFBR<S,T>::_updateViewFilter(bool isPreprocess)
{
  vuString             name    = m_CHOICEfilter->GetStringSelection().c_str();
  vuSphericFilter<S,T> *filter = m_Data->getFilter();
  bool                 isNew   = false;

  if (filter->getFilterName() != name) {
    filter = vuSphericFilterFactory<S,T>::getFilter(name);
    isNew  = true;
  }

  if (name == "Nearest") {
    vuSphLfFlt_Nearest<S,T> *flt = (vuSphLfFlt_Nearest<S,T> *)filter;
    unsigned long num;

    m_TEXTfilterHint->GetValue().ToULong(&num);
    if (num < 1) num = 1;
    if (num > m_Data->getNumberOfViews()) num = m_Data->getNumberOfViews();

    if (num != flt->getNumberOfViews()) {
      flt->setNumberOfViews(num);
      isPreprocess = true;
    }
    m_TEXTfilterHint->SetValue(vuString(num).c_str());
  }
  else if (name == "Fit Angle") {
    vuSphLfFlt_FitAngle<S,T> *flt = (vuSphLfFlt_FitAngle<S,T> *)filter;
    double angle;

    m_TEXTfilterHint->GetValue().ToDouble(&angle);
    flt->setAngle(angle);
    m_TEXTfilterHint->SetValue(vuString((dword)flt->getAngle()).c_str());
    isPreprocess = true;
  }
  
  vuSphVwFlt_Fourier<S,T> *fbr = getFourierViewFilter();

  fbr->setFourierFilter(m_FourierFilter);

  if (isNew) {
    fbr->setSliceFilter(m_SliceFilter);
    m_Data->setFilter(filter);
    m_Data->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
  }
  else if (isPreprocess) {
    fbr->setSliceFilter(m_SliceFilter);
  }
  return (isNew || isPreprocess);
}

template <int S, class T>
void vuSimpleFBR<S,T>::_updateFourierFilter()
{
  vuTorstensFourierFilter *tmp = m_FourierFilter;
  vuString name = m_CHOICEfourierFilter->GetStringSelection().c_str();

  if (m_FourierFilter) {
    if (m_FourierFilter->getFilterName() == name) return;
  }

  m_FourierFilter = vuTorstensFourierFilter::getFilter(name);

  CHECKNDELETE(tmp);
}

template <int S, class T>
bool vuSimpleFBR<S,T>::_updateSliceFilter()
{
  vuSliceFilter *tmp = m_SliceFilter;
  vuString      name = m_CHOICEsliceFilter->GetStringSelection().c_str();
  bool     isUpdated = (m_SliceFilter == NULL);
  float   oldLowPass = (tmp == NULL) ? 1.0 : tmp->getLowPassFactor();

  if (m_SliceFilter) {
    isUpdated = (m_SliceFilter->getFilterName() != name);
  }

  if (isUpdated) {
    m_SliceFilter = vuSliceFilter::getFilter(name.c_str());
    CHECKNDELETE(tmp);
  }
  
  // update lowPass factor
  {
    double newLowPass;

    if (m_CHOICElowPass->GetSelection() == 0)
      newLowPass = 1.0;
    else {
      wxString str = m_CHOICElowPass->GetStringSelection().BeforeLast('%');
      str.ToDouble(&newLowPass);
      newLowPass /= 100;
    }

    if (fabs(oldLowPass - newLowPass) > 0.0001) {
      isUpdated = true;
      m_SliceFilter->setLowPassFactor(newLowPass);
    }
  }
  
  // update filter width
  {  
    dword size = m_CHOICEsliceFilterSize->GetSelection() + 1;

    if (m_SliceFilter->getWidth() != size) {
      m_SliceFilter->setWidth(size);
      isUpdated = true;
    }
  }
  
  // update filter type
  {
    byte kind = m_CHOICEsliceFilterKind->GetSelection();

    if (m_SliceFilter->getKind() != kind) {
      m_SliceFilter->setKind(kind);
      isUpdated = true;
    }
  }

  return isUpdated;
}

template <int S, class T>
void vuSimpleFBR<S,T>::_filterHintTextUpdate()
{
  vuString name = m_CHOICEfilter->GetStringSelection().c_str();

  m_SIZERhint->Remove(0);
  m_SIZERhint->Remove(0);

  if (name == "Fit Angle") { 
    m_CHOICEfilter->SetSize(96,20);
    m_SIZERhint->Add(m_CHOICEfilter,   0, wxALL|wxALIGN_LEFT,1);
    m_SIZERhint->Add(m_TEXTfilterHint, 0, wxALL|wxALIGN_LEFT,1);
    m_TEXTfilterHint->Show(true);
    m_TEXTfilterHint->SetValue("45");
    m_TEXTfilterHint->SetInsertionPointEnd();
  }
  else if (name == "Nearest") {
    m_CHOICEfilter->SetSize(96,20);
    m_SIZERhint->Add(m_CHOICEfilter,   0, wxALL|wxALIGN_LEFT,1);
    m_SIZERhint->Add(m_TEXTfilterHint, 0, wxALL|wxALIGN_LEFT,1);
    m_TEXTfilterHint->Show(true);
    m_TEXTfilterHint->SetValue("1");
    m_TEXTfilterHint->SetInsertionPointEnd();
  }
  else {
    m_CHOICEfilter->SetSize(130,20);
    m_SIZERhint->Add(m_CHOICEfilter, 0, wxALL|wxALIGN_LEFT,1);
    m_TEXTfilterHint->Show(false);    
  }
  m_SIZERhint->Layout();
  m_SIZERhint->Fit(this);
}

template <int S, class T>
void vuSimpleFBR<S,T>::_resetInteractivePreprocessing()
{
  bool isPreprocess = false;

  _updateFourierFilter();
  isPreprocess = _updateSliceFilter();
  isPreprocess = _updateViewFilter(isPreprocess);

  vuSphVwFlt_Fourier<S,T> *filter = getFourierViewFilter();
  cerr << "prepare for interactive reconstruction..." << endl;
  filter->prepareForInteractive(m_Data->getWidth(), m_Data->getHeight());
  m_Data->getFilter()->applyFilteredViews(m_Data->getViewFilter());
  m_Interactive->reset();
}

template <int S, class T>
void vuSimpleFBR<S,T>::_addViewWithIndex(dword idx) //, int idxInFilter)
{
  wxStopWatch watch;
  // vuFourierCluster<T> *cluster = getFourierCluster(); // ???

  watch.Start();
  // cluster->addView(m_Data->getView(idx));
  watch.Pause();

  cerr << "add view: ";
  cerr.width(3);
  cerr << idx;
  cerr << "   (";
  cerr.width(3);
  cerr << m_Interactive->getNumberOfVisited();
  cerr << " of ";
  cerr << m_Data->getViewFilter()->getNumberOfViews();
  cerr << ")   [";
  cerr << watch.Time();
  cerr << " ms]" << endl;
}

template <int S, class T>
void vuSimpleFBR<S,T>::_addChannelControls(wxFlexGridSizer *sizer)
{
  vuString str;

  if (S == 1)
    return;
  else if (S == 2)
    str = "IA";
  else if (S == 3)
    str = "RGB";
  else
    return;

  wxBoxSizer *channelSpacer = new wxBoxSizer(wxHORIZONTAL);

  m_CHECKBOXchannels = new wxCheckBox*[S];

  for (int i=0; i<S; i++) {
    m_CHECKBOXchannels[i] = new wxCheckBox(this, idCHANNELS, str[(dword)i],
                                          wxDefaultPosition, wxSize(130/S,15));
    m_CHECKBOXchannels[i]->SetValue(true);
    channelSpacer->Add(m_CHECKBOXchannels[i]);
  }

  sizer->Add(new wxStaticText(this, -1, "Channels:"), 0, wxALIGN_RIGHT);
  sizer->Add(channelSpacer,0,wxALL|wxALIGN_LEFT,1);
}


template <int S, class T>
void vuSimpleFBR<S,T>::_interactiveReconstruction()
{
  vuSphVwFlt_Fourier<S,T> *filter = getFourierViewFilter();

  // do interactive preprocessing if it's asked for
  if (filter->isPreparedForInteractive()) {
    int  idx   = m_Interactive->indexOfNearestView(); 
    bool isNew = !m_Interactive->isVisited(idx);

    if (isNew && idx >= 0) {
      m_Interactive->setIsVisited(idx,true);
      //filter->setFilter(m_FourierFilter);
      //filter->setSliceFilter(m_SliceFilter);
      //_addViewWithIndex(idx);
    }
  }
}

template <int S, class T>
void vuSimpleFBR<S,T>::_renderImage()
{
  if (m_Data->IsReRendering()) {
    int method = m_CHOICErenderMethod->GetSelection();

    if (method < 0) return;

    wxStopWatch watch;

    watch.Start();

    vuCamera *camera = m_Data->getCameraPtr();

    getFourierViewFilter()->computeUnscaledImage(camera, m_Image, method);
    m_Image->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
    m_Data->setIsReRendering(false);

    watch.Pause();

    // --- update status text ------------------
    wxString statusText = "Render Time: ";
    int      idx        = m_Interactive->indexOfNearestView();
    
    statusText += vuString(watch.Time()).c_str();
    statusText += "ms";
    if (idx >= 0) {  
      statusText += " [view=";
      statusText += vuString(idx).c_str();
      statusText += "]";
    }    
    SetStatusText(statusText);

    // redraw sub viewer
  }
}

template <int S, class T>
void vuSimpleFBR<S,T>::_displayImage()
{
  float scale = (float)m_SLIDERimageScale->GetValue() / 10;

  // chose the channels from m_Image and display them
  {
    vuFixelMap<S,float> *img = NULL;

    img = new vuFixelMap<S,float>(m_Image->getWidth(), m_Image->getHeight());

    bool status = true;

    if (m_CHECKBOXchannels != NULL) {
      for (int i=0;i<S;i++) status=status && m_CHECKBOXchannels[i]->GetValue();
    }

    if (status)
      *img = *m_Image;
    else {
      vuFixelMap<1,float> *imag = NULL;

      imag = new vuFixelMap<1,float>(m_Image->getWidth(),m_Image->getHeight());

      for (int i=0; i<S; i++) {
	if (m_CHECKBOXchannels[i]->GetValue()) {
	  m_Image->getChannel(imag,i);
	  img->copyMapToChannel(imag,i);
	}
      }
      CHECKNDELETE(imag);
    }

    *img /= scale;
    img->glRender();
    CHECKNDELETE(img);
  }
}
