#include "vuSphericRevolver.h"
#include "Volume/Lightfield/Unimodal/Spheric/vuSphericFilterFactory.h"
#include "General/vuLightfield/vuSphericViewFilterFactory.h"

template <int SIZE, class TYPE>
vuSphericRevolver<SIZE,TYPE>::vuSphericRevolver()
{
  m_Data = NULL;
}

template <int SIZE, class TYPE>
vuSphericRevolver<SIZE,TYPE>::~vuSphericRevolver()
{
  CHECKNDELETE(m_Data);
}

template <int SIZE, class TYPE>
const char *vuSphericRevolver<SIZE,TYPE>::_titleString()
{
  return "Spherical Lightfield Revolver";
}

template <int SIZE, class TYPE>
bool vuSphericRevolver<SIZE,TYPE>::init(const char* DataFile)
{
  SetEvtHandlerEnabled(true);

  //Set up the window
  SetTitle(_titleString());
  CreateStatusBar();

  //Create a volume data instance.
  // the "m_Data = new vu1611?2" is called by child classes
  m_Data->setFileName(DataFile);

  //Read in the data.
  bool success = m_Data->read();
  if (success) {
    m_glCanvas->SetSize(512,512);
    vuString str;
    str  = _titleString();
    str += ": [";
    str += m_Data->getWidth();
    str += "x";
    str += m_Data->getHeight();
    str += "] x ";
    str += m_Data->getNumberOfViews();
    str += " Views <";
    str += SIZE;
    str += " ";
    str += m_Data->_typeName();
    str += ">";
    SetTitle(str.c_str());
    Fit();
    m_Data->setViewFilter(vuSphericViewFilterFactory<SIZE,TYPE>::
			  getFilter("Blend"));
    m_Data->setFilter(vuSphericFilterFactory<SIZE,TYPE>::getFilter("Nearest"));

    m_Data->setIsReRendering(true);
  }
  else {
    wxMessageDialog dlg(this,m_Data->getErrorMessage(),
			"vuSphericRevolver",wxOK);
    dlg.ShowModal();
  }
  return success;
}


/* ------------------------------------------------------------------------ */
/* --- add some ui elements ----------------------------------------------- */
/* ------------------------------------------------------------------------ */

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::addBottom(wxSizer *sizer)
{
  //Add some control elements
  wxBoxSizer *verSpacer = new wxBoxSizer(wxVERTICAL);
  wxBoxSizer *horSpacer = new wxBoxSizer(wxHORIZONTAL);

  m_ImageScale = new wxSlider(this, idIMAGESCALE, 100, 1,
			      1500, 
			      wxDefaultPosition, wxSize(300,18),
			      wxSL_HORIZONTAL,wxDefaultValidator,"imageScale");

  horSpacer->Add(10,0);
  horSpacer->Add(new wxStaticText(this, -1, "Scale:"));
  horSpacer->Add(5,0);
  horSpacer->Add(m_ImageScale);
  horSpacer->Add(5,0);
  horSpacer->Add(new wxButton(this, idIMAGESCALE2ONE, "fit",
		 wxDefaultPosition, wxSize(40,18)));
  horSpacer->Add(18,0);

  // number of nearest views
  {  
    m_CHOICEnumberOfViews = 
      new wxChoice(this, idNUMBEROFVIEWS, wxDefaultPosition, wxSize(45,20),
		   0, NULL, wxMAXIMIZE_BOX, wxDefaultValidator, "Views");

    // lightfield filter choice
    m_CHOICEnumberOfViews->Append("1");
    m_CHOICEnumberOfViews->Append("2");
    m_CHOICEnumberOfViews->Append("3");
    m_CHOICEnumberOfViews->Append("4");
    m_CHOICEnumberOfViews->SetSelection(0);

    horSpacer->Add(new wxStaticText(this, -1, "Views:"));
    horSpacer->Add(5,0);
    horSpacer->Add(m_CHOICEnumberOfViews,0,wxALL|wxALIGN_LEFT,1);
  }

  verSpacer->Add(1,10);
  verSpacer->Add(horSpacer);
  verSpacer->Add(1,10);
  sizer->Add(verSpacer);
}

template <int SIZE, class TYPE>
bool vuSphericRevolver<SIZE,TYPE>::glInit()
{
  if (m_Data == NULL) return false;
  m_Data->initOpenGL();
  m_Data->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
  m_Data->setIsReRendering(true);
  return true;
}

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::glRender()
{
  wxStopWatch watch;

  watch.Start();
  vuFixelMap<SIZE,TYPE> map = *(m_Data->computeAndReturnImage());
  map.scaleAndBias((float)m_ImageScale->GetValue()/100);
  map.glRender();
  watch.Pause();
  
  wxString statusText = "";
  int      idx        = -1;

  vuSphericViewFilter<SIZE,TYPE> *filter = m_Data->getViewFilter();
  if (filter) {
    idx = m_Data->getIndexOfView(filter->getView(0));
  }
  
  {
    char msg[64];
    sprintf(msg,"Render Time: %3lu ms", watch.Time());
    statusText += msg;
  }
  
  if (idx >= 0) {
    char msg[32];
    sprintf(msg,"  --  view: %4lu", (dword)idx);
    statusText += msg;
  }

  statusText += "  --  Press '?' for help";

  SetStatusText(statusText);
}

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::glResize()
{
  m_Data->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
}

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::onMouse(wxMouseEvent &ev)
{
  // here is the place for additional mouse bevaviour.
}

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::onKeyboard(wxKeyEvent& event)
{
  switch(event.GetKeyCode()) {
    case '>': { // plus
      int delta = (event.m_controlDown) ? 1 : 10;
      _updateScaleAndRefresh(m_ImageScale->GetValue() + delta);
      break;
    }
    case '<': { // minus
      int delta = (event.m_controlDown) ? 1 : 10;
      _updateScaleAndRefresh(m_ImageScale->GetValue() - delta);
      break;
    }
    case ' ': { // space --> set image scaling to one
      _updateScaleAndRefresh(100);
    }
  }
}

template <int SIZE, class TYPE>
wxString vuSphericRevolver<SIZE,TYPE>::helpText()
{
  wxString str("");

  str += 
    "\n  Keyboard bindings:\n\n"
    "\t ?\t\t\t this help window\n"
    "\t k\t\t\t Keyframer\n"
    "\t right\t\t\t rotate right\n"
    "\t LEFT\t\t\t rotate left\n"
    "\t UP\t\t rotate up\n"
    "\t DOWN\t\t rotate down\n"
    "\t SHIFT-RIGHT\t rotate slowly right\n"
    "\t SHIFT-LEFT\t\t rotate slowly left\n"
    "\t SHIFT-UP\t\t rotate slowly up\n"
    "\t SHIFT-DOWN\t rotate slowly down\n"
    "\t <\t\t\t decrease image scaling\n"
    "\t >\t\t\t increase image scaling\n"
    "\t CONTROL+<\t\t slowly decrease image scaling\n"
    "\t CONTROL+>\t\t slowly increase image scaling\n"
    "\t SPACE\t\t\t set image scaling to one (same as 'fit' button)\n";

  return str;
}


template <int SIZE, class TYPE>
vu1 *vuSphericRevolver<SIZE,TYPE>::getVolume()
{
  return (vu1 *)m_Data;
}

/* --------------------------------------------------------------------- */
/* --- some GUI callbacks ---------------------------------------------- */
/* --------------------------------------------------------------------- */

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::OnChoiceNumberOfViews(wxCommandEvent& event)
{
  vuSphLfFlt_Nearest<SIZE,TYPE> *filter = NULL;

  filter = (vuSphLfFlt_Nearest<SIZE,TYPE>*)m_Data->getFilter();
  
  if (filter == NULL) return;
  
  dword idx = m_CHOICEnumberOfViews->GetSelection();
 
  filter->setNumberOfViews(idx+1);
  
  m_Data->setIsReRendering(true);
  m_glCanvas->redraw();
}

template<int SIZE,class TYPE>
void vuSphericRevolver<SIZE,TYPE>::OnScaleImage(wxScrollEvent& event)
{
  m_glCanvas->redraw();
}

template<int SIZE,class TYPE>
#if wxMINOR_VERSION < 5
void vuSphericRevolver<SIZE,TYPE>::OnScaleImage2One(wxScrollEvent& event)
#else
void vuSphericRevolver<SIZE,TYPE>::OnScaleImage2One(wxCommandEvent& event)
#endif
{
  m_ImageScale->SetValue(100);
  m_glCanvas->redraw();
}

template <int SIZE, class TYPE>
vuCamera* vuSphericRevolver<SIZE,TYPE>::getCamera()
{
  return m_Data->getCameraPtr ();
}

template <int SIZE, class TYPE>
vuImage* vuSphericRevolver<SIZE,TYPE>::getCurrentImage()
{
  return NULL;
}

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::DrawFromImage()
{
  m_glCanvas->redraw ();
}

template <int SIZE, class TYPE>
void vuSphericRevolver<SIZE,TYPE>::DrawAgain()
{
}

/* ------------------------------------------------------------------------- */
/* --- internal methods ---------------------------------------------------- */
/* ------------------------------------------------------------------------- */

template<int SIZE,class TYPE>
void vuSphericRevolver<SIZE,TYPE>::_updateScaleAndRefresh(int scale)
{
  if (scale >= 1 && scale <= 1500) {
    m_ImageScale->SetValue(scale);
    m_glCanvas->redraw();
  }
}
