#include "vuSphericBrowser.h"

#define BROWSER_MAX 10000

template<int SIZE,class TYPE>
vuSphericBrowser<SIZE,TYPE>::vuSphericBrowser()
{
  m_Data = NULL;
}

template<int SIZE,class TYPE>
vuSphericBrowser<SIZE,TYPE>::~vuSphericBrowser()
{
  if (m_Data != NULL) {
    delete m_Data;
    m_Data = NULL;
  }
}

template<int SIZE,class TYPE>
const char *vuSphericBrowser<SIZE,TYPE>::_titleString()
{
  return "Spheric Lightfield Browser";
}

template<int SIZE,class TYPE>
bool vuSphericBrowser<SIZE,TYPE>::init(const char* DataFile)
{
  SetEvtHandlerEnabled(true);

  //Set up the window
  SetTitle(_titleString());
  CreateStatusBar();

  //Create a volume data instance.
  // the "m_Data = new vu1611?1" is called by the child class
  m_Data->setFileName(DataFile);

  //Read in the data.
  bool success = m_Data->read();
  if (success) {
    m_glCanvas->SetSize(512,512);
    vuString str(m_Data->getNumberOfViews() - 1);
    m_ViewsText->SetLabel(str.c_str());
    str  = "";
    str += _titleString();
    str += ": [";
    str += m_Data->getWidth();
    str += "x";
    str += m_Data->getHeight();
    str += "] x ";
    str += m_Data->getNumberOfViews();
    str += " Views <";
    str += SIZE;
    str += " ";
    str += m_Data->_typeName();
    str += ">";
    SetTitle(str.c_str());
    Fit();
  }
  else {
    wxMessageDialog dlg(this,m_Data->getErrorMessage(),
			_titleString(), wxOK);
    dlg.ShowModal();
  }
  return success;
}

template<int SIZE,class TYPE>
vuCamera* vuSphericBrowser<SIZE,TYPE>::getCamera()
{
  return NULL;
}

template<int SIZE,class TYPE>
vuImage* vuSphericBrowser<SIZE,TYPE>::getCurrentImage()
{
  return NULL;
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::DrawFromImage()
{
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::DrawAgain()
{
}

template<int SIZE,class TYPE>
wxString vuSphericBrowser<SIZE,TYPE>::helpText()
{
  wxString str("");

  str += 
    "\n  Keyboard bindings:\n\n"
    "\t ?\t\t\t\t this help window\n"
    "\t UP\t\t\t previous view\n"
    "\t DOWN\t\t\t next view\n"
    "\t LEFT\t\t\t decrease image scaling\n"
    "\t RIGHT\t\t\t increase image scaling\n"
    "\t SHIFT+LEFT\t\t slowly decrease image scaling\n"
    "\t SHIFT+RIGHT\t\t slowly increase image scaling\n"
    "\t SPACE\t\t\t set image scaling to one (same as 'fit' button)\n";

  return str;
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::addRight(wxSizer *sizer)
{
  //Add some control elements
  sizer->Add(new wxStaticText(this, -1, "0", wxDefaultPosition,
			      wxSize(40,20), wxALIGN_CENTRE), 0,
	     wxEXPAND, 5);
  
  m_ViewSlider = new wxSlider(this, idBROWSELIGHTFIELD, 0, 0,
			      BROWSER_MAX, 
			      wxDefaultPosition, wxSize(30,300),
			      wxSL_VERTICAL, wxDefaultValidator,"browser");
  sizer->Add(m_ViewSlider, 1, wxALL, 10);

  m_ViewsText = new wxStaticText(this, -1, "0", wxDefaultPosition,
				 wxSize(40,20), wxALIGN_CENTRE);
  sizer->Add(m_ViewsText, 0, wxEXPAND, 5);
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::addBottom(wxSizer *sizer)
{
  //Add some control elements
  wxBoxSizer *verSpacer = new wxBoxSizer(wxVERTICAL);
  wxBoxSizer *horSpacer = new wxBoxSizer(wxHORIZONTAL);

  m_ImageScale = new wxSlider(this, idIMAGESCALE, 100, 1,
			      1500, 
			      wxDefaultPosition, wxSize(300,18),
			      wxSL_HORIZONTAL,wxDefaultValidator,"imageScale");

  horSpacer->Add(10,0);
  horSpacer->Add(new wxStaticText(this, -1, "Scale:"));
  horSpacer->Add(5,0);
  horSpacer->Add(m_ImageScale);
  horSpacer->Add(5,0);
  horSpacer->Add(new wxButton(this, idIMAGESCALE2ONE, "fit",
		 wxDefaultPosition, wxSize(40,18)));
  horSpacer->Add(40,0);

  verSpacer->Add(1,10);
  verSpacer->Add(horSpacer);
  verSpacer->Add(1,10);
  sizer->Add(verSpacer);
}


template<int SIZE,class TYPE>
bool vuSphericBrowser<SIZE,TYPE>::glInit()
{
  if (m_Data == NULL) return false;
  m_Data->initOpenGL();
  return true;
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::glRender()
{
  m_Data->render();
  vuFixelMap<SIZE,TYPE> map = *(m_Data->getMap());
  
  map.scaleAndBias((float)m_ImageScale->GetValue()/100);
  map.glRender();

  char msg[512];
  vuVector lookFrom = m_Data->getCurrentView()->getLookFrom();
  sprintf(msg,"view: %lu  x=%6.3f  y=%6.3f  z=%6.3f",
	  m_Data->getViewIndex(), lookFrom[0], lookFrom[1], lookFrom[2]);
  wxString statusText = msg;
  statusText += " -- Press '?' for help";

  SetStatusText(statusText);
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::glResize()
{
  m_Data->glResize(m_glCanvas->getWidth(), m_glCanvas->getHeight());
  m_glCanvas->redraw();
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::glOnMouse(wxMouseEvent &ev)
{
 // do nothing
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::OnChar(wxKeyEvent &event)
{
  switch(event.GetKeyCode()) {
    case '?' :
      onKeyboardHelp(event);
      break;
    case 317: { // up
      dword idx = m_Data->getViewIndex();
      if (idx > 0)
        _updateIndexAndRefresh(idx - 1);
      break;
    }
    case 319: { // down
      dword idx = m_Data->getViewIndex() + 1;
      if (idx < m_Data->getNumberOfViews())
        _updateIndexAndRefresh(idx);
      break;
    }
    case 318: { // right
      int delta = (event.m_shiftDown) ? 1 : 10;
      _updateScaleAndRefresh(m_ImageScale->GetValue() + delta);
      break;
    }
    case 316: { // left
      int delta = (event.m_shiftDown) ? 1 : 10;
      _updateScaleAndRefresh(m_ImageScale->GetValue() - delta);
      break;
    }
    case ' ': { // space --> set image scaling to one
      _updateScaleAndRefresh(100);
    }
  }
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::OnBrowseViews(wxScrollEvent& event)
{
  dword idx;

  idx = m_ViewSlider->GetValue()*m_Data->getNumberOfViews()/BROWSER_MAX;
  m_Data->setViewIndex(idx);
  m_glCanvas->redraw();
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::OnScaleImage(wxScrollEvent& event)
{
  m_glCanvas->redraw();
}

template<int SIZE,class TYPE>
#if wxMINOR_VERSION < 5
void vuSphericBrowser<SIZE,TYPE>::OnScaleImage2One(wxScrollEvent& event)
#else
void vuSphericBrowser<SIZE,TYPE>::OnScaleImage2One(wxCommandEvent& event)
#endif
{
  m_ImageScale->SetValue(100);
  m_glCanvas->redraw();
}

/* ------------------------------------------------------------------------- */
/* --- internal methods ---------------------------------------------------- */
/* ------------------------------------------------------------------------- */

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::_updateIndexAndRefresh(dword idx)
{
  m_Data->setViewIndex(idx);
  m_ViewSlider->SetValue(BROWSER_MAX*idx/(m_Data->getNumberOfViews()-1));
  m_glCanvas->redraw();
}

template<int SIZE,class TYPE>
void vuSphericBrowser<SIZE,TYPE>::_updateScaleAndRefresh(int scale)
{
  if (scale >= 1 && scale <= 1500) {
    m_ImageScale->SetValue(scale);
    m_glCanvas->redraw();
  }
}
