#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "vuGuiParser.h"

//----------------------------------------------------------------------------
//------------------------- Gui script tokens --------------------------------
//----------------------------------------------------------------------------

#define GUI_OPEN       " gui %s : %s %n"
#define SECTION_OPEN   "%s { %n"
#define CONTROL_DEFINE "%s %s { %n"
#define STRING         "%s %n"


//----------------------------------------------------------------------------
//------------------------- vuGuiParser() ------------------------------------
//----------------------------------------------------------------------------

vuGuiParser::vuGuiParser() : vuParser()
{
}

//----------------------------------------------------------------------------
//------------------------- ~vuGuiParser() -----------------------------------
//----------------------------------------------------------------------------

vuGuiParser::~vuGuiParser()
{
}

//----------------------------------------------------------------------------
//------------------------- public Parse() -----------------------------------
//----------------------------------------------------------------------------

void vuGuiParser::Parse(const char* filename) throw (const char *)
{
    //read in the file
    readBuffer(filename);
    FixBuffer();
  
    //parse the gui declaration
    ParseGui();
}

//----------------------------------------------------------------------------
//------------------------- private: ParseGui() ------------------------------
//----------------------------------------------------------------------------

void vuGuiParser::ParseGui()
{
    int len=0, ret;
    char name[64]="";
    char base[64]="";

    //Read in the gui declaration
    ret = sscanf(&m_Buffer[m_Pos], GUI_OPEN, name, base, &len);
    if (ret != 2) throw ("Gui declaration is missing or invalid.");
    m_Pos+=len; len = 0;
    m_Name = name; m_Base = base;

    //Create the output files for the code
    m_fHeader.open(m_Name + ".h",ios::out);
    m_fImpl.open(m_Name + ".cpp",ios::out);
    if (!m_fHeader.is_open()||!m_fImpl.is_open())
        throw ("Could not open files for output.");

    //Declare the gui class in the header file
    m_fHeader << "#include \"" << m_Base << ".h\"\n\n";
    m_fHeader << "class " << m_Name <<": public " << m_Base << "\n{\n";
    m_fHeader << "public:\n";

    //Parse the includes for the gui class
    ParseAPIs();

    //Parse the rest of the file.
    ParseSections();

    //Write the control, event, and api data.
    WriteData();

    //Close the output files
    m_fHeader.close();
    m_fImpl.close();
}

//----------------------------------------------------------------------------
//------------------------- private: ParseAPIs() -----------------------------
//----------------------------------------------------------------------------

void vuGuiParser::ParseAPIs()
{
    StringList Apis;

    //Parse the api list from the input file
    if (readToken('['))
    { 
        Apis = ParseList();
        if (!readToken(']')) 
            throw ("Include header listing not closed properly.");
    }

    //Now output the api data to the files.
    m_fImpl << "#include \"" << m_Name << ".h\"\n";
    int num = Apis.getLength();
    for (int i=0; i < num; i++)
        m_fImpl << "#include \"" << Apis[i] << "\"\n";
    m_fImpl << "\n";
};

//----------------------------------------------------------------------------
//------------------------- private: ParseSections() -------------------------
//----------------------------------------------------------------------------

void vuGuiParser::ParseSections()
{
    char name[256] = "";
    int len = 0;

    //Check for the gui declaration opening.
    if (!readToken('{')) return;

    //Read all the sections from the file 
    while (!readToken('}') && !finished())
    {
        if (sscanf(&m_Buffer[m_Pos],SECTION_OPEN,name,&len)!=1)
            throw ("Invalid gui section declaration.");
        m_Pos += len; len = 0;

        //Parse the section based on its type
        if (strcmp(name,"init") == 0)
            ParseMethod("init()");
        else if (strcmp(name,"close") == 0)
            ParseMethod("close()");
        else if (strcmp(name,"Top")==0  || strcmp(name,"Bottom")==0 ||
                 strcmp(name,"Left")==0 || strcmp(name,"Right")==0)
            ParseControls(name);
        else
            throw ("Undefined name for gui section declaration.");
    }
}

//----------------------------------------------------------------------------
//------------------------- private: ParseControls() -------------------------
//----------------------------------------------------------------------------

void vuGuiParser::ParseControls(const char *Location)
{
    char name[256],type[256];
    int len = 0;
    vuString addFunction;

    while (!readToken('}') && !finished())
    {
        if (sscanf(&m_Buffer[m_Pos],CONTROL_DEFINE,type,name,&len)!=2)
            throw ("Invalid user interface control declaration.");
        m_Pos+=len; len = 0;

        if (strcmp(type,"listbox")==0)
            addFunction += ParseListBox(name);
        else if (strcmp(type,"button")==0)
            addFunction += ParseButton(name);
        else if (strcmp(type,"slider")==0)
            addFunction += ParseSlider(name);
        else if (strcmp(type,"checkbox")==0)
	        addFunction += ParseCheckBox(name);
        else if (strcmp(type,"radiobox")==0)
            addFunction += ParseRadioBox(name);
		addFunction += '\n';
    }
    
    //declare the control function in the header file
    m_fHeader << "    void add" << Location << "(wxSizer *Sizer);\n";

    //implement the function in the implementation file
    m_fImpl << "void " << m_Name << "::add" << Location;
    m_fImpl << "(wxSizer *Sizer)\n{\n" << addFunction << "};\n\n";
}

//----------------------------------------------------------------------------
//------------------------- private: ParseList() -----------------------------
//----------------------------------------------------------------------------

StringList vuGuiParser::ParseList()
{
    bool read = true;
    int len = 0;
    char value[256];
    StringList slist;
    
    while (read)
    {
        read = false;
        if (readToken(',')) 
            read = true;
        else if (readToken(';')) {}
        else if (readToken('}',false) || finished()) {}
        else if (readToken(']',false)) {}
        else if (sscanf(&m_Buffer[m_Pos],STRING,value,&len)==1)
        {
            m_Pos += len; len = 0;
            slist.add(value);
            read = true;
        }
    }
    return slist;
}

//----------------------------------------------------------------------------
//------------------------- private: ParseMethod() ---------------------------
//----------------------------------------------------------------------------

void vuGuiParser::ParseMethod(const char *Header)
{
        //declare the method in the header file
    m_fHeader << "    void " << Header << ";\n";

        //declare the method in the implementation file
    m_fImpl << "void " << m_Name << "::" << Header << "\n{\n";

        //Parse all the statements in the code block
    while (!readToken('}') && !finished())
        m_fImpl << "    " << ParseStatement() << ";\n";

    m_fImpl << "};\n\n";
}

//----------------------------------------------------------------------------
//------------------------- private: ParseStatement() ------------------------
//----------------------------------------------------------------------------

vuString vuGuiParser::ParseStatement()
{
    int len = 0;
    char name[256];
    vuString statement = "m_Data->";

        //check to see if it is an empty statement.
    if (readToken(';')||finished()) return "";

        //read in the statement.
    sscanf(&m_Buffer[m_Pos],STRING,name,&len);
    m_Pos+=len; len = 0;
        //translate the name into a method call
    statement << name << "(";
        //check if there is a parameter given
    if (readToken('('))
    {
        if (!readToken(')',false))
        {
            sscanf(&m_Buffer[m_Pos],STRING,name,&len);
            m_Pos+=len; len = 0;
                //change the parameter into a value
            if (m_Controls.isMember(vuString("l")+name))
                statement << "m_l" << name << "->GetSelection()";
            else if (m_Controls.isMember(vuString("s")+name))
                statement << "m_s" << name << "->GetValue()";
            else if (m_Controls.isMember(vuString("c")+name))
                statement << "m_c" << name << "->GetValue()";
            else if (m_Controls.isMember(vuString("r")+name))
                statement << "m_r" << name << "->GetStringSelection()";
            else if (strcmp(name,"glwidth")==0)
                statement << "m_glCanvas->getWidth()";
            else if (strcmp(name,"glheight")==0)
                statement << "m_glCanvas->getHeight()";
            else if (m_Controls.isMember(vuString("b")+name))
                throw ("Can't pass a button value as a parameter.");
            else
                throw ("Undefined identifier passed as a parameter.");
        }
            //read the closing paranthesis
        if (!readToken(')')) throw ("Missing closing paranthesis in code block");
    }
    statement << ")";
        //Read in the ending semicolon.
    readToken(';');
    return statement;
}

//----------------------------------------------------------------------------
//------------------------- private: WriteData() -----------------------------
//----------------------------------------------------------------------------

void vuGuiParser::WriteData()
{
    //Declare the control variables in the header file
    m_fHeader << "\nprivate:\n";
    unsigned int num = m_Controls.getLength();
    for (unsigned int i=0; i < num; i++)
    {
        if (m_Controls[i][(long unsigned int) (0)]=='l')
            m_fHeader << "    wxListBox* m_" << m_Controls[i].c_str() << ";\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='s')
            m_fHeader << "    wxSlider* m_" << m_Controls[i].c_str() << ";\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='b')
            m_fHeader << "    wxButton* m_" << m_Controls[i].c_str() << ";\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='c')
            m_fHeader << "    wxChoiceBox* m_" << m_Controls[i].c_str() << ";\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='r')
            m_fHeader << "    wxRadioBox* m_" << m_Controls[i].c_str() << ";\n";
    }
        //add the event declartion
    m_fHeader << "\n    DECLARE_EVENT_TABLE()\n";
        //Close off the header
    m_fHeader << "};\n";

        //Declare the enums for the controls in the implementation file
    m_fImpl << "enum\n{\n";
    m_fImpl << "    START_VALUE = 100,\n";
    for (int i=0; i < int (num-1); i++)
        m_fImpl << "    id_" << m_Controls[i].c_str() << ",\n";
    m_fImpl << "    id_" << m_Controls[num-1].c_str() << "\n};\n\n";

        //Declare the event table for the controls
    m_fImpl << "BEGIN_EVENT_TABLE(" << m_Name << ", " << m_Base << ")\n";
    for (unsigned int i=0; i < num; i++)
    {
        if (m_Controls[i][(long unsigned int) (0)]=='l')
            m_fImpl << "    EVT_LISTBOX(id_" << m_Controls[i].c_str() << ", "
                  << m_Name.c_str() << "::On" << m_Controls[i].c_str() << ")\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='s')
            m_fImpl << "    EVT_SLIDER(id_" << m_Controls[i].c_str() << ", "
                  << m_Name.c_str() << "::On" << m_Controls[i].c_str() << ")\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='b')
            m_fImpl << "    EVT_BUTTON(id_" << m_Controls[i].c_str() << ", "
                  << m_Name.c_str() << "::On" << m_Controls[i].c_str() << ")\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='c')
            m_fImpl << "    EVT_CHECKBOX(id_" << m_Controls[i].c_str() << ", "
                  << m_Name.c_str() << "::On" << m_Controls[i].c_str() << ")\n";
        else if (m_Controls[i][(long unsigned int) (0)]=='r')
            m_fImpl << "    EVT_RADIOBOX(id_" << m_Controls[i].c_str() << ", "
                      << m_Name.c_str() << "::On" << m_Controls[i].c_str() << ")\n";
    }
    m_fImpl << "END_EVENT_TABLE()\n\n";
    m_fImpl.seekp(0,ios::end);
}

//----------------------------------------------------------------------------
//------------------------- private: ParseListBox() --------------------------
//----------------------------------------------------------------------------

vuString vuGuiParser::ParseListBox(const char *cname)
{
    vuString name = vuString("l") + cname;
    if (m_Controls.isMember(name))
        throw ("Defined two controls of the same name.");
    m_Controls.add(name);

    //Parse the data for the listbox
    StringList values = ParseList();
    StringList initial = ParseList();
    if ((values.getLength()==0)||(initial.getLength()!=1))
        throw ("Invalid definition of a listbox control.");

    //Make the code to create the listbox
    vuString create;
    create << "    m_" << name << " = new wxListBox(this, ";
    create << "id_" << name << ");\n";
    int num = values.getLength();
    for (int i=0; i < num; i++)
        create << "    m_" << name << "->Append(\"" << values[i].c_str() << "\");\n";
    create << "    m_" << name << "->SetStringSelection(\"";
    create << initial[0].c_str() << "\");\n";
    create << "    Sizer->add(m_" << name << ");\n";

    //Finally process any code that goes along with the control
    ParseMethod("On" + name + "(wxCommandEvent &event)");

    //And return the creation code
    return create;
}

//----------------------------------------------------------------------------
//------------------------- private: ParseSlider() ---------------------------
//----------------------------------------------------------------------------

vuString vuGuiParser::ParseSlider(const char *cname)
{
    vuString name = vuString("s") + cname;
    if (m_Controls.isMember(name))
        throw ("Defined two controls of the same name.");
    m_Controls.add(name);

        //Parse the data for the slider
    StringList settings = ParseList();
    if (settings.getLength()!=4)
        throw ("Invalid definition of a slider control.");

        //Make the code to create the slider
    vuString create;
    create << "    m_" << name << " = new wxSlider(this, ";
    create << "id_" << name << "," << settings[2] << "," << settings[0].c_str();
    create << "," << settings[1].c_str() << ",wxDefaultPosition,wxDefaultSize,";
    if (strcmp(settings[3].c_str(),"vertical")==0)
        create << "wxSL_VERTICAL";
    else
        create << "wxSL_HORIZONTAL";
    create << ");\n    Sizer->add(m_" << name << ");\n";

        //Finally process any code that goes along with the control
    ParseMethod("On" + name + "(wxCommandEvent &event)");

        //And return the creation code
    return create;
}
//----------------------------------------------------------------------------
//------------------------- private: ParseRadioBox() -------------------------
//----------------------------------------------------------------------------

vuString vuGuiParser::ParseRadioBox(const char *cname)
{
    vuString name = vuString("r") + cname;
    if (m_Controls.isMember(name))
        throw ("Defined two controls of the same name.");
    m_Controls.add(name);

        //Parse the data for the radiobox
    StringList label = ParseList();
    StringList choices = ParseList();
    StringList initial = ParseList();
    if ((label.getLength()==0)||(choices.getLength()==0)
	    ||(initial.getLength()!=1))
        throw ("Invalid definition of a radiobox control.");

        //Make the code to create the radiobox
    vuString create;

    create << "    vuString label" << name << " = \"";
    int num = label.getLength();
    for (int i=0; i < num-1; i++)
        create << label[i].c_str() << " ";
    create << label[num-1].c_str() << "\";\n";
    create << "    wxString choices[" << choices.getLength() << "];\n";
    num = choices.getLength();
    for (int i=0; i < num; i++)
       create << "    choices[" << i << "] = \"" << choices[i].c_str() << "\";\n";
    create << "    m_" << name << " = new wxRadioBox(this, ";
    create << "id_" << name << ",label" << name << ",wxDefaultPosition,";
    create << "wxDefaultSize," << num << ",choices);\n";
    create << "    Sizer->add(m_" << name << ");\n";

        //Finally process any code that goes along with the control
    ParseMethod("On" + name + "(wxCommandEvent &event)");

        //And return the creation code
    return create;
}
//----------------------------------------------------------------------------
//------------------------- private: ParseCheckbox() -------------------------
//----------------------------------------------------------------------------

vuString vuGuiParser::ParseCheckBox(const char *cname)
{
    vuString name = vuString("c") + cname;
    if (m_Controls.isMember(name))
        throw ("Defined two controls of the same name.");
    m_Controls.add(name);

        //Parse the data for the checkbox
    StringList label = ParseList();
    StringList initial = ParseList();
    if ((label.getLength()==0)||(initial.getLength()==0))
        throw ("Invalid definition of a checkbox control.");

        //Make the code to create the checkbox
    vuString create;
    create << "    wxString label" << name << " = \"";
    int num = label.getLength();
    for (int i=0; i < num-1; i++)
        create << label[i].c_str() << " ";
    create << label[num-1].c_str() << "\";\n";
    create << "    m_" << name << " = new wxCheckBox(this, ";
    create << "id_" << name << ",label" << name << ");\n";
    create << "    m_" << name << "->SetValue(" << initial[0];
    create << ");\n    Sizer->add(m_" << name << ");\n";

        //Finally process any code that goes along with the control
    ParseMethod("On" + name + "(wxCommandEvent &event)");

        //And return the creation code
    return create;
}
//----------------------------------------------------------------------------
//------------------------- private: ParseButton() ---------------------------
//----------------------------------------------------------------------------

vuString vuGuiParser::ParseButton(const char *cname)
{
    vuString name = vuString("b") + cname;
    if (m_Controls.isMember(name))
        throw ("Defined two controls of the same name.");
    m_Controls.add(name);

        //Parse the data for the button
    StringList label = ParseList();
    if (label.getLength()==0)
    throw ("Invalid definition of a button control.");

        //Make the code to create the button
    vuString create;
    create << "    wxString label" << name << " = \"";
    int num = label.getLength();
    for (int i=0; i < num-1; i++)
        create << label[i].c_str() << " ";
    create << label[num-1].c_str() << "\";\n";
    create << "    m_" << name << " = new wxButton(this, ";
    create << "id_" << name << ",label" << name << ");\n";
    create << "    Sizer->add(m_" << name << ");\n";

        //Finally process any code that goes along with the control
    ParseMethod("On" + name + "(wxCommandEvent &event)");

        //And return the creation code
    return create;
}
