/***************************************************************************
                          vuCellProjector.cpp  -  description
                             -------------------
    begin                : Thu May 1 2003
    copyright            : (C) 2003 by tmeng
    email                : tmeng@sfu.ca
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "vuCellProjector.h"
#include "../../wxUIElements/vuGLCanvas.h"
#include "../../wxUIElements/vuTFDialogSpec.h"
#include "vuColourRGBa.h"

//----------------------------------------------------------------------------

enum IDEnum //for the event table and wx controls initilization.
{
  idCheckBoxIsSort,
  idListBoxSubdivScheme,
  idButtonRender
};

//----------------------------------------------------------------------------
//----------------- The vuCellProjector event table --------------------------
//----------------------------------------------------------------------------

//Message map, very similar to MFC.
BEGIN_EVENT_TABLE(vuCellProjector, vuBasicUtility)
  EVT_CHECKBOX(idCheckBoxIsSort, vuCellProjector::onCheckBoxIsSort)
  EVT_LISTBOX(idListBoxSubdivScheme, vuCellProjector::onListBoxSubdivScheme)
  EVT_BUTTON(idButtonRender, vuCellProjector::onButtonRender)
END_EVENT_TABLE();

//----------------------------------------------------------------------------
//------------------------- The constructor ----------------------------------
//----------------------------------------------------------------------------

vuCellProjector::vuCellProjector() 
  : vuBasicUtility(), 
    m_Data(new vu111211a), //default cell projector.
    m_ViewScale(1.0f), //Default zooming is 100%
    m_TFunc(4), //4 means RGBA (4 channels)
    m_TFDialog(this, m_TFunc), //construct transfer function dialog.
    m_x(0), m_y(0) //sensible values before actual mouse clicks.
{
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

/*virtual*/ vuCellProjector::~vuCellProjector()
{
  if (m_Data != 0) delete m_Data;
}

//----------------------------------------------------------------------------
//------------------------- public: getFileType() ----------------------------
//----------------------------------------------------------------------------

/*static*/ const char* vuCellProjector::getFileType()
{
  return "11121";
}

//----------------------------------------------------------------------------
//------------------------- public: addRight() -------------------------------
//----------------------------------------------------------------------------


/*virtual*/ void vuCellProjector::addRight(wxSizer* sizer)
{
  //top alignment by default.
  addCheckBoxIsSort(sizer);
  addListBoxSubdivScheme(sizer);
  addButtonRender(sizer);
}

//----------------------------------------------------------------------------
//------------------------- public: init() -----------------------------------
//----------------------------------------------------------------------------

/*virtual*/ bool vuCellProjector::init(const char* DataFile)
  //Initializes the cellProjector utility.
  //Initializes the utility window.  A cellProjector object is created and
  //the volume data is read.  The window appears when finished.
{
  //Set up the window for the cellProjectorter.
  SetTitle("vuCellProjector");
  CreateStatusBar(); //shows render time.
    
  //Create a volume data instance.
  //  m_Data = new vu111211a; //allocate space for a cell projector.
  m_Data->setFileName(DataFile); //will project the volume data in the specified file.

  //see readme.txt within vuVolume/TFunc for more info
  m_TFunc.loadTF("TFunc/default.tf");
  m_TFunc.setOpacitySmoothing(0);
  m_TFunc.setColourSmoothing(1);
  m_TFunc.generateFunction();
  m_Data->setTransferFunc(m_TFunc); 
  //the CellProjector (not the utility window) now has a transfer function.

  //Read in the data.
  bool success = m_Data->read();
  if (success)
    {
      m_glCanvas->SetSize(512,512); //default utility size.
      Fit(); //likely a wxWindows command.
    }
  else
    {
      //Show the error message.
      wxMessageDialog dlg(this, m_Data->getErrorMessage(), "vuCellProjector",wxOK);
      dlg.ShowModal();
    }
  
  //Not sure if false should be returned anywhere, since as I recall in MFC
  //false is never returned in init() either. So may have to look back to this later.
  return success; //initialization successful.
};

//----------------------------------------------------------------------------
//------------------------- public: notifyDataChanged() ----------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::notifyDataChanged()
  //Acts kind of like Invalidate() in MFC, which does a redraw.
{
  m_glCanvas->redraw();
}

//----------------------------------------------------------------------------
//------------------------- public: DrawAgain() ------------------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::DrawAgain()
  //Rerenders the screen from the current camera position.
{
  m_glCanvas->redraw();
}

//----------------------------------------------------------------------------
//------------------------- public: DrawFromImage() --------------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::DrawFromImage()
  //Draws on the screen the image contained in the image buffer.
  //Basically, it's a flush() function.
{
  m_Data->drawPic();
}

//----------------------------------------------------------------------------
//------------------------- public: getCurrentImage() ------------------------
//----------------------------------------------------------------------------

/*virtual*/ vuImage* vuCellProjector::getCurrentImage()
  //This will return a pointer to the image buffer.
{
  return m_Data->getBuffer ();
}

//----------------------------------------------------------------------------
//------------------------- public: getCamera() ------------------------------
//----------------------------------------------------------------------------

/*virtual*/ vuCamera* vuCellProjector::getCamera()
  //This will return a pointer to the camera that this class is using
  //(note that this may be derived from vuCamera...) 
{
  return &m_Camera;
}

//----------------------------------------------------------------------------
//------------------------- public: IsReRendering() --------------------------
//----------------------------------------------------------------------------

/*virtual*/ bool vuCellProjector::IsReRendering()
  //This will return whether or not the cellProjector is rerendering to the screen.
{
  return m_Data->IsReRendering ();
}

//----------------------------------------------------------------------------
//------------------------- public: setIsReRendering() -----------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::setIsReRendering (bool isit)
  //This will set the state of rerendering to be the same as "isit". 
{
  m_Data->setIsReRendering(isit);
}

//----------------------------------------------------------------------------
//------------------------- protected: glInit() ------------------------------
//----------------------------------------------------------------------------

/*virtual*/ bool vuCellProjector::glInit()   
  //Initializes openGL for the cellProjector utility.
  //The method calls the CellProjector initGL() method so it can do the
  //proper initialization.
{
  if (m_Data == 0) return false;
    
  //Paint a black background to the utility window.
  const GLfloat RED = 0.0f;
  const GLfloat GREEN = 0.0f;
  const GLfloat BLUE = 0.0f;
  const GLfloat ALPHA = 1.0f;
  glClearColor(RED, GREEN, BLUE, ALPHA);
    
  //Enable one light for now.
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);

  //Enable lighting calculation for backfaces as well.
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
    
  //Call CellProjector::initGL() which does initilization particular
  //to the CellProjector algorithm.
  m_Data->initOpenGL();
  return true;
};

//----------------------------------------------------------------------------
//------------------------- protected: glRender() ----------------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::glRender()
{
  wxStopWatch watch; //used to time how long it takes to render each frame.
  watch.Start();

  //Clears the color buffer.
  glClear(GL_COLOR_BUFFER_BIT);

  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  m_Camera.gluLookAt(); //applied to the model view matrix.

  glTranslatef((float)m_Data->getDim1Size()/(-2.0f),
	       (float)m_Data->getDim2Size()/(-2.0f),
	       (float)m_Data->getDim3Size()/(-2.0f));
  
  m_Data->setViewVectors(m_Camera.getLookAtVector(),
			 m_Camera.getUpVector(),
			 m_Camera.getRightVector());
  
  //This is not necessary every frame, but done anyways,
  //because the non-modal tfuncDlg might have changed it.
  //The overhead is negligible so it's ok to keep.
  m_Data->setTransferFunc(m_TFunc);
    
  //Render the volume.
  m_Data->render();

  //print out the time it took to render the frame onto the status bar.  
  watch.Pause();
  SetStatusText(wxString("Render Time: ") + vuString(watch.Time()).c_str() + "ms");
};

//----------------------------------------------------------------------------
//------------------------- protected: glResize() ----------------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::glResize()
{
  //Retrieves the new width and height of the window from the window itself.
  GLint newWidth = (GLint)m_glCanvas->getWidth();
  GLint newHeight = (GLint)m_glCanvas->getHeight();

  //Calculate an aspect ratio.
  GLfloat aspect_ratio = ((float) newWidth) / ((float) newHeight);

  //Set the viewport(left, bottom, width, height).
  glViewport(0, 0, newWidth, newHeight);
    
  //Find the largest dimension of the data, so it fits inside the window.
  dword max = m_Data->getDim1Size();
  if (m_Data->getDim2Size() > max)
    max = m_Data->getDim2Size();
  if (m_Data->getDim3Size() > max)
    max = m_Data->getDim3Size();

  //Switch the openGL projection matrix, required before calling glOrtho.
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity(); //reset the projection matrix.

  //The following if/else loop gets rid of the horizontal or vertical
  //distortions associated with stretching the window.
  if(newHeight >= newWidth)
    //This means aspect_ratio <= 1.0
    //Output image stretched along y direction, so need to divide top and 
    //bottom by aspect_ratio (i.e. need more "slots" in the y direction to
    //not distort image)
    {
      //Sets up orthographic projection matrix and defines a viewing volume
      //that is a right parallelepiped.
      //glOrtho(left, right, bottom, top, near, far)
      glOrtho((float)max/(-1.0f*m_ViewScale), 
	      (float)max/m_ViewScale,
	      (float)max/aspect_ratio/(-1.0f*m_ViewScale), 
	      (float)max/aspect_ratio/m_ViewScale,
	      10000.0f, -10000.0f); 
      //10000 SHOULD be large enough, data sets that are larger
      //probably can't be rendered in available memory anyway.
    }
  else
    //This means aspect_ratio > 1.0
    //Output image stretched along x direction, so need to divide left and 
    //right by aspect_ratio (i.e. need more "slots" in the x direction to
    //not distort image)
    {
      glOrtho((float)max*aspect_ratio/(-1.0f*m_ViewScale), 
	      (float)max*aspect_ratio/m_ViewScale,
	      (float)max/(-1.0f*m_ViewScale), 
	      (float)max/m_ViewScale,
	      10000.0f, -10000.0f);

    }
    
  //Return to the model view matrix (for manipulating objects).
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity(); //this line probably can be removed.
  
  //Set the opengl light info.
  //float lpos[4] = {0.0, 0.0, 1024.0, 1.0};
  //glLightfv(GL_LIGHT0, GL_POSITION, lpos);
}

//----------------------------------------------------------------------------
//------------------------- protected: onMouse() -----------------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::glOnMouse(wxMouseEvent &ev)
  //Eventually, this could extend vuBasicUtility::glOnMouse(event).
  //For now, I have followed what everyone else has done.
{
  //vuBasicUtility::glOnMouse(ev); 
  //extend the parent function, which deals with rotation and zooming
  //already. 

  if (ev.LeftDown() || ev.RightDown())
    {
      //Store the click position.
      m_x = (int) ev.GetX();
      m_y = (int) ev.GetY();
    }
  else if (ev.LeftIsDown() && ev.Moving())
    //Rotate the camera (same as rotating the volume)
    { 
      //get the camera location.
      vuVector t = m_Camera.getPosition();
      
      //get distance from camera to origin. Same as d = (t-origin).norm()
      float d = t.norm(); 

      m_Camera.translateXYZ(0.0f, 0.0f, d);
      m_Camera.rotateAboutUp(ev.GetX() - m_x);
      m_Camera.rotateAboutRight(ev.GetY() - m_y);
      m_Camera.translateXYZ(0.0f, 0.0f, -d);
      m_glCanvas->redraw();

      //Store the click position.
      m_x = (int) ev.GetX();
      m_y = (int) ev.GetY();
    }
  else if (ev.RightIsDown() && ev.Moving())
    //Zooming.
    {
      m_ViewScale -= ((float)ev.GetY() - m_y)/500.0f;
      this->glResize();
      m_glCanvas->redraw();

      //Store the click position.
      m_x = (int) ev.GetX();
      m_y = (int) ev.GetY();
    }
  else if (ev.LeftDClick())
    //Pop up the transfer function editor.
    {
      m_TFDialog.Show(true);
    }
}

//----------------------------------------------------------------------------
//------------------------- protected: getVolume() ---------------------------
//----------------------------------------------------------------------------

/*virtual*/ vu1* vuCellProjector::getVolume()
  // Returns a parent pointer to the current algorithm.
{
  return m_Data; 
  //m_Data is a pointer to the current algorithm, and vu1* is a parent
  //pointer so the upcasting is auto.
}

//----------------------------------------------------------------------------
//------------------------- protected: OnChar() ------------------------------
//----------------------------------------------------------------------------

/*virtual*/ void vuCellProjector::OnChar(wxKeyEvent& event)
{
  //vuBasicUtility::OnChar(event); //chain to parent.
  //the above function does not work right now, CellProjector has both
  //getVolume() and getCameraPtr() but does not work, not quite sure why.
  //should look into it later.

  dword dataSize = m_Data->m_DataSize;
  char key = event.GetKeyCode();
  if (dataSize == 8) 
    //Possibly enable manual display of tets.
    {
      switch(key)
	{
	case 'm': //manual tet enable
	  m_Data->m_state = MANUAL_TET;
	  break;
	case 'n': //normal (default)
	  m_Data->m_state = NORMAL;
	  break;
	};
      if (key >= '0' && key <= '9')
	//if 8 data points, then only tets avail for display.
	//if 12 data points, 10 tets avail for display
	{
	  m_Data->m_tetIndex = key - '0';
	  //the above line has been tested with cout statements
	  if (m_Data->m_subdivScheme == SIX_FOLD && m_Data->m_tetIndex > 5)
	    {
	      m_Data->m_tetIndex = 5; //clamp, else out of bound index.
	      cout<<"tet index: "<<m_Data->m_tetIndex<<endl;
	    }
	}
    }
  //the following line seems to call glRender() from somewhere
  //already so no need to call glRender() again.
  this->DrawAgain(); //or this->notifyDataChanged();
}

//----------------------------------------------------------------------------
//--------------- protected: onCheckBoxIsSort() ------------------------------
//----------------------------------------------------------------------------

#if wxMINOR_VERSION < 5
void vuCellProjector::onCheckBoxIsSort()
#else
void vuCellProjector::onCheckBoxIsSort(wxCommandEvent&)
#endif
{
  m_Data->m_isSort = (bool) m_checkBoxIsSort->GetValue();
  this->notifyDataChanged();
}

//----------------------------------------------------------------------------
//--------------- protected: onListBoxSubdivScheme() -------------------------
//----------------------------------------------------------------------------

#if wxMINOR_VERSION < 5
void vuCellProjector::onListBoxSubdivScheme()
#else
void vuCellProjector::onListBoxSubdivScheme(wxCommandEvent&)
#endif
{  
  m_Data->m_subdivScheme = (SubdivEnum) m_listBoxSubdivScheme->GetSelection();
  this->notifyDataChanged();
}

//----------------------------------------------------------------------------
//--------------- protected: onButtonRender() --------------------------------
//----------------------------------------------------------------------------

#if wxMINOR_VERSION < 5
void vuCellProjector::onButtonRender()
#else
void vuCellProjector::onButtonRender(wxCommandEvent&)
#endif
{
  m_Data->m_isRender = true;
  this->notifyDataChanged();
}

//----------------------------------------------------------------------------
//--------------- protected: addCheckBoxIsSort() -----------------------------
//----------------------------------------------------------------------------

void vuCellProjector::addCheckBoxIsSort(wxSizer* sizer)
{
  m_checkBoxIsSort = new wxCheckBox(this, //parent window pointer
				    idCheckBoxIsSort, //unique control id
				    "Sort and Triangulate?");
  m_checkBoxIsSort->SetValue(m_Data->m_isSort);

  //if no following line, the checkbox will show up at the top left hand
  //corner of the utility window.
  sizer->Add(m_checkBoxIsSort, //check box added to right panel (sizer)
	     0, //not stretcheable
	     wxALL | wxALIGN_LEFT, //wxALL->all 4 borders,
	     1); //border width = 1
}

//----------------------------------------------------------------------------
//--------------- protected: addListBoxSubdivScheme() ------------------------
//----------------------------------------------------------------------------

void vuCellProjector::addListBoxSubdivScheme(wxSizer* sizer)
{
  char scheme1[] = "5-Fold";
  char scheme2[] = "6-Fold";
  wxString listBoxEntries[] = {scheme1, scheme2};
  m_listBoxSubdivScheme = 
    new wxListBox(this, //parent window pointer
		  idListBoxSubdivScheme, //unique id
		  wxDefaultPosition,
		  wxSize(150,70), //arbitrary, in pixels
		  2, //2 entries initially
		  listBoxEntries,
		  wxLB_SINGLE | wxLB_ALWAYS_SB); 
  //can only select one entry at once and always show vert scrolling bar.

  m_listBoxSubdivScheme->SetSelection((int)(m_Data->m_subdivScheme),
				      TRUE); //select subdiv scheme.

  sizer->Add(m_listBoxSubdivScheme , //check box added to right panel (sizer)
	     0, //not stretcheable
	     wxALL | wxALIGN_LEFT, //wxALL->all 4 borders,
	     1); //border width = 1
}

//----------------------------------------------------------------------------
//--------------- protected: addButtonRender() -------------------------------
//----------------------------------------------------------------------------

void vuCellProjector::addButtonRender(wxSizer* sizer)
{
  sizer->Add(new wxButton(this, idButtonRender, "Render"), //ptr to button
	     0, //unstretcheable
	     wxALL | wxALIGN_LEFT, //all borders
	     10); //border width = 10.
}
