/** \file
 * 
 * Implements a program which generates the vuUtilityFactory class 
 * implementation for the GUI Tool.
 *
 * It gets passed the Utility names for the specific vuUtilityFactory 
 * implementation and makes a class that can create these utilities.  
 * Specifically, it makes the .cpp file that implements the vuUtilityFactory 
 * class interface.  The names of the utilities are passed by the command line.
 *
 * Syntax:
 *
 * generateFactory Utility1 Utility2 ...
 *
 * where "Utility#" is both the name of the Utility's header file, and of the 
 * class itself.  The program will generate code to #include the header and to 
 * instantiate the class.  This will be saved as vuUtilityFactory.cpp which 
 * then can be compiled with the GUI Tool program.
 *    
 * This program is meant to be run by the GUI Tool's makefile.
 */

#include <fstream.h>
#include <iostream.h>
#include <string.h>
#include "vuString.h"
#include "vuDVector.h"

typedef vuDVector<vuString> StringList;

//The following write different parts of the vuUtilityFactory.cpp file

//!Write a comment for the vuUtilityFactory implementation.
void writeComment(ofstream &fout,const StringList &list);
//!Writes the include statements for the factory.
void writeIncludes(ofstream &fout,const StringList &list);
//!Implements the create() method of the vuUtilityFactory
void writeCreate(ofstream &fout,const StringList &list);
//!Implements the isAvailable() method of the vuUtiltyFactory.
void writeIsAvailable(ofstream &fout,const StringList &list);
//!Implements the listAvailable() method of the vuUTiltiyFactory.
void writeListAvailable(ofstream &fout,const StringList &list);
//!Strips any ".o" extensions from the name and returns the string.
vuString stripExt(const char *name);
//!Strips any directory paths from the name and returns the pointer.
const char *stripPath(const char *name);

//----------------------------------------------------------------------------
//------------------------- The main program ---------------------------------
//----------------------------------------------------------------------------

int main(int argc, char* argv[])
{
        //Make sure that command line parameters were passed
    if (argc == 1)
    {
        cout << "Error: No output file specified.  vuUtilityFactory.cpp not created.\n";
        return -1;
    }
    else if (argc == 2)
    {
        cout << "Error: No utilities were listed.  vuUtilityFactory.cpp not created.\n";
        return -1;
    }

    //Open the output file, truncating/replacing it if it exists
    ofstream fout(argv[1]);
    
    //Make a string list out of the arguments
    StringList list;    
    for (int i=2;i< argc;i++)
        list.add(stripExt(argv[i]));
    
    //Now write the class file
    writeComment(fout,list);
    writeIncludes(fout,list);
    writeCreate(fout,list);
    writeIsAvailable(fout,list);
    writeListAvailable(fout,list);
    
    fout.close();
    cout.flush();
    return 0;
};


//----------------------------------------------------------------------------
//------------------------- void writeComment() ------------------------------
//----------------------------------------------------------------------------

void writeComment(ofstream &fout,const StringList &list)
    //!Writes a comment on the generated cpp file so that developers know where it came from.
{
    fout << "//This implementation file was generated by the GenerateFactory utility.\n\n";
    fout << "//The utilities supported by this implementation are:\n";
    for (dword i=0; i < list.getLength() ;i++)
        fout << "//    " << list[i] << "\n";
    fout << "\n";
}

//----------------------------------------------------------------------------
//------------------------- void writeIncludes() -----------------------------
//----------------------------------------------------------------------------

void writeIncludes(ofstream &fout,const StringList &list)
    //!Writes the include statements for the vuUtilityFactory implementation into the output file.
    //The command line parameters are passed to the function and list the utilities
    //that the class will create.
{
    fout << "#include \"vuUtilityFactory.h\"\n";
    for (dword i = 0; i < list.getLength(); i++)
        fout << "#include \"wxUtilities/" << list[i] << "/" << list[i] << ".h\"\n";
    fout << "\n";
}

//----------------------------------------------------------------------------
//------------------------- void writeCreate() -------------------------------
//----------------------------------------------------------------------------

void writeCreate(ofstream &fout,const StringList &list)
    //!Writes the vuUtilityFactory::create() method into the output file.
    //The command line parameters are passed to the function and list the 
    //utilities that the class will create.
{
    fout << "vuUtilityWindow *vuUtilityFactory::create(const char *name)\n";
    fout << "{\n";
    fout << "    vuUtilityWindow *window=0;\n\n";

    for (dword i = 0; i < list.getLength() ; i++)
    {
        if (i == 0)
            fout << "    if ";
        else
            fout << "    else if ";
        fout << "(strcmp(name,\"" << stripPath(list[i]) << "\")==0)\n"
             << "        window = new " << stripPath(list[i]) << "();\n";
    };
    fout << "\n    return window;\n}\n\n";
};

//----------------------------------------------------------------------------
//------------------------- void writeIsAvailable() --------------------------
//----------------------------------------------------------------------------

void writeIsAvailable(ofstream &fout,const StringList &list)
    //!Writes the vuUtilityFactory::isAvailable() method into the output file.
    //The command line parameters are passed to the function and list the 
    //utilities that the class can create.
{
    fout << "bool vuUtilityFactory::isAvailable(const char *name)\n";
    fout << "{\n";

    for (dword i = 0; i < list.getLength(); i++)
    {
        if (i == 0)
            fout << "    if ";
        else
            fout << "    else if ";
        fout << "(strcmp(name,\"" << stripPath(list[i]) << "\")==0)\n";
        fout << "        return true;\n";
    };
    fout << "    else\n        return false;\n}\n\n";
};

//----------------------------------------------------------------------------
//------------------------- void writeListAvailable() ------------------------
//----------------------------------------------------------------------------

void writeListAvailable(ofstream &fout,const StringList &list)
    //!Writes the vuUtilityFactory::listAvailable() method into the output file.
    //The command line parameters are passed to the function and list the 
    //utilities that the class can create.
{
    fout << "wxStringList vuUtilityFactory::listAvailable(const char* FileType)\n{\n";
    fout << "    wxStringList utilities;\n\n";
    for (dword i = 0; i < list.getLength(); i++)
        fout << "    if (strcmp(" << stripPath(list[i]) << "::getFileType(),FileType)==0)\n"
             << "        utilities.Add(\"" << stripPath(list[i]) << "\");\n";

    fout << "\n    return utilities;\n}\n\n";
};

//----------------------------------------------------------------------------
//------------------------- vuString stripExt() -------------------------------
//----------------------------------------------------------------------------

vuString stripExt(const char *name)
    //!Strips any ".o" extensions from the name of the utilities
{
    vuString Name = name;
    dword length = Name.getLength();
    if ((name[length-2]=='.') && (name[length-1]=='o'))
        return Name.substr(0,length-3);
    else
        return Name;
};

//----------------------------------------------------------------------------
//------------------------- vuString stripPath() -------------------------------
//----------------------------------------------------------------------------

const char* stripPath(const char *file)
    //!Strips the path of the filename
{
    const char *name=strrchr(file,'\\');
    if (name==NULL) name = strrchr(file,'/');
    if (name==NULL) name = file;
    else name++;
    return name;
};
