#include "vuTransferDialog.h"

#include <iostream>
#include "vuColourRGBa.h"
#include "../vuUtilityWindow.h"
#include <wx/colordlg.h>

//----------------------------------------------------------------------------
//------------------------- The vuTransferDialog event table -----------------
//----------------------------------------------------------------------------

BEGIN_EVENT_TABLE(vuTransferDialog, wxDialog)
  EVT_COMMAND(vuTransferDialog::idCANVAS,
	      vuEVT_TRANSFER_NODE_SELECT, vuTransferDialog::OnSelect)
  EVT_COMMAND(vuTransferDialog::idCANVAS,
	      vuEVT_TRANSFER_NODE_OPEN, vuTransferDialog::OnOpen)
  EVT_COMMAND(vuTransferDialog::idCANVAS,
	      vuEVT_TRANSFER_CHANGE, vuTransferDialog::OnSelect)
  EVT_BUTTON(vuTransferDialog::idUPDATE, vuTransferDialog::OnUpdate)
  EVT_BUTTON(vuTransferDialog::idLOADTF, vuTransferDialog::OnLoadTF)
  EVT_BUTTON(vuTransferDialog::idSAVETF, vuTransferDialog::OnSaveTF)
  EVT_BUTTON(wxID_OK, vuTransferDialog::OnOK)
  EVT_BUTTON(wxID_CANCEL, wxDialog::OnCancel)
END_EVENT_TABLE()

//----------------------------------------------------------------------------
//------------------------- The constructor ----------------------------------
//----------------------------------------------------------------------------

vuTransferDialog::vuTransferDialog(wxWindow *parent, vuTFDesign &tf)
    : wxDialog(parent,-1, wxString("Transfer Function Editor"),
	       wxDefaultPosition,wxDefaultSize,wxDEFAULT_DIALOG_STYLE)
{
    //Create the OpenGL canvas.
    m_Canvas = new vuTransferCanvas(tf,this,idCANVAS,true);
    m_DoSpectral = (tf.getNComponents() != 4);

    //m_Canvas->setTransferFunc(tf);
    m_Canvas->SetSize(400,300);

    //Create the text controls, all accepting only numeric input.
    wxSize size(35,-1);
    tInte = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
			   wxTextValidator(wxFILTER_NUMERIC));
    tOpac = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
			   wxTextValidator(wxFILTER_NUMERIC));
    tOpacSmooth = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
				 wxTextValidator(wxFILTER_NUMERIC));
    tColSmooth = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
				wxTextValidator(wxFILTER_NUMERIC));
    if(m_DoSpectral) {
	//choose small size to make them invisible...
	tr = new wxTextCtrl(this,-1,"",wxDefaultPosition,wxSize(1,1),0,
			    wxTextValidator(wxFILTER_NUMERIC));
	tg = new wxTextCtrl(this,-1,"",wxDefaultPosition,wxSize(1,1),0,
			    wxTextValidator(wxFILTER_NUMERIC));
	tb = new wxTextCtrl(this,-1,"",wxDefaultPosition,wxSize(1,1),0,
			    wxTextValidator(wxFILTER_NUMERIC));
    } else {
	tr = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
			    wxTextValidator(wxFILTER_NUMERIC));
	tg = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
			    wxTextValidator(wxFILTER_NUMERIC));
	tb = new wxTextCtrl(this,-1,"",wxDefaultPosition,size,0,
			    wxTextValidator(wxFILTER_NUMERIC));
    }
    tr->Enable(false);
    tg->Enable(false);
    tb->Enable(false);

    //Set up the initial values
    tInte->Enable(false);
    tOpac->Enable(false);
    wxString temp;
    temp.Printf("%g",tf.getOpacitySmoothing());
    tOpacSmooth->SetValue(temp);
    temp.Printf("%g",tf.getColourSmoothing());
    tColSmooth->SetValue(temp);

    //Now create the UI for the dialog
    //First the sizers.
    m_MainSizer = new wxBoxSizer(wxHORIZONTAL);
    wxBoxSizer *lefSizer = new wxBoxSizer(wxVERTICAL);
    wxBoxSizer *rigSizer = new wxBoxSizer(wxVERTICAL);

    //Add the glCanvas.
    lefSizer->Add(m_Canvas,1,wxEXPAND);

    //Add the editing controls
    wxBoxSizer *inteSizer = new wxBoxSizer(wxHORIZONTAL);
    inteSizer->Add(new wxStaticText(this,-1,"Intensity:"),0,wxALL|wxALIGN_LEFT,1);
    inteSizer->Add(tInte,1,wxALL|wxEXPAND,1);
    rigSizer->Add(inteSizer,0,wxALIGN_CENTER);
    wxBoxSizer *opacSizer = new wxBoxSizer(wxHORIZONTAL);
    opacSizer->Add(new wxStaticText(this,-1,"Opacity:"),0,wxALL|wxALIGN_LEFT,1);
    opacSizer->Add(tOpac,1,wxALL|wxEXPAND,1);
    rigSizer->Add(opacSizer,0,wxALIGN_CENTER);
    if(!m_DoSpectral) {
	wxBoxSizer *colourSizer = new wxStaticBoxSizer(new wxStaticBox(this,-1,"Colour"),
						       wxHORIZONTAL);
	colourSizer->Add(tr,1,wxALL|wxEXPAND,1);
	colourSizer->Add(tg,1,wxALL|wxEXPAND,1);
	colourSizer->Add(tb,1,wxALL|wxEXPAND,1);
	rigSizer->Add(colourSizer,0,wxALIGN_CENTER);
    }
    wxBoxSizer *opasmSizer = new wxBoxSizer(wxHORIZONTAL);
    opasmSizer->Add(new wxStaticText(this,-1,"Opacity Smoothing:"),0,wxALIGN_LEFT|wxALL,1);
    opasmSizer->Add(tOpacSmooth,1,wxALL|wxEXPAND,1);
    rigSizer->Add(opasmSizer,0,wxALIGN_CENTER);
    wxBoxSizer *colsmSizer = new wxBoxSizer(wxHORIZONTAL);
    colsmSizer->Add(new wxStaticText(this,-1,"Colour Smoothing:"),0,wxALIGN_LEFT|wxALL,1);
    colsmSizer->Add(tColSmooth,1,wxALL|wxEXPAND,1);
    rigSizer->Add(colsmSizer,0,wxALIGN_CENTER);
    rigSizer->Add(20,5,0,wxALIGN_CENTER|wxALL);
    rigSizer->Add(new wxButton(this,idUPDATE,"Update"),0,wxALIGN_CENTER|wxALL,1);
    rigSizer->Add(new wxButton(this,idLOADTF,"Load..."),0,wxALIGN_CENTER|wxALL,1);
    rigSizer->Add(new wxButton(this,idSAVETF,"Save..."),0,wxALIGN_CENTER|wxALL,1);

    m_CustomSizer = new wxBoxSizer(wxVERTICAL);
    buildCustomSizer();	//since it's virtual the basic version will be called
    rigSizer->Add(m_CustomSizer,0,wxALIGN_CENTER);
    
    rigSizer->Add(20,20,1,wxALIGN_CENTER|wxALL|wxEXPAND);
    wxBoxSizer *butSizer = new wxBoxSizer(wxHORIZONTAL);
    butSizer->Add(new wxButton(this,wxID_OK,"OK"),0,wxALIGN_CENTER|wxALL,1);
    butSizer->Add(new wxButton(this,wxID_CANCEL,"Hide"),0,wxALIGN_CENTER|wxALL,1);
    rigSizer->Add(butSizer,0,wxALIGN_CENTER);

    //Construct the Dialog architecture.
    m_MainSizer->Add(lefSizer,1,wxEXPAND);
    m_MainSizer->Add(rigSizer,0,wxEXPAND);

    //Set the Dialog up to use the sizers
    SetSizer(m_MainSizer);
    SetAutoLayout(true);
    m_MainSizer->Layout();
        
    m_MainSizer->SetSizeHints(this);
    m_MainSizer->Fit(this);
    m_IsUpdated = true;
}

void vuTransferDialog::rebuildCustomSizer()
{
    if(!m_CustomSizer) return;
    //remove all elements from the sizer
    while(m_CustomSizer->GetChildren().GetCount() > 0 )
		m_CustomSizer->Remove(0);
    
	buildCustomSizer();

    //Set the Dialog up to use the sizers
    m_CustomSizer->Layout();	// recalc layout
    m_MainSizer->Layout();
    m_MainSizer->SetSizeHints(this);
    m_MainSizer->Fit(this);
}

//----------------------------------------------------------------------------
//------------------------- private: OnSelect() ------------------------------
//----------------------------------------------------------------------------

void vuTransferDialog::OnSelect(wxCommandEvent &ev)
{
    const vuTFDesign::OpacityNode *con = m_Canvas->getActiveOpacity();
    const vuTFDesign::ColourNode *col = m_Canvas->getActiveColour();
    wxString temp;

    if (con != 0) 
    {
        temp.Printf("%i",con->intensity);
        tInte->SetValue(temp);
        tInte->Enable(true);

        temp.Printf("%g",con->opacity);
        tOpac->SetValue(temp);
        tOpac->Enable(true);

        tr->SetValue("");
        tr->Enable(false); 

        tg->SetValue("");
        tg->Enable(false); 

        tb->SetValue("");
        tb->Enable(false);
    }
    else if (col != 0 && !m_DoSpectral)
    {
        temp.Printf("%i",col->intensity);
        tInte->SetValue(temp);
        tInte->Enable(true);

        tOpac->SetValue("");
        tOpac->Enable(false);

        temp.Printf("%g",col->col[0]);
        tr->SetValue(temp);
        tr->Enable(true);

        temp.Printf("%g",col->col[1]);
        tg->SetValue(temp);
        tg->Enable(true);

        temp.Printf("%g",col->col[2]);
        tb->SetValue(temp);
        tb->Enable(true);
    }
    else
    {
        tInte->SetValue("");
        tInte->Enable(false);

        tOpac->SetValue("");
        tOpac->Enable(false);

        tr->SetValue("");
        tr->Enable(false); 

        tg->SetValue("");
        tg->Enable(false); 

        tb->SetValue("");
        tb->Enable(false);
    }
}

//----------------------------------------------------------------------------
//------------------------- private: OnOpen() --------------------------------
//----------------------------------------------------------------------------

void vuTransferDialog::OnOpen(wxCommandEvent &ev)
{
    const vuTFDesign::ColourNode *coln = m_Canvas->getActiveColour();
    
    if (coln != 0)
    {
      if(!m_DoSpectral) {
        wxColourData Data;
        wxColour Col(byte(coln->col[0]*256),byte(coln->col[1]*256),byte(coln->col[2]*256));
        Data.SetCustomColour(0,Col);
        Data.SetColour(Col);
        wxColourDialog dialog(this,&Data);

        if (dialog.ShowModal() == wxID_OK)
        {
            Data = dialog.GetColourData();
            Col = Data.GetColour();

            vuTFDesign::ColourNode cn(coln->intensity, 4,
				      vuColourRGBa((float)Col.Red()/256,
						   (float)Col.Green()/256,
						   (float)Col.Blue()/256));
            m_Canvas->setActiveColour(&cn);
        }
      } else {
	  vuTFDesign &tf = m_Canvas->getTransferFunc();
	  editSpecColour(tf.getColourNodeIndex(*coln));
      }
    }
    m_IsUpdated = true;
}

//----------------------------------------------------------------------------
//------------------------- private: OnOk() ----------------------------------
//----------------------------------------------------------------------------

void vuTransferDialog::OnOK(wxCommandEvent &ev)
{
    m_IsUpdated = true;
    EndModal(wxID_OK);
}

//----------------------------------------------------------------------------
//------------------------- private: OnUpdate() ------------------------------
//----------------------------------------------------------------------------

void vuTransferDialog::OnUpdate(wxCommandEvent &ev)
{
    long intensity; 
    double opacity;
    double r,g,b;
    double OpacSmooth, ColSmooth;

    //Figure out what kind of control node is being edited
    //by the text controls that are enabled
    if (tOpac->IsEnabled() && tInte->GetValue().ToLong(&intensity) &&
        tOpac->GetValue().ToDouble(&opacity))
    {
        vuTFDesign::OpacityNode cn((byte)intensity,(float)opacity);
        m_Canvas->setActiveOpacity(&cn);
    }
    else if (tr->IsEnabled() && tInte->GetValue().ToLong(&intensity) &&
        tr->GetValue().ToDouble(&r) && tg->GetValue().ToDouble(&g) &&
        tb->GetValue().ToDouble(&b))
    {
      vuTFDesign::ColourNode cn((byte)intensity, 4,
				vuColourRGBa((float)r,(float)g,(float)b));
      m_Canvas->setActiveColour(&cn);
    }

    //Update the opacity and colour smoothing
    if (tOpacSmooth->GetValue().ToDouble(&OpacSmooth) &&
        tColSmooth->GetValue().ToDouble(&ColSmooth))
    {
        m_Canvas->setSmoothing((float)OpacSmooth,(float)ColSmooth);
    }
    m_IsUpdated = true;
}

//----------------------------------------------------------------------------
//------------------------- private: OnLoadTF() ------------------------------
//----------------------------------------------------------------------------

void vuTransferDialog::OnLoadTF(wxCommandEvent &ev)
{
  //open load dialog, ...
  wxFileDialog fd(this,"Choose a file","","","*.tf",wxOPEN);
  if(fd.ShowModal() == wxID_OK)
    {

      cout<<"loading... "<<fd.GetPath()<<endl;
      try {
	  vuTFDesign &tf = m_Canvas->getTransferFunc();
	  if(tf.loadTF(fd.GetPath()))
	  {
	      cout<<"successful."<<endl;
	      tf.generateFunction();
		  rebuildCustomSizer();
	  } else {
	      cout<<"failed."<<endl;
	  }
      } catch (char *msg) {
	  printf("%s\n",msg);
      }
      m_Canvas->redraw();
    }
    m_IsUpdated = true;
}

//----------------------------------------------------------------------------
//------------------------- private: OnSaveTF() ------------------------------
//----------------------------------------------------------------------------

void vuTransferDialog::OnSaveTF(wxCommandEvent &ev)
{
  //open save dialog, ...
  wxFileDialog fd(this,"Choose a file","","","*.tf",wxSAVE|wxOVERWRITE_PROMPT);
  if(fd.ShowModal() == wxID_OK)
    {

      cout<<"saving... "<<fd.GetPath()<<endl;
      try {
	  vuTFDesign &tf = m_Canvas->getTransferFunc();
	  if(tf.saveTF(fd.GetPath()))
	      cout<<"successful."<<endl;
	  else
	      cout<<"failed."<<endl;
      } catch (char *msg) {
	  printf("%s\n",msg);
      }
    }
}


//----------------------------------------------------------------------------
//------------------------- public: getTransferFunc() ------------------------
//----------------------------------------------------------------------------

const vuTFDesign& vuTransferDialog::getTransferFunc() const
{
    return m_Canvas->getTransferFunc();
}

void vuTransferDialog::repaintParent() const
{
    ((vuUtilityWindow*)GetParent())->notifyDataChanged();
}

