#ifndef _vuTransferCanvas_h_
#define _vuTransferCanvas_h_

#include <wx/wx.h>
#include "vuGLCanvas.h"
#include "vuTFunc/vuTFDesign.h"

//!A viewer and editor for the vuTFDesign transfer function.

/*!
   This class implements an Open GL viewer for the vuTFDesign transfer function.
   It can be run in both modes: as strictly a viewer, or as a viewer and editor.
   The class inherits from the wxWindows gl canvas, and is therefore a full-
   fledged wxWindows widget.  It can be plugged into other windows and panels
   just like any other widget, or it can be made to fill the whole window.

   As a viewer, the class displays the transfer function and has the option of
   going into editing mode.  Displaying is done using a method similar to 
   Riemann integrals:  a rectangle is drawn for each intensity, where the 
   height corresponds to the opacity at that intensity. The rectangle is 
   assigned the same colour as the intensity.  This method gives good results
   both for the continuity of the function and the continuity in colour
   gradients.  The viewer can also be taken into editing mode.  This is
   done when the user double clicks on it.  This action pops up the
   vuTransferDialog for editing the function.  As a viewer, this class could
   easily be used for any intensity transfer function, though right now it
   only accepts vuTFDesign transfer functions.

   The class can also be run as an editor for the vuTFDesign transfer function.
   In this mode, it renders the opacity polyline which defines the transfer
   function.  The control nodes defining the opacity polyline and also the
   colour regions are also drawn.  The user can then click on these control
   regions to select them, can move them around, and can add and delete them.
   No user interface widgets are provided for this editing, for this kind
   of functionality look at vuTransferDialog.  However, to help with such
   editing, the class also generates events corresponding to the user's 
   interaction with the window.  These events indicate when control nodes
   have been selected or opened up, and when the transfer function has
   been changed.

   The editing functionality can only be provided for the vuTFDesign transfer
   function. This is because every transfer function is defined with different
   parameters, and the control nodes mean different things.  However, this
   class can be used as a model for other transfer function editors.
*/

class vuTransferCanvas : public vuGLCanvas
{
public:
    //!The constructor for the transfer function canvas.
    /*!The transfer canvas can be used in either edit or display mode
       (as set by the edit parameter).  In edit mode, the control nodes
       are displayed and are moveable; in display mode, simply the transfer
       function is showed.

       \param parent The parent window of the transfer canvas
       \param id The wxWindows id to assign to the transfer canvas.
       \param edit Sets the edit mode for the transfer canvas, true for edit mode. 
       \param number of components in colour model for transfer function
       \param range Range of transfer function, which always begins with zero
    */
    vuTransferCanvas(vuTFDesign &tf, wxWindow *parent,wxWindowID id=-1,bool edit=false);
    //!The destructor.
    ~vuTransferCanvas();

    //!Returns the transfer function displayed by the canvas.
    vuTFDesign& getTransferFunc();
    //!Sets the transfer function to display on the canvas.
    void setTransferFunc(const vuTFDesign &tf);

    //!Returns the active control node, or 0 if there is none.
    const vuTFDesign::OpacityNode *getActiveOpacity() const;
    //!Sets the active control node to the given value.
    void setActiveOpacity(const vuTFDesign::OpacityNode *);

    //!Returns the active colour 
    const vuTFDesign::ColourNode *getActiveColour() const;
    //!Sets the active colour node to the given value.
    void setActiveColour(const vuTFDesign::ColourNode *);

    //!Sets the smoothing of the transfer function.
    void setSmoothing(float opacity, float colour);

protected:
    //!Initializes openGL for the transfer function canvas
    bool glInit();
    //!Renders the transfer function.
    void render();
    //!Sets up the coordinates for the canvas.
    void resize();
    //!Handles the mouse commands.
    void OnMouse(wxMouseEvent &ev);

    //!Helper function that posts the passed in event to the event handler
    void postEvent(wxEventType ev);

protected:
    //!The transfer function being shown in the Canvas.
    vuTFDesign &m_TFunc;
    //!Whether the transfer function is editeable or not.
    bool m_Edit;

    //!The coordinates of the screen.
    float m_xScreenMin, m_xScreenMax, m_yScreenMin, m_yScreenMax;
    //!Grid widths and heights in graph coordinates
    float m_dx, m_dy;

    //!The opacity node selected, if any (if none then value is -1).
    dword m_Opacity;
    //!The colour node selected, if any (if none then value is -1).
    dword m_Colour;

protected:
    //!These are the ranges for the graph, ie the scales on the axes.
    const dword m_xMin;
    const dword m_xMax;
    const float m_yMin;
    const float m_yMax;

    //!Indicates whether spectral colours are to be designed or RGBa
    bool m_DoSpectral;

    DECLARE_EVENT_TABLE()
};

//!These are the event identifiers for vuTransferCanvas events
enum
{
    //!A control node has been selected
    vuEVT_TRANSFER_NODE_SELECT = 12000,

    //!A control node has been double clicked
    vuEVT_TRANSFER_NODE_OPEN,
    
    //!The transfer function has changed
    vuEVT_TRANSFER_CHANGE
};

#endif
