#include "vuTransferCanvas.h"
#include "vuTransferDialog.h"
#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>

#include "vuColourRGBa.h"

BEGIN_EVENT_TABLE(vuTransferCanvas, vuGLCanvas)
    EVT_MOUSE_EVENTS(vuTransferCanvas::OnMouse)
END_EVENT_TABLE()

//----------------------------------------------------------------------------
//------------------------- The constructor ----------------------------------
//----------------------------------------------------------------------------

vuTransferCanvas::vuTransferCanvas(vuTFDesign &tf, wxWindow *parent,wxWindowID id,bool edit)
  : vuGLCanvas(parent,id,wxDefaultPosition,wxDefaultSize,0,"vuTransferCanvas",NULL)
    , m_TFunc(tf), m_Edit(edit),m_xMin(0),m_xMax(255),m_yMin(0),m_yMax(1) 
{
    m_Opacity = m_Colour = (dword) -1;

    m_xScreenMin = m_xMin;
    m_xScreenMax = m_xMax;
    m_yScreenMin = m_yMin;
    m_yScreenMax = m_yMax;

    m_DoSpectral = (tf.getNComponents() != 4);
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vuTransferCanvas::~vuTransferCanvas()
{
}

//----------------------------------------------------------------------------
//------------------------- public: getTransferFunc() ------------------------
//----------------------------------------------------------------------------

vuTFDesign& vuTransferCanvas::getTransferFunc()
{
    return m_TFunc;
}

//----------------------------------------------------------------------------
//------------------------- public: setTransferFunc() ------------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::setTransferFunc(const vuTFDesign& tf)
{
    m_TFunc = tf;
    redraw();
}

//----------------------------------------------------------------------------
//------------------------- public: setOpacitiesmoothing() -------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::setSmoothing(float opacity, float colour)
{
    m_TFunc.setOpacitySmoothing(opacity);
    m_TFunc.setColourSmoothing(colour);
    m_TFunc.generateFunction();
    redraw();
}

//----------------------------------------------------------------------------
//------------------------- public: getActiveOpacity() -----------------------
//----------------------------------------------------------------------------

const vuTFDesign::OpacityNode *vuTransferCanvas::getActiveOpacity() const
{
    if (m_Opacity==(dword)-1)
        return 0;
    else
        return &m_TFunc.getOpacity(m_Opacity);
}

//----------------------------------------------------------------------------
//------------------------- public: setActiveOpacity() -----------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::setActiveOpacity(const vuTFDesign::OpacityNode *cn)
{
    if (m_Opacity!=(dword)-1)
    {
        m_TFunc.removeOpacity(m_Opacity);
        m_Opacity = m_TFunc.addOpacity(cn->intensity,cn->opacity);

        m_TFunc.generateFunction();
        redraw();
    }
}

//----------------------------------------------------------------------------
//------------------------- public: getActiveColour() ------------------------
//----------------------------------------------------------------------------

const vuTFDesign::ColourNode *vuTransferCanvas::getActiveColour() const
{
    if (m_Colour==(dword)-1)
        return 0;
    else
        return &m_TFunc.getColour(m_Colour);
}

//----------------------------------------------------------------------------
//------------------------- public: setActiveColour() ------------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::setActiveColour(const vuTFDesign::ColourNode *cn)
{
    if (m_Colour!=(dword)-1)
    {
        m_TFunc.removeColour(m_Colour);
        m_Colour = m_TFunc.addColour(cn->intensity,cn->col);

        m_TFunc.generateFunction();
        redraw();
    }
}

//----------------------------------------------------------------------------
//------------------------- protected: glInit() ------------------------------
//----------------------------------------------------------------------------

bool vuTransferCanvas::glInit(void)
{
    glClearColor(0.0f, 0.0f, 0.0f, 0.0f);

    //Set up the coordinate system.
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluOrtho2D(m_xScreenMin,m_xScreenMax,m_yScreenMin,m_yScreenMax);

    return true;
};

//----------------------------------------------------------------------------
//------------------------- protected: resize() ------------------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::resize()
{
    glViewport(0, 0, (GLint)getWidth(),(GLint)getHeight());

    //Recalculate the grid values
    m_dx = float(m_xMax-m_xMin) / getWidth();
    m_dy = float(m_yMax-m_yMin) / getHeight();

    if (m_Edit)
    {
        //Fix the coordinate system.
        m_xScreenMin = m_xMin-10*m_dx;
        m_xScreenMax = m_xMax+10*m_dx;
        m_yScreenMin = m_yMin-30*m_dy;
        m_yScreenMax = m_yMax+10*m_dy;

        glMatrixMode(GL_MODELVIEW);
        glLoadIdentity();
        glMatrixMode(GL_PROJECTION);
        glLoadIdentity();
        gluOrtho2D(m_xScreenMin,m_xScreenMax,m_yScreenMin,m_yScreenMax);
    }
}

//----------------------------------------------------------------------------
//------------------------- protected: render() ------------------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::render()
{
    glClear(GL_COLOR_BUFFER_BIT);

    //First fill in the graph.
    //Draw the colour regions
    glBegin(GL_QUADS);
    for (dword i = m_xMin; i <= m_xMax; i++)
    {
        vuColourRGBa rgba;
	m_TFunc.getRGBa(i,rgba);

        glColor3fv(rgba.getData());
        glVertex2f(i-0.5,-20*m_dy);
        glVertex2f(i+0.5,-20*m_dy);
        glVertex2f(i+0.5,m_TFunc.getOpacityAtPos(i+1));
        glVertex2f(i-0.5,m_TFunc.getOpacityAtPos(i));
    }
    glEnd();

    //When editing, draw all the helper lines
    if (m_Edit)
    {
        //Draw the surrounding box
        glBegin(GL_LINE_LOOP);
        glColor3f(1,1,1);
        glVertex2f(m_xMin,m_yMin-20*m_dy);
        glVertex2f(m_xMax,m_yMin-20*m_dy);
        glVertex2f(m_xMax,m_yMax);
        glVertex2f(m_xMin,m_yMax);
        glEnd();

        //Draw the polyline opacity function.
        glBegin(GL_LINE_STRIP);
        glColor3f(1,1,1);
        dword numOpacities = m_TFunc.getNumOpacities();
        for (dword i = 0; i < numOpacities; i++)
        {
            vuTFDesign::OpacityNode cp = m_TFunc.getOpacity(i);
            glVertex2f(cp.intensity,cp.opacity);
        }
        glEnd();

        //Draw the opacity control nodes.
        for (dword i = 0; i < numOpacities; i++)
        {
            if (i == m_Opacity)
                glColor4f(1,0.5,0.5,0.01);
            else
                glColor4f(1,1,1,0.01);

            vuTFDesign::OpacityNode cp = m_TFunc.getOpacity(i);

            glBegin(GL_LINE_LOOP);
            glVertex2f(cp.intensity-3*m_dx,cp.opacity-3*m_dy);
            glVertex2f(cp.intensity+3*m_dx,cp.opacity-3*m_dy);
            glVertex2f(cp.intensity+3*m_dx,cp.opacity+3*m_dy);
            glVertex2f(cp.intensity-3*m_dx,cp.opacity+3*m_dy);
            glEnd();
        }

        //Draw the colour control nodes.
        dword numColours = m_TFunc.getNumColours();
        for (dword i = 0; i < numColours; i++)
        {
            if (i == m_Colour)
                glColor4f(1,0.5,0.5,0.01);
            else
                glColor4f(1,1,1,0.01);

            vuTFDesign::ColourNode cp = m_TFunc.getColour(i);

            glBegin(GL_LINE_LOOP);
            glVertex2f(cp.intensity-3*m_dx,-23*m_dy);
            glVertex2f(cp.intensity+3*m_dx,-23*m_dy);
            glVertex2f(cp.intensity+3*m_dx,-17*m_dy);
            glVertex2f(cp.intensity-3*m_dx,-17*m_dy);
            glEnd();
        }
    }
};

//----------------------------------------------------------------------------
//------------------------- protected: OnMouse() -----------------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::OnMouse(wxMouseEvent &ev)
{
    if (ev.ButtonDClick() && !m_Edit)
    {
        //Pop up dialog Canvas that lets you edit
        vuTransferDialog dlg(this,m_TFunc);

        if (dlg.ShowModal() == wxID_OK)
        {
            m_TFunc = dlg.getTransferFunc();

            postEvent(vuEVT_TRANSFER_CHANGE);
            redraw();
        }
    }
    else if (ev.LeftDown() && m_Edit) //Select the clicked control Node
    {
        //Translate into graph coordinates
        float x = (float(ev.GetX()) / getWidth()) * float(m_xScreenMax - m_xScreenMin) + m_xScreenMin;
        float y = (float(getHeight() - ev.GetY()) / getHeight()) * float(m_yScreenMax - m_yScreenMin) + m_yScreenMin;

        m_Opacity = m_Colour = (dword)-1;

        //Find out if there is an associated Opacity Node and store it
        for (dword i = 0; i < m_TFunc.getNumOpacities(); i++)
        {
            const vuTFDesign::OpacityNode &cp = m_TFunc.getOpacity(i);
            if ((fabs(cp.intensity-x) < 4*m_dx) && (fabs(cp.opacity-y) < 4*m_dy))
            {
                m_Opacity = i;
                break;
            }
        }

        //Find out if there is an associated colour Node and store it
        for (dword i = 0; i < m_TFunc.getNumColours(); i++)
        {
            const vuTFDesign::ColourNode &cp = m_TFunc.getColour(i);
            if ((fabs(cp.intensity-x) < 4*m_dx) && (fabs(-m_dy*20-y) < 4*m_dy))
            {
                m_Colour = i;
                break;
            }
        }

        postEvent(vuEVT_TRANSFER_NODE_SELECT);
        redraw();
    }
    else if (ev.LeftDClick() && m_Edit) //Create and/or open up control Node
    {
        //Translate click into graph coordinates
        float x = (float(ev.GetX()) / getWidth()) * float(m_xScreenMax - m_xScreenMin) + m_xScreenMin;
        float y = (float(getHeight() - ev.GetY()) / getHeight()) * float(m_yScreenMax - m_yScreenMin) + m_yScreenMin;

        //If in the graph range, open the clicked Opacity node, or add a new one
        if ((m_Opacity != (dword)-1) && 
            (fabs(m_TFunc.getOpacity(m_Opacity).intensity-x) < 4*m_dx) &&
            (fabs(m_TFunc.getOpacity(m_Opacity).opacity - y) < 4*m_dy))
        {
            postEvent(vuEVT_TRANSFER_NODE_OPEN);
        }
        else if (( x >= m_xMin && x <= m_xMax) && (y >= m_yMin && y <= m_yMax))
        {
            m_Opacity = m_TFunc.addOpacity((byte)x,y);
            m_Colour = (dword)-1;

            m_TFunc.generateFunction();
            postEvent(vuEVT_TRANSFER_CHANGE);
            postEvent(vuEVT_TRANSFER_NODE_OPEN);
            redraw();
        }

        //If in the colour range, open the clicked colour node, or add a new one
        if ((m_Colour != (dword)-1) && 
            (fabs(m_TFunc.getColour(m_Colour).intensity-x) < 4*m_dx) &&
            (-20*m_dy - y < 4*m_dy))
        {
            postEvent(vuEVT_TRANSFER_NODE_OPEN);
        }
        else if (( x >= m_xMin && x <= m_xMax) && (fabs(-20*m_dy-y) < 4*m_dy))
        {
	  if(m_DoSpectral) {
	    // This has to be changed!
            m_Colour = m_TFunc.addColour((byte)x,vuColourRGBa(0.f));
	  } else {
            m_Colour = m_TFunc.addColour((byte)x,vuColourRGBa(0.f));
	  }
            m_Opacity = (dword)-1;
            
            m_TFunc.generateFunction();
            postEvent(vuEVT_TRANSFER_CHANGE);    
            postEvent(vuEVT_TRANSFER_NODE_OPEN);
            redraw();
        }
    }
    else if (ev.RightDClick() && m_Edit) //Remove control Node
    {
        //Translate click into graph coordinates
        float x = (float(ev.GetX()) / getWidth()) * float(m_xScreenMax - m_xScreenMin) + m_xScreenMin;
        float y = (float(getHeight() - ev.GetY()) / getHeight()) * float(m_yScreenMax - m_yScreenMin) + m_yScreenMin;

        //Find out if there is an associated Opacity Node and remove it
        for (dword i = 0; i < m_TFunc.getNumOpacities(); i++)
        {
            const vuTFDesign::OpacityNode &cp = m_TFunc.getOpacity(i);
            if ((fabs(cp.intensity-x) < 4*m_dx) && (fabs(cp.opacity-y) < 4*m_dy))
            {
                m_TFunc.removeOpacity(i);
                if (i==m_Opacity) m_Opacity = (dword)-1;
                
                m_TFunc.generateFunction();
                postEvent(vuEVT_TRANSFER_CHANGE);
                redraw();
            }
        }

        //Find out if there is an associated colour Node and remove it
        for (dword i = 0; i < m_TFunc.getNumColours(); i++)
        {
            const vuTFDesign::ColourNode &cp = m_TFunc.getColour(i);
            if ((fabs(cp.intensity-x) < 4*m_dx) && (fabs(-20*m_dy-y) < 4*m_dy))
            {
                m_TFunc.removeColour(i);
                if (i==m_Colour) m_Colour = (dword)-1;

                m_TFunc.generateFunction();
                postEvent(vuEVT_TRANSFER_CHANGE);
                redraw();
            }
        }
    }
    else if (ev.LeftIsDown() && ev.Dragging() && m_Edit) //Drag the selected control node
    {
        //Translate into graph coordinates
        float x = (float(ev.GetX()) / getWidth()) * float(m_xScreenMax - m_xScreenMin) + m_xScreenMin;
        float y = (float(getHeight() - ev.GetY()) / getHeight()) * float(m_yScreenMax - m_yScreenMin) + m_yScreenMin;

        //Update the dragged Opacity Node, if any.
        if ((m_Opacity != (dword)-1) && 
            (x >= m_xMin) && (x <= m_xMax) && (y >= m_yMin) && (y <= m_yMax) && 
            (m_Opacity == 0 || (byte)x > m_TFunc.getOpacity(m_Opacity-1).intensity) && 
            (m_Opacity == m_TFunc.getNumOpacities()-1 || (byte)x < m_TFunc.getOpacity(m_Opacity+1).intensity))
        {
            if (m_Opacity == 0)
                m_TFunc.addOpacity(m_xMin,y);
            else if (m_Opacity == m_TFunc.getNumOpacities()-1)
                m_TFunc.addOpacity(m_xMax,y);
            else
            {
                m_TFunc.removeOpacity(m_Opacity);
                m_TFunc.addOpacity((byte)x,y);
            }
           
            m_TFunc.generateFunction();
            postEvent(vuEVT_TRANSFER_CHANGE);
            redraw();
        }

        //Update the dragged colour Node, if any.
        if ((m_Colour != (dword)-1) && (m_Colour != m_TFunc.getNumColours() - 1) &&
            (x >= m_xMin) && (x <= m_xMax) && 
            (m_Colour == 0 || (byte)x > m_TFunc.getColour(m_Colour-1).intensity) && 
            ((byte)x < m_TFunc.getColour(m_Colour+1).intensity))
        {
            vuTFDesign::ColourNode cp = m_TFunc.getColour(m_Colour);
            m_TFunc.removeColour(m_Colour);
            m_TFunc.addColour((byte)x, cp.col);
            
            m_TFunc.generateFunction();
            postEvent(vuEVT_TRANSFER_CHANGE);
            redraw();
        }
    }
}

//----------------------------------------------------------------------------
//------------------------- protected: postEvent() ---------------------------
//----------------------------------------------------------------------------

void vuTransferCanvas::postEvent(wxEventType ev)
{
     wxCommandEvent commandEvent(ev, GetId());
     commandEvent.SetEventObject( this );
     commandEvent.SetClientData(&m_TFunc);
     GetEventHandler()->ProcessEvent(commandEvent);   
}
