#ifndef _vuSpecPalette_h_
#define _vuSpecPalette_h_

#include <wx/wx.h>
#include <wx/dialog.h>
#include <wx/textctrl.h>
#include <wx/checkbox.h>
#include <wx/listbox.h>

#include "vuColourXYZa.h"
#include "vuSpectral/SPalette.h"
#ifdef WIN32
#include <wx/msw/winundef.h>
#endif

//! A graphical interface for designing a spectral palette

class vuSpecPalette : public wxDialog
{
 public:

    enum
	{
	    idCREATESP,
	    idLOADPAL,
	    idSAVEPAL,
	    idLOADSP,
	    idSAVESP,
	    idRED_SLIDER,
	    idGREEN_SLIDER,
	    idBLUE_SLIDER,
	    idWEIGHT_SLIDER,
	    idDCDES,
	    idDCGET,
	    idDCGETALL,
	    idDCFC,
	    idFCUSEB,
	    idFCUB,
	    idFCLB,
	    idSPNAME,
	    idSPDES,
	    idSPUSEB,
	    idSPUB,
	    idSPLB,
	    idADDREF,
	    idADDLIG,
	    idWSMOOTH,
	    idWERROR,
	    idUSEV7,
	    idMUL,
	    idDIV,
	    idSETPLANCK,
	    idNORMNORM,
	    idNORMLUM,
	    idLAST
	};
    
    //!Constructor for the dialog window.
    /*!This creates the dialog window and it's controls, but doesn't show it.
       Call the standard ShowModal() method to display it.

       \param parent The parent window showing the dialog.
       \param pal the palette to be designed by this dialog
    */
    vuSpecPalette(wxWindow *parent, SPalette &pal);

    //! Destructor.
    ~vuSpecPalette();
    
    //! returns a the spectral palette used by this design dialog
    const SPalette& getSPalette() const;
    
    /** select a spectrum
	if indices are too high, new entries will be created */
    void selectSpec(int rid, int lid);
    
    /** Updates the values in the widgets.
	Call _after_ CreateWidgets() */
    void UpdateWidgets();
    
    /** Fits all desired colours to the actual combination colours */
    void FitAllDesignColours() {
#if wxMINOR_VERSION < 5
        OnGetAllDCColours();
#else
	wxCommandEvent ev;
        OnGetAllDCColours(ev);
#endif
    }

 protected:
	 enum TYPE_COLOUR_CLICK {CC_GET, CC_PUT, CC_TOGGLE_DESIGN,};
    //! called if a colours is clicked
    /*! Sets or gets values from palette entry.  If refl or light is -1
        then the click refers to a spectrum (accordingly, light or
        reflectance).  If get is true the values will be loaded from
        entry. */
    void OnColourClick(int refl, int light, TYPE_COLOUR_CLICK what);
    //! sets up the measurement variables
    void CalculateMeasurements();
    //! created widgets
    void CreateWidgets();
    //! clears background and paints the highlight
    void ClearAndHighlight(wxDC& dc);
    //! paints the colour table
    void PaintColours(wxDC& dc);
    //! paints the selected colour
    void PaintColour(wxDC& dc);
    //! paints the spectrum
    void PaintSpectrum(wxDC& dc);

// event handlers   
    //! mouse handler
    void OnMouseEvent(wxMouseEvent& event);
    //! customized handle for paint events
    void OnPaint(wxPaintEvent& event);
    //!Sets the proper return codes for the ok button.
    void OnOK(wxCommandEvent &ev);
    /** Loads a transfer function */
    void OnLoadPal(wxCommandEvent &ev);
    /** Saves a transfer function */
    void OnSavePal(wxCommandEvent &ev);
    /** Loads a spectrum (reflectance or light) */
    void OnLoadSP(wxCommandEvent &ev);
    /** Saves a spectrum (reflectance or light) */
    void OnSaveSP(wxCommandEvent &ev);
    /** Multiply by value found in m_SScale */
    void OnMultiplyScalar(wxCommandEvent &ev);
    /** Divide by value found in m_SScale */
    void OnDivideScalar(wxCommandEvent &ev);
    /** Copying the value to m_SScale if Enter is pressed. 
        Works for both m_SP_Y and m_SP_Norm. */
    void OnNormVal(wxCommandEvent &ev) {
	switch(ev.GetId()) {
	    case idNORMNORM :
		m_SScale->SetValue(m_SP_Norm->GetValue());
		break;
	    case idNORMLUM:
		m_SScale->SetValue(m_SP_Y->GetValue());
		break;
	}
	OnDivideScalar(ev); // just some dummy event
    }
    /** Adds a (white) reflectance */
    void OnAddReflectance(wxCommandEvent &ev) {
		m_Palette->addReflectance(vuColour31a(1.0f));
		UpdateWidgets();
    };
    /** Adds a (white) light */
    void OnAddLight(wxCommandEvent &ev) {
	m_Palette->addLight(vuColour31a(1.0f));
	UpdateWidgets();
    };
    /** Set the currently selected light to be a Planckian black body radiator. 
     Opens a sub dialog. */
    void OnCreatePlanckian(wxCommandEvent &ev);

    //! Is a valid spectrum selected?
    bool validSpecSelected() {
	return ((m_SelRefl==-1) ^ (m_SelLight==-1) &&
		(m_SelRefl<(int)getNRefls()) && (m_SelLight<(int)getNLights()));
    }
    //! event handler
#if wxMINOR_VERSION < 5
    void OnCBSPDesign(void) {
#else
    void OnCBSPDesign(wxCommandEvent&) {
#endif
		if(validSpecSelected()) {
			bool &des = m_Palette->getSpecDesignState(m_SelRefl,m_SelLight);
			des = m_SP_design->GetValue();
		}
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnSPUpperBound(void) {
#else
    void OnSPUpperBound(wxCommandEvent&) {
#endif
	if(validSpecSelected()) {
	    double val;
	    m_SP_upperBound->GetValue().ToDouble(&val);
	    m_Palette->getSpecUB(m_SelRefl,m_SelLight) = val;
	}
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnSPLowerBound(void) {
#else
    void OnSPLowerBound(wxCommandEvent&) {
#endif
	if(validSpecSelected()) {
	    double val;
	    m_SP_lowerBound->GetValue().ToDouble(&val);
	    m_Palette->getSpecLB(m_SelRefl,m_SelLight) = val;
	}
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnSPName(void) {
#else
    void OnSPName(wxCommandEvent&) {
#endif
	if(validSpecSelected())
	    m_Palette->setSpecName(m_SelRefl,m_SelLight,
				   m_SP_name->GetValue());
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnCBSPUseB(void) {
#else
    void OnCBSPUseB(wxCommandEvent&) {
#endif
	if(validSpecSelected()) {
	    bool &b = m_Palette->useSpecBounds(m_SelRefl,m_SelLight);
	    b = m_SP_useBounds->GetValue();
	}
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnCBDCDesign(void) {
#else
    void OnCBDCDesign(wxCommandEvent&) {
#endif
	if(m_SelDesColourR>=0 && m_SelDesColourL>=0)
	    m_Palette->getDesignState(m_SelDesColourR,m_SelDesColourL)
		= m_DC_design->GetValue();
    };
	//! get design colour from actual light/refl combination colour
#if wxMINOR_VERSION < 5
	void OnGetDCColour(void) {
#else
	void OnGetDCColour(wxCommandEvent&) {
#endif
		if(m_SelDesColourR>=0 && m_SelDesColourL>=0) {
			m_Palette->getDesignRGBW(m_SelDesColourR,m_SelDesColourL) 
				= m_Palette->getRLColour(m_SelDesColourR,m_SelDesColourL, m_useV7->GetValue());
			OnColourClick(m_SelDesColourR,m_SelDesColourL,CC_GET);
		}
    };
	//! fit all design colours to the actual combination colours
#if wxMINOR_VERSION < 5
	void OnGetAllDCColours() {
#else
	void OnGetAllDCColours(wxCommandEvent&) {
#endif
		for (int refl = 0; refl < (int)getNRefls(); refl++) {
			for (int light = 0; light < (int)getNLights(); light++) {
			m_Palette->getDesignRGBW(refl,light) 
				= m_Palette->getRLColour(refl,light, m_useV7->GetValue());
			}
		}

		if(m_SelDesColourR>=0 && m_SelDesColourL>=0) 
			OnColourClick(m_SelDesColourR,m_SelDesColourL,CC_GET);
	}
    //! gets called when the design colour - RGBW changes
#if wxMINOR_VERSION < 5
    void OnChangeDCRGBW(void);
#else
    void OnChangeDCRGBW(wxCommandEvent&);
#endif
    //! event handler
#if wxMINOR_VERSION < 5
    void OnDCFC(void);
#else
    void OnDCFC(wxCommandEvent&);
#endif
    //! event handlercatscam
#if wxMINOR_VERSION < 5
    void OnCBFCUseB(void);
#else
    void OnCBFCUseB(wxCommandEvent&);
#endif
    //! event handler
#if wxMINOR_VERSION < 5
    void OnFCUpperBound(void);
#else
    void OnFCUpperBound(wxCommandEvent&);
#endif
    //! event handler
#if wxMINOR_VERSION < 5
    void OnFCLowerBound(void);
#else
    void OnFCLowerBound(wxCommandEvent&);
#endif
    //! event handler
#if wxMINOR_VERSION < 5
    void OnSmoothW(void) {
#else
    void OnSmoothW(wxCommandEvent&) {
#endif
	double val;
	m_SmoothW->GetValue().ToDouble(&val);
	m_Palette->setSmoothnessWeight(val);
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnErrorW(void) {
#else
    void OnErrorW(wxCommandEvent&) {
#endif
	double val;
	m_ErrorW->GetValue().ToDouble(&val);
	m_Palette->setErrorMinWeight(val);
    };
    //! event handler
#if wxMINOR_VERSION < 5
    void OnCBUseV7(void) {
#else
    void OnCBUseV7(wxCommandEvent&) {
#endif
	m_Palette->useV7(m_useV7->GetValue());
    };
    
#if wxMINOR_VERSION < 5
    void OnCreateSpectrum() {
#else
    void OnCreateSpectrum(wxCommandEvent& ev) {
#endif
	m_Palette->createSpectrum();
	UpdateWidgets();
    };
    
    //! event handler for sliders
#if wxMINOR_VERSION < 5
    void OnCompSlider( wxScrollEvent& event);
#else
    void OnCompSlider( wxCommandEvent& event);
#endif

// custom stuff    
    //!Part of the wxWindows architecture -- there is no data to transfer.
    bool TransferDataToWindow() {return true;};
    //!Part of the wxWindows architecture -- there is no data to transfer
    bool TransferDataFromWindow() {return true;};

    dword getNLights() {return m_Palette->getNLights();};
    dword getNRefls() {return m_Palette->getNRefls();};
    
protected:
    //!The palette designed by this dialog
    SPalette *m_Palette;

    //! Is a design colour selected?
    bool m_DCSelected;
    //! selected design colour (reflectance (y) and light (x))
    int m_SelDesColourR, m_SelDesColourL;
    
    int m_SelRefl;		//!< selected reflectance
    int m_SelLight;		//!< selected light
    
    //! Area reserved for grids of colours
    wxRect m_PalRect;
    //! Size of each colour rectangle
    wxPoint m_ColourSize;
    //! border around palette
    wxPoint m_Border;
    
    //! Grid spacing (between rectangles)
    int m_GridSpacing;
    //! Section spacing (between left and right halves of dialog box)
    int m_SectionSpacing;

    //! sliders (rgbw)
    wxSlider *m_CompSlider[4];

    //! checkboxes
    wxCheckBox *m_DC_design, *m_FC_useBounds, *m_SP_useBounds,
	*m_SP_design, *m_useV7;
    //! text controls
    wxTextCtrl *m_FC_upperBound, *m_FC_lowerBound,
	*m_SP_upperBound, *m_SP_lowerBound, *m_SP_name,
	*m_SmoothW, *m_ErrorW, *m_SScale,
	*m_SP_Y, *m_SP_Norm, *m_PlanckT;
    //! list box
    wxListBox *m_DC_FreeCol;

    /** Spaceholder for self drawn elements.
	e.g. drawing canvas, desing colour, palette) */
    wxSizer *m_DiagSpacer, *m_DColSpacer, *m_PalSpacer;
    //! Contains position and size of drawing area after layout
    wxRect m_DiagRect, m_DColRect;
    //! root sizer, needed for redesign
    wxSizer* m_TopSizer;
    
    DECLARE_EVENT_TABLE()
};

#endif
