#include <wx/wx.h>
#include <wx/colordlg.h>
#include <wx/event.h>
#include <wx/statline.h>
#include <wx/dc.h>

#include "vuSpecPalette.h"
#include <iostream.h>
#include "vuColourRGBa.h"

//----------------------------------------------------------------------------
//------------------------- The vuSpecPalette event table -----------------
//----------------------------------------------------------------------------

BEGIN_EVENT_TABLE(vuSpecPalette, wxDialog)
    EVT_BUTTON(vuSpecPalette::idCREATESP, vuSpecPalette::OnCreateSpectrum)
    EVT_BUTTON(vuSpecPalette::idLOADPAL, vuSpecPalette::OnLoadPal)
    EVT_BUTTON(vuSpecPalette::idSAVEPAL, vuSpecPalette::OnSavePal)
    EVT_BUTTON(vuSpecPalette::idLOADSP, vuSpecPalette::OnLoadSP)
    EVT_BUTTON(vuSpecPalette::idMUL, vuSpecPalette::OnMultiplyScalar)
    EVT_BUTTON(vuSpecPalette::idDIV, vuSpecPalette::OnDivideScalar)
    EVT_BUTTON(vuSpecPalette::idSAVESP, vuSpecPalette::OnSaveSP)
    EVT_BUTTON(vuSpecPalette::idADDREF, vuSpecPalette::OnAddReflectance)
    EVT_BUTTON(vuSpecPalette::idADDLIG, vuSpecPalette::OnAddLight)
    EVT_BUTTON(wxID_OK, vuSpecPalette::OnOK)
    EVT_BUTTON(wxID_CANCEL, wxDialog::OnCancel)
    EVT_BUTTON(vuSpecPalette::idSETPLANCK, vuSpecPalette::OnCreatePlanckian)
    EVT_SLIDER(vuSpecPalette::idRED_SLIDER, vuSpecPalette::OnCompSlider)
    EVT_SLIDER(vuSpecPalette::idGREEN_SLIDER, vuSpecPalette::OnCompSlider)
    EVT_SLIDER(vuSpecPalette::idBLUE_SLIDER, vuSpecPalette::OnCompSlider)
    EVT_SLIDER(vuSpecPalette::idWEIGHT_SLIDER, vuSpecPalette::OnCompSlider)
    EVT_PAINT(vuSpecPalette::OnPaint)
    EVT_MOUSE_EVENTS(vuSpecPalette::OnMouseEvent)
    //EVT_CLOSE(vuSpecPalette::OnCloseWindow)
    EVT_BUTTON(vuSpecPalette::idDCGET, vuSpecPalette::OnGetDCColour)
    EVT_BUTTON(vuSpecPalette::idDCGETALL, vuSpecPalette::OnGetAllDCColours)
    EVT_CHECKBOX(vuSpecPalette::idDCDES, vuSpecPalette::OnCBDCDesign)
    EVT_LISTBOX(vuSpecPalette::idDCFC, vuSpecPalette::OnDCFC) 
    EVT_CHECKBOX(vuSpecPalette::idFCUSEB, vuSpecPalette::OnCBFCUseB)
    EVT_TEXT_ENTER(vuSpecPalette::idFCUB, vuSpecPalette::OnFCUpperBound)
    EVT_TEXT_ENTER(vuSpecPalette::idFCLB, vuSpecPalette::OnFCLowerBound)
    EVT_CHECKBOX(vuSpecPalette::idSPDES, vuSpecPalette::OnCBSPDesign)
    EVT_CHECKBOX(vuSpecPalette::idSPUSEB, vuSpecPalette::OnCBSPUseB)
    EVT_TEXT_ENTER(vuSpecPalette::idSPUB, vuSpecPalette::OnSPUpperBound)
    EVT_TEXT_ENTER(vuSpecPalette::idSPLB, vuSpecPalette::OnSPLowerBound)
    EVT_TEXT_ENTER(vuSpecPalette::idSPNAME, vuSpecPalette::OnSPName)
    EVT_TEXT_ENTER(vuSpecPalette::idWSMOOTH, vuSpecPalette::OnSmoothW)
    EVT_TEXT_ENTER(vuSpecPalette::idWERROR, vuSpecPalette::OnErrorW)
    EVT_BUTTON(vuSpecPalette::idNORMNORM, vuSpecPalette::OnNormVal)
    EVT_BUTTON(vuSpecPalette::idNORMLUM, vuSpecPalette::OnNormVal)
    EVT_CHECKBOX(vuSpecPalette::idUSEV7, vuSpecPalette::OnCBUseV7)
END_EVENT_TABLE();

enum {
    SP_RED,
    SP_GREEN,
    SP_BLUE,
    SP_WEIGHT
};

//----------------------------------------------------------------------------
//------------------------- The constructor ----------------------------------
//----------------------------------------------------------------------------

vuSpecPalette::vuSpecPalette(wxWindow *parent, SPalette &pal)
    : wxDialog(parent, -1, wxString("Spectral palette designer"),
	       wxPoint(0, 0), wxSize(900, 900),
	       //wxDefaultPosition,wxDefaultSize,
	       wxDEFAULT_DIALOG_STYLE), m_Palette(&pal)
{
    m_SelDesColourR = 0;
    m_SelDesColourL = 0;
    m_SelRefl = 0;
    m_SelLight = -1;
    
    CalculateMeasurements();
    CreateWidgets();
}

vuSpecPalette::~vuSpecPalette()
{
}

//----------------------------------------------------------------------------
//------------------------- private: OnOk() ----------------------------------
//----------------------------------------------------------------------------

void vuSpecPalette::OnOK(wxCommandEvent &ev)
{
    EndModal(wxID_OK);
}

//----------------------------------------------------------------------------
//------------------------- private: OnLoadPal() ------------------------------
//----------------------------------------------------------------------------

void vuSpecPalette::OnLoadPal(wxCommandEvent &ev)
{
  //open load dialog, ...
  wxFileDialog fd(this,"Choose a palette","","","*.pal",wxOPEN);
  if(fd.ShowModal() == wxID_OK)
  {
      cout<<"loading... "<<fd.GetPath()<<endl;
      try {
	  if(m_Palette->load(fd.GetPath()))
	    {
		cout<<"successful."<<endl;
	    } else {
		cout<<"failed."<<endl;
	    }
      } catch (char *msg) {
	  printf("%s\n",msg);
      }
      if(getNLights()==0) m_Palette->addLight(vuColour31a(1.0f));
      if(getNRefls()==0) m_Palette->addReflectance(vuColour31a(1.0f));
      m_SelRefl = 0;
      m_SelLight = -1;
      UpdateWidgets();
  }
}

//----------------------------------------------------------------------------
//------------------------- private: OnSavePal() ------------------------------
//----------------------------------------------------------------------------

void vuSpecPalette::OnSavePal(wxCommandEvent &ev)
{
  //open save dialog, ...
  wxFileDialog fd(this,"Save Palette","","","*.pal",wxSAVE|wxOVERWRITE_PROMPT);
  if(fd.ShowModal() == wxID_OK)
    {
      cout<<"saving... "<<fd.GetPath()<<endl;
      try {
	  if(m_Palette->save(fd.GetPath()))
	      cout<<"successful."<<endl;
	  else
	      cout<<"failed."<<endl;
      } catch (char *msg) {
	  printf("%s\n",msg);
      }
    }
}

void vuSpecPalette::OnLoadSP(wxCommandEvent &ev)
{
  if(!validSpecSelected()) return;
    //open load dialog, ...
  wxFileDialog fd(this,"load spectrum","","","*.spc",wxOPEN);
  if(fd.ShowModal() == wxID_OK)
    {
      try {
	  if(!m_Palette->loadSpectrum(m_SelRefl,m_SelLight,fd.GetPath()))
	  {
	      cerr<<"failed to load spectrum."<<endl;
	  }
      } catch (char *msg) {
	  cerr << msg << endl;
      }
      UpdateWidgets();
    }
}

void vuSpecPalette::OnSaveSP(wxCommandEvent &ev)
{
  if(!validSpecSelected()) return;
  //open save dialog, ...
  wxFileDialog fd(this,"Save spectrum","","","*.spc",
		  wxSAVE|wxOVERWRITE_PROMPT);
  if(fd.ShowModal() == wxID_OK)
    {
      cout<<"saving... "<<fd.GetPath()<<endl;
      try {
	  if(m_Palette->saveSpectrum(m_SelRefl,m_SelLight,fd.GetPath()))
	      cout<<"successful."<<endl;
	  else
	      cout<<"failed."<<endl;
      } catch (char *msg) {
	  printf("%s\n",msg);
      }
    }
}

/** Multiply by value found in m_SScale */
void vuSpecPalette::OnMultiplyScalar(wxCommandEvent &ev)
{
	if(validSpecSelected()) {
	    double val;
	    m_SScale->GetValue().ToDouble(&val);
	    m_Palette->getSpec(m_SelRefl,m_SelLight) *= val;
		OnColourClick(m_SelRefl,m_SelLight,CC_GET);
	}
}

/** Divide by value found in m_SScale */
void vuSpecPalette::OnDivideScalar(wxCommandEvent &ev)
{
	if(validSpecSelected()) {
	    double val;
	    m_SScale->GetValue().ToDouble(&val);
		if(val!=0.0) {
			m_Palette->getSpec(m_SelRefl,m_SelLight) /= val;
			OnColourClick(m_SelRefl,m_SelLight,CC_GET);
		}
	}
}

/** select a spectrum
    if indices are too high, new entries will be created */
void vuSpecPalette::selectSpec(int rid, int lid)
{
    if((rid==-1) ^ (lid==-1) &&
       (rid<(int)getNRefls()) &&lid<(int)getNLights())
    {
		OnColourClick(rid,lid,CC_GET);
    }
}

const SPalette& vuSpecPalette::getSPalette() const
{
    return *m_Palette;
}

//-----------------------------------------------------------------------------
//  stuff copied from wxColourDialog
//-----------------------------------------------------------------------------

void vuSpecPalette::CalculateMeasurements()
{
    m_ColourSize.x = 40; //18;
    m_ColourSize.y = 40; //14;
    m_GridSpacing = 15;
    m_SectionSpacing = 15;
    m_Border.x = m_Border.y = 10;
        
    // make additional row and column for pure light/relf colours
    m_PalRect.width = ( ((getNLights()+1)*m_ColourSize.x)
			    + ((getNLights())*m_GridSpacing)
			    + 2*m_Border.x);
    m_PalRect.height = ( ((getNRefls()+1)*m_ColourSize.y)
			     + ((getNRefls())*m_GridSpacing) 
			     + 2*m_Border.y);
    
    m_DColRect.width = m_ColourSize.x+ 3*m_Border.x;
    m_DColRect.height = m_ColourSize.y*2+m_GridSpacing
	+ 2*m_Border.y + 100;
    

    m_DiagRect.width = 200;
    m_DiagRect.height = 100;
}

void vuSpecPalette::CreateWidgets()
{
    wxBeginBusyCursor();

    int sliderX = m_DColRect.width + m_SectionSpacing;
#if defined(__WXMOTIF__)
    int sliderSpacing = 65;
    int sliderHeight =  130;
#else
    int sliderSpacing = 65;
    int sliderHeight =  130;
#endif

// create objects ------------------------------
    wxSize size(100,30);
    wxSize tbsize(50,30);
    m_SmoothW = new wxTextCtrl(this,idWSMOOTH,"0.00", 
			       wxDefaultPosition,tbsize,
			       wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			       wxTextValidator(wxFILTER_NUMERIC));
    m_ErrorW = new wxTextCtrl(this,idWERROR,"0.00", 
			      wxDefaultPosition,tbsize,
			      wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			      wxTextValidator(wxFILTER_NUMERIC));
    m_SP_name = new wxTextCtrl(this,idSPNAME,"<name>", 
			       wxDefaultPosition,size,
			       wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			       wxTextValidator(wxFILTER_ASCII));
    m_SP_upperBound = new wxTextCtrl(this,idSPUB,"10000", 
				     wxDefaultPosition,tbsize,
				     wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
				     wxTextValidator(wxFILTER_NUMERIC));
    m_SP_lowerBound = new wxTextCtrl(this,idSPLB,"0", 
				     wxDefaultPosition,tbsize,
				     wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
				     wxTextValidator(wxFILTER_NUMERIC));
    m_FC_upperBound = new wxTextCtrl(this,idFCUB,"1.00", 
				     wxDefaultPosition,tbsize,
				     wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
				     wxTextValidator(wxFILTER_NUMERIC));
    m_FC_lowerBound = new wxTextCtrl(this,idFCLB,"0.00", 
				     wxDefaultPosition,tbsize,
				     wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
				     wxTextValidator(wxFILTER_NUMERIC));
    m_SScale = new wxTextCtrl(this,-1,"1.00", 
			       wxDefaultPosition,tbsize,
			       wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			       wxTextValidator(wxFILTER_NUMERIC));
    m_SP_Norm = new wxTextCtrl(this, -1,"0.00", 
			       wxDefaultPosition,tbsize,
			       wxTE_READONLY);
			       //wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			       //wxTextValidator(wxFILTER_NUMERIC));
    m_PlanckT  = new wxTextCtrl(this,-1,"6500", 
			       wxDefaultPosition,tbsize,
			       wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			       wxTextValidator(wxFILTER_NUMERIC));
    m_SP_Y   = new wxTextCtrl(this, -1,"0.00", 
			      wxDefaultPosition,tbsize,
			      wxTE_READONLY);
			      //wxTE_PROCESS_ENTER|wxTE_PROCESS_TAB,
			      //wxTextValidator(wxFILTER_NUMERIC));
    
    m_CompSlider[SP_RED] = new wxSlider(this, idRED_SLIDER, 0, 0, 255,
					wxPoint(sliderX, 10), 
					wxSize(-1, sliderHeight), 
					wxVERTICAL|wxSL_LABELS);
    m_CompSlider[SP_GREEN] = new wxSlider(this, idGREEN_SLIDER, 0, 0, 255,
					  wxPoint(sliderX+sliderSpacing, 10), 
					  wxSize(-1, sliderHeight), 
					  wxVERTICAL|wxSL_LABELS);
    m_CompSlider[SP_BLUE] = new wxSlider(this, idBLUE_SLIDER, 0, 0, 255,
					 wxPoint(sliderX+2*sliderSpacing,10), 
					 wxSize(-1, sliderHeight), 
					 wxVERTICAL|wxSL_LABELS);
    m_CompSlider[SP_WEIGHT] = new wxSlider(this, idWEIGHT_SLIDER, 255, 0, 255,
					 wxPoint(sliderX+2*sliderSpacing,10), 
					 wxSize(-1, sliderHeight), 
					 wxVERTICAL|wxSL_LABELS);
    m_DC_FreeCol = new wxListBox(this, idDCFC, wxDefaultPosition,
				wxSize(100,75), 0, NULL, wxLB_SINGLE);
    m_DC_FreeCol->Append("<none>");
    m_DC_FreeCol->Append("A");
    m_DC_FreeCol->Append("B");
    
    m_DC_design = new wxCheckBox(this,idDCDES,"use colour");
    m_DC_design->SetValue(false);
    m_SP_design = new wxCheckBox(this,idSPDES,"design spectrum");
    m_SP_design->SetValue(false);
    m_SP_useBounds = new wxCheckBox(this,idSPUSEB,"use bounds");
    m_SP_useBounds->SetValue(true);
    m_FC_useBounds = new wxCheckBox(this,idFCUSEB,"use bounds");
    m_FC_useBounds->SetValue(true);
    m_useV7 = new wxCheckBox(this,idUSEV7,"optimize V7");
    m_Palette->useV7(false);
    m_useV7->SetValue(false);
    //sizers used as place holders
    m_DiagSpacer = new wxBoxSizer(wxHORIZONTAL);
    m_DiagSpacer->Add(1,1);
    //m_DiagSpacer->Add((int)m_DiagRect.width,(int)m_DiagRect.height);
    m_DColSpacer = new wxBoxSizer(wxHORIZONTAL);
    m_DColSpacer->Add(1,1);
    m_PalSpacer = new wxBoxSizer(wxHORIZONTAL);
    m_PalSpacer->Add(1,1);
    
// create sizers ------------------------------
    wxBoxSizer *leftSizer = new wxBoxSizer(wxVERTICAL );
    wxBoxSizer *rightSizer = new wxBoxSizer(wxVERTICAL );

// fill left sizer ------------------------------
    wxSizer* palette = new wxFlexGridSizer(2,2,2);
    palette->Add(1,1);
    palette->Add(new wxStaticText(this,-1,"lights"),
		 0,wxALL|wxALIGN_CENTER,1);
    palette->Add(new wxStaticText(this,-1,"reflectances"),
		 0,wxALL|wxALIGN_CENTER,1);
    palette->Add(m_PalSpacer,0,wxALL,3);
    
    leftSizer->Add(palette,0,wxRIGHT, 20);
    wxSizer *spnameSizer = new wxBoxSizer(wxHORIZONTAL);
    spnameSizer->Add( m_SP_name, 0, wxALL, 10 );
    spnameSizer->Add( m_SP_design, 0, wxALL, 10 );
    leftSizer->Add(spnameSizer);

	wxSize buttonSize(wxButton::GetDefaultSize());
	buttonSize.SetWidth(120);
    wxSizer *lbSizer = new wxBoxSizer(wxHORIZONTAL);
    wxSizer *llSizer = new wxBoxSizer(wxVERTICAL);
	wxSizer *loadSaveSizer = new wxGridSizer(2, 10, 10);
    loadSaveSizer->Add( new wxButton(this, idLOADSP, _("Load Spectrum"), 
						       wxDefaultPosition, buttonSize ));
    loadSaveSizer->Add( new wxButton(this, idSAVESP, _("Save Spectrum"), 
							   wxDefaultPosition, buttonSize  ));
    loadSaveSizer->Add( new wxButton(this, idADDREF, _("Add Reflectance"), wxDefaultPosition, buttonSize  ));
    loadSaveSizer->Add( new wxButton(this, idADDLIG, _("Add Light"), wxDefaultPosition, buttonSize  ));
	llSizer->Add( loadSaveSizer );

	wxSizer *spInfoSizer = new wxBoxSizer(wxHORIZONTAL);
	llSizer->Add(spInfoSizer);
    spInfoSizer->Add(m_DiagSpacer, 0, wxBOTTOM|wxTOP,10); 	// space for diagram

	wxSizer *spInfo = new wxFlexGridSizer( 3 );
	spInfo->Add(new wxStaticText(this, -1, _("Luminance")), 0, 
		    wxALIGN_RIGHT|wxLEFT, 5);
	spInfo->Add(m_SP_Y, 0, wxALIGN_LEFT, 10);
	spInfo->Add( new wxButton( this, idNORMLUM, _("1"), wxDefaultPosition, wxSize(20, buttonSize.GetHeight())), 0, 
		     wxLEFT|wxALIGN_LEFT, 5 );

	spInfo->Add(new wxStaticText(this, -1, _("Norm")), 0, wxALIGN_RIGHT|wxLEFT, 5);
	spInfo->Add(m_SP_Norm, 0, wxALIGN_LEFT, 10);
	spInfo->Add( new wxButton( this, idNORMNORM, _("1"), wxDefaultPosition, wxSize(20, buttonSize.GetHeight())), 0, 
		     wxLEFT|wxALIGN_LEFT, 5 );

	spInfo->Add( new wxButton( this, idSETPLANCK, _("Make Planck")), 0, wxLEFT|wxALIGN_CENTRE, 10 );
	spInfo->Add(m_PlanckT, 0, wxALIGN_LEFT, 10);
	spInfo->Add(new wxStaticText(this, -1, _("K")), 0, wxALIGN_LEFT|wxLEFT, 5);

	spInfoSizer->Add( spInfo, 0, wxLEFT|wxBOTTOM|wxALIGN_BOTTOM, 10  );

	wxSizer *scaleSizer = new wxBoxSizer(wxHORIZONTAL);
	scaleSizer->Add( new wxStaticText(this,-1,_("scale by")), 0, wxALIGN_CENTRE, 10 );
	scaleSizer->Add( m_SScale, 0, wxLEFT|wxALIGN_CENTRE, 10 );
	scaleSizer->Add( new wxButton( this, idMUL, _("MUL")), 0, wxLEFT|wxALIGN_CENTRE, 10 );
	scaleSizer->Add( new wxButton( this, idDIV, _("DIV")), 0, wxLEFT|wxALIGN_CENTRE, 10 );
	llSizer->Add( scaleSizer, 0, wxTOP, 5 );

	wxSizer *lrSizer = new wxFlexGridSizer( 2 );
    lrSizer->Add( 1, 1);
    lrSizer->Add( m_SP_useBounds );
    lrSizer->Add(new wxStaticText(this,-1,"upper bound"),
		 0,wxALIGN_RIGHT|wxTOP,5);
    lrSizer->Add( m_SP_upperBound );
    lrSizer->Add(new wxStaticText(this,-1,"lower bound"),
		 0,wxALIGN_RIGHT|wxTOP,5);
    lrSizer->Add( m_SP_lowerBound );
    lrSizer->Add(new wxStaticText(this,-1,"smoothness"),
		 0,wxALIGN_RIGHT|wxTOP,5);
    lrSizer->Add( m_SmoothW );
    lrSizer->Add(new wxStaticText(this,-1,"error min."),
		 0,wxALIGN_RIGHT|wxTOP,5);
    lrSizer->Add( m_ErrorW );
    lrSizer->Add(1,1);
    lrSizer->Add( m_useV7 );
    lbSizer->Add(llSizer, 0, wxALL, 10);
    lbSizer->Add(lrSizer, 0, wxALL, 10);
    leftSizer->Add(lbSizer);
        
// fill right sizer ------------------------------
	wxSizer *colGetSizer = new wxGridSizer(2,10,10);
    colGetSizer->Add( new wxButton(this, idDCGET, _("Fit Colour"), 
							   wxDefaultPosition, buttonSize));
    colGetSizer->Add( new wxButton(this, idDCGETALL, _("Fit All"), 
							   wxDefaultPosition, buttonSize));
	rightSizer->Add( colGetSizer, 0, wxTOP, 10 );

    wxSizer *colourSizer = new wxBoxSizer(wxHORIZONTAL);
    rightSizer->Add(colourSizer);
	wxSizer *lcolourSizer = new wxBoxSizer(wxVERTICAL);
	colourSizer->Add(lcolourSizer);
	lcolourSizer->Add( m_DC_design, 0, wxTOP, 10 );
    lcolourSizer->Add(m_DColSpacer);
    colourSizer->Add(m_CompSlider[SP_RED],0,wxALL,10);
    colourSizer->Add(m_CompSlider[SP_GREEN],0,wxALL,10);
    colourSizer->Add(m_CompSlider[SP_BLUE],0,wxALL,10);
    colourSizer->Add(m_CompSlider[SP_WEIGHT],0,wxALL,10);
    
	//lines don't work under windows :-(
	//rightSizer->Add( new wxStaticLine(this,-1,wxDefaultPosition,
	//			   wxSize(200,20),wxLI_HORIZONTAL),
	//	      0, wxLEFT|wxRIGHT, 10 );
	//rightSizer->Add(200,20);

	wxSizer *fcSizer = new wxFlexGridSizer(2);
	wxSizer *lfcSizer = new wxBoxSizer(wxVERTICAL);
	fcSizer->Add(lfcSizer);
    lfcSizer->Add(new wxStaticText(this,-1,"Free Colour"),
				  0,wxALIGN_LEFT|wxALIGN_TOP);
    lfcSizer->Add( m_DC_FreeCol );
    wxSizer *rfcSizer = new wxBoxSizer(wxVERTICAL);
	fcSizer->Add(rfcSizer);
	rfcSizer->Add( m_FC_useBounds, 0, wxTOP|wxBOTTOM, 5 );
	rfcSizer->Add( 1, 1 );
    rfcSizer->Add(new wxStaticText(this,-1,"Upper bound"),
				  0,wxALL|wxALIGN_RIGHT,1);
    rfcSizer->Add( m_FC_upperBound );
    rfcSizer->Add(new wxStaticText(this,-1,"Lower bound"),
				  0,wxALL|wxALIGN_RIGHT,1);
    rfcSizer->Add( m_FC_lowerBound );
	rightSizer->Add( fcSizer );
    //rightSizer->Add( new wxStaticLine(this,-1,wxDefaultPosition,
	//			      wxSize(200,20),wxLI_HORIZONTAL),
	//	     0, wxLEFT|wxRIGHT, 10 );
	rightSizer->Add(200,20);

    // buttons
    wxSizer *buttonSizer = new wxGridSizer( 2 , 10, 10);
    buttonSizer->Add( new wxButton(this, idLOADPAL, _("Load Palette"), wxDefaultPosition, buttonSize  ));
    buttonSizer->Add( new wxButton(this, idSAVEPAL, _("Save Palette"), wxDefaultPosition, buttonSize  ));
    buttonSizer->Add( new wxButton(this, idCREATESP, _("Create Spectra"), wxDefaultPosition, buttonSize ));
    buttonSizer->Add( new wxButton(this, wxID_OK, _("Done"), wxDefaultPosition, buttonSize ));
    rightSizer->Add(buttonSizer, 0, wxBOTTOM|wxRIGHT, 10);

// finish ------------------------------
    m_TopSizer = new wxBoxSizer( wxHORIZONTAL );
    m_TopSizer->Add( leftSizer );
    m_TopSizer->Add( rightSizer );
    //m_TopSizer->Add( rightSizer, 0, wxCENTRE | wxALL, 10 );

// calc layout
    m_TopSizer->RecalcSizes();
    SetAutoLayout( true );
    SetSizer( m_TopSizer );
    m_TopSizer->SetSizeHints( this );
    
    Centre( wxBOTH );
    
    wxEndBusyCursor();

    UpdateWidgets();			// update the layout
}

/** Updates the values in the widgets.
    Call _after_ CreateWidgets() */
void vuSpecPalette::UpdateWidgets()
{
    if(m_SelRefl>=(int)getNRefls() ||
       m_SelLight>=(int)getNLights() ||
       m_SelDesColourR>=(int)getNRefls() ||
       m_SelDesColourL>=(int)getNLights())
    {
	m_SelRefl = 0;
	m_SelLight = -1;
	m_SelDesColourR = m_SelDesColourR = 0;
    }
    
// update values in widgets
    OnColourClick(m_SelRefl, m_SelLight, CC_GET);	
    OnColourClick(m_SelDesColourR, m_SelDesColourL, CC_GET);

// recalc layout
    CalculateMeasurements();
    m_DiagSpacer->Remove(0);
    m_DiagSpacer->Add(m_DiagRect.width,m_DiagRect.height);
    m_DColSpacer->Remove(0);
    m_DColSpacer->Add(m_DColRect.width,m_DColRect.height);
    m_PalSpacer->Remove(0);
    m_PalSpacer->Add(m_PalRect.width, m_PalRect.height);
    
    Layout();	// recalc layout for sizers in this window
    m_TopSizer->Fit(this);	// tell the window to fit the sizer
	Refresh(true);

// get information about new layout
    m_PalRect.x= m_PalSpacer->GetPosition().x + m_Border.x;
    m_PalRect.y= m_PalSpacer->GetPosition().y + m_Border.y;
    m_DiagRect.x= m_DiagSpacer->GetPosition().x + m_Border.x;
    m_DiagRect.y= m_DiagSpacer->GetPosition().y + m_Border.y;
    m_DColRect.x= m_DColSpacer->GetPosition().x + m_Border.x;
    m_DColRect.y= m_DColSpacer->GetPosition().y + m_Border.y;
}

// misc event handlers

//! event handler
#if wxMINOR_VERSION < 5
void vuSpecPalette::OnDCFC(void) {
#else
void vuSpecPalette::OnDCFC(wxCommandEvent&) {
#endif
    int fcid = m_DC_FreeCol->GetSelection()-1;
    m_Palette->attachFreeColour(m_SelDesColourR,m_SelDesColourL,fcid);
        
    bool enable = (fcid!=-1);
    m_FC_useBounds->Enable(enable);
    m_FC_lowerBound->Enable(enable);
    m_FC_upperBound->Enable(enable);
    if(enable) {
	m_FC_useBounds->SetValue(m_Palette->getFreeColBoundState(fcid));
	m_FC_upperBound->SetValue(wxString::Format("%.2f", m_Palette->
						   getFreeColUB(fcid)[0]));
    m_FC_lowerBound->SetValue(wxString::Format("%.2f", m_Palette->
					       getFreeColLB(fcid)[0]));
    } else {
	m_FC_useBounds->SetValue(false);
	m_FC_upperBound->SetValue(_(""));
	m_FC_lowerBound->SetValue(_(""));
    }
}

//! event handler
#if wxMINOR_VERSION < 5
 void vuSpecPalette::OnCBFCUseB(void)
#else
 void vuSpecPalette::OnCBFCUseB(wxCommandEvent&)
#endif
{
    int fcid = m_Palette->getFreeColourID(m_SelDesColourR,
					  m_SelDesColourL);
    if(fcid>=0)
	m_Palette->getFreeColBoundState(fcid) = m_FC_useBounds->GetValue();
}

//! event handler
#if wxMINOR_VERSION < 5
void vuSpecPalette::OnFCLowerBound(void)
#else
void vuSpecPalette::OnFCLowerBound(wxCommandEvent&)
#endif
{
    double val;
    m_FC_lowerBound->GetValue().ToDouble(&val);
    int fcid = m_Palette->getFreeColourID(m_SelDesColourR,
					  m_SelDesColourL);
    if(fcid>=0) {
	m_Palette->getFreeColLB(fcid) = val;
	m_FC_lowerBound->SetValue(wxString::Format("%.2f",val));
    }
}

#if wxMINOR_VERSION < 5
void vuSpecPalette::OnFCUpperBound(void)
#else
void vuSpecPalette::OnFCUpperBound(wxCommandEvent&)
#endif
{
    double val;
    m_FC_upperBound->GetValue().ToDouble(&val);
    int fcid = m_Palette->getFreeColourID(m_SelDesColourR,
					  m_SelDesColourL);
    if(fcid>=0) {
	m_Palette->getFreeColUB(fcid) = val;
	m_FC_upperBound->SetValue(wxString::Format("%.2f",val));
    }
};

//! gets called when the design colour - RGBW changes
#if wxMINOR_VERSION < 5
void vuSpecPalette::OnChangeDCRGBW(void) 
#else
void vuSpecPalette::OnChangeDCRGBW(wxCommandEvent&) 
#endif
{
    if(m_SelDesColourR>=0 && m_SelDesColourL>=0) {
	vuColourRGBa &rgbw = m_Palette->
	    getDesignRGBW(m_SelDesColourR,m_SelDesColourL);
	rgbw[0] = m_CompSlider[SP_RED]->GetValue()/255.0f;
	rgbw[1] = m_CompSlider[SP_GREEN]->GetValue()/255.0f;
	rgbw[2] = m_CompSlider[SP_BLUE]->GetValue()/255.0f;
	rgbw[3] = m_CompSlider[SP_WEIGHT]->GetValue()/255.0f;
    }
}

#if wxMINOR_VERSION < 5
void vuSpecPalette::OnCompSlider(wxScrollEvent& event)
#else
void vuSpecPalette::OnCompSlider(wxCommandEvent& event)
#endif
{
#if wxMINOR_VERSION < 5
    OnChangeDCRGBW();
#else
    wxCommandEvent ev;
    OnChangeDCRGBW(ev);
#endif
    wxClientDC dc(this);
    PaintColours(dc);
    PaintColour(dc);
}

void vuSpecPalette::OnCreatePlanckian(wxCommandEvent &ev)
{
    if(validSpecSelected()) {
	double val;
	m_PlanckT->GetValue().ToDouble(&val);
	vuColour31a &spec = m_Palette->getSpec(m_SelRefl,m_SelLight);
	spec.planckian(val);
	float luminance = vuColourXYZa(spec)[1];
	if(fabs(luminance) > 0.0000001) spec /= luminance;
	OnColourClick(m_SelRefl,m_SelLight,CC_GET);
    }
}

// Internal functions
void vuSpecPalette::OnMouseEvent(wxMouseEvent& event)
{
    int x = (int)event.GetX();
    int y = (int)event.GetY();
    
    if ((x >= m_PalRect.x && 
		x <= (m_PalRect.x + m_PalRect.width)) &&
        (y >= m_PalRect.y && 
		y <= (m_PalRect.y + m_PalRect.height)))
    {
		int indX = (int)(x - m_PalRect.x) / 
		  (m_ColourSize.x + m_GridSpacing);
		int indY = (int)(y - m_PalRect.y) /
			(m_ColourSize.y + m_GridSpacing);
		int selRefl = indY-1;
		int selLight = indX-1;
		if(selRefl<(int)getNRefls() && selLight<(int)getNLights())
		{
			if (event.ButtonDown(1) || event.ButtonDown(3))
			{
				TYPE_COLOUR_CLICK what = event.ButtonDown(1) ? CC_GET : CC_PUT; //LB-get RB-set
				OnColourClick(selRefl,selLight, what);
			} else if (event.LeftDClick())
			{
				OnColourClick(selRefl,selLight, CC_TOGGLE_DESIGN);
			}
		}
	}
}

void vuSpecPalette::OnColourClick(int refl, int light, 
								  TYPE_COLOUR_CLICK what)
{
  wxClientDC dc(this);
  if(light==-1 || refl==-1) {
      if(light*refl!=1) {
		  // a light or a reflectance spectrum has been selected
		  m_SelLight = m_SelRefl = -1;
		  if(light>=0) m_SelLight = light;
		  else m_SelRefl = refl;
		  switch(what) {
		  case CC_GET:
			  {
				  m_SP_name->SetValue(wxString(m_Palette->
							   getSpecName(m_SelRefl,m_SelLight)));
				  m_SP_design->SetValue(m_Palette->
							getSpecDesignState(m_SelRefl,m_SelLight));
				  m_SP_useBounds->SetValue(m_Palette->
							   useSpecBounds(m_SelRefl,m_SelLight));
				  m_SP_upperBound->
					  SetValue(wxString::
						   Format("%.2f",m_Palette->getSpecUB(m_SelRefl,
											  m_SelLight)[0]));
				  m_SP_lowerBound->
					  SetValue(wxString::Format
						   ("%.2f", m_Palette->getSpecLB(m_SelRefl,
										 m_SelLight)[0]));
				  SVector spec(m_Palette->getSpec(m_SelRefl,m_SelLight));
				  vuColourXYZa specXYZ(m_Palette->getSpec(m_SelRefl,m_SelLight));
				  m_SP_Norm->SetValue(wxString::Format("%.2f", spec.norm()));
				  m_SP_Y->SetValue(wxString::Format("%.2f", specXYZ[1]));
			  }
			  break;
		  case CC_PUT: {
#if wxMINOR_VERSION < 5
			  OnSPName();
			  OnCBSPUseB();
			  OnCBSPDesign();
			  OnSPLowerBound();
			  OnSPUpperBound();
#else
    			  wxCommandEvent ev;
			  OnSPName(ev);
			  OnCBSPUseB(ev);
			  OnCBSPDesign(ev);
			  OnSPLowerBound(ev);
			  OnSPUpperBound(ev);
#endif
		  }
		  break;
		  case CC_TOGGLE_DESIGN:
				if(validSpecSelected()) {
					bool &des = m_Palette->
							getSpecDesignState(m_SelRefl,m_SelLight);
					des = !des;
					m_SP_design->SetValue(des);
				}
			  break;
		  }
      }
  } else
  {
	  // a design colour has been selected
      m_SelDesColourR = refl;
      m_SelDesColourL = light;
      switch(what)
	  {
	  case CC_GET:
		  {
			  m_DC_design->SetValue(m_Palette->getDesignState(m_SelDesColourR, 
									  m_SelDesColourL));
			  vuColourRGBa &rgbw = m_Palette->
				  getDesignRGBW(m_SelDesColourR, m_SelDesColourL);
			  m_CompSlider[SP_RED]->SetValue(byte(255.0f*rgbw[0]));
			  m_CompSlider[SP_GREEN]->SetValue(byte(255.0f*rgbw[1]));
			  m_CompSlider[SP_BLUE]->SetValue(byte(255.0f*rgbw[2]));
			  m_CompSlider[SP_WEIGHT]->SetValue(byte(255.0f*rgbw[3]));
			  m_DC_FreeCol->
			  SetSelection(m_Palette->getFreeColourID(m_SelDesColourR, 
								  m_SelDesColourL)+1);
#if wxMINOR_VERSION < 5
			  OnDCFC();	// will update all 'free colour' stuff
#else
			  wxCommandEvent ev;
			  OnDCFC(ev);	// will update all 'free colour' stuff
#endif
		  }
		  break;
	  case CC_PUT: {
#if wxMINOR_VERSION < 5
		  OnCBDCDesign();
		  OnDCFC();
		  OnChangeDCRGBW();
#else
		  wxCommandEvent ev;
		  OnCBDCDesign(ev);
		  OnDCFC(ev);
		  OnChangeDCRGBW(ev);
#endif
	  }
	  break;
	  case CC_TOGGLE_DESIGN:
			{
				bool &destate = m_Palette->getDesignState(m_SelDesColourR, m_SelDesColourL);
				destate = !destate;
				m_DC_design->SetValue(destate);
			} 
		  break;
      }
  }
  
  PaintColours(dc);
  PaintColour(dc);
  PaintSpectrum(dc);
}

void vuSpecPalette::OnPaint(wxPaintEvent& event)
{
#if !defined(__WXMOTIF__) && !defined(WIN32)
  wxDialog::OnPaint(event);
#endif
  //Clear();

  wxPaintDC dc(this);

  PaintColours(dc);
  PaintColour(dc);
  PaintSpectrum(dc);
}

void vuSpecPalette::ClearAndHighlight(wxDC& dc)
{
  dc.BeginDrawing();
  //clear background
  dc.SetPen(*wxTRANSPARENT_PEN);
  //wxBrush lgreyb(wxColour(221,221,221),wxSOLID);
  wxBrush lgreyb(dc.GetBackground());
  dc.SetBrush(lgreyb);	//*wxLIGHT_GREY_BRUSH);
  dc.DrawRectangle(m_PalRect.x-m_GridSpacing,m_PalRect.y-m_GridSpacing,
		   m_PalRect.width+m_GridSpacing,m_PalRect.height+m_GridSpacing);
  
  // Number of pixels bigger than the standard rectangle size
  // for drawing a highlight
  //int deltaX = 2;
  //int deltaY = 2;

  if((m_SelRefl==-1) ^ (m_SelLight==-1))
  {
      int x = m_SelLight+1;
      int y = m_SelRefl+1;

    x = (x*(m_ColourSize.x + m_GridSpacing) 
	 + m_PalRect.x) - m_GridSpacing/2;
    y = (y*(m_ColourSize.y + m_GridSpacing) 
	 + m_PalRect.y) - m_GridSpacing/2;

    wxColour ibg(200, 200, 250);
	//wxColour bg(dc.GetBackground().GetColour());
	//wxColour ibg(255-bg.Red(),255-bg.Green(),255-bg.Blue());
    wxBrush specbg(ibg,wxSOLID);
    dc.SetPen(*wxTRANSPARENT_PEN);
    dc.SetBrush(specbg);
    if(m_SelRefl==-1)			//light selected
	dc.DrawRectangle( x, y, 
			  m_ColourSize.x + (1*m_GridSpacing),
			  (getNRefls()+1)*(m_ColourSize.y + m_GridSpacing)
			  +(m_GridSpacing/2));
    else						//reflectance selected
	dc.DrawRectangle( x, y, 
			  (getNLights()+1)*(m_ColourSize.x + m_GridSpacing) +
			  (m_GridSpacing/2), m_ColourSize.y + (1*m_GridSpacing));
  }

  if(m_SelDesColourR!=-1 || m_SelDesColourL!=-1)
  {
      int x = m_SelDesColourL+1;
      int y = m_SelDesColourR+1;

    x = (x*(m_ColourSize.x + m_GridSpacing) 
	    + m_PalRect.x) - m_GridSpacing/4;
    y = (y*(m_ColourSize.y + m_GridSpacing) 
		+ m_PalRect.y) - m_GridSpacing/4;

    wxBrush selbg(wxColour(100, 255, 0),wxSOLID);
    dc.SetPen(*wxTRANSPARENT_PEN);
    dc.SetBrush(selbg);
    dc.DrawRectangle( x, y, (m_ColourSize.x + m_GridSpacing/2), 
		      (m_ColourSize.y + m_GridSpacing/2));
  }

  dc.EndDrawing();
}

void vuSpecPalette::PaintColours(wxDC& dc)
{
  dc.BeginDrawing();
  ClearAndHighlight(dc);
  
//  wxPen despen(wxColour(255,255,255),1,wxSHORT_DASH);
  wxPen despen(wxColour(255,0,0),2,wxSOLID);
  wxPen nodespen(*wxBLACK_PEN);
  for (dword j = 0; j < getNLights(); j++)
  {
      int x = ((j+1)*(m_ColourSize.x+m_GridSpacing)) + m_PalRect.x;
      int y = (0*(m_ColourSize.y+m_GridSpacing)) + m_PalRect.y;
      vuColourRGBa col(m_Palette->getSpec(-1,j));
      col.clampTo01();
      wxColour wxc(byte(255*col[0]),byte(255*col[1]),byte(255*col[2]));
      dc.SetPen(m_Palette->getSpecDesignState(-1,j)?despen:nodespen);
      wxBrush brush(wxc, wxSOLID);
      dc.SetBrush(brush);
      dc.DrawRectangle( x, y, 
			m_ColourSize.x, m_ColourSize.y);
  }
  
  for (dword i = 0; i < getNRefls(); i++)
  {
      int x = (0*(m_ColourSize.x+m_GridSpacing)) + m_PalRect.x;
      int y = ((i+1)*(m_ColourSize.y+m_GridSpacing)) + m_PalRect.y;
      vuColourRGBa col(m_Palette->getSpec(i,-1));
      col.clampTo01();
      wxColour wxc(byte(255*col[0]),byte(255*col[1]),byte(255*col[2]));
      dc.SetPen(m_Palette->getSpecDesignState(i,-1)?despen:nodespen);
      wxBrush brush(wxc, wxSOLID);
      dc.SetBrush(brush);
      dc.DrawRectangle( x, y, 
			m_ColourSize.x, m_ColourSize.y);
  }

  int halfx = 2*m_ColourSize.x/3;	//not really half
  int halfy = m_ColourSize.y/2;
  for (dword i = 0; i < getNRefls(); i++)
  {
      for (dword j = 0; j < getNLights(); j++)
      {
	  int x = ((j+1)*(m_ColourSize.x+m_GridSpacing)) 
	      + m_PalRect.x;
	  int y = ((i+1)*(m_ColourSize.y+m_GridSpacing)) 
	      + m_PalRect.y;
	  
	  dc.SetPen(*wxTRANSPARENT_PEN);
	  vuColourRGBa ac31_rgb(m_Palette->getRLColour(i,j,false));
	  ac31_rgb.clampTo01();
	  wxColour wxc(byte(255.0f*ac31_rgb[0]),
		       byte(255.0f*ac31_rgb[1]),
		       byte(255.0f*ac31_rgb[2]));
	  wxBrush abrush(wxc, wxSOLID);
	  dc.SetBrush(abrush);
	  dc.DrawRectangle( x, y, 
			    halfx, m_ColourSize.y-halfy+1);
	  
	  vuColourRGBa ac7_rgb(m_Palette->getRLColour(i,j,true));
	  ac7_rgb.clampTo01();
	  wxColour bwxc(byte(255.0f*ac7_rgb[0]),
		       byte(255.0f*ac7_rgb[1]),
		       byte(255.0f*ac7_rgb[2]));
	  wxBrush bbrush(bwxc, wxSOLID);
	  dc.SetBrush(bbrush);
	  dc.DrawRectangle( x, y+halfy, 
			    halfx, m_ColourSize.y-halfy);

	  vuColourRGBa &rgbw = m_Palette->getDesignRGBW(i,j);
	  rgbw.clampTo01();
	  wxColour dwxcol(byte(255.0f*rgbw[0]),
			  byte(255.0f*rgbw[1]),
			  byte(255.0f*rgbw[2]));
	  wxBrush dbrush(dwxcol, wxSOLID);
	  dc.SetBrush(dbrush);
	  
	  dc.DrawRectangle( x+halfx-1, y, 
			    m_ColourSize.x-halfx+1, m_ColourSize.y);
	  dc.SetBrush(*wxTRANSPARENT_BRUSH);
	  dc.SetPen(m_Palette->getDesignState(i,j) ? despen : nodespen);
	  dc.DrawRectangle( x, y, m_ColourSize.x, m_ColourSize.y);
      }
  }
  dc.SetPen(wxNullPen);
  dc.SetBrush(wxNullBrush);
  dc.EndDrawing();
}

void vuSpecPalette::PaintColour(wxDC& dc)
{
  dc.BeginDrawing();

  dc.SetPen(*wxBLACK_PEN);

  wxColour col(m_CompSlider[SP_RED]->GetValue(),
	       m_CompSlider[SP_GREEN]->GetValue(),
	       m_CompSlider[SP_BLUE]->GetValue());

  wxBrush *brush = new wxBrush(col, wxSOLID);
  dc.SetBrush(*brush);

  dc.DrawRectangle( m_DColRect.x, m_DColRect.y,
                    m_ColourSize.x, m_ColourSize.y);

  dc.SetBrush(wxNullBrush);
  delete brush;

  dc.EndDrawing();
}

void vuSpecPalette::PaintSpectrum(wxDC& dc)
{
  dc.BeginDrawing();

  dc.SetPen(*wxBLACK_PEN);

  //wxBrush *brush = new wxBrush(wxColour(200,150,150), wxSOLID);
  dc.SetBrush(*wxWHITE_BRUSH);

  dc.DrawRectangle( m_DiagRect.x, m_DiagRect.y,
                    m_DiagRect.width, m_DiagRect.height);

  dc.SetBrush(wxNullBrush);
  //delete brush;

  if(validSpecSelected()) {
      vuColour31a &spec = m_Palette->getSpec(m_SelRefl,m_SelLight);
      float max = spec.maxComponent();
      float min = spec.minComponent();
      if( max == min ) {	// constant spectrum?
		  if( max>0 ) {
			  max *= 1.1;
		  } else if ( max<0 ) {
			  min *= 1.1;
		  } else {  // spectrum is constant 0
			  max=1;
			  min=-1;
		  } 
      }
      max = 0>max?0:max;
      min = 0<min?0:min;
      
      float sy = float(m_DiagRect.height)/(max-min);
      float sx = float(m_DiagRect.width)/31;
      int middle = m_DiagRect.y+int(max*sy);
      wxBrush lightredbrush(wxColour(255,230,230),wxSOLID);
      dc.SetBrush(lightredbrush);
      dc.SetPen(*wxTRANSPARENT_PEN);
      dc.DrawRectangle( m_DiagRect.x+1, middle,
			m_DiagRect.width-2, m_DiagRect.height-1
			-int(max*sy));
      dc.SetPen(*wxGREY_PEN);
      dc.DrawLine(m_DiagRect.x, middle,
		  m_DiagRect.x+m_DiagRect.width, middle);

      wxPen greenpen(wxColour(0,100,0),1,wxSOLID);
      dc.SetPen(greenpen);
      wxPoint points[31];
      for(int c=0;c<31;c++)
	  points[c] = wxPoint(m_DiagRect.x+int(sx*c),middle-int(sy*spec[c]));
      dc.DrawSpline(31,(wxPoint*)&points);
      
      dc.SetPen(wxNullPen);
      dc.SetBrush(wxNullBrush);
	
	  //draw scale information
	  char number[1024];
	  wxCoord wx, wy;
#if wxMINOR_VERSION < 5
	  wxFont &oldFont = dc.GetFont();
#else
	  wxFont oldFont = dc.GetFont();
#endif
	  wxFont font(6,wxSWISS,wxNORMAL,wxLIGHT);
	  //dc.SetFont(*wxSMALL_FONT);
	  dc.SetFont(font);
	  sprintf(number,"%3.2f",max);
	  dc.DrawText(wxString(number),m_DiagRect.x,m_DiagRect.y);
	  sprintf(number,"%3.2f",min);
	  dc.GetTextExtent(number,&wx,&wy);
	  dc.DrawText(wxString(number),m_DiagRect.x,m_DiagRect.y+m_DiagRect.height-wy);

	  dc.SetFont(oldFont);
  }
  
  dc.EndDrawing();
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------

