//!A specialization of the wxGLCanvas to our project.

/*! The vuGLCanvas class implements a special wxGLCanvas
    for use with the vuVolume project. The wxGLCanvas class
    is very barebone, and there are a lot of common things
    you have to program when using it.  This class makes
    as high level of an OpenGL canvas as possible, abstracting
    the methods into simple init() and render() commands.  Also,
    the issues with painting to windows and initializing openGL
    are taken care of and hidden from the callers.
*/

/*! Steven Bergner, 02Oct01: direct framebuffer access
    useOpenGL(true/false) can deactivate OpenGL drawing context
    then static vuGLCanvas functions provide basic drawing tools
*/

/*! Steven Bergner, 30May02: redraw() now just triggers PAINT event,
    not calling render() directly. This caused problems in windows.
 */


#ifndef _vuGLCanvas_h_
#define _vuGLCanvas_h_

#include <wx/glcanvas.h>

class vuGLCanvas : public wxGLCanvas
{
public:
    //!Initializes the canvas with the same parameters as the wxGLCanvas parent.
    /*!The constructor acts in the same way as the wxGLCanvas constructor, with
       the same parameters.  Any open gl initialization commands must be made
       by overriding the glInit() method.

       \note Even though the glCanvas is constructed after this call, it is
             not necessarily the case that openGL has been fully initialized
             for the canvas.  This is because under some windowing systems,
             the opengl classes are not fully created until after the canvas
             has been shown on the screen.  To initialize openGl then, the
             glInit() method must be overridden.  It will get called
             automatically as soon as open gl calls are possible.  Any
             opengl calls before this can result in segmentation faults.
    */
    vuGLCanvas(wxWindow *parent, const wxWindowID id = -1, 
	       const wxPoint& pos = wxDefaultPosition, 
	       const wxSize& size = wxDefaultSize, long style = 0, 
	       const wxString& name = "vuGLCanvas", int* gl_attrib = NULL);

    //!Redraws the contents of the canvas.
    /*!This method is called when the contents of canvas should be redrawn.
       calls Refresh(), which creates a PAINT event, which triggers onPaint()
    */
    void redraw();

    //!Returns the width of the OpenGL canvas.
    int getWidth() const;
    //!Returns the height of the OpenGL canvas.
    int getHeight() const;

    //!activate OpenGL Support
    /*!OpenGL is active in a vuGLCanvas by default. This function is intended
       to reactivate the OpenGL context.
     */
    void enableOpenGL();

    //!deactivate OpenGL Support
    /*!This is useful to hold the drawing canvas free for other drawing 
       functions than OpenGL, e.g. if you want to use the GUI blit function
       instead of glDrawPixels() (which is said to be rather slow on PCs)
       Alternatively you can use vuDrawTools then, which should provide 
       more direct frame buffer access.
     */
    void disableOpenGL();

protected:

    //!Initializes openGL for use in the canvas.
    /*!This method is called by vuGLCanvas as soon as the canvas' open gl
       classes are created, and openGL calls can be made.  All initializiation 
       should be done by overriding this method.
       
       \return True if successful, false if the canvas could not yet be initialized.
    */
    virtual bool glInit() { return true; };

    //!Renders the scene to the canvas using OpenGL.
    /*!This is an interior method for the canvas that encapsulates all the
       rendering calls.  It is called both when the user wants a redraw,
       and when the operating system requests one.  These two cases need
       different preparation, and so are implemented in the redraw() and
       onPaint() methods.  
      
       The method should be overriden and implement all the rendering calls
       to OpenGL.  The two methods that call it set up the needed device 
       contexts, activate the canvas' GL context, and flush the buffers, so no
       setting up is necessary.  The method encapsulates just the openGL calls.
    */
    virtual void render() {};

    //!Alters the OpenGL state after a resize event.
    /*!This method is called by the vuGLCanvas interior OnSize() method when
       it gets a canvas resize event.  The OnSize() method handles the
       interior management related to the resize event, then passes control
       to this method so that any application specific calculation can
       be done.  These are calls such as setting view matrices or updating
       necessary states.
    */
    virtual void resize() {};

private:
    //!Implements the OnPaint event.
    /*!It prepares the window's onPaint() drawing context, as required
       for an onPaint() event. It creates a clientDC for the window,
       as required for drawing to it, calls the render() method,
       then flushes the GL buffers to the canvas.
    */
    void onPaint(wxPaintEvent& event);

    //!Implements the default OnSize event.
    /*!It updates the internal vuGLCanvas variables and resizes the openGL 
       viewport to the new canvas area.  If different behaviour is desired,
       the derived class should implement it's own onSize event handler.
       The onPaint() event is called by wxWindows immediately afterwards,
       so the glCanvas does not need to be explicitly refreshed.
    */
    void onSize(wxSizeEvent& event);
 
    //!Implements the OnEraseBackground event.
    /*!This method actually eliminates the EraseBackground event from the 
       event stack so that the windowing system doesn't erase the window 
       itself.  This causes flickering because the paint event is put right 
       after the background is erased.
    */
    void onEraseBackground(wxEraseEvent &event);

private:
    //!The width of the canvas.
    int m_Width;
    //!The height of the canvas.
    int m_Height;
    //!Whether the canvas has been initialized or not.
    bool m_init;
    //!Whether the OpenGL should be used or not.
    bool m_useOGL;

DECLARE_EVENT_TABLE()
};

#endif
