
#ifndef _vuMainWindow_h_
#define _vuMainWindow_h_

#include "vuUtilityWindow.h"
#include "Volume/volume.h"
#include <wx/list.h>

//Decide on the base class of the main window.  MDI windows under the GTK 
//toolkit do not behave how we want, so under this toolkit use a different
//type of window

#if defined(__WXGTK__)

#include <wx/frame.h>
#define wxPARENTWINDOW wxFrame

#else

#include <wx/mdi.h>
#define wxPARENTWINDOW wxMDIParentFrame

#endif

// We have to move the documentation here that is recognized by doxygen.
//! The main class of the GUI Tool.  It implements the main window and the general functionality.
/*! 
    This class primarily represents to the main window of the GUI Tool; it is 
    created and used as a windowing class.  As such, it creates the main window
    and implements its functions. The implementation of the main window depends
    on the windowing system.  In most systems, the main window is implemented
    as an MDI parent window, and so is derived from the wxWindows 
    wxMDIParentFrame class.  In gtk, the wxWindow's implementation of the mdi 
    architecture is not suitable to the gui tool, and so the main window is 
    implemented as a regular window, deriving from the wxFrame class.
 
    The main window's other task is to manage all the utility windows.  It is 
    responsible for creating them, supplying them with the volume data file, 
    and notifying them of any changes in the data.  All the utility windows
    implement the vuUtilityWindow interface, and this is what the main window
    sees.

    Finally, the class also manages the volume data for the application.  This
    is because the main window corresponds to a viewing session of volume data.
    Therefore the vuMainWindow class keeps track of this data and supplies 
    functions for operating on it.  For many utilities, this simply constitutes
    passing around the data file name.  These are the utilites which need to 
    open their own data set for efficiency reasons.  The main window can also 
    keep track of a shared data set which a number of utilities operate on.

    The vuMainWindow, and so the main program, can be started with command line
    parameters.  When the vuMainWindow initializes itself, it will 
    automatically open up the volume data and create the desired utility 
    windows.  The syntax is as follows

        programname VolumeData Utility1 Utility2 ...

    programName - name of the executable

    VolumeData - the file name of the volume data
    
    Utility# - a list of the utilities to be opened up, by name
*/
class vuMainWindow : public wxPARENTWINDOW
{
public:
    //!The vuMainWindow constructor.  Does default initialization.
    vuMainWindow();
    //!The vuMainWindow destructor.  Responsible for cleaning up memory use.
    ~vuMainWindow();

    //!Initializes and starts the vuMainWindow, with the option of command line parameters.
    /*!This method intializes the windowing environment for the vuMainWindow,
       creates the main window, and starts it.  Once this method has been 
       called, the window will be visible and accepting events from the user.
       The user can then choose to open data and so on.
        
       The main window can be optionally initialized with command line 
       parameters.  These tell it the name of a file containing volume data, 
       and the utility windows to start with.  If the volume file or utility 
       name is invalid, the main window asks the user if it should quit.

       \return True if initialization was successful, false otherwise.
    */
    bool init(int argc = 0, char* argv[] = 0);

public:
    //These functions are called by vuUtilityWindow implementations.

    //!Notifies the vuMainWindow that a change has been made to the volume data.
    /*!This is called by utility windows when they change the data and is 
       necessary for program synchronization.  First of all, the vuMainWindow 
       window will then notify the user of the change.  Also, it will let other
       utility windows know of the change so that they can update their data 
       views.

       \note This method is not used right now, but may be in future expansions.
    */
    void notifyDataChanged();

    //!Notifies the vuMainWindow that a utility window has been closed by the user.
    /*!This is called by utility windows when they have been closed by the 
       user, as opposed to being closed by the vuMainWindow itself.  In GUI 
       environments, this is often done without the main window being aware 
       that the child has been closed.
    */
    void notifyClosed(vuUtilityWindow *Window);

private:
    //These methods control the volume data owned by the program and so it's 
    //opened/closed state.

    //!Opens a volume data file.
    /*!This puts the main window into the opened state.  Menus and functions 
       which depend on volume data are now enabled.  This includes adding 
       utilities, managing windows and so on.

       \param file The name of the volume data file.
       \return True if successful, false if the file does not exist or was invalid.
    */
    bool open(const char *file);

    //!Closes the volume data file.
    /*!This puts the main window into the closed state.  The main window is 
       open but it's empty and has no data.  This implies that no utility 
       windows can be added anymore.  The user can only quit or open up another
       set of data.
    */
    void close();

    //!Tells whether the window is in the opened or closed state.
    /*!\return True if the window is in the opened state.
    */
    bool isOpened();

    //!Saves the volume data to a file.
    /*!
       \param file The name of the file to save the volume data to.
       \return True if successfule, false otherwise.

       \note This method is not used right now, but may be in future expansions.
    */
    bool save(const char *file);

    //!Indicates whether the current volume data is saved to disk.
    /*!This would be false if one of the utilities changed the volume data in 
       memory, and thereby created new data.  By checking this function, the 
       caller can find out whether to provide the option of saving the data to
       disk.  Once the data has been saved, this method again returns true.

       \return True if the data has been saved to disk, false if it has been changed since opening or saving.

       \note This method is not used right now, but may be in future expansions.
    */
    bool isSaved();

private:
    //These are internal methods of the class.

    //! Parses the command line and returns whether it was successful or not.
    bool parseCommandLine(int argc,char *argv[]);
    //! Creates and initializes a utility window of the given Name.
    bool addUtility(const char *Name);
    //! Pops up a file dialog and saves to the chosen file.
    /*!
       \note This method is not used right now, but may be in future expansions.
    */
    void saveData();

private:
    //These methods operate on the window's user interface

    //! Creates and sets up the main window.
    /*! This method sets up the user interface for the main window.  It does not
        put the window into neither the open nor the closed state, that has to be done
        afterwards.
        \return True if initialization was successful.
    */
    bool initWindow();
    //! Puts the window into the opened state.
    /*! This is the state of the window when there is an opened volume data file.
        Under this state, the utitily menu is enabled and the user can add utilities
    */
    void setOpened();
    //! Puts the window into the closed state.
    /*! This is the state of the window when there is no opened volume data file.
        Under this state, the user can only open a volume data file.
    */
    void setClosed();
    //! Creates a utility menu for the opened file type.
    /*! The method checks with the vuUtilityFactory to see what utilities
        are available for the opened file type, and creates a menu that
        lets the user choose from among them.
    */
    wxMenu* createUtilityMenu();

private:
    //These are the wxWindows/toolkit specific declarations.
    //They respond to events in the user interface.

    //!Opens a file dialog and opens the chosen file.
    /*!This method handles the Open Data menu command. */
    void OnOpenData(wxCommandEvent& event);
    //!Opens a file dialog and saves to the chosen file.
    /*!This method handles the Save Data menu command. */
    void OnSaveData(wxCommandEvent& event);
    //!Puts the window in a closed state.
    /*!This method handles the Close Data menu command. */
    void OnCloseData(wxCommandEvent& event);
    //!Adds the utility that the user chose.
    /*!This method handles the Add Utility menu commands. */
    void OnAddUtility(wxCommandEvent &event);
    //!Handles the window resize event.
    void OnSize(wxSizeEvent& event);
    //!Handles the window exit event.
    void OnExit(wxCommandEvent& event);
    //!Handles the window close event.
    void OnClose(wxCloseEvent& event);

    // Declare the wxWindows event table.
    DECLARE_EVENT_TABLE()

private:
    //The data owned by the main window.

    //!A pointer to the opened volume data, if any -- currently unused.
    vuVolume *m_Data;
    //!The name of the file storing the opened volume data 
    wxString m_FileName;
    //!The known file type of the chosen data file.
    wxString m_FileType;

    //!Indicates whether the program is in the open state.
    bool m_Opened;
    //!Indicates whether the opened volume data is saved to the disk.
    bool m_Saved;

    //!A list of the utility windows.
    wxList m_Windows;
};

#endif
