#include "vuMainWindow.h"
#include "vuUtilityFactory.h"
#include <wx/menu.h>
#include <wx/filedlg.h>
#include <wx/msgdlg.h>
#include <wx/choicdlg.h>
#include "vuFileHelper.h"

//----------------------------------------------------------------------------
//------------------------- The vuMainWindow event table ---------------------
//----------------------------------------------------------------------------

enum
{
    MENU_EXIT = 100,
    MENU_OPEN,
    MENU_SAVE,
    MENU_CLOSE,
    MENU_ADD,
    MENU_ADDLAST = MENU_ADD + 100,   //These are reserved for utility id's.
};

BEGIN_EVENT_TABLE(vuMainWindow, wxPARENTWINDOW)
    EVT_MENU(MENU_EXIT, vuMainWindow::OnExit)
    EVT_MENU(MENU_OPEN, vuMainWindow::OnOpenData)
    EVT_MENU(MENU_SAVE, vuMainWindow::OnSaveData)
    EVT_MENU(MENU_CLOSE, vuMainWindow::OnCloseData)
    EVT_MENU_RANGE(MENU_ADD, MENU_ADDLAST, vuMainWindow::OnAddUtility)

    EVT_SIZE(vuMainWindow::OnSize)
    EVT_CLOSE(vuMainWindow::OnClose)
END_EVENT_TABLE()

//----------------------------------------------------------------------------
//------------------------- public: vuMainWindow() ---------------------------
//----------------------------------------------------------------------------

vuMainWindow::vuMainWindow()
{
    m_Opened = false;
    m_Saved = true;
    m_Data = 0;
};

//----------------------------------------------------------------------------
//------------------------- public: ~vuMainWindow() --------------------------
//----------------------------------------------------------------------------

vuMainWindow::~vuMainWindow()
{
    if (isOpened()) close();
};

//----------------------------------------------------------------------------
//------------------------- public: init() -----------------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::init(int argc, char* argv[])
{
	//Create the main window
    if (!initWindow()) return false;
    //Put it in the closed state.
    setClosed();
    //Parse any command line parameters now that the main window is
	//initialized.
    if (!parseCommandLine(argc,argv)) return false;
    //Finally show the main window.
    Show(true);
    return true;
}

//----------------------------------------------------------------------------
//------------------------- private: initWindow() ----------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::initWindow()
    //This method is resposible for laying out and initializing the main window
{
    //Create the window
    bool success = Create(NULL,-1,"vuGUI");
    if (!success) return false;

    // Make a menubar
    // The file menu...
    wxMenu *file_menu = new wxMenu;
    file_menu->Append(MENU_OPEN, "&Open...", "Open a volume data file");
    //file_menu->Append(MENU_SAVE, "&Save As...", "Save volume data to a file");
    file_menu->Append(MENU_CLOSE, "&Close", "Close volume data");
    file_menu->AppendSeparator();
    file_menu->Append(MENU_EXIT, "&Exit", "Exit the program");

    // Add the menus to the menu bar
    wxMenuBar *menu_bar = new wxMenuBar;
    menu_bar->Append(file_menu, "&File");
    menu_bar->Append(createUtilityMenu(),"&Utility");
    SetMenuBar(menu_bar);

    return true;
}

//----------------------------------------------------------------------------
//------------------------- private: setOpened() -----------------------------
//----------------------------------------------------------------------------

void vuMainWindow::setOpened()
    //This method puts the window into the opened state.
{
    //Update the window menu bar
    wxMenuBar *menubar = GetMenuBar();
    menubar->Enable(MENU_OPEN,false);
    //menubar->Enable(MENU_SAVE,true);
    menubar->Enable(MENU_CLOSE,true);

    //Create and add a new utility menu based on the opened file
    int utilindex = menubar->FindMenu("&Utility");
    delete menubar->Replace(utilindex,createUtilityMenu(),"&Utility");

    //Now add the name of the file to the window title.
    const char *name=strrchr(m_FileName,'\\');
    if (name==NULL) name = strrchr(m_FileName,'/');
    if (name==NULL) name = m_FileName;
    else name++;
    wxString title;
    title << "vuGUI - [" << name << "]";
    SetTitle(title);
};

//----------------------------------------------------------------------------
//------------------------- private: setClosed() -----------------------------
//----------------------------------------------------------------------------

void vuMainWindow::setClosed()
    //This method puts the window into the Closed state.
{
    //Update the window menu bar
    wxMenuBar *menubar = GetMenuBar();
    menubar->Enable(MENU_OPEN,true);
    //menubar->Enable(MENU_SAVE,false);
    menubar->Enable(MENU_CLOSE,false);
    menubar->Enable(MENU_ADD,false);
    
    //Now set the title to the generic name.
    SetTitle("vuGUI");
};

//----------------------------------------------------------------------------
//------------------------- private: createUtilityMenu() ---------------------
//----------------------------------------------------------------------------

wxMenu* vuMainWindow::createUtilityMenu()
{
    wxMenu *utility_menu = new wxMenu;
    wxMenu *utility_add_menu = new wxMenu;
    
    //Find out which utilities are available for the file
    wxStringList l_utilities = vuUtilityFactory::listAvailable(m_FileType);
    int num = l_utilities.GetCount();
    char **utilities = l_utilities.ListToArray();
    
    //Add them to the submenu
    utility_add_menu->Append(MENU_ADD+1, "Add Multiple...", 
			     "Add multiple utilities");
    utility_add_menu->AppendSeparator();
    for (int i= 0; i < num; i++)
        utility_add_menu->Append(MENU_ADD+i+2, utilities[i],
          "Add the chosen utility");
    utility_menu->Append(MENU_ADD, "&Add", utility_add_menu, 
			 "Add a utility window");

    return utility_menu;
}

//----------------------------------------------------------------------------
//------------------------- private: addUtility() ----------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::addUtility(const char *Name)
    //Creates and initializes a utility window of the given name.  The window 
    //is opened up and available to the user.  The method returns true if 
    //successful, false if not.  (This would happen if the Name does not refer
    //to an existing utility).
{
	//First check if the program is in the opened state.  This is the only
    //state that utility windows can be added in.
    if (!isOpened()) return false;

    //Ask the utility factory to create the utility window
    vuUtilityWindow *window = vuUtilityFactory::create(Name);
    if (window == 0) return false;
    
    //Initialize the window.
    bool success = window->init(this,m_FileName);
    
    if (success)
        //Add the utility to the utility list
        m_Windows.Append((wxObject *)window);
    else
        window->Destroy();
    
    return success;
}

//----------------------------------------------------------------------------
//------------------------- private: parseCommandLine() ----------------------
//----------------------------------------------------------------------------

bool vuMainWindow::parseCommandLine(int argc,char *argv[])
{
	//Check if there are any parameters given
    if (argc <= 1) return true;

    //Open the volume data.
    bool success = open(argv[1]);
    if (!success) 
    {
        cout <<"Error: Could not read the volume data file " <<argv[1]<< ".\n";
        return false;
    }

    //Now create the desired utility windows.
    //First check if the named utilities are available.
    for (int i = 2; i < argc; i++)
    {
	if (!addUtility(argv[i])) 
        {
            cout <<"Error: Could not initialize the "<<argv[i]<< " utility.\n";
            success = false;
        }
    }
    return success;
}

//----------------------------------------------------------------------------
//------------------------- public: notifyDataChanged() ----------------------
//----------------------------------------------------------------------------

void vuMainWindow::notifyDataChanged()
{
    //Indicate that the data has been changed.
    m_Saved = false;
    
    //Go through all the utility windows and notify them of the change.
    for (size_t i = 0; i < m_Windows.GetCount(); i++)
        ((vuUtilityWindow *) m_Windows.Item(i)->GetData())->notifyDataChanged();
}

//----------------------------------------------------------------------------
//------------------------- public: notifyClosed() ---------------------------
//----------------------------------------------------------------------------

void vuMainWindow::notifyClosed(vuUtilityWindow *window)
{
	//Take the window out of the window list.
	//Note that wxWindows owns the memory, so we don't delete it ourselves!
    m_Windows.DeleteObject((wxObject *)window);
}

//----------------------------------------------------------------------------
//------------------------- protected: open() --------------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::open(const char *file)
{
    //First make sure that the window is in the closed state.
    if (isOpened()) close();

    //Check to see whether the chosen file is useable by the program.
    m_FileName = file;
    m_FileType = vuFileHelper::getFileType(m_FileName);
    cerr << "getFileType=" << m_FileType << endl;
    if (m_FileType.IsEmpty()) return false;

    //*********Open the volume data here.************
    //********This is not used right now.************

    //Update the internal variables. 
    m_Opened = true;
    m_Saved = true;

    //Finally update the user interface.
    setOpened();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- protected: close() -------------------------------
//----------------------------------------------------------------------------

void vuMainWindow::close()
{
    //Check if the window is already in the closed state
    if (!isOpened()) return;

    //Close all the utility windows
    vuUtilityWindow *window;
    while (m_Windows.GetCount() > 0)
    {
        window = (vuUtilityWindow *) m_Windows.GetFirst()->GetData();
        window->close();
	    //Note that wxWindows owns the memory,so we don't delete ourselves!
        m_Windows.DeleteObject((wxObject *)window);
    };
    
    //Update the internal variables.
    if (m_Data != 0)
    {
        delete m_Data;
        m_Data = 0;
    }
    m_FileName = "";
    m_FileType = "";
    m_Opened = false;
    m_Saved = true;
    
    //Finally update the user interface.
    setClosed();
}

//----------------------------------------------------------------------------
//------------------------- protected: isOpened() ----------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::isOpened()
{
    return m_Opened;
}

//----------------------------------------------------------------------------
//------------------------- protected: save() --------------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::save(const char *file)
{
        //*********Save the volume data here.************
        //*********This is not used right now.***********

        //Notify that the data is now saved to file.
    m_Saved = true;
    return true;
}

//----------------------------------------------------------------------------
//------------------------- protected: isSaved() -----------------------------
//----------------------------------------------------------------------------

bool vuMainWindow::isSaved()
{
    return m_Saved;
}

//----------------------------------------------------------------------------
//------------------------- private: saveData() ------------------------------
//----------------------------------------------------------------------------

void vuMainWindow::saveData()
{
        //Ask the user for a file name
    wxFileDialog dialog(this,"Save Volume Data","./","","*.*",
			wxSAVE|wxOVERWRITE_PROMPT);
    if (dialog.ShowModal() == wxID_OK)
    {
            //Save the data file
        bool success = save(dialog.GetPath());
        if (!success)
        {
                //If the file could not be saved, tell the user.
            wxMessageDialog mdlg(this,"Couldn't save to the chosen file name.",
				 "Error",wxOK|wxICON_EXCLAMATION);
            mdlg.ShowModal();
        }
    }
}

//----------------------------------------------------------------------------
//------------------------- private: OnOpenData() ----------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnOpenData(wxCommandEvent& WXUNUSED(event))
{
    //Ask the user for a file name
    wxFileDialog dialog(this,"Open Volume Data", wxGetCwd(), "",
			"CG-TUWien Dat File (*.dat)|*.dat|"
			"Volume Data (*.vud)|*.vud|"
			"Lightfield Data (*.vul)|*.vul|"
			"Frequency Data (*.vuf)|*.vuf",
			wxOPEN);

    if (dialog.ShowModal() == wxID_OK)
    {
        //Open the data file
        bool success = open(dialog.GetPath());
        if (!success)
        {
            //If the file could not be read, tell the user.
            wxString message;
            message << dialog.GetFilename() 
                    << " is not a data file recognized by the program.";
            wxMessageDialog mdlg(this,message,"Error",wxOK|wxICON_EXCLAMATION);
            mdlg.ShowModal();
        }
    }
}

//----------------------------------------------------------------------------
//------------------------- private: OnSaveData() ----------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnSaveData(wxCommandEvent& WXUNUSED(event))
{
    saveData();
}

//----------------------------------------------------------------------------
//------------------------- private: OnCloseData() ---------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnCloseData(wxCommandEvent& WXUNUSED(event))
{
        //Check whether the data has been saved first.
    if (!isSaved())
    {
            //If not then offer the user the chance to save it.
        wxMessageDialog mdlg(this,
        "The volume data has been changed.  Do you want to save the changes?",
        "Save Data",wxYES_NO|wxCANCEL|wxICON_EXCLAMATION);

        int answer = mdlg.ShowModal();
        if (answer == wxID_YES) saveData();
        else if (answer == wxID_CANCEL) return;
    }
    close();
}

//----------------------------------------------------------------------------
//------------------------- private: OnAddUtility() --------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnAddUtility(wxCommandEvent& event)
{
    int id = event.GetId();
    wxString name;

    if (id==MENU_ADD)
        return; //in case there is an event for the Add submenu popping up
    else if (id==MENU_ADD+1)
    {
        wxStringList slist = vuUtilityFactory::listAvailable(m_FileType);
	wxString utilist[slist.GetCount()];
	dword e;
	for(e=0;e<slist.GetCount();e++)
	  utilist[e]=slist[e];
	
        //Bring up a choice dialog.
        wxSingleChoiceDialog dialog(this,
				    wxString("Select which utility to add:"),
				    wxString("Add Utility"),
				    (int)slist.GetCount(),
				    utilist);
        if (dialog.ShowModal() != wxID_OK) return;
        name = dialog.GetStringSelection();
    }
    else
        //Get the name corresponding to the menu choice
        name= vuUtilityFactory::listAvailable(m_FileType).ListToArray()[id-(MENU_ADD+2)];

    addUtility(name);
}
    
//----------------------------------------------------------------------------
//------------------------- private: OnSize() --------------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnSize(wxSizeEvent& WXUNUSED(event))
{
    //If we are working with MDI windows, then the client window needs to be
    //resized when the main window size is changed.
#ifndef __WXGTK__

        //Get the size of the drawing area.
    int w, h;
    GetClientSize(&w,&h);
        //Set the client window to this size
    GetClientWindow()->SetSize(0,0,w,h);

#endif
}

//----------------------------------------------------------------------------
//------------------------- private: OnQuit() --------------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnExit(wxCommandEvent& WXUNUSED(event))
{
    Close();
};

//----------------------------------------------------------------------------
//------------------------- private: OnClose() -------------------------------
//----------------------------------------------------------------------------

void vuMainWindow::OnClose(wxCloseEvent& event)
{
        //Check whether the data has been saved.
    if (event.CanVeto() && !isSaved())
    {
        wxMessageDialog mdlg(this,"The volume data has been changed.  Do you want to save the changes before exiting?",
			     "Save Data",wxYES_NO|wxCANCEL|wxICON_EXCLAMATION);
        int answer = mdlg.ShowModal();
        if (answer == wxID_YES) saveData();
        else if (answer == wxID_CANCEL)
        {
            event.Veto();
            return;
        }
    }
    event.Skip();
}

