/*	Modified April 2002
	Modified by Christopher Steinbach

	Modified to add functions to enable/disable rerendering
	so that animations will play back correctly.
*/

#ifndef _VUVOLUME_H_
#define _VUVOLUME_H_

#include "vuSimpleTypes.h"
#include "vuString.h"
#include "vuVector.h"
#include "vuCamera.h"
#include "vuTFunc/vuTFIntensity.h" //for transfer functions 
#include "vuTFunc/vuTFDesign.h"

//Needed for the FILE* declaration.
#include <stdio.h>


class vu1;
typedef vu1 vuVolume;

//! The root class for all volume data in the vuVolume project.

/*! It defines the most general interface for all possible volume data.
    Most interaction with volume data is through this interface.  The interface
    also defines the actions that are possible on volume data, such as render(),
    read() and write().

    To use the volume tree, a proper leaf class of the tree must be instantiated.
    The class instantiated specifies the type of volume data to be worked with,
    usually up to the type of rendering method desired.  Once instantiated, the
    volume data file name is set with the setFileName() method.  The volume
    data can then be read with the read() method, which also initializes the
    whole class so that it's fully functional.  Finally, the volume data can
    be viewed and operated on using the other class methods, such as render().
*/

class vu1
{
public:
    //!Default constructor.
    /*!Creates an empty volume with no data.
    */
    vu1();
    //!A copy constructor doing a deep copy of the volume data.
    /*!All the volume data is copied to the new class.  This can be both
       time and resource consuming, so be careful when coding.
    */
    vu1(const vu1 &inst);
    //!Destructor.
    virtual ~vu1();

    //!A deep assignment operator.
    /*!All the volume data is copied to the new class.  This can be both
       time and resource consuming, so be careful when coding.
    */
    vu1& operator=(const vu1 &rhs);
    
    //!Sets the file name that the read and write methods access.
    /*!This method must be called before the read and write methods are
       called.  It sets the data file that the instance is associated with.
       If the fileName is set a second time, the read method must be called
       before the new data is read in.
    */
    void setFileName(const char* val);
    //!Returns the file name that the volume class is associated with.
    /*!If no filename has been set for the instance, then 0 is returned.*/
    const char *getFileName(void) const;

    //!Reads volume data from the data file set with setFileName().
    /*!This is the abstract method for reading in all types of volume data.  It
       can only be called once a specific node of the volume tree has been
       instantiated, a node which corresponds to the type of data being read.

       If the read is successful, the function returns true.  If the file does
       not exist or is in an invalid format, the function returns false.  In
       this case, the error information is set for the instance.

       Once the read method is called, the volume instance is fully initialized
       and all the other methods may be called on it.  This includes initializing
       OpenGL for the render method.

       \return True if successful, false otherwise.
    */
    virtual bool read(void) = 0;
    //!Writes volume data to the data file set with setFileName().
    /*!\return True if successful, false otherwise.
    */
    virtual bool write(void) = 0;

    //!Gets the descriptive header for the volume data.
    /*!The header is a one line description of the volume data that is written
       into the file.  It provides a short description of what is stored in the
       data file. This architecture is according to the vtk file type.  For more
       information, see the .vu file format document.
       If no volume data has not been read, the method returns 0.
    */
    const char *getHeader(void) const;

    /** Returns true if the method is allowed to rerender, false otherwise */
    virtual bool IsReRendering ();

    /** Sets the return value of IsReRendering to be the same as the value of isit. */
    virtual void setIsReRendering(bool isit=true);

public:

    //! Gets the size of the data. 
    /*! Might be useful for optimization by case.
     */
    dword getDataSize();

    // Every subclass of volume (i.e. every algorithm) has a transfer function.
    // See comments for volume::m_TFunc.
    //void setTransferFunc(const vuTFDesign& tf);

    //!Sets the render state.
    /*!Currently undefined for the volume tree.*/
    void setRenderState(dword val);
    //!Returns the render state.
    /*!Currently undefined for the volume tree.*/
    dword getRenderState(void) const;

    //!Sets the camera viewing position for the render method.
    /*!\param view The offset of the view from the origin.
       \param up A vector specifying the up direction of the camera.
       \param right A vector specifying the right direction of the camera.
    */
    virtual void setViewVectors(const vuVector& view,const vuVector& up,const vuVector& right)=0;

    //!Renders the volume to OpenGL, using the appropriate render method.
    /*!This is an abstract method that is implemented at the bottom of the
       volume tree according to the specific rendering algorithm defined.
       It can be called after the volume data has been read from a file.  
       The method renders directly to the current OpenGL buffer, which then
       has to be swapped to be shown in a window.
    */
    virtual void render(void) = 0;

    //!Indicates whether an error occurred with the volume class.
    /*!Currently, errors can only arrise from reading and writing volume data.
       \return True if an error has occurred.
    */
    bool errorOccurred(void) const;
    //!Returns the level of the error.
    /*!Errors that can occur have different levels, staring with 0 for no error,
       and increasing according to the severity of the level.
       Currently there are only two error levels, 0 and 1.
    */
    byte getErrorLevel(void) const;
    //!Returns an error message associated with an error that occured, if any.
    /*!This is a user readable error message that can be displayed in an
       error window.
    */
    const char *getErrorMessage(void) const;

protected:
    //!Closes any opened data and clears the error information.
    /*!This method closes all the data for the instance.  The instance still 
       points to the same file as before, but all the volume data is deleted 
       and freed.  This is useful for freeing up memory if the data is no 
       longer needed.  The method also resets any error information that might
       have occurred during reading or processing.
    */
    void close(void);

    //!Reads the general volume info from the file
    /*!This includes the standard vuVolume header, the description header 
       for the data file, and whether the data is stored in binary or ascii
       format.
    */
    virtual bool read(FILE *file);
    //!Writes the general volume info to the file
    /*!This is the same information as outlined for the read() method.
    */
    virtual bool write(FILE *file);

    //!Sets the error flag in the class.
    /*!
       \param Message An error message for the user.
       \param ErrorLevel The level of the error, with increasing severity.
       \return Always returns false for programming conveniance.
    */
    bool setError(const char *Message,byte ErrorLevel = 1);
    //!A shortcut to setting an invalid data format error while reading.
    bool setInvalidFormatError();
    //!A shortcut to setting a writing to a file error.
    bool setWriteError();

protected:
    //!The volume data
    byte* m_Data;
    //!The size of the volume data in bytes.
    dword m_DataSize;

    //!The render state--currently not used.
    dword m_RenderState;

    //!The file name that the volume data is associated with
    vuString m_FileName;
    //!The description header for the volume data
    vuString m_Header;
    //!Whether the volume data file is in binary or ascii format.
    bool m_Binary;

    //!True when an error with the volume data has occurred.
    bool m_Error;
    //!The error message associated with the error.
    vuString m_ErrorMessage;
    //!The severity level of the error.
    byte m_ErrorLevel;

    bool m_rerendering;
    //!The Camera
    vuCamera *m_Camera;

    //The transfer function.
    /*
      The vast majority of subclasses of volume should have a transfer
      function, just like the vast majority of subclasses will need a 
      camera set up. Why declare m_TFunc in every algorithm class and
      reimplement its behavior in every utility window when it can be
      done in one place? To disable volume's transfer function dialog,
      simply overwrite onLeftDoubleClick() in vuBasicUtility.

      This attribute, however, is not yet used, because conceivably,
      each algorithm could demand a variation of the transfer function
      (subclasses), and because current algorithms already follows that
      architecture (though as far as I can see every transfer function is 
      of the same type). This might be something to consider in the next
      architectural revision of vuVolume.
    */
    //vuTFDesign m_TFunc;

public:
    //!Provides a (fast renderable) preview (ms)
    virtual void preview(int hint=0);
    //!Set the camera  (ms)
    void setCamera(vuCamera *mycamera);
    //!Returns the camera (ms)
    vuCamera& getCamera();
    //!Returns a pointer to the camera (ms)
    vuCamera *getCameraPtr();
    //!returns the center of the volume
    virtual vuVector getCenter() const;


    virtual void glInit();
    virtual void glResize(dword width, dword height);
};

#endif
