#include "volume.h"
#include "vuString.h"
#include <stdio.h>
#include <string.h>
#include "vuSimpleTypes.h"

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vu1::vu1()
{
    m_Data        = NULL;
    m_DataSize    = 0;

    m_RenderState = 0;

    m_Error       = false;
    m_ErrorLevel  = 0;

    m_rerendering = true;
    m_Camera      = NULL;
}

//----------------------------------------------------------------------------
//------------------------- The copy constructor -----------------------------
//----------------------------------------------------------------------------

vu1::vu1(const vu1 &inst)
{
    //Copy over the instance data.
    m_DataSize = inst.m_DataSize;
    m_RenderState = inst.m_RenderState;
    m_FileName = inst.m_FileName;
    m_Header = inst.m_Header;
    m_Binary = inst.m_Binary;
    m_Error = inst.m_Error;
    m_ErrorMessage = inst.m_ErrorMessage;
    m_ErrorLevel = inst.m_ErrorLevel;
    
    //Copy the volume data, if any.
    if (inst.m_Data != 0)
    {
        m_Data = new byte[m_DataSize];
        memcpy(m_Data,inst.m_Data,m_DataSize);
    }
    else
        m_Data = 0;
    //Copy the camera, if any
    if (inst.m_Camera != NULL)
    {
        throw "Copy constructor of vu1 not fully implemented yet!";
    }
    else
        m_Camera = NULL;
}

//----------------------------------------------------------------------------
//------------------------- The destructor -----------------------------------
//----------------------------------------------------------------------------

vu1::~vu1()
{
    if (m_Data != NULL) {
      delete [] m_Data;
      m_Data = NULL;
    }
    if (m_Camera != NULL) {
      delete m_Camera;
      m_Camera = NULL;
    }
}

//----------------------------------------------------------------------------
//------------------------- The assignment operator --------------------------
//----------------------------------------------------------------------------

vu1& vu1::operator=(const vu1 &rhs)
{
    if (this != &rhs)
    {
        //Copy over the instance data.
        m_DataSize = rhs.m_DataSize;
        m_RenderState = rhs.m_RenderState;
        m_FileName = rhs.m_FileName;
        m_Header = rhs.m_Header;
        m_Binary = rhs.m_Binary;
        m_Error = rhs.m_Error;
        m_ErrorMessage = rhs.m_ErrorMessage;
        m_ErrorLevel = rhs.m_ErrorLevel;

        //Copy over the volume data..
        if (m_Data != 0) delete [] m_Data;
        if (rhs.m_Data != 0)
        {
            m_Data = new byte[m_DataSize];
            memcpy(m_Data,rhs.m_Data,m_DataSize);
        }
        else
            m_Data = 0;
    }
    return *this;
}

//----------------------------------------------------------------------------
//------------------------- public errorOccurred() ---------------------------
//----------------------------------------------------------------------------

bool vu1::errorOccurred() const
{
    return m_Error;
}

//----------------------------------------------------------------------------
//------------------------- public getErrorLevel() ---------------------------
//----------------------------------------------------------------------------

byte vu1::getErrorLevel() const
{
    return m_ErrorLevel;
}

//----------------------------------------------------------------------------
//------------------------- public getErrorMessage() -------------------------
//----------------------------------------------------------------------------

const char *vu1::getErrorMessage() const
{
    return m_ErrorMessage.c_str();
}

//----------------------------------------------------------------------------
//------------------------- public setTransferFunc() --------------------------
//----------------------------------------------------------------------------

//see comment in .h and for volume::m_TFunc
/*
void vu1::setTransferFunc(const vuTFDesign& tf)  
// Every subclass of volume (i.e. every algorithm) has a transfer function.
{
    m_TFunc = tf;
}
*/

//----------------------------------------------------------------------------
//------------------------- public setRenderState() --------------------------
//----------------------------------------------------------------------------

void vu1::setRenderState(dword val)
{
    m_RenderState = val;
}

//----------------------------------------------------------------------------
//------------------------- public getRenderState() --------------------------
//----------------------------------------------------------------------------

dword vu1::getRenderState(void) const
{
    return m_RenderState;
}

//----------------------------------------------------------------------------
//------------------------- public getDataSize() -----------------------------
//----------------------------------------------------------------------------

dword vu1::getDataSize()
  //Function useful because m_DataSize is protected and cannot always be
  //accessed.
{
  return m_DataSize;
}

//----------------------------------------------------------------------------
//------------------------- public setFileName() -----------------------------
//----------------------------------------------------------------------------

void vu1::setFileName(const char *val)
{
    m_FileName = val;
}

//----------------------------------------------------------------------------
//------------------------- public getFileName() -----------------------------
//----------------------------------------------------------------------------

const char *vu1::getFileName() const
{
    return m_FileName.c_str();
}

//----------------------------------------------------------------------------
//------------------------- public getHeader() -------------------------------
//----------------------------------------------------------------------------

const char *vu1::getHeader() const
{
    return m_Header.c_str();
}

//----------------------------------------------------------------------------
//------------------------- protected close() --------------------------------
//----------------------------------------------------------------------------

void vu1::close()
{
    //Delete the volume data.
    if (m_Data != 0) delete [] m_Data;
    m_Data = 0;
    m_DataSize = 0;
    
    //Clear the data header.
    m_Header.empty();

    //Clear any error information.
    m_Error = false;
    m_ErrorMessage.empty();
    m_ErrorLevel = 0;
}

//----------------------------------------------------------------------------
//------------------------- protected read() ---------------------------------
//----------------------------------------------------------------------------

bool vu1::read(FILE *file)
{
    int len = 0;
    int ret = 0;
    char header[257]="";

    //Read in the standard vuVolume header to assert the file type.
    ret = fscanf(file,"# vu DataFile Version 1.0%n",&len);
    if ((len != 25) || (fgetc(file) != '\n')) return setInvalidFormatError();
    ret = 0;  len = 0;

    //Read in the data header and assign it to the header string.
    if ( fgets(header,257,file) == NULL ) return setInvalidFormatError();
    m_Header = header;

	//There is no m_Header.trim() yet, so
	while(m_Header.getLength()>0 && 
	      m_Header[m_Header.getLength()-1]<=32)
	    m_Header[m_Header.getLength()-1] = 0;
		
    //Read in the format of the file.
    if (( fscanf(file,"ASCII %n",&len) == 0 ) && (len >= 6) )
        m_Binary = false;
    else if (( fscanf(file,"BINARY %n",&len) == 0 ) && (len >= 7) )
        m_Binary = true;
    else 
        return setInvalidFormatError();

    return true;
}

//----------------------------------------------------------------------------
//------------------------- protected write() --------------------------------
//----------------------------------------------------------------------------

bool vu1::write(FILE *file) 
{
    int ret = 0;

    //Write the standard vuVolume header.
    fprintf(file,"# vu DataFile Version 1.0\n");

    //Write the data header.
    if (!m_Header.isEmpty())
        fprintf(file,"%s\n",m_Header.c_str());
    else
        fprintf(file,"Volume Data\n");

    //Write the file format.
    if (m_Binary)
        ret = fprintf(file,"BINARY\n");
    else
        ret = fprintf(file,"ASCII\n");

    if (ret > 0)
        return true;
    else
        return setWriteError();
};

//----------------------------------------------------------------------------
//------------------------- protected setError() -----------------------------
//----------------------------------------------------------------------------

bool vu1::setError(const char *Message, byte ErrorLevel)
{
    m_ErrorMessage = Message;
    m_ErrorLevel = ErrorLevel;
    m_Error = true;

    //Return false for programming conveniance
    return false;
}

//----------------------------------------------------------------------------
//------------------------- protected setInvalidFormatError() ----------------
//----------------------------------------------------------------------------

bool vu1::setInvalidFormatError()
{
    return setError("Unsupported data format.");
}

//----------------------------------------------------------------------------
//------------------------- protected setWriteError() ------------------------
//----------------------------------------------------------------------------

bool vu1::setWriteError()
{
    return setError("Could not write data to the file.");
}

bool vu1::IsReRendering ()
{
	return m_rerendering;
}

void vu1::setIsReRendering (bool isit)
{
	m_rerendering = isit;
}

//----------------------------------------------------------------------------
//------------------------------ public preview() ----------------------------
//----------------------------------------------------------------------------

void vu1::preview(int hint)
{
  // sub implementations can provide a (fast renderable) preview here (ms)
}

void vu1::setCamera(vuCamera *myCamera)
{
  if (m_Camera == myCamera) return;

  if (m_Camera != NULL) delete m_Camera;

  m_Camera = myCamera;
}

vuCamera& vu1::getCamera()
{
  return *m_Camera;
}

vuCamera* vu1::getCameraPtr()
{
  return m_Camera;
}

vuVector vu1::getCenter() const
{
  cerr << "Warning: vu1::getCenter() was not overwritten." << endl;
  return vuVector();
}


void vu1::glInit() 
{
  cerr<<"Warning: glInit unimplemented."<<endl;
}

void vu1::glResize(dword width, dword height)
{
  cerr<<"Warning: glResize default implementation."<<endl;
}

