#ifndef _VUTFINTENSITY_H_
#define _VUTFINTENSITY_H_

#include "vuSimpleTypes.h"
#include "vuColourRGBa.h"
#include "vuMap.h"

//!A base class for transfer functions operating only on the intensity of data.
/*!This class defines the way that intensity transfer functions are used by 
   rendering methods.  In this case, it is with a simple look-up function.  

   The class also sets up the data architecture for storing the transfer 
   function, since this is uniform over all the transfer functions.  To 
   implement a transfer function, you only need to add methods which
   set up the transfer function.  The data is stored in a float array of
   length (255*4).  For each possible intensity value, from 0 to 255, four
   float values are stored, corresponding to the rgba for the intensity.
   The values for the points are stored sequentially, so the architecture is
   (rgba[0]rgba[1]...)

   Without changing the above explanation this class has been modified to handle 
   different colour models and value ranges. TODO: change the above explanation.
*/
class vuTFIntensity
{
public:
    //!Default constructor initializing all colour components (+ alpha) to zero.
    /*! \param ncomp number of components  RGBA should be built with ncomp = 4
      \param range range for which the look up table should be defined */
    vuTFIntensity();
    vuTFIntensity(dword ncomp, dword range);

    /** Copy constructor. */
    vuTFIntensity(const vuTFIntensity& inst);

    /** Destructor.  Frees the memory for the LUT */
    virtual ~vuTFIntensity();

    /**Assignment operator.
       If the transfer function specified in rhs has a different number of components,
       a conversion to RGBa is performed. By now this conversion is only implemented 
       for 31 -> RGB. If the ranges (default 256) don't match, the function does nothing.
     */
    vuTFIntensity& operator=(const vuTFIntensity& rhs);

    //!Access operator for converting intensity to an rgba value.
    /*!\param intensity An intensity value, from 0 to 255.
       \return A pointer to the rgba values for the intensity, stored as an rgba array.
    */
    const float* operator[](dword intensity) const
	{ // inlined to speed it up
	    return &m_Table[intensity*m_NComp];
	}
    
    //! Returns number of components for colour model (RGBA -> 4).
    dword getNComponents()  const { return m_NComp; };
    //! Returns range for which the LUT should be defined (8-bit -> 256).
    dword getRange()  const { return m_Range; };

    /** returns an interpolated colour from the look-up table. */
    virtual void getRGBa(dword i, vuColourRGBa& rgba) const;
    /** returns an interpolated opacity value according to the entry in the look-up table. */
    virtual float getOpacityAtPos(dword i);

    /** changes the size of the lookup table */
    bool resize(dword ncomp, dword range);

    /** returns a float* to the spectral components of the light.
	getNComponents() tells how many components are used. */
    float* getLight() {return m_Light;};
    
    /** Weight all colours by their alpha values and set alpha to one.
	This can be used if you don't want to use alpha values but, still
	want to have weighted colours, eg. for coloured FVR. */
    void normalizeAlphaToOne();

    /** Create a transfer function from a given vuMap.
     The alpha values will all be set to one.*/
    void fromMap(const vuMap& map);
    
 protected:
    /** allocate memory for the LUT
	made virtual to allow extensions in the sub classes
	called during construction and in resize()
    */
    virtual bool init(dword ncomp, dword range);
    /** Free memory from the LUT.
	Made virtual to allow extensions in the sub classes.
	Called during construction and in resize().
    */
    virtual void cleanup();
    
    dword m_NComp;	//!< number of components for colour model (RGBA -> 4)
    dword m_Range;	//!< range for which the LUT should be defined (8-bit -> 256)
    dword m_TableLength;	//!< results from m_NComp * m_Range

    //!The function table for the transfer function.
    float *m_Table;
    /** The colour (or spectrum) of the illuminating light. 
     For spectral to RGBa converted transfer functions the light is already contained in the 
     colour of the LUT. */
    float *m_Light;
};

#endif
