#ifndef _VUTFDESIGNSPEC_H_
#define _VUTFDESIGNSPEC_H_

#include "vuTFDesign.h"
#include "vuSpectral/SPalette.h"

//! A transfer function based on vuTFDesign with specialization on spectral colour models.

/*! The getRGBa() function of vuTFIntensity base class is overridden
    to provide a proper transformation of the spectral colours to
    RGBa.  */

class vuTFDesignSpec : public vuTFDesign
{
 public:
    //-------------------------------------------------------------------------
    //!The structure for representing light nodes.
    struct LightNode 
    {
	LightNode(float _weight = 0, dword ncomp = 32, const float val = 0) :
	    weight(_weight), wopade(-1)
	    {
		dword c;
		for(c=0;c<ncomp;c++) {
		    col[c] = val;
		}
		for(;c<32;c++) col[c] = 0;
	    };

	LightNode(float _weight, dword ncomp, const vuColourN& _col) :
	    weight(_weight), wopade(-1)
	    {
		if((dword)_col.nComponents() == ncomp) {
		    dword c;
		    for(c=0;c<ncomp;c++) {
			col[c] = _col[c];
		    }
		    for(;c<32;c++) col[c] = 0;
		}
	    };
      
	LightNode(float _weight, dword ncomp, const float* _col) :
	    weight(_weight), wopade(-1)
	    {
		for(dword c=0;c<ncomp;c++) {
		    col[c] = _col[c];
		}
		
	    };
	
	float weight;
	float col[32];
	int wopade;	// opacity node to be weighted by this light 
    };
    
 public:
    /** This default constructor creates a 32 by 256 look up table for intensities */
    vuTFDesignSpec(dword ncomp = 32, dword range = 256) : vuTFDesign(ncomp,range) {};

    //!Generates the transfer function using the defined control nodes.
    /*!Extends vuTFDesign::generateFunction()
     */
    void generateFunction();

    /** Interpolates the opacity nodes as alpha values into the transfer function. */
    void generateOpacities();

    /* * returns an interpolated colour from the look-up table. */
    //void getRGBa(dword i, vuColourRGBa& rgba);

    //!Adds a light source.
    /*!The spectrum is defined by an array of floating point
      components.  The number of elements in the array has to match
      the number of components for the light model used in the
      transfer function (see constructor vuTFDesign() ) \returns The
      index of the new light source.  */
    dword addLight(const float* _col);
    /** returns a light of given index */
    const LightNode& getLightNode(dword index) const;
    /** Returns the number of lights */
    dword getNumLights() const;
    /** Set the weight for one light.  The weight for each light is
	used as coefficient in the weighted sum to determine the final
	light source.  This function does not call generateLight() */
    void setLightWeight(dword index, float _weight);
    float getLightWeight(dword index ) const
	{ return index < getNumLights() ? m_Lights[index].weight : 0; }

    /** Define the weights for the lights.  The light source used for
	the scene is then a weighted sum of the given lights
	ATTENTION: The length of the weights array has to match the
	number of light sources given by getNumLights().  This
	function also calls generateLight() */
    void weightLights(const float* weights);
    /** Sets a scalar that is used to rescale the intensity of the light */
    void setLightIntensity(float intensity) { m_LightIntensity = intensity; };
    /** Returns the light intensity scaling factor */
    float getLightIntensity() { return m_LightIntensity; };
    /** Set index of the node whose opacity should be controled by the weight of the light
     \param lindex index of the light
     \param intensity intensity of the opacity node */
    void setLightOpacityNode(word lindex, int intensity);
    
    /** Calculate a light source from a weighted sum of the added light sources.
	The resulting light is stored in m_Light and is used to make 'colours' out of the 
	reflectances, which is performed in getRGBa().
    */
    void generateLight();

    /**Adds opacity nodes to span triangles over the materials.
       This is used for the alpha manipulation by light weights in connection
       with multiple spectral light sources. */
    void setupMtlTriAlphaNodes();

    /** Sets the alpha values of some nodes by the weights of the according light source.
	This is just an idea for handling multiple alpha values in a different way. */
    void setAlphaByLight();
    
    /** Remove all colour and opacity nodes and insert default nodes at 0 and 255. */
    void clearAllNodes();

    /** returns a handle to the spectral palette */
    SPalette& getPalette() {return m_Palette;};

    /** sets entries in palette according to lights and materials in transfer func. */
    void updatePalette();

    /** sets materials and lights in transfer function from values in palette */
    void updateFromPalette();
    
 protected:
    
    /** Parses the transfer function description file. */
    void parseTFunc();
    /** Save the specified transfer function to output stream.
        Override to save extended properties of derived transfer function.
	On error throw const char *
    */
    void writeTFunc(ofstream &ofp) const;

    /** Read a line defining a light source.
	If there is a syntactic error a char* exception is thrown.
	On succesful reading the light node is added to the transfer function.
	\return true if an light line was found otherwise false.
    */
    bool readLight();
    
    /** Writes information about all registered lights to the output stream. */
    bool writeLights(ofstream &ofp) const;

//------------------------------------------------------------------------------
// member variables
    
    vuDVector<LightNode> m_Lights;
    float m_LightIntensity;
    SPalette		m_Palette;
};

#endif
 







