#include <stdio.h>
#include <iostream>
#include "vuTFDesign.h"
#include "vuColour31a.h"

//----------------------------------------------------------------------------
//------------------------- The default constructor --------------------------
//----------------------------------------------------------------------------

vuTFDesign::vuTFDesign(dword ncomp, 
		       dword range) : vuTFIntensity(ncomp,range)
{
    //Add the essential nodes.
    m_Opacities.add(OpacityNode(0,0));
    m_Opacities.add(OpacityNode(255,0));
    m_Colours.add(ColourNode(255, m_NComp, (float)0));

    //Set the default smoothing
    m_OpacitySmoothing = 0;
    m_ColourSmoothing = 0;
    generateFunction();
}

//----------------------------------------------------------------------------
//------------------------- public addOpacity() ------------------------------
//----------------------------------------------------------------------------

dword vuTFDesign::addOpacity(byte intensity, float opacity)
{
    //Find where to insert the Opacity node
    dword l = m_Opacities.getLength();
    dword i = 0;
    for (i=0; i < l; i++)
        if (m_Opacities[i].intensity >= intensity) break;

    //Add the new Opacity node.
    if ((i < l) && m_Opacities[i].intensity==intensity)
        m_Opacities[i].opacity = opacity;
    else
        m_Opacities.insert(i,OpacityNode(intensity,opacity));

    return i;
}

//----------------------------------------------------------------------------
//------------------------- public removeOpacity() ---------------------------
//----------------------------------------------------------------------------

void vuTFDesign::removeOpacity(dword index)
{
    //Never delete the two Opacity nodes at the end
    if (index == 0 || index >= m_Opacities.getLength()-1) return;

    //Remove the Opacity node
    m_Opacities.remove(index);
}

//----------------------------------------------------------------------------
//------------------------- public getNumOpacities() -------------------------
//----------------------------------------------------------------------------

dword vuTFDesign::getNumOpacities() const
{
    return m_Opacities.getLength();
}

//----------------------------------------------------------------------------
//------------------------- public getOpacity() ------------------------------
//----------------------------------------------------------------------------

const vuTFDesign::OpacityNode& vuTFDesign::getOpacity(dword index) const
{
    return m_Opacities[index];
}

//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
void vuTFDesign::clearAllNodes()
{
  m_Opacities.removeRange(0,m_Opacities.getLength());
  m_Colours.removeRange(0,m_Colours.getLength());
  
  //Add the essential nodes.
  m_Opacities.add(OpacityNode(0,0));
  m_Opacities.add(OpacityNode(m_Range-1,0));
  m_Colours.add(ColourNode(m_Range-1, m_NComp, (float)0));
  
  //Set the default smoothing
  m_OpacitySmoothing = 0;
  m_ColourSmoothing = 0;
}

//----------------------------------------------------------------------------
//------------------------- public getOpacitySmoothing() ---------------------
//----------------------------------------------------------------------------

float vuTFDesign::getOpacitySmoothing() const
{
    return m_OpacitySmoothing;
}

//----------------------------------------------------------------------------
//------------------------- public setOpacitySmoothing() ---------------------
//----------------------------------------------------------------------------

void vuTFDesign::setOpacitySmoothing(float val)
{
    if (val > 1) val = 1;
    if (val < 0) val = 0;
    m_OpacitySmoothing = val;
}

//----------------------------------------------------------------------------
//------------------------- public addColour() -------------------------------
//----------------------------------------------------------------------------

dword vuTFDesign::addColour(dword intensity, const vuColourN& _col)
{
    //Check if this overwrites any other colours
    dword l = m_Colours.getLength();
    dword i = 0;
    for (i = 0; i < l; i++)
        if (m_Colours[i].intensity >= intensity) break;

    //Create the colour node
    ColourNode cp(intensity,m_NComp,_col);

    //Add the new colour node.
    if ((i < l) && m_Colours[i].intensity==intensity)
        m_Colours[i] = cp;
    else
        m_Colours.insert(i,cp);

    return i;
}

//----------------------------------------------------------------------------
//------------------------- public addColour() -------------------------------
//----------------------------------------------------------------------------

dword vuTFDesign::addColour(dword intensity, const float* _col)
{
    //Check if this overwrites any other colours
    dword l = m_Colours.getLength();
    dword i = 0;
    for (i = 0; i < l; i++)
        if (m_Colours[i].intensity >= intensity) break;

    //Create the colour node
    ColourNode cp(intensity,m_NComp,_col);

    //Add the new colour node.
    if ((i < l) && m_Colours[i].intensity==intensity)
        m_Colours[i] = cp;
    else
        m_Colours.insert(i,cp);

    return i;
}

//----------------------------------------------------------------------------
//------------------------- public removeColour() ----------------------------
//----------------------------------------------------------------------------

void vuTFDesign::removeColour(dword index)
{
    //Never delete the colour node at the end
    if (index >= m_Colours.getLength() - 1) return;

    //Remove the colour node.
    m_Colours.remove(index);
}

//----------------------------------------------------------------------------
//------------------------- public getNumColours -----------------------------
//----------------------------------------------------------------------------

dword vuTFDesign::getNumColours() const
{
    return m_Colours.getLength();
}

dword vuTFDesign::getColourNodeIndex(const ColourNode& cnode) const
{
    dword i=0;		// we have to search for the index :-|
    while(m_Colours[i].intensity<cnode.intensity) i++;
    return i;
}

//----------------------------------------------------------------------------
//------------------------- public getColour() -------------------------------
//----------------------------------------------------------------------------

const vuTFDesign::ColourNode& vuTFDesign::getColour(dword index) const
{
    return m_Colours[index];
}


//----------------------------------------------------------------------------
//------------------------- public getColourSmoothing() ----------------------
//----------------------------------------------------------------------------

float vuTFDesign::getColourSmoothing() const
{
    return m_ColourSmoothing;
}


//----------------------------------------------------------------------------
//------------------------- public setColourSmoothing() ----------------------
//----------------------------------------------------------------------------

void vuTFDesign::setColourSmoothing(float val)
{
    if (val > 1) val = 1;
    if (val < 0) val = 0;
    m_ColourSmoothing = val;
}

//----------------------------------------------------------------------------
//------------------------- public generateFunction() ------------------------
//----------------------------------------------------------------------------

void vuTFDesign::generateFunction()
{
    dword index;
    dword numColours = m_Colours.getLength();
    generateOpacities();
    
    //Set up the array representing the colour functions
    //Note that we always have at least one colour Node
    index = 0;
    for (dword i = 0; i < numColours; i++)
    {
        for (; index <= m_Colours[i].intensity; index ++)
        {
	  dword cind = index*m_NComp;
	  for(dword c=0;c<m_NComp-1;c++) {
            m_Table[cind+c] = m_Colours[i].col[c];
	  }
        }
    }

    //Apply the smoothing functions, if any, to the functions
    smooth(&m_Table[m_NComp-1], m_OpacitySmoothing);
    for(dword c=0;c<m_NComp-1;c++) {
      smooth(&m_Table[c], m_ColourSmoothing);
    }

}

//----------------------------------------------------------------------------
void vuTFDesign::generateOpacities()
{
    dword index;
    dword numOpacities = m_Opacities.getLength();
//    dword numColours = m_Colours.getLength();
    //First set up the arrays representing the intensity function
    //Note that we always have at least two Nodes
    index = 0;
    for (dword i = 0; i < numOpacities-1; i++)
    {
	float slope = float(m_Opacities[i+1].opacity - m_Opacities[i].opacity)/
	    float(m_Opacities[i+1].intensity - m_Opacities[i].intensity);
	
	for (; index <= m_Opacities[i+1].intensity; index ++)
	    m_Table[index*m_NComp+(m_NComp-1)] = m_Opacities[i].opacity +
		slope * (index-m_Opacities[i].intensity);
    }
}


//----------------------------------------------------------------------------
//------------------------- private smooth() ---------------------------------
//----------------------------------------------------------------------------

void vuTFDesign::smooth(float *data,float percent) const
{
    const int num = (int)m_Range;

    float *in = new float[num];
    
    //Smooth the function using a low pass filter, achieved by averaging.

    //Copy the data.
    for (int i = 0, index = 0; i < num; i++, index+=m_NComp)
        in[i] = data[index];

    //Average it out
    if (percent < 0) percent = 0;
    if (percent > 1) percent = 1;
    int offset = (int)(num * percent / 2);
    for (int i = 0, index = 0; i < num; i++, index+=m_NComp)
    {
        data[index] = 0;
        for (int j = i-offset; j <= i + offset; j++)
        {
            if (j >= 0)
	    {
		if(j < num)
		    data[index] += in[j];
		else data[index] += in[num-1];
	    } else data[index] += in[0];
        }
        data[index] /= (offset*2 + 1);
    }
    delete [] in;
}

//----------------------------------------------------------------------------
bool vuTFDesign::loadTF(const char* fname)
{
  try {
    Parse(fname);
  } catch(const char *msg) {
    cout << msg << endl;
    return false;
  }
  return true;
}

//----------------------------------------------------------------------------
bool vuTFDesign::saveTF(const char* fname) const
{
  ofstream ofp;
  ofp.open(fname, ios::out);

  writeTFunc(ofp);
  
  return true;
}

void vuTFDesign::writeTFunc(ofstream &ofp) const
{
    writeNComp(ofp);
    writeOpacities(ofp);
    writeColours(ofp);
}

bool vuTFDesign::writeNComp(ofstream &ofp) const
{
    ofp << "components("<< m_NComp-1 << ")" << endl;
    return true;
}

bool vuTFDesign::readNComp()
{
    if(readString("components")) {
	float ncomp;
	if(!readNumber(" ( ", ncomp)) throw "Error reading number of components.";
	else readString(" ) ");
	if(m_NComp != ncomp+1) {
	    m_NComp = int(ncomp)+1;		// add alpha channel
	    cout<<"Changing transfer function format to "<<m_NComp-1<<"+1 components."<<endl;
	    init(m_NComp,m_Range);
	}
	return true;
    } else return false;
}

bool vuTFDesign::writeOpacities(ofstream &ofp) const
{
    //opacities
    dword nopa = getNumOpacities();
    for(dword o=0;o<nopa;o++) {
	ofp<<"opacity("<<getOpacity(o).intensity<<", "<<getOpacity(o).opacity<<")"<<endl;
    }
    ofp<<endl;
    return true;
}

bool vuTFDesign::writeColours(ofstream &ofp) const
{
  //colours
  dword ncol = getNumColours();
  for(dword o=0;o<ncol;o++) {
    ofp<<"colour("<<getColour(o).intensity;
    for(dword c=0;c<m_NComp;c++)
      ofp<<", "<<getColour(o).col[c];
    ofp<<")"<<endl<<endl;
  }
  ofp<<endl;
  return true;
}

//----------------------------------------------------------------------------
//------------------------- ColourNode constructor ---------------------------
//----------------------------------------------------------------------------

vuTFDesign::ColourNode::ColourNode(byte inten, dword ncomp, const vuColourN& _col)
{
    intensity = inten;
    if((dword)_col.nComponents() == ncomp) {
      for(dword c=0;c<ncomp;c++) {
	col[c] = _col[c];
      }
    }
}

//----------------------------------------------------------------------------
//------------------------- ColourNode constructor ---------------------------
//----------------------------------------------------------------------------

vuTFDesign::ColourNode::ColourNode(byte inten, dword ncomp, const float* _col)
{
    intensity = inten;
    for(dword c=0;c<ncomp;c++) {
      col[c] = _col[c];
    }
}

//----------------------------------------------------------------------------
//------------------------- ColourNode constructor ---------------------------
//----------------------------------------------------------------------------

vuTFDesign::ColourNode::ColourNode(byte inten, dword ncomp, float val)
{
    intensity = inten;

    for(dword c=0;c<ncomp;c++)
      col[c] = val;
}

//----------------------------------------------------------------------------
// functions for the parser
//----------------------------------------------------------------------------
void vuTFDesign::Parse(const char* filename) throw (const char *)
{
  readBuffer(filename);
  FixBuffer();
  parseTFunc();
}

void vuTFDesign::parseTFunc()
{
  clearAllNodes();

  bool read = true;
  while (read) {
      read = false;
      if ( readOpacity() ) { read = true; }
      else if ( readColour() ) { read = true; }
      else if ( readNComp() ) { read = true; };
  }
}

bool vuTFDesign::readColour()
{
  if( readString("colour") ) {
    float intensity;
    float col[32];
    if(!readNumber(" ( ",intensity)) throw "Error reading colour.";
    bool read = true;
    dword c = 0;
    while (read && c<32) {
      read = false;
      if(readNumber(" , ",col[c])) {
	read = true;
	c++;
      } else if(readString(" ) ")) read = false;
    }
    if(c==32) readString(" ) ");
    else if(c==0) throw "Error reading colour components";
    while(c<32) col[c++] = 0.0f;
    addColour((dword)intensity,col);
    return true;
  } else return false;
}

bool vuTFDesign::readOpacity()
{
  if( readString("opacity") ) {
    float intensity;
    float opa;
    if(readNumber(" ( ",intensity) &&
       readNumber(" , ",opa) && 
       readString(" ) ")) {
      addOpacity((dword)intensity,opa);
      return true;
    } else throw "Error reading opacity.";
  } else return false;
}

